/* Socket server implementation.  Fairly generic.
 * Copyright (c) 1989, 1992  Leonard Dickens
 * Intelligent Data Management, NASA Goddard Space Flight Center
 * 
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
 * to anyone for the consequences of using it or for whether it serves any 
 * particular purpose or works at all.
 * 
 * You may copy, modify and generally use this software freely, as long as
 * this copyright notice is left intact in every copy or derivative work.
 */
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <malloc.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>
#include <stdio.h>
#include "socket_server.h"

#define NULL 0

#ifdef ALLEGRO_BUG_HACK
#ifdef __STDC__
long lisp_call(int index, ...);
#else
long lisp_call();
#endif
#endif

/* The number of file descriptors available on this system: */
static int n_fds;

/* The set of fds whose input we are interested in.
   A strict superset of the set of client fds. */
static fd_set server_input_fds;

/* The function that will be used to read non-socket input to the server.
   Register file descriptors you are interested in via add_input_fd. */
void (*other_input_procedure)();

static struct connection **connect_record = NULL;

static struct connection *active_connection = NULL;


/*
 * Notify the socket server that there is a new fd of interest.
 */
void socket_server_add_input_fd(fd)
     int fd;
{
  FD_SET(fd, &server_input_fds);
  }


/*
 * This function should be called by any user program to close any fds
 * that are being used here.
 */
void socket_server_end_of_file(fd)
     int fd;
{
  struct connection *c = connect_record[fd];

  if (c != NULL) {
    if (fd == c->connect_fd)
      c->connect_fd = -1;
    else {
#ifdef ALLEGRO_BUG_HACK
      lisp_call((int)c->end_of_file_procedure, fd);
#else
      if (c->end_of_file_procedure != NULL)
	(*(c->end_of_file_procedure))(fd);
#endif
      if ((c->n_clients-- == c->max_clients) && (c->connect_fd != -1))
	FD_SET(c->connect_fd, &server_input_fds);
      }
    if ((c->n_clients == 0) && (c->connect_fd == -1))
      free(c);
    close(fd);
    connect_record[fd] = NULL;
    }
  FD_CLR(fd, &server_input_fds);
  }


/*
 * Returns record describing the currently active connection.
 * This can be useful for changing things in that structure.
 */
struct connection *socket_server_get_connection()
{
  return active_connection;
  }


/*
 * Initialize local data structures.
 */
void socket_server_initialize(other_input_proc, other_input_fds)
     void (*other_input_proc)();
     fd_set *other_input_fds;
{
  int fd;

  if (connect_record != NULL)
    return;

  n_fds = 64;   /* ulimit(4,0); */
  connect_record = (struct connection **) 
    malloc(n_fds * sizeof(struct connection *));
  for (fd = 0; fd < n_fds; fd++)
    connect_record[fd] = NULL;
  other_input_procedure = other_input_proc;
  if (other_input_fds != NULL)
    for (fd = 0; fd < n_fds; fd++)
      if (FD_ISSET(fd, other_input_fds))
	FD_SET(fd, &server_input_fds);
  }


/*
 * Shut down the server's client connections and connect socket.
 */
int socket_server_shutdown()
{
  struct connection *c;
  int fd;

  /* Take our number out of the directory, so people won't try to call. */

  /* Close all of our client connections first: */
  for (fd = 0; fd < n_fds; fd++) {
    if (NULL == (c = connect_record[fd]))
      continue;
#ifdef DEBUG
    fprintf(stderr, "closing client %d. \n",fd);
#endif
    if (fd != c->connect_fd) {
      socket_server_end_of_file(fd);
      }
    }
  /* Close all of our accept sockets: */
  for (fd = 0; fd < n_fds; fd++) {
    if (NULL == (c = connect_record[fd]))
      continue;
#ifdef DEBUG
    fprintf(stderr, "closing accept socket %d.\n", fd);
#endif
    socket_server_end_of_file(fd);
    }
  free(connect_record);
  connect_record = NULL;
  return SS_NORMAL_END;
} 


/*
 * Startup a new acceptance port/client server.
 */
int socket_server_startup(socket_port, input_proc, max_clients_in,
			  new_connection_proc, socket_shutdown_proc) 
     char  *socket_port;
     void (*input_proc)();
     int    max_clients_in;
     void (*new_connection_proc)();
     void (*socket_shutdown_proc)();
{
  char *strdup();
  struct sockaddr_in addr_in;
  struct hostent *hp;
  char *addr;
  int len, t, x;
  struct connection *c;

  c = (struct connection *) 
    malloc(sizeof(struct connection));

  /* Initialize connection record to sensible values: */
  c->connect_fd = -1;
  c->max_clients = ((0 < max_clients_in) && (max_clients_in <= n_fds)) 
    ? max_clients_in : n_fds;
  c->n_clients = 0;
  c->port = strdup(socket_port);
  c->end_of_file_procedure    = socket_shutdown_proc;
  c->input_procedure          = input_proc;
  c->new_connection_procedure = new_connection_proc;

  if (-1 == (c->connect_fd = socket(AF_INET, SOCK_STREAM, 0))) {
    free(c);
    return SS_FAILED_SOCKET;
    }
  /* Prepare INET binding: refer Sun IPC Primer p. 20 */
  addr = (char *) &addr_in;
  len = sizeof(struct sockaddr_in);
  bzero(addr, len);
  addr_in.sin_family = AF_INET;
  addr_in.sin_addr.s_addr = INADDR_ANY;
  addr_in.sin_port = atoi(socket_port);

  /* Bind the socket to the port: */
  if (x = bind(c->connect_fd, addr, len)) {
    write(2, "$ Socket server: bind returned -1: port already in use.\n", 56);
    free(c);
    return SS_FAILED_BIND;
    }
  /* Start accepting connections to the socket: */
  if (x = listen(c->connect_fd, 5)) {
    free(c);
    return SS_FAILED_LISTEN;
    }
  /* Success!  Twiddle local data structures. */
  connect_record[c->connect_fd] = c;
  FD_SET(c->connect_fd, &server_input_fds);
  return SS_NORMAL_END;
  }



/*
 * Poll to see if there is any input, and call handlers as 
 * appropriate.  See the "select(3)" man page for further explanation of 
 * how timeout works.
 */
int socket_server_poll(timeout)
     struct timeval *timeout;
{
  struct connection *c;
  struct linger linger;
  fd_set active_fds;
  int t, x, fd;

  while (connect_record != NULL) {
#ifdef DEBUG
#if 1
    /* Very useful for debugging programs, to see if they are returning
       to this loop as they should be. */
    socket_server_printf();
#endif
#endif
    bcopy(&server_input_fds, &active_fds, sizeof(fd_set));
    if (0 > (x = select(n_fds, &active_fds, NULL, NULL, timeout))) {
      if (errno == EINTR)
	continue;
      else
	return SS_FAILED_SELECT;
      }
    /* Check for timeout on the select call: */
    if (x == 0) {
      return SS_NORMAL_END;
      }
    
    /* Look at all fds to see which one(s) have input: */
    for (fd = 0; fd < n_fds; fd++) {
      if (FD_ISSET(fd, &active_fds)) {
	c = connect_record[fd];
	if (c == NULL) {
	  /* It is not any of our sockets, connect or client.
	     I.e., it must be an other input socket. */
	  (*other_input_procedure)(fd);
	  }
	else if (fd != c->connect_fd) {
	  /* Fd must be a client connection with input to server; handle it. */
#ifndef ALLEGRO_BUG_HACK
	  if (c->input_procedure)
#endif
	    {
	    active_connection = c;
#ifdef ALLEGRO_BUG_HACK
	    lisp_call((int)c->input_procedure, fd);
#else
	    (*c->input_procedure)(fd);
#endif
	    active_connection = NULL;
	    }
	  }
	else {  /* Accept a new connection. */
	  if (-1 == (t = accept(fd, 0, 0)))
	    return SS_FAILED_ACCEPT;
	  
	  /* Add t to the set of connections that we find interesting. */
	  FD_SET(t, &server_input_fds);
	  c->n_clients++;
	  if (c->n_clients == c->max_clients)
	    FD_CLR(fd, &server_input_fds);
	  connect_record[t] = c;
	  linger.l_onoff = 1;
	  linger.l_linger = 0;
	  setsockopt(t, SOL_SOCKET, SO_LINGER, (char*)&linger);
#ifndef ALLEGRO_BUG_HACK
	  if (c->new_connection_procedure != NULL) 
#endif
	    {
	    active_connection = c;
#ifdef DEBUG
	    printf("C: about to call [%ld](%d)\n", 
		   (long) c->new_connection_procedure, t);
#endif
#ifdef ALLEGRO_BUG_HACK
	    lisp_call((int)c->new_connection_procedure, t);
#else
	    (*c->new_connection_procedure)(t);
#endif
#ifdef DEBUG
	    printf("C: just returned from to call [%ld](%d)\n", 
		   (long) c->new_connection_procedure, t);
#endif
	    active_connection = NULL;
	    }
	  }
	}
    }
  }
  return SS_NORMAL_END;
  }



int socket_server_simple_startup(socket_port, input_proc)
     char *socket_port;
     void (*input_proc)();
{
  socket_server_initialize(NULL, NULL);
  return socket_server_startup(socket_port, input_proc, 256,
			       NULL, NULL);
  }


int socket_server_simple_loop(socket_port, input_proc)
     char *socket_port;
     void (*input_proc)();
{
  socket_server_simple_startup(socket_port, input_proc);
  return socket_server_poll(NULL);
  }


#ifdef DEBUG
#if 1
/*#include <stdio.h>*/

int socket_server_printf()
{
  struct connection *c;
  int x, fd;

  /* For debugging it can be useful to see which fds are set where. */
  fprintf(stderr, "All:0x");
  fflush(stderr);
  for (x=howmany(64,NFDBITS)-1; x>=0; x--) {
    fprintf(stderr, "%04X", server_input_fds.fds_bits[x]);
    fflush(stderr);
  }
  for (fd = 0; fd < n_fds; fd++) {
    c = connect_record[fd];
    if (c == NULL)
      continue;
    if (c->connect_fd == fd) {
      fprintf(stderr, "   %s [%d]", c->port, c->connect_fd);
      fflush(stderr);
      }
    }
  fprintf(stderr, "\n");
  fflush(stderr);
  }
#endif
#endif
