" (c) 1990, 1991 Copyright (c) University of Washington
  Written by Stephen Soderland, Tony Barrett and Daniel Weld.

  All rights reserved. Use of this software is permitted for non-commercial
  research purposes, and it may be copied only for that use.  All copies must
  include this copyright message.  This software is made available AS IS, and
  neither the authors nor the University of Washington make any warranty about
  the software or its performance.

  When you first acquire this software please send mail to
  bug-snlp@cs.washington.edu; the same address should be used for problems."

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  This file contains the code used to run the statistical tests
;;;  mentioned in the paper: "Partial Order Planning: Evaluating Possible
;;;  Efficiency Gains" by A. Barrett and D. Weld (University of Washington
;;;  CSE TR 92-05-01).
;;;
;;;  A test is performed by calling a function starting with "RAND-"
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(use-package 'plan-utils)

(defvar *stats* nil)
(defvar *old-stats* nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Tyre world tests

(defun tyre-tests (cycles)
  (format t "~%:::::::::::::::::::::::::::both serializable:::::::::::")
  (tyre2-test cycles)
  (format t "~%:::::::::::::::::::::::::::POCL serializable:::::::::::")
  (tyre3-test cycles)
  (format t "~%:::::::::::::::::::::::::::nonserializable:::::::::::")
  (tyre1-test cycles))
  
;;;;;;;;;;;;;;;;;;;;;;
;;; not serializations

(defun tyre1-test (cycles)
  (tyre-test cycles #'tyre1-test*))

(defun tyre1-test* (goals)
  (not (pocl-seri goals)))

;;;;;;;;;;;;;;;;;;;;;;
;;; serializations for TOCL (and POCL)

(defun tyre2-test (cycles)
  (tyre-test cycles #'tocl-seri))

;;;;;;;;;;;;;;;;;;;;;;
;;; serializations for POCL (and not TOCL)

(defun tyre3-test (cycles)
  (tyre-test cycles #'tyre3-test*))

(defun tyre3-test* (goals)
  (and (pocl-seri goals) (not (tocl-seri goals))))

;;;;;;;;;;;;;;;;;;;;;;
;;;  General test routine

(defun tyre-test (cycles test)
  (init-flat-tire)
  (setf plan-utils::*verbose* t
	plan-utils::*search-limit* 1000000)
  (dotimes (c cycles)
    (let* ((is (permute '((closed boot) (unlocked boot) (in jack boot) 
			  (in pump boot) (in wheel2 boot) (intact wheel2)
			  (in wrench boot) (flat wheel2) (flat wheel1)
			  (on wheel1 hub) (on-ground wheel1) (on-ground hub)
			  (tight nuts wheel1))))
	   (gs (permute '((on wheel2 hub)(inflated wheel2)(tight nuts wheel2)
			  (in jack boot) (in pump boot) (in wheel1 boot)
			  (in wrench boot)(closed boot)))))
      (do ((g gs (permute g)))
	  ((funcall test g) 
	   (setf gs g)))
      (system::gc t)
      (format t "~%~%;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;")
      (format t "~%Goal     : ~a ~%" gs)
      (format t "~%SNLP ():~%")
      (system::gc t)
      (multiple-value-bind (plan-t mct-stat)
	  (snlp:plan is gs
		     :search-fun #'plan-utils::id-bf-search
		     :rank-fun #'snlp::df-tire-rank)
	(snlp::print-plan plan-t)
	(print-stat mct-stat))
      (format t "~%TOCL ():~%")
      (system::gc t)
      (multiple-value-bind (plan-l slp-stat)
	  (tocl:plan  is gs 
		      :search-fun #'plan-utils::id-bf-search
		      :rank-fun #'tocl::df-tire-rank)
	(tocl::print-plan plan-l)
	(print-stat slp-stat)))))

;;;;;;;;;;;;;;;;;;;;;;
;;; Testing for serialization orders

(defun POCL-SERI (goals)
  (and (< (min (position '(ON WHEEL2 HUB) goals :test #'equal)
	       (position '(IN WHEEL1 BOOT) goals :test #'equal)
	       (position '(TIGHT NUTS WHEEL2) goals :test #'equal))
	  (min (position '(IN WRENCH BOOT) goals :test #'equal)
	       (position '(IN JACK BOOT)  goals :test #'equal)))
       (< (min (position '(ON WHEEL2 HUB) goals :test #'equal)
	       (position '(TIGHT NUTS WHEEL2) goals :test #'equal))
	  (position '(IN JACK BOOT)  goals :test #'equal))
       (< (position '(INFLATED WHEEL2) goals :test #'equal)
	  (position '(IN PUMP BOOT) goals :test #'equal))
       (not (equal (car goals) '(closed boot)))))

(defun TOCL-SERI (goals)
  (and (pocl-seri goals)
       (< (position '(TIGHT NUTS WHEEL2) goals :test #'equal)
	  (position '(IN WRENCH BOOT) goals :test #'equal))
       (< (position '(IN WRENCH BOOT) goals :test #'equal)
	  (position '(CLOSED BOOT) goals :test #'equal))
       (< (position '(IN PUMP BOOT) goals :test #'equal)
	  (position '(CLOSED BOOT) goals :test #'equal))))

(defun count-seri (test &aux (ret 0))
  (labels ((count* (part gs)
	     (if (null gs) 
		 (when (funcall test part) (incf ret))
	       (dolist (g gs)
		 (count* (cons g part) (remove g gs :test #'eq))))))
    (count* nil '((on wheel2 hub)(inflated wheel2)(tight nuts wheel2)
		  (in jack boot) (in pump boot) (in wheel1 boot)
		  (in wrench boot)(closed boot))))
  ret)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Artificial world where any order is ok
(defun INIT-D0S1 ()
  (reset-domain)
  (dolist (prop '(a7 a6 a5 a4 a3 a2 a1 a b c d e f g h i j))
     (defstep :action `(,prop ?x)
       :precond  `((,prop i ?x))
       :add `((,prop g ?x)))))

(defun RAND-D0S1 (cycles)
  (init-d0s1)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (format t "~%~%~%Randomized Artificial World Test1 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 15)))

(defun CONVERT-domain (&optional (templates *templates*))
  (setf *templates* nil)
  (dolist (op templates)
    (push (list (make-plan-step 
		 :action (cons 'y (plan-step-action (car op)))
		 :precond (cons '(p y) (plan-step-precond (car op)))
		 :add (plan-step-add (car op))
		 :dele (plan-step-dele (car op)))
		nil)
	  *templates*)
    (push (list (make-plan-step 
		 :action (cons 'x (plan-step-action (car op)))
		 :precond (cons '(p x) (plan-step-precond (car op)))
		 :add (plan-step-add (car op))
		 :dele (plan-step-dele (car op)))
		nil)
	  *templates*))
  (push (list (make-plan-step 
	       :action '(ax)
	       :add '((ga ?x))
	       :dele (cons '(P y)
			   (mapcar #'(lambda (op)
				       (car (plan-step-add (car op))))
				   templates)))
	      nil)
	*templates*)
  (setf *templates* (permute *templates*)))

(defun RAND-D0S1a (cycles)
  (init-d0s1)
  (CONVERT-domain)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (format t "~%~%~%Randomized Artificial World Test1 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((p x) (p y)
	      (a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((ga x) (a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 10)))

(defun RAND-DMS1a (cycles)
  (init-dms1)
  (CONVERT-domain)
  (setq topi::*lazy* t)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((p x) (p y)
	      (a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((ga x)(a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 10)))

(defun RAND-D1S1a (cycles)
  (init-d1s1)
  (CONVERT-domain)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (setq *snlp-limit* 1500)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((p x) (p y)
	      (a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((ga x)(a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 8)))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Art world where things clobber each other in a more subtle manner.

;;; world where no preconditions, but plan steps maximally constrained in
;;; order - must do steps in lexicographic order
(defun INIT-DMS1 ()
  (reset-domain)
  (do ((prop '(a7 a6 a5 a4 a3 a2 a1 a b c d e f g h i j) (cdr prop)))
      ((null prop))
    (defstep :action `(,(car prop) ?x)
       :precond  `((,(car prop) i ?x))
       :add `((,(car prop) g ?x))
       :dele (mapcar #'(lambda (x) `(,x i ?x)) (cdr prop)))))

(defun RAND-DMS1 (cycles)
  (init-dms1)
  (setq topi::*lazy* t)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 15)))

(defun RAND-TEST (is gs cycles n)
  (dotimes (c cycles)
    (setq is (permute is))
    (dotimes (i n)			; was (length gs)
      (setf *templates* (permute *templates*))
      (let ((goals (permute (subseq gs 0 (1+ i))))
	    (test (+ 1 i (* c (length gs)))))
	(format t "~%~%;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;")
	(format t "~%Goal     : ~a ~%" goals)
	(format t "~%SNLP (~a):~%" test)
	(system::gc t)
	(multiple-value-bind (plan-t mct-stat)
	    (snlp:plan is goals :rank-fun #'snlp::df-rank)
	  (snlp::print-plan plan-t)
	  (print-stat mct-stat)
	  (format t "~%SLP (~a):~%" test)
	  (system::gc t)
	  (multiple-value-bind (plan-l slp-stat)
	      (tocl:plan is goals :rank-fun #'tocl::df-rank)
	    (tocl::print-plan plan-l)
	    (print-stat slp-stat)
	    (when (< i 6)
	      (format t "~%McSTRIPS (~a):~%" test)
	      (system::gc t)
	      (multiple-value-bind (plan-s mcs-stat)
		  (topi:plan is goals :rank-fun #'topi::McS-DF-RANK)
		(topi::print-plan plan-s)
		(print-stat mcs-stat)
		(push (list test (list is goals) mct-stat mcs-stat slp-stat) 
		      *stats*)))))))))

(defun INIT-DMS2 ()
  (reset-domain)
  (let ((props '(a7 a6 a5 a4 a3 a2 a1 a b c d e f g h i j)))
    (do ((prop props (cdr prop)))
	((null prop) nil)
	(defstep :action `(,(car prop) 2 ?x)
	  :precond  `((,(car prop) - ?x))
	  :add `((,(car prop) g ?x))
	  :dele (append (mapcar #'(lambda (x) `(,x - ?x)) (cdr prop))
			(mapcar #'(lambda (x) `(,x i ?x)) props)))
	(defstep :action `(,(car prop) 1 ?x)
	  :precond  `((,(car prop) i ?x))
	  :add `((,(car prop) - ?x))
	  :dele (mapcar #'(lambda (x) `(,x i ?x)) (cdr prop))))))

(defun RAND-DMS2 (cycles)
  (init-dms2)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (rand-test is gs cycles 7)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Art world where things clobber each other in a more subtle manner.

;;; world where no preconditions, but plan steps maximally constrained in
;;; order - must do steps in lexicographic order
(defun INIT-D1S1 ()
  (reset-domain)
  (do ((prop '(a7 a6 a5 a4 a3 a2 a1 a b c d e f g h i j) (cdr prop)))
      ((null prop))
    (defstep :action `(,(car prop) ?x)
       :precond  `((,(car prop) i ?x))
       :add `((,(car prop) g ?x))
       :dele (when (cdr prop) `((,(cadr prop) i ?x))))))

(defun RAND-D1S1 (cycles)
  (init-d1s1)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (setq *snlp-limit* 1500)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
              ))
        (gs '((a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x) (j g x)
              (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x))))
    (dotimes (c cycles)
      (rand-test is gs cycles 13))))

(defun INIT-D1S2 ()
  (reset-domain)
  (let ((props '(a7 a6 a5 a4 a3 a2 a1 a b c d e f g h i j)))
    (do ((prop props (cdr prop)))
	((null prop) nil)
	(defstep :action `(,(car prop) 2 ?x)
	  :precond  `((,(car prop) - ?x))
	  :add `((,(car prop) g ?x))
	  :dele (append (when (cdr prop) `((,(cadr prop) - ?x)))
			(mapcar #'(lambda (x) `(,x i ?x)) props)))
	(defstep :action `(,(car prop) 1 ?x)
	  :precond  `((,(car prop) i ?x))
	  :add `((,(car prop) - ?x))
	  :dele (when (cdr prop) `((,(cadr prop) i ?x)))))))

;;; To test whether nonserializability is the problem or whether 
;;; it is simply step ordering that is the problem. 

;;; The idea is that the initial state might be something like:
;;; ((a i x) (b i x) (c i x))
;;; And the goals might be: ((a g x) (b g x) (c g x))
;;; And to achieve (b g x) one needs to do TWO actions: (b x) and (bb x) 
;;; (rather than just (b x) as in D1S1)
;;; BUT one can't just do (b x) and (bb x) right after each other because
;;; That will clobber things.
;;; In art3 one would have had to do the sequence (c x) (b x) (a x)
;;; here one needs to do: (c x) (b x) (a x) (cc x) (bb x) (aa x) 
;;; Thus the subgoals are nonserializable

(defun RAND-D1S2 (cycles)
  (init-d1s2)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (setq *snlp-limit* 1500)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) (a6 i x) (a7 i x) 
              ))
        (gs '((j g x) (i g x) (h g x) (g g x) (f g x) (e g x) (d g x)
	      (c g x) (b g x)
	      (a g x) (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x)
	      (a6 g x) (a7 g x)
	      )))
    (rand-test is gs cycles 13)))

(defun INIT-D1S* ()
  (reset-domain)
  (do ((prop '(j i h g f e d c b a a1 a2 a3 a4 a5 a6 a7) (cdr prop)))
      ((null prop))
    (defstep :action `(,(car prop) ?x)
       :precond  `((,(car prop) i ?x))
       :add `((,(car prop) g ?x))
       :dele (cons '(a* i ?x) (when (cdr prop) `((,(cadr prop) i ?x))))))
  (defstep :action '(a* 1 ?x)
    :precond '((a* i ?x))
    :add '((a* - ?x))
    :dele nil)
  (defstep :action '(a* 2 ?x)
    :precond '((a* - ?x))
    :add '((a* g ?x))
    :dele '((j i ?x))))

(defun RAND-D1S* (cycles)
  (init-d1s*)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (setq *snlp-limit* 1500)
  (format t "~%~%~%Randomized Artificial World Test2 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((a* i x) (a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) (a6 i x) (a7 i x) 
              ))
        (gs '((a* g x) (j g x) (i g x) (h g x) (g g x) (f g x) (e g x)
	      (d g x) (c g x) (b g x)
	      (a g x) (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x)
	      (a6 g x) (a7 g x)
	      )))
    (rand-test is gs cycles 13)))

(defvar *old-stat-array* nil)
(defvar *stat-array* nil)

(defun INIT-DMS2* (n)
  (reset-domain)
  (dotimes (i n)
    (defstep :action `(s1 ,i ?x)
      :precond `((I ,i ?x))
      :add `((P ,i ?x))
      :dele (let ((j nil))
	      (dotimes (k i j)
		(push `(P ,k ?x) j))))
    (defstep :action `(s2 ,i ?x)
      :precond `((P ,i ?x))
      :add `((G ,i ?x))
      :dele (let ((j nil))
	      (dotimes (k i j)
		(push `(P ,k ?x) j)))))
  (defstep :action `(middle ?x)
    :precond '((middle I ?x))
    :add '((middle G ?x))
    :dele (let ((j nil))
	    (dotimes (i n j)
	      (push `(I ,i ?x) j)
	      (push `(G ,i ?x) j)))))

(defun RAND-DMS2* (cycles &optional (n 10))
  (init-dms2* n)
  (let ((is (let ((j '((middle I x))))
	      (dotimes (k n j)
		(push `(I ,k x) j))))
	(gs (let ((j '((middle G x))))
	      (dotimes (k n (reverse j))
		(push `(G ,k x) j)))))
    (rand-test is gs cycles n)))

(defun RAND-TIRE (cycles &optional (n 10))
  (init-flat-tire)
  (let ((is '((closed boot) (unlocked boot) (in jack boot) (in pump boot)
	      (in wheel2 boot) (intact wheel2) (in wrench boot) (flat wheel2)
	      (flat wheel1) (on wheel1 hub) (on-ground wheel1)
	      (on-ground hub) (tight nuts wheel1)))
	(gs '((in wheel1 boot) 
	      (inflated wheel2)
	      (on wheel2 hub)
	      (tight nuts wheel2)
	      (in jack boot) (in pump boot) 
	      (in wrench boot) 
	      (closed boot)
	      )))
    (rand-test is gs cycles (length gs))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Artificial world where any order is ok
(defun INIT-ern1 ()
  (reset-domain)
  (dolist (prop '(a b c d e f g h i))
     (defstep :action `(,prop A ?x)    ; This is A_i
       :precond  `((j ?x))
       :add `((,prop p ?x))
       :dele `((j ?x)))
     (defstep :action `(,prop B ?x)    ; This is B_i
       :precond  `((,prop i ?x)(,prop p ?x))
       :add `((,prop g ?x)(j ?x)))))

(defun RAND-ern1 (cycles)
  (init-ern1)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (format t "~%~%~%Randomized Artificial World Test1 on ~a~%" (today))
  (forma
   t t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((j x)(a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) 
              (h i x) (i i x)
              ))
        (gs '((a g x) (b g x) (c g x) (d g x) (e g x) (f g x) (g g x) 
              (h g x) (i g x))))
    (rand-test is gs cycles 8)))
          
(defun INIT-ern2 ()
  (reset-domain)
  (defstep :action '(a) :precond '((r)) :add '((p)))
  (defstep :action '(b) :precond '((q)) :add '((g 1)) :dele '((r)))
  (defstep :action '(c) :precond '((p)) :add '((g 2)) :dele '((q)))

  (dolist (prop '(3 4 5 6 7 8 9))
     (defstep :action `(D ,prop)    ; This is A_i
       :add `((G ,prop)))))

(defun RAND-ern2 (cycles)
  (init-ern2)
  (setq *old-stats* *stats*)
  (setq *stats* nil)
  (setq topi::*lazy* t)
  (format t "~%~%~%Randomized Artificial World Test1 on ~a~%" (today))
  (format t "~&McSTRIPS has lazy set to ~a~%" topi::*lazy*)
  (let ((is '((r) (q)))
        (gs '((g 1) (g 2) (g 3) (g 4) (g 5) (g 6) (g 7) (g 8) (g 9))))
    (rand-test is gs cycles 9)))

(defun PERMUTE (list)
  (let ((l (copy-list list))
        (ret nil))
    (do ()
        ((null l) ret)
      (let ((i (random (length l))))
        (push (nth i l) ret)
        (setf l (delete (nth i l) l))))))
    
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun INIT-D0S1-DMS2* ()
  (reset-domain)
  (do ((i '(a1 a2 a3 a4 a5 a6) (cdr i)))
      ((null i))
    (defstep :action `(,(car i) 1 ?x)
      :precond `((,(car i) i ?x))
      :add `((,(car i) p ?x))
      :dele (mapcar #'(lambda (x) `(,x p ?x)) (cdr i)))
    (defstep :action `(,(car i) 2 ?x)
      :precond `((,(car i) p ?x))
      :add `((,(car i) g ?x))
      :dele (mapcar #'(lambda (x) `(,x p ?x)) (cdr i)))
  (defstep :action `(a ?x)
    :precond '((a I ?x))
    :add '((a G ?x))
    :dele (let ((j nil))
	    (dolist (i '(a1 a2 a3 a4 a5 a6) j)
	      (push `(,i i ?x) j)
	      (push `(,i g ?x) j)))))
  (dolist (i '(b c d e f g h i j))
    (defstep :action `(,i ?x)
      :precond `((,i i ?x))
      :add `((,i G ?x))
      :dele `())))
    

(defun RAND-D0S1-DMS2* (cycles &key (ordered 6) (unordered 9))
  (init-d0s1-dms2*)
  (setq *mcs-limit* 400)
  (setq topi::*lazy* t)
  (setf *old-stat-array* *stat-array*)
  (setf *slp-stat-array*
	(make-array (list ordered unordered) :initial-element nil))
  (setf *snlp-stat-array*
	(make-array (list ordered unordered) :initial-element nil))
  (setf *mcs-stat-array*
	(make-array (list ordered unordered) :initial-element nil))
  (let ((is '((a i x) (b i x) (c i x) (d i x) (e i x) (f i x) (g i x) (h i x)
	      (i i x) (j i x)
              (a1 i x) (a2 i x) (a3 i x) (a4 i x) (a5 i x) 
	      (a6 i x)
              ))
	(ngs '((b g x) (c g x) (d g x) (e g x) (f g x) (g g x) (h g x) (i g x)
	       (j g x)))
	(ogs '((a g x) (a1 g x) (a2 g x) (a3 g x) (a4 g x) (a5 g x) (a6 g x)))
	(test 0))
    (dotimes (c cycles)
      (dotimes (i ordered)
	(dotimes (j unordered)
	  (let ((goals (permute (append (subseq  ogs 0 i) (subseq  ngs 0 j)))))
	    (format t "~%~%;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;")
	    (format t "~%SLP (~a):~%" (incf test))
	    (system::gc t)
	    (multiple-value-bind (plan-l slp-stat)
		(tocl:plan is goals :rank-fun #'tocl::df-rank)
	      (tocl::print-plan plan-l)
   	      (print-stat slp-stat)
	      (push slp-stat (aref *slp-stat-array* i j)))
	    (format t "~%SNLP (~a):~%" (incf test))
	    (system::gc t)
	    (multiple-value-bind (plan-t snlp-stat)
		(snlp:plan is goals :rank-fun #'snlp::df-rank)
	      (snlp::print-plan plan-t)
   	      (print-stat snlp-stat)
	      (push snlp-stat (aref *snlp-stat-array* i j)))
	    (format t "~%McSTRIPS (~a):~%" (incf test))
	    (system::gc t)
	    (multiple-value-bind (plan-s mcs-stat)
		(topi:plan is goals :rank-fun #'topi::McS-DF-RANK)
	      (topi::print-plan plan-s)
   	      (print-stat mcs-stat)
	      (push mcs-stat (aref *mcs-stat-array* i j)))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;

(defun RAND-block (cycles &optional (n 4))
  (blocks-world-domain)
  (let ((is (random-bw-state n))
	(ig (random-bw-state n)))
    (format t "~%is = ~a" is)
    (format t "~%ig = ~a" ig)
    (rand-test is ig cycles n)))

(defun random-bw-state (n &aux (state nil) (ret '((clear table))))
  (dolist (b (permute (subseq '(a b c d e f) 0 n)))
    (let ((r (random (1+ (length state)))))
      (if (zerop r) (push (list b) state)
	(push b (nth (1- r) state)))))
  (dolist (tower state (permute ret))
    (push `(clear ,(car tower)) ret)
    (do ((b tower (cdr b)))
	((null b))
      (push `(on ,(car b) ,(if (cdr b) (cadr b) 'table)) ret))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Print functions

(defun VIEW-STATS (fn &optional (st *stat-array*))
  (let ((ordered (car (array-dimensions st)))
	(unordered (cadr (array-dimensions st))))
    (format t "          <-----unordered----->~%")
    (format t "{")
    (dotimes (i ordered)
      (format t "{")
      (dotimes (j unordered)
	(format t "~F" (averages (aref st i j) fn))
        (if (= (1+ j) unordered) (format t "}") (format t ", ")))
      (if (= (1+ i) ordered) (format t "}~%") (format t ",~%")))))

(defun AVERAGES (list fn)
  (if list
      (do* ((l list (cdr l))
	    (sum (funcall fn (car l)) (+ sum (funcall fn (car l))))
	    (num 1 (1+ num)))
	   ((null (cdr l)) (/ sum num)))
    nil))


;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Summarize Statistics

;;; Compare performance of two algos
(defun COMPARE-STAT (s1 s2)
  (format t "~& ~a/~a on problem ~3a  (Init = ~2a ; Goals = ~2a)"
          (stat-algo s1) (stat-algo s2)
          (stat-prob-num s1) (stat-num-init s1) (stat-num-goal s1))
  (format t "~%   CPU    ~8,5f;    NodesVisit~10,5f;     BindAdd    ~8,5f"
          (float (div* (stat-time s1) (stat-time s2)))
          (float (div* (stat-visited s1) (stat-visited s2)))
          (float (div* (stat-add-bindings s1) (stat-add-bindings s2))))
  (format t "~%   Branch ~8,5f;    Visit-&-Q ~10,5f;     UseUnifies ~8,5f"
          (float (div* (stat-ave-branch s1) (stat-ave-branch s2)))
          (float (div* (+ (stat-visited s1) (stat-q-len s1))
                    (+ (stat-visited s2) (stat-q-len s2))))
          (float (div* (- (stat-unify-count s1) (stat-rank-unifies s1))
                    (- (stat-unify-count s2) (stat-rank-unifies s2)))))
  (format t "~%                       ")
  (format t "SS V/Made ~10,5f;     TotUnifies ~8,5f"
          (float (div* (div* (stat-visited s1)
                       (+ (stat-visited s1) (stat-q-len s1)))
                    (div* (stat-visited s2)
                       (+ (stat-visited s2) (stat-q-len s2)))))
          (float (div* (stat-unify-count s1) (stat-unify-count s2))))
  (format t "~&")
  )


;;; Print out stats averaged over an entire testing run 
(defun AVE-STATS (&optional (ss-arg *stats*))
  (let ((inits 0) (goals 0) (win1 0) (win2 0) (win3 0) (cpu1 0) (cpu2 0) (cpu3 0)
        (nodev1 0) (nodev2 0) (nodev3 0) (nodevq1 0) (nodevq2 0) (nodevq3 0)
        (branch1 0) (branch2 0) (branch3 0) (bindadd1 0) (bindadd2 0) (bindadd3 0)
        (useu1 0) (useu2 0) (useu3 0) (totu1 0) (totu2 0) (totu3 0)
        (plen1 0) (plen2 0) (plen3 0) (nodec1 0) (nodec2 0) (nodec3 0))
    (dolist (test ss-arg)
      (let ((s1 (caddr test))
            (s2 (cadddr test))
	    (s3 (caddddr test)))
        (setq inits (+ inits (stat-num-init s1)))
        (setq goals (+ goals (stat-num-goal s1)))
        (setq win1 (+ win1 (if (stat-complete? s1) 1 0)))
        (setq win2 (+ win2 (if (stat-complete? s2) 1 0)))
	(setq win3 (+ win3 (if (stat-complete? s3) 1 0)))
        (setq plen1 (+ plen1 (stat-plan-len s1)))
        (setq plen2 (+ plen2 (stat-plan-len s2)))
	(setq plen3 (+ plen3 (stat-plan-len s3)))
        (setq cpu1 (+ cpu1 (stat-time s1)))
        (setq cpu2 (+ cpu2 (stat-time s2)))
	(setq cpu3 (+ cpu3 (stat-time s3)))
        (setq nodev1 (+ nodev1 (stat-visited s1)))
        (setq nodev2 (+ nodev2 (stat-visited s2)))
        (setq nodev3 (+ nodev3 (stat-visited s3)))
        (setq nodec1 (+ nodec1 (stat-created s1)))
        (setq nodec2 (+ nodec2 (stat-created s2)))
        (setq nodec3 (+ nodec3 (stat-created s3)))
        (setq nodevq1 (+ nodevq1 (stat-visited s1) (stat-q-len s1)))
        (setq nodevq2 (+ nodevq2 (stat-visited s2) (stat-q-len s2)))
	(setq nodevq3 (+ nodevq3 (stat-visited s3) (stat-q-len s3)))
        (setq branch1 (+ branch1 (stat-ave-branch s1)))
        (setq branch2 (+ branch2 (stat-ave-branch s2)))
	(setq branch3 (+ branch3 (stat-ave-branch s3)))
        (setq bindadd1 (+ bindadd1 (stat-add-bindings s1)))
        (setq bindadd2 (+ bindadd2 (stat-add-bindings s2)))
	(setq bindadd3 (+ bindadd3 (stat-add-bindings s3)))
        (setq useu1 (+ useu1 (- (stat-unify-count s1)
                                (stat-rank-unifies s1))))
        (setq useu2 (+ useu2 (- (stat-unify-count s2)
                                (stat-rank-unifies s2))))
	(setq useu3 (+ useu3 (- (stat-unify-count s3)
                                (stat-rank-unifies s3))))
        (setq totu1 (+ totu1 (stat-unify-count s1)))
        (setq totu2 (+ totu2 (stat-unify-count s2)))
	(setq totu3 (+ totu3 (stat-unify-count s3)))))
    (flet ((avef (num) 
             (div* (float num) (length ss-arg))))
      (setq inits (funcall #'avef inits))
      (setq goals (funcall #'avef goals))
      (setq win1 (funcall #'avef win1))
      (setq win2 (funcall #'avef win2))
      (setq win3 (funcall #'avef win3))
      (setq plen1 (funcall #'avef plen1))
      (setq plen2 (funcall #'avef plen2))
      (setq plen3 (funcall #'avef plen3))
      (setq cpu1 (funcall #'avef cpu1))
      (setq cpu2 (funcall #'avef cpu2))
      (setq cpu3 (funcall #'avef cpu3))
      (setq nodev1 (funcall #'avef nodev1))
      (setq nodev2 (funcall #'avef nodev2))
      (setq nodev3 (funcall #'avef nodev3))
      (setq nodec1 (funcall #'avef nodec1))
      (setq nodec2 (funcall #'avef nodec2))
      (setq nodec3 (funcall #'avef nodec3))
      (setq nodevq1 (funcall #'avef nodevq1))
      (setq nodevq2 (funcall #'avef nodevq2))
      (setq nodevq3 (funcall #'avef nodevq3))
      (setq branch1 (funcall #'avef branch1))
      (setq branch2 (funcall #'avef branch2))
      (setq branch3 (funcall #'avef branch3))
      (setq bindadd1 (funcall #'avef bindadd1))
      (setq bindadd2 (funcall #'avef bindadd2))
      (setq bindadd3 (funcall #'avef bindadd3))
      (setq useu1 (funcall #'avef useu1))
      (setq useu2 (funcall #'avef useu2))
      (setq useu3 (funcall #'avef useu3))
      (setq totu1 (funcall #'avef totu1))
      (setq totu2 (funcall #'avef totu2))
      (setq totu3 (funcall #'avef totu3)))
    (format t "~%~%AVERAGE (~3a tests) (Init ~3,3f;  Goals  ~3,3f)"
            (length ss-arg) inits goals)
    (format t "~%Variable            ~a   /   ~a   /  ~a"
            (stat-algo (caddr (car ss-arg))) (stat-algo (cadddr (car ss-arg)))
	    (stat-algo (caddddr (car ss-arg))))
    (format t "~%  Win Percent    ~11,4f   ~11,4f   ~11,4f" 
            win1 win2 win3)
    (format t "~%  Plan Length    ~11,4f   ~11,4f   ~11,4f" 
            plen1 plen2 plen3)
    (format t "~%  CPU Time       ~11,4f   ~11,4f   ~11,4f" 
            cpu1 cpu2 cpu3)
    (format t "~%  Branch Factor  ~11,4f   ~11,4f   ~11,4f" 
            branch1 branch2 branch3)
    (format t "~%  Nodes Visited  ~11,4f   ~11,4f   ~11,4f" 
            nodev1 nodev2 nodev3)
    (format t "~%  Total Nodes    ~11,4f   ~11,4f   ~11,4f" 
            nodevq1 nodevq2 nodevq3)
    (format t "~%  Created Nodes  ~11,4f   ~11,4f   ~11,4f" 
            nodec1 nodec2 nodec3)
    (format t "~%  Bindings Added ~11,4f   ~11,4f   ~11,4f" 
            bindadd1 bindadd2 bindadd3)
    (format t "~%  Gener. Unifies ~11,4f   ~11,4f   ~11,4f" 
            useu1 useu2 useu3)
    (format t "~%  Total Unifies  ~11,4f   ~11,4f   ~11,4f" 
            totu1 totu2 totu3)
    ))

;;; Compute average statistics for different subpopulations
;;; Cadr of a test is (init goal)
(defun MULTI-AVE (&optional (ss-arg *stats*))
  (labels ((num-inits (test) 
             (length (caadr test)))
           (num-goals (test)
             (length (cadadr test)))
           (num-ig (test)
             (+ (length (caadr test)) (length (cadadr test)))))
    (multi-ave-helper ss-arg #'num-inits "INITIAL CONDITIONS")
    (multi-ave-helper ss-arg #'num-goals "GOALS")
;   (multi-ave-helper ss-arg #'num-ig "INITS PLUS GOALS")
    ))

(defun caddddr (x) (car (cdr (cdr (cdr (cdr x))))))

(defun MULTI-AVE-HELPER (ss-arg f doc)
  (let* ((ss (sort (copy-list ss-arg) #'< :key f)) 
         (rank (funcall f (car ss)))
         (subseq nil))
    (format t "~%~%~%=====> AVERAGE BY NUMBER OF ~a" doc)
    (dolist (test ss)
      (cond
        ((eql (funcall f test) rank)
         (push test subseq))
        (t
         (format t "~%~%")
         (ave-stats subseq)
         (setq subseq (list test))
         (setq rank (funcall f test)))))
    (ave-stats subseq)))

;;; Print out a summary on an entire testing run
(defun ANAL-STATS (&optional (ss-arg *stats*))
  (flet ((test< (p1 p2)                 ; p1 = (num-init num-goal)
           (or (< (length (cadr p1)) (length (cadr p2)))
               (and (= (length (cadr p1)) (length (cadr p2)))
                    (< (length (car p1))  (length (car p2)))))))
    (let ((ss (sort (copy-list ss-arg) #'test< :key #'cadr)))
      (dolist (test ss)
        (compare-stat (caddr test) (cadddr test))))))

