;;
;;      Name: Term Unification Routines
;;
;;      Functions:  contains routines that are specific to term unification
;;
;;      Author: Alan W Black September 1984
;;              Dept of A.I.
;;              University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;

(declare
   (special
      D-CATEGORYDEFINITIONS)
   (localf
      D-ParseCatDef
      D-BuildCategoryDefinition
      D-ParseFeatureClass
      D-ParseTopCategory
      D-MakeCanMakeTree 
      D-FindAppropriateRules
      D-FindAllAppropriateTypes 
      D-FindMatchingLeftDaughters
      D-BuildGrammarDiscrimminationNet
      D-ParseLCategory
      D-CheckDistCat
   )
)

(defun D-ParseCatDef ()
;;
;;   parses a category definition
;;
   (let ( cat values feat name )
      (setq name D-CurrentSym)
      (D-CheckAtom name)            ;; ensure not a bracket etc
      (D-GetNextSym)
      (D-MustHave (DK-has))
      (setq values (D-ParseSimpleSet))
      (mapc                     ;; check the values are declared features
	 #'(lambda (fname)
	      (cond
		 ((D-FeatureP fname) t)
		 (t   
		    (D-FindCurrentLine)
		    (error (concat "\"" fname 
				   "\" is not a valid feature name")))))
         values)
      (setq D-CATEGORYDEFINITIONS 
	 (cons
	    (cons name values)
	    D-CATEGORYDEFINITIONS)))
)

(defun D-BuildCategoryDefinition (name skeleton)
;;
;;  Builds a category definition.  This consists of 
;;  a name, a list of mandatory features and values, and
;;  a list of features that are also part of the category
;;
   t
)

(defun D-ParseFeatureClass ()
;;
;;  This takes a feature classes declaration and adds
;;  to the appropriate global list
;;  In term unification only one Feature Class is allowed
;;    MorphologyOnly.
;;  consistancy checks should be done later
;;
   (cond
      ((eq D-CurrentSym (DK-WHead))
	 (D-FindCurrentLine)
	 (error (concat "Invalid feature class in term unification  "
			D-CurrentSym))
      )
      ((eq D-CurrentSym (DK-WDaughter))
	 (D-FindCurrentLine)
	 (error (concat "Invalid feature class in term unification  "))
      )
      ((eq D-CurrentSym (DK-MorphologyOnly))
	 (D-GetNextSym)    ;; skip to class list
	 (D-MustHave '=)
	 (setq D-MORPHOLOGYONLY (D-ParseSimpleSet)) ;; list of morpho-features
      )
      (t
	 (D-FindCurrentLine)
	 (error (concat "Unknown Feature Class name " D-CurrentSym))
      )
   )
)

(defun D-ParseTopCategory ()
;;
;;   Parses the definition of a top category that is 
;;   used in parsing to define what a word is
;;   In term unification this is a list of category types.  This
;;   is because in tu you can't specify things that range over 
;;   different categories.
;;
;;   All parses that span the chart and has labels that are categories that
;;   are definied in the list D-DISTINGUISHEDCATEGORY are valid parses
;;
   (D-MustHave '=)
   (setq D-DISTINGUISHEDCATEGORY
         (D-ParseSimpleSet))     ;; list of categroy names 
)

(defun D-MakeCanMakeTree (grammar)
;;
;;  This takes the list of grammar rules and produces an assoc list
;;  of the form, (<category type> <rulename> <rulename> ...)
;;  Where a category type is the left daughter of a rule or is
;;  the left daughter of a rule whose mother is the left daughter
;;  of a rule etc.  This (Steve's trick) is so that during parsing
;;  proposing only gets done when adding a lexical entry.
;;
   (let ( (types (mapcar #'(lambda (type) (car type))
			   D-CATEGORYDEFINITIONS)) 
          (expandedgram (D-ProcessGRules grammar)) )
      (mapcar
	 #'(lambda (type)
	       (cons
		  type
		  (D-FindAppropriateRules
		     (D-FindAllAppropriateTypes
			(ncons type)
			nil         ;; rules found so far
			expandedgram)
                     expandedgram)))
	 types
      )
   )
)

(defun D-FindAppropriateRules (types rules)
;;
;;  returns a rlist of the names of all grammar rules that have
;;  any of the types as their left daughter
;;
   (mapcan
      #'(lambda (type)
	 (mapcan
	    #'(lambda (rule)
	       (cond
		  ((eq type (D-GetCategoryType (D-GetGRuleFirstDaughter rule)))
		     (ncons (D-GetGRuleName rule)))
                  (t nil)))
            rules))
      types)
)

(defun D-FindAllAppropriateTypes (types typesfound rules)
;;
;;  finds all rules with type as left daughter and then for each 
;;  of their mothers checks them
;;
;;  returns a list of category types
;;
   (cond
      ((null types)
	 typesfound)
      ((memq (car types) typesfound)
	 (D-FindAllAppropriateTypes
	    (cdr types)
	    typesfound
	    rules))
      (t
	 (D-FindAllAppropriateTypes
	    (append (D-FindMatchingLeftDaughters (car types) rules)
		    (cdr types))
            (cons (car types) typesfound)
	    rules)
      )
   )
)

(defun D-FindMatchingLeftDaughters (type rules)
;;
;;   searches the list of rules and returns the mother of all rules
;;   that have type as their left daughter
;;
   (mapcan
      #'(lambda (rule)
	 (cond
	    ((eq type (D-GetCategoryType (D-GetGRuleFirstDaughter rule)))
	       (ncons (D-GetCategoryType (D-GetGRuleMother rule)))
            )
	    (t nil)))
      rules)
)

(defun D-BuildGrammarDiscrimminationNet (grammar)
;;
;;  Used in Bay Area Unification version 
;;
   nil
)

(defun D-ParseLCategory ()
;; 
;;   LCategory definitions are only allowed in the bay area version of the 
;;   system so this gives an error if they are defined in a term unification
;;   file
;;
   (D-FindCurrentLine)
   (error (concat "LCategory definitions are only allowed in the Bay Area "
		  "Unification System"))
)

(defun D-CheckDistCat (distcat)
;;
;;   does nothing in the term unification version
;;
   distcat
)

(defun D-SetTop ()
;;
;;
;;  This is part of the command interpreter but is put in here
;;  as it differs between term and bay area unification
;;
;;  Changes the distinguished set of categories.  Note this over writes
;;  the current distinguished list of category types
;;
;;
   (let ( newcategory )
      (cond
	 ((null (assq 'gr D-LOADEDPARTS))
	    (error "Must have word grammar loaded to change top categories")))
      (princ "Enter new category set: ") (drain)
      (setq D-fileid t)   ;; so parse category will get the right file
      (D-MakeEOLNSymbol)
      (D-GetNextSym)      ;; read first symbol
      (setq newcategory (D-ParseSimpleSet))
      (D-MakeEOLNNotSymbol)
      (cond
	 ((D-Subset newcategory 
	      (mapcar #'car D-CATEGORYDEFINITIONS))
	    (setq D-DISTINGUISHEDCATEGORY newcategory)
	    (princ "New top category types set") (terpri))
         (t    ;; illegal new category
	    (princ "New top categores NOT changed because")
	    (princ " given categories are not valid") (terpri)))
   )
)

(defun D-CheckTopCat ()
;;
;;   This checks to see if a distinguished category has been
;;   defined.  If it is not set at all and error is signalled and
;;   the compilation aborted.  This is because there must be some
;;   definition of a word.
;;
   (cond
      ((eq D-DISTINGUISHEDCATEGORY 'UNSET)
	 (D-FindCurrentLine)
	 (error "NO Top Categories set - so can be no valid words")
      )
      (t
	 t    ;; no problem
      )
   )
)

