;;
;;
;;      Title : morphan
;;
;;      Function : A morphological analyser and dictionary
;;                 Dictionary Command Interpreter
;;
;;      Author : Alan W Black  Nov 1985 
;;               Dept of A.I. University of Edinburgh
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Version : $Revision: 3.0 $
;;                $Date: 87/08/04 14:21:26 $
;;
;;       1.1 Version small upgrade to  increase speed mainly, and
;;           fixed bug in parser regarding the left recursion check
;;           Changes of some recursion to mapcars
;;       1.5 Change the rule interpretation to allow for instantiation of
;;           variables in the categories.
;;           Note that 1.5 does not include the 1.4 upgrade (expansion of
;;           automata) because the upgrade did not save much time. (this
;;           my be incorporated in later versions)
;;       1.6 This adds in feature passing conventions of WHead WDaughter
;;           and WSister.
;;       1.7 Changed top level to be more command like so words are not
;;           directly looked up but have to be specifically requested.
;;           Added errset catches to cope with error (I hope). Minor
;;           bugs fixed
;;       1.8 Fix the hyphen problem and remove the path build up in
;;           autorun.  
;;       1.11 October 1985
;;           Extend the number of commands and incorporate the compiling
;;           mechanism into this (formerly called macomp).
;;       1.18 June 1986
;;           Rationalised the reader by writting my own so it can port
;;           to the vax.  Also it give more control over reading
;;       2.4  February 1987
;;           Added the spelling rule debugger.  ParseFor command
;;     
;;
;;      Description :
;;         This forms the prototype of a morphological analyser and
;;         dictionary system developed at Dept of AI Edinburgh and
;;         Computing Laboratory, Cambridge by Graeme Ritchie, Alan
;;         Black, Steve Pulman and Graham Russell. This work is supported
;;         by an Alvey/SERC research grant, number GR/C/79114.
;;        
;;         The system is built on two levels, a morphographemic
;;         (or spelling) analyser based on KIMMO (Kartunnen et al
;;         Texas Linguistic Forum 32) and GPSG(ish) grammar driving
;;         the morphology proper (inflectional and derivational).
;;
;;         See "A Morphology Analyser and Dictionary - User Manual"
;;
;;         This command interface allows access to all the functions in 
;;         the analyser and dictionary.  h gives a list of all the 
;;         commands available, see the User Manual for more detail
;;
;;      Parameters :
;;         NONE
;;         can also be called from UNIX command line with the 
;;         command dci
;;
;;      Returns :
;;         ?
;;      Side Effect :
;;         
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;         Please remember this is a prototype and is likely to have
;;         problems, but please also remember to report these problems
;;         to me (Alan W Black, Dept of AI, University of Edinburgh)
;;         so they can be dealt with.
;;           JANET  awb@uk.ac.edinburgh.aiva
;;           UUCP   ...ukc!aiva!awb  (from US ...!seismo!mcvax!ukc!aiva!awb)
;;           ARPA   awb%uk.ac.edinburgh.aiva@ucl.cs
;;

(declare
   (special
      D-NEXTCHAR
      D-TRACEFLAGSWITCH
      D-LOOKUPFORMAT
      D-LEFTPAREN
      D-RIGHTPAREN
      D-SIMPLEQUOTE
      D-DCICurrentSym
   )
   (localf                     ;; local functions
      D-ClearDictionary
      D-DebugParseCommand
      D-LoadLexiconSet
      D-LoadFiles                     
      D-CompileSp                     
      D-CompileWG                     
      D-CompileLe                     
      D-Loadls                        
      D-Loadlg                        
      D-Loadll                        
      D-Addll                        
      D-ChangeOutputFormat            
      D-WordAnalyse                   
      D-WordSegment                   
      D-MorphemeLookUp                
      D-ConcatMorphemes
      D-PrintParses                   
      D-PrintEachParse                
      D-NicePrint                     
      D-WordTreePP                    
      D-Pause                         
      D-toggle                        
      D-ShowHelp                      
      D-ReadWord
   )
)
      
(include "macros")
(include "keywords")
(include "subrout")
(include "catrouts")
(include "parserouts")  ;; also includes readatom
(include "specrouts")   ;; some specific parsing stuff
(sstatus translink on)
(allocate 'list 2000)
(allocate 'bignum 50)
(allocate 'fixnum 150)
(allocate 'symbol 200)
;(allocate 'vector 400)

(defun D-Start (&optional dir)
;;
;;   This is the morphological analyser, this reads in the code and
;;   then calls the basic command interpreter
;;
   (cond                   ;; load code in
      ((and dir (atom dir))
	 (load (concat dir "/maload"))  ;; load in basic bootstrap code
	 (d-maload dir)    ;; directory specified for code
      )
      (t
	 (load "maload")  ;; load in basic bootstrap routine
	 (d-maload)      ;; current directory or lib-search-path member 
      )
   )
   (D-Initialise)        ;; This is also done in d-maload
   (D-InitReader)
   (D-Restart)           ;; the basic command line interpreter
)


(defun D-Restart ()
;;
;;    This is the command line interpreter for the morphological analyser
;;
   (terpri)
   (princ "Dictionary Command Interpreter")
   (terpri)
   (D-VersionHeading)
   (terpri)
   (princ "type h<CR> for help")
   (terpri)
   (D-ReadToEndOfLine 'EOF)    ;; skip over last EOLN char
   (let ( (command '(INIT)) )
      (D-while (not (equal command 'EXIT))
	 (princ "> ")   (drain) ;; prompt
	 (setq command (D-Command)) ;; get command 
	 (cond
	    ((null command)       ;; nothing typed in
	       nil     ;; do nothing
            )
	    ((or (eq command 'EOF) 
		 (memq (car command) (DK-COM-EXITS))) ;; exit from analyser
	       (princ "EXITING FROM ANALYSER")
	       (terpri)
	       (setq command 'EXIT)
            )
	    ((memq (car command) (DK-COM-HELPS))  ;; display help
	       (D-ShowHelp)
            )
	    ((equal (car command) (DK-COM-TRACE))  ;; D-toggle tracing
	       (setq D-TRACEFLAGSWITCH (D-toggle D-TRACEFLAGSWITCH))
	       (princ "TRACING ")
	       (princ D-TRACEFLAGSWITCH)
	       (terpri)
            )
	    ((equal (car command) (DK-COM-LOAD));; a set of dictionary files
	       (errset (D-LoadFiles (cdr command)))
            )
	    ((equal (car command) (DK-COM-LS))  ;; load a set of spelling rules
	       (errset (D-Loadls (cdr command)))
            )
	    ((equal (car command) (DK-COM-LL))  ;; load a lexicon tree etc
	       (errset (D-Loadll (cdr command)))
            )
	    ((equal (car command) (DK-COM-AL))  ;; add a lexicon tree to current
	       (errset (D-Addll (cdr command)))
            )
	    ((equal (car command) (DK-COM-LG))  ;; load a word grammar
	       (errset (D-Loadlg (cdr command)))
            )
	    ((equal (car command) (DK-COM-CS))  ;; compile spelling rules
	       (errset (D-CompileSp (cdr command)))
            )
	    ((equal (car command) (DK-COM-CG))  ;; compile word grammar  
	       (errset (D-CompileWG (cdr command)))
            )
	    ((equal (car command) (DK-COM-CL))  ;; compile lexicon       
	       (errset (D-CompileLe (cdr command)))
            )
	    ((equal (car command) (DK-COM-W))   ;; analyse word
	       (errset (D-WordAnalyse (cdr command)))
            )
	    ((equal (car command) (DK-COM-S))   ;; segment  word
	       (errset (D-WordSegment (cdr command)))
            )
	    ((equal (car command) (DK-COM-F))   ;; change look up format
	       (errset (D-ChangeOutputFormat (cdr command)))
            )
	    ((equal (car command) (DK-COM-M))   ;; look up morpheme
	       (errset (D-MorphemeLookUp (cdr command)))
            )
	    ((equal (car command) (DK-COM-CLEAR));; clear system by initialising
	       (errset (D-ClearDictionary))
            )
	    ((equal (car command) (DK-COM-DS))      ;; display sections
	       (errset (D-PrintLoadedSections))
            )
	    ((equal (car command) (DK-COM-EVAL))    ;; lisp escape
	       (errset (print (eval (cdr command))))
	       (terpri)
            )
	    ((equal (car command) (DK-COM-LA))      ;; load all         
	       (errset (D-LoadLexiconSet (cdr command)))
            )
	    ((equal (car command) (DK-COM-CM))      ;; concat morphemes
	       (errset (D-ConcatMorphemes (cdr command)))
            )
	    ((equal (car command) (DK-COM-SPDEBUG)) ;; Spelling rule debug
	       (errset (D-SpDebug))
            )
	    ((equal (car command) (DK-COM-SETTOP)) ;; Set Top Category
	       (errset (D-SetTop))
            )
	    ((equal (car command) (DK-COM-DEBUG))  ;; debug parse
	       (errset (D-DebugParseCommand (cdr command)))
            )
	    (t     
	       (princ "ERROR >>>> input unknown")
	       (terpri)
	       (princ "enter h for help")
	       (terpri)
            )
         )
      )
   )
)

(defun D-ClearDictionary ()
;;
;;   Asks the user to confirm before clearing the dictionary,
;;   grammar and spelling rules
;;
   (let  ( (ans nil) )
      (princ "Do you really want to initialise the dictionary ?? [n|y] ")
      (setq ans (D-ReadToEndOfLine 'EOF))
      (cond
         ((memq (car ans) (DK-COM-YES))
	    (D-Initialise)
	    (princ "Dictionary and Analyser System emptied")
	    (terpri)
         )
	 (t
	    (princ "Dictionary NOT initialised")
	    (terpri)
         )
      )
   )
)

(defun D-DebugParseCommand (params)
;;
;;  Like look up but enters the debugger after look up
;;
   (let ( word )
      (cond
	 ((null params)
            (princ "Enter word ? ")
            (setq word (D-ReadWord))
         )
	 (t
	    (setq word (car params))))
      (cond
	 ((and (listp word) word) 
	    nil    ;; got end of file or nothing
         )
	 (t
	    (princ "Looking up word with debug")
	    (terpri)
	    (D-DebugParse word)
         )
      )
   )
)

(defun D-LoadLexiconSet (params)
;;
;;   Loads in a spelling rule set, word grammar and lexicon all
;;   with the given name.
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF))))
	 (t
	    (setq name (car params))))
      (D-LoadAll name name name)
      (princ "Lexicon set ") (princ name) (princ " have been loaded")
      (terpri)
   )
)

(defun D-LoadFiles (params)
;;
;;  This reads in the file name and loads in a set of files
;;  of the form suitable for the analyser
;;
   (let ()
      (cond
	 ((null params)
            (D-Loadls nil)             ;; spelling rules
            (D-Loadlg nil)             ;; word grammar
            (D-Loadll nil)             ;; load lexicon
         )
	 ((eq (length params) 1)
            (D-Loadls params)          ;; spelling rules
            (D-Loadlg nil)             ;; word grammar
            (D-Loadll nil)             ;; load lexicon
         )
	 ((eq (length params) 2)
            (D-Loadls params)          ;; spelling rules
            (D-Loadlg (cdr params))    ;; word grammar
            (D-Loadll nil)             ;; load lexicon
         )
	 ((eq (length params) 3)
            (D-Loadls params)          ;; spelling rules
            (D-Loadlg (cdr params))    ;; word grammar
            (D-Loadll (cddr params))   ;; load lexicon
         )
	 (t
	    (error "Too many arguments")
         )
      )
      (princ "Loaded Analyser files")
      (terpri)

      ;;;   Consistancy check required

   )
)


(defun D-CompileSp (params)
;;
;;  gets the name of the spelling rules and compiles them, 
;;  automatically loading them
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter spelling rule name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF))))
	 (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
            (D-MakeSpRules name)
            (princ "Spelling Rules ") (princ name) (princ " have been compiled")
            (terpri)))
   )
)

(defun D-CompileWG (params)
;;
;;  gets the name of the word grammar and compiles it, 
;;  automatically loading it
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter word grammar name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF))))
	 (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
	    (D-MakeWordGrammar name)
	    (princ "Word Grammar ") (princ name) (princ " has been compiled")
	    (terpri)))
   )
)

(defun D-CompileLe (params)
;;
;;  gets the name of the dictionary file and compiles it, 
;;  automatically loading it afterwards
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter dictionary name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF))))
         (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
	    (D-MakeLexicon name)
	    (princ "Lexicon ") (princ name) (princ " has been compiled")
	    (terpri)))
   )
)

(defun D-Loadls (params)
;;
;;  gets the name of the files and loads the spelling rules
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter spelling rule name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF)))
	 )
         (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
            (D-LoadSpRules name)))
   )
)
   
(defun D-Loadlg (params)
;;
;;  gets the name of the files and loads the word grammar  
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter word grammar name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF)))
         )
	 (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
            (D-LoadWordGrammar name)))
   )
)
   
(defun D-Loadll (params)
;;
;;  gets the name of the files and loads the lexicon and opens the 
;;  port to the entry data file
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter dictionary name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF)))
	 )
         (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
            (D-LoadLexicon name)))
   )
)
   
(defun D-Addll (params)
;;
;;  gets the name of the files and adds the lexicon to the list
;;  of current lexicons. And opens the 
;;  port to the entry data file
;;
   (let (name)
      (cond
	 ((null params)
            (princ "Enter dictionary name ? ")
            (setq name (car (D-ReadToEndOfLine 'EOF)))
	 )
         (t
	    (setq name (car params))))
      (cond
	 ((null name)    ;; check for name
	    (princ "No name given") (terpri))
         (t
            (D-AddLexicon name)))
   )
)
   
(defun D-ChangeOutputFormat (params)
;;
;;  change the output format four formats are possible:
;;     top category
;;     whole word-structure
;;     segment string returning edges with top category as label
;;     segment string returning edges with word structure as label
;;
   (let (choice)
      (cond
         ((null params)    ;; no parameter given
	    (princ "Change current output format") (terpri)
	    (princ "   1 - Category Form ") (terpri)
	    (princ "   2 - Word Structure Form ") (terpri)
	    (princ "   3 - String Segment with Category Form ") (terpri)
	    (princ "   4 - String Segment with Word Structure Form ") (terpri)
	    (princ "Enter choice ? ")
	    (setq choice (car (D-ReadToEndOfLine 'EOF)))
         )
	 (t
	    (setq choice (car params))))
      (cond
         ((eq choice '|1|)
	    (D-ChangeLookUpFormat 'D-CATEGORYFORM)
	    (princ "Format is now CATEGORYFORM") (terpri)
         )
         ((eq choice '|2|)
	    (D-ChangeLookUpFormat 'D-WORDSTRUCTURE)
	    (princ "Format is now WORDSTRUCTURE") (terpri)
         )
         ((eq choice '|3|)
	    (D-ChangeLookUpFormat 'D-STRINGSEGMENTCAT)
	    (princ "Format is now STRINGSEGMENTCAT") (terpri)
         )
         ((eq choice '|4|)
	    (D-ChangeLookUpFormat 'D-STRINGSEGMENTWS)
	    (princ "Format is now STRINGSEGMENTWS") (terpri)
         )
	 (t
	    (princ "Reply unknown - format is unchanged")
	    (terpri)
         )
      )
   )
)

(defun D-WordAnalyse (params)
;;
;;  analyse word and look up in lexicon
;;
   (let ( word (starttime (ptime)) answer)
      (cond
	 ((null params)
            (princ "Enter word ? ")
            (setq word (D-ReadWord))
         )
	 (t
	    (setq word (car params))))
      (cond
	 ((and (listp word) word) 
	    nil    ;; got end of file or nothing
         )
	 (t
	    (princ "Looking up word")
	    (terpri)
	    (setq answer (D-LookUp word))
	    (princ "Time taken ") (princ (- (car (ptime)) (car starttime)))
	    (terpri)
	    (D-PrintParses
	       answer
	       D-LOOKUPFORMAT     ;; type of printing
	    )
         )
      )
   )
)

(defun D-WordSegment (params)
;;
;;  segment word using morphographemic analyser only
;;
   (let (word)
      (cond
	 ((null params)
            (princ "Enter word ? ")
            (setq word (D-ReadWord))
         )
	 (t
	    (setq word (car params))))
      (cond
	 ((and (listp word) word) 
	    nil    ;; got end of file or nothing
         )
	 (t
	    (princ "Segmenting word")
	    (terpri)
	    (D-PrintParses
	       (D-Segment word)    ;; morphographemicize word 
	       'segmentation          ;; type of printing
	    )
         )
      )
   )
)

(defun D-ConcatMorphemes (params)
;;
;;  concat the given morphemes and return a list of surface forms
;;  if no params are given only ONE morpheme can be given - but this
;;  is really just a lexical tape which can be a long as you wish
;;
   (let (morphemes)
      (cond
	 ((null params)
	    (princ "Enter lexical string ? ")
	    (setq morphemes (D-ReadToEndOfLine 'EOF)))
         (t
	    (setq morphemes params))
      )
      (cond
	 ((null morphemes) 
	    nil    ;; got end of file or nothing
         )
	 (t
	    (princ "Generating Surface Forms")
	    (terpri)
	    (D-PrintParses
	       (D-MorphemeConcat morphemes nil)
	       'surfaceform
	    )
	 )
      )
   )
)

(defun D-MorphemeLookUp (params)
;;
;;  look a morphem up directly in lexicon
;;
   (let (morpheme)
      (cond
	 ((null params)
            (princ "Enter morpheme ? ")
            (setq morpheme (D-ReadWord))
         )
	 (t
	    (setq morpheme (car params))))
      (cond
	 ((and (listp morpheme) morpheme) 
	    nil    ;; got end of file or nothing
         )
	 (t
	    (princ "Looking up morpheme")
	    (terpri)
	    (D-PrintParses
	       (D-Morpheme
		  morpheme
	       )
	       'morpheme    ;; type
            )
         )
      )
   )
)

(defun D-PrintParses (parseslist ptype)
;;
;;     This pretty prints each parse of the word or edge in the 
;;     analyses
;;
   (cond
      ((eq ptype 'morpheme)
	 (print (length parseslist))
	 (princ " morphemes found")
	 (terpri)
	 (terpri)
	 (D-PrintEachParse
	    parseslist 1 'morpheme)
      )
      ((eq ptype 'segmentation)
	 (print (length parseslist))
	 (princ " segmentations found")
	 (terpri)
	 (terpri)
	 (D-PrintEachParse
	    parseslist 1 'segmentation)
      )
      ((eq ptype 'surfaceform)
	 (print (length parseslist))
	 (princ " surface forms found") (terpri) (terpri)
	 (D-PrintEachParse
	    parseslist 1 ptype)
      )
      ((memq ptype '(D-STRINGSEGMENTCAT D-STRINGSEGMENTWS))
	 (print (length (cdr parseslist)))
	 (princ " edges found ") (terpri)
	 (princ "Start vertex: ") (princ (caar parseslist))
	 (princ "  End vertex: ") (princ (cadar parseslist))
	 (terpri)
	 (terpri)
	 (D-PrintEachParse
	    (cdr parseslist) 1 'edge)
      )
      ((memq ptype '(D-CATEGORYFORM D-WORDSTRUCTURE))
         (print (length parseslist))
         (princ " analyses found")
         (terpri)
         (terpri)
         (D-PrintEachParse
            parseslist 1 'analysis)
      )
      (t     ;; God Knows ????
	 (print (length parseslist))
	 (princ " analyses found")
	 (terpri)
	 (terpri)
	 (D-PrintEachParse
	    parseslist 1 'analysis)
      )
   )
)

(defun D-PrintEachParse (parselist parsenum ptype)
;;
;;  recurses down the parselist and prints the parse tree with a pause
;;  between each parse (or each edge)
;;
   (cond
      ((null parselist)
	 nil            ;; no parses
      )
      (t
	 (princ ptype) (princ " : ")
	 (princ parsenum)
	 (terpri)
	 (D-NicePrint (car parselist) ptype)
	 (cond
	    ((null (cdr parselist))
	       (terpri)
	       nil)  ;; just shown last one so return to top level
	    ((memq (D-Pause) (DK-COM-EXITS)) nil)   ;; don't show any more
	    (t
	       (D-PrintEachParse 
	          (cdr parselist) (add 1 parsenum) ptype)))
      )
   )
)

(defun D-NicePrint (parse ptype)
;;
;;  pretty print the given parse to the terminal followed by a newline
;;  This uses a different pretty printer depending on the type of
;;  parse given
;;
   ;(print ptype) (terpri)
   (cond
      ((memq ptype '(surfaceform morpheme))
	 (terpri)
	 (princ parse)
	 (terpri)
      )
      ((equal ptype 'segmentation)
	 (mapcar
	    #'(lambda (segment)
		 (princ segment) (terpri))
            parse
         )
      )
      ((eq D-LOOKUPFORMAT 'D-WORDSTRUCTURE)
	 (D-WordTreePP ""  parse)
      )
      ((equal D-LOOKUPFORMAT 'D-CATEGORYFORM)
	 (princ parse)
	 (terpri)
      )
      (t            ;; not sure what this is  (chart options)
         (pp-form parse)
      )
   )
   (terpri)
)

(defun D-WordTreePP (margin wtree)
;;
;;   pretty prints the word tree format of output from analyser
;;
   (cond
      ((equal (cadr wtree) 'ENTRY)   ;; lexical entry
	 (terpri)
	 (princ margin) (princ "(")
	 (princ (car wtree)) (terpri)
	 (princ margin) (princ 'ENTRY) (terpri)
	 (princ margin) (princ "   ") (princ (caddr wtree)) 
	 (princ ")")
      )
      (t
	 (terpri)
	 (princ margin)
	 (princ "(")
	 (princ (car wtree)) (terpri)  ;; category
	 (princ margin)
	 (princ (cadr wtree))          ;; rule number
	 (mapcar
	    #'(lambda (daughter)
		  (D-WordTreePP
		     (concat margin "   ")
		     daughter))
            (cddr wtree)   ;; the daughters
         )
	 (princ ")")
      )
   )
)

(defun D-Pause ()
;;
;;   prints message and waits for the user to press return.  
;;   Any atom typed before the cr is returned
;;
   (princ "Press RETURN key to continue")
   (terpri)
   (let ((reply (D-ReadToEndOfLine 'EOF)))
      (cond
	 ((null reply) nil)
	 ((memq (car reply) (DK-COM-EXITS)) (car reply))
	 (t reply)))
)

(defun D-toggle (tflag)
;;
;;  Toggles the tracing flag from ON to OFF or vice versa
;;
   (cond
      ((equal tflag (DK-ON))
	 (DK-OFF)
      )
      (t
	 (DK-ON)
      )
   )
)

(defun D-ShowHelp ()
;;
;;    Display help information
;;
   (D-VersionHeading)
   (terpri)
   (princ " the commands will prompt for more information where required ")
   (terpri)
   (terpri)
   (princ "h  or ? : help                e q exit quit or EOF : for exit")
   (terpri)
   (princ "l : load all three parts      ll : load lexicon")
   (terpri)
   (princ "ls : load spelling rules      lg : load word grammar")
   (terpri)
   (princ "al : add a new lexicon to current lexicon") 
   (terpri)
   (princ "la : load a lexicon set all with the same name")
   (terpri)
   (princ "cl : compile lexicon ")
   (terpri)
   (princ "cs : compile spelling rules")
   (terpri)
   (princ "cg : compile word grammar")
   (terpri)
   (princ "t : toggle trace setting      f : change output format")
   (terpri)
   (princ "w : look up word in dictionary with morphological analysis")
   (terpri)
   (princ "m : look up morpheme directly in lexicon")
   (terpri)
   (princ "s : display segmentation of word into morphemes")
   (terpri)
   (princ "st : set new top category for word parser")
   (terpri)
   (princ "cm : concatenate morphemes using spelling rules")
   (terpri)
   (princ "ds : display currently loaded sections")
   (terpri)
   (princ "spd : spelling rule debugger   db: debug lookup ")
   (terpri)
   (princ "clear : clear dictionary system (make empty)")
   (terpri)
   (princ "! <s-expression>: evaluate a lisp s-expression")
   (terpri)
   (terpri)
)

(defun D-ReadWord ()
;;
;;  Reads a the next word (atom) and zaps the rest of the line
;;  this is a replacement for ratom and zapline returns a cons cell
;;
   (let ( (ans (D-Command)) )
       (cond
	  ((and (listp ans) ans)   ;; has been given an answer
	     (car ans))
          (t
	     (cons nil nil))))
)

(defun D-Command ()
;;
;;   Reads a list of atoms from the standard input up to and
;;   including the cr.  Returns the atoms as a list
;;   nil if none and EOF if EOF
;;
   (D-ReadToEndOfLine 'EOF)
)

