;;
;;
;;      Title : D-MakeWordGrammar
;;
;;      Function : Expands the variables and aliases in the
;;                 word structure grammar
;;
;;      Author :  Alan W Black  Dec 1984
;;                Dept of A.I. University of Edinburgh
;;     
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description :
;;         Reads in the user specified grammar (GPSG type rules) from
;;         inputfile  and expands the rules by expanding variables and 
;;         replacing aliases.  The result is saved in outputfile
;;         The variables and aliases and features used in the grammar 
;;         must be declared in file dcls.
;;
;;         1.2 Update  13th February 1985
;;         classify the rules by type: noun verb preposition adjective
;;         and other so the searching of rules in the parser is faster
;;         1.3  1st March 1985
;;         parser changed to run bottom up so rules have to be primarily
;;         typed by there first category in the RHS.  
;;         1.5  29th March 1985
;;         feature value variables, now no longer to be expanded as they
;;         are instantiated in the parser on the fly.
;;         1.6  24th April 1985
;;         removed the sorting of rules and categories
;;         1.8  11th June 1985
;;         put in consistancy checks for the categories, ensuring
;;         that features values etc are valid.
;;         1.11 3rd October 1985
;;         modified so that grammar compilation can be done separately
;;         from lexicon building and compiling of spelling rules.
;;         1.14 24th January 1986
;;         Made the reader different so it checks the syntax of the
;;         grammar and declarations better
;;         1.15 21st February 1986
;;         modified the grammar definition to include a top category
;;         now more than just binary branching rules are allowed
;;         1.18 17th June  1986
;;         Changed the stuff that is printed to <name>.gr.ma to be a 
;;         number fo s-expressions rather than a number of setq's  This
;;         means the globals names used in the system never appear in files
;;         outside the system and hence allow this code to ported easier.
;;         Added another featureclass, MorphologyOnly.  These are removed
;;         from the syntactic category (when lookup format is CATEGORYFORM)
;;         2.1  30th July 1986
;;         Removed closures and replaced with simpler system of actually 
;;         saving and restoring globals.  Changed syntax of declarations
;;         and rules to that of GDE.  Added LCategory declarations
;;         2.2  9th October 1986
;;         Made keywords macros
;;         
;;      Parameters :
;;         name : atom  where in <name>.gr is the name of the file containing
;;                the grammar (and the feature declarations)
;;      Returns :
;;         ?
;;      Side Effect :
;;         creates or modifies <name>.gr.ma
;;
;;      External references :
;;
;;      System functions used :
;;        Name     type      Args #  type                 Comment
;;
;;      Restrictions :
;;
;;

(declare
   (special
      D-fileid
      D-GRAMMAR
      D-FSD
      D-VARIABLES
      D-BASICVARIABLES
      D-WDAUGHTER
      D-WHEAD
      D-MORPHOLOGYONLY
      D-ALIASES
      D-DISTINGUISHEDCATEGORY
      D-LCATEGORIES
      D-NUMBER
      D-CATEGORYDEFINITIONS
   )
   (localf
      D-InitGramGlobals               
      D-ParseWGram                    
      D-ParseGrammarRule              
      D-ConvGrammar                   
      D-AddUniqueRuleName             
      D-MarkVarRule
      D-SubsAliases                   
      D-SubsAliasRule                 
      D-CheckCategoryAndReturnIt
      D-ExpandRule                    
      D-ExpandEachCategory            
      D-ExpandCategory                
      D-ExpandCat                
      D-GetBindings                   
      D-VarBoundP                     
   )
)

(include "macros")
(include "keywords")
(include "subrout")
(include "catrouts")
(include "unify")
(include "parserouts")
(include "specrouts")  ;; routines specific to unification type
(include "dclsconv")

(defun d-makewordgrammar (name) (D-MakeWordGrammar name))

(defun D-MakeWordGrammar (name)
;;
;;  Reads in the users word grammar file and expands variables
;;  etc.  And produces a new file <name>.gr.ma which ontains all
;;  the simple expnasion so they can be loaded later.
;;
   (let ()
      (D-SaveGlobalEnvironment)
      (D-InitGramGlobals)
      (setq D-fileid (infile (concat name (DK-GR))))
      (setq D-INCLUDEFILES (ncons (concat name (DK-GR))))

      (D-GetNextSym) 

      (D-ParseWGram)    ;; process word grammar rules rules

      (close D-fileid)
      (setq D-fileid (outfile (concat name (DK-GR-MA))))
      (print (DK-VERSION-STAMP) D-fileid) (terpri D-fileid)
      (print D-GRAMMAR D-fileid) (terpri D-fileid)
      (print D-ALIASES D-fileid) (terpri D-fileid)
      (print D-FEATURES D-fileid) (terpri D-fileid)
      (setq D-BASICVARIABLES D-VARIABLES)
      (print D-BASICVARIABLES D-fileid) (terpri D-fileid)
      (print D-CATVALFEAT D-fileid) (terpri D-fileid)
      (print D-WHEAD D-fileid) (terpri D-fileid)
      (print D-WDAUGHTER D-fileid) (terpri D-fileid)
      (print D-FSD D-fileid) (terpri D-fileid)
      (print D-DISTINGUISHEDCATEGORY D-fileid) (terpri D-fileid)
      (print D-MORPHOLOGYONLY D-fileid) (terpri D-fileid)
      (print D-LCATEGORIES D-fileid) (terpri D-fileid)
      ;; for Bay Area Unification
      (print (D-BuildGrammarDiscrimminationNet D-GRAMMAR) D-fileid) 
      (terpri D-fileid)
      (print D-CATEGORYDEFINITIONS D-fileid) (terpri D-fileid)
      ;; for Term Unification
      (print (D-MakeCanMakeTree D-GRAMMAR) D-fileid) (terpri D-fileid)
      (terpri D-fileid)
      (close D-fileid)
      (D-RestoreGlobalEnvironment)
      t
   )
)

(defun D-InitGramGlobals ()
;;
;;  This initialises the global variables so that no previous 
;;  junk can appear in the analyser
;;
   (D-InitReader)
   (setq D-GRAMMAR nil)
   (setq D-ALIASES nil)
   (setq D-FEATURES nil)
   (setq D-VARIABLES nil)
   (setq D-CATVALFEAT nil)      ;; category-valued features
   (setq D-WHEAD nil)             ;; head features
   (setq D-WDAUGHTER nil)         ;; daughter features
   (setq D-MORPHOLOGYONLY nil)    ;; morphologically unique  features
   (setq D-FSD nil)               ;; feature specification defaults
   (setq D-INCLUDESTACK nil)     ;; initialise list of include fileids
   (setq D-DISTINGUISHEDCATEGORY 'UNSET) ;; top category in grammar
   (setq D-LASTSYMBOL nil)       ;; no previous symbol
   (setq D-LCATEGORIES nil)      ;; lexical category definitions
   (setq D-CATEGORYDEFINITIONS nil)
   (setq D-CANMAKE nil)
)

(defun D-ParseWGram ()
;;
;;  main routine that reads in the feature declarations and the 
;;  the word grammar rules etc
;;  and does the necessary expansion
;;
   (let ()
      (D-MustHave (DK-Declarations))
	 (D-ParseDeclarations)
      (D-MustHave (DK-Grammar))
      (D-while (not (eq D-CurrentSym 'EOF))
	 (setq D-GRAMMAR
	    (append
	       (D-ConvGrammar (D-ParseGrammarRule))
	       D-GRAMMAR))
      )
   )
)

(defun D-ParseGrammarRule ()
;;
;;   Parses a grammar rule
;;   returns a rule in the form of 
;;     (name mother_cat daughter1 daughter2 ....)
;;
;;   Syntax is now (Name mother -> daughter1 , daughter2 , ...)
;;
   (let ( (name nil) (lhs nil) (children nil) )
      (D-MustHave D-LEFTPAREN)
	 (setq name D-CurrentSym) (D-GetNextSym)
	 (setq lhs (D-ParseCategory))
	 (D-MustHave '-)
	 (D-MustHave '>)
	 (setq children (cons (D-ParseCategory) children));;compulsory daughter
	 (D-while (eq D-CurrentSym D-COMMA)
	    (D-MustHave D-COMMA)
	    (setq children
	       (cons (D-ParseCategory) children))
         )
         (D-MustHave D-RIGHTPAREN)
         (cons name (cons lhs (nreverse children)))
   )
)

(defun D-ConvGrammar (rule)
;;
;;  This converts a grammar rules to one or more rules of
;;  the correct format.  Category Variables are expanded
;;  increasing the number of rules, and then aliases are
;;  expanded.  Note that feature value variables are left
;;  asis.  In addition to the rule itself  flags are set
;;  to say whether this rule contains feature variables or 
;;  not
;;
   (let ()
      (setq D-CATCONTAINSVAR nil)
      (princ "Rule ") (princ (car rule))
      (terpri)
      (D-MarkVarRule
         (D-AddUniqueRuleName      ;; add the rule number onto each 
            (car rule)         ;; rule name
            (D-SubsAliases     ;; remove aliases
               (D-ExpandRule   ;; remove variables
                  (cdr rule)    ;; the mother and daughters in a rule
               )
            )
	 )
      )
   )
)

(defun D-MarkVarRule (rules)
;;
;;   Adds info to say if this rule contains variables.
;;   If so uniquify variables is used on it during parsing 
;;   otherwise that part can be skipped
;;   Containing variables was checked by the substituting alias
;;   function.  The global D-CATCONTAINSVAR will be set to a t or false
;;   depending if the rule. contain variables or not
;;
;;   Note rules is a list rules derived from the same user specified
;;   rule
;;
   (mapcar
      #'(lambda (basicrule)
	 (cons 
	    D-CATCONTAINSVAR 
	    basicrule))
      rules)
)

(defun D-AddUniqueRuleName (name rules)
;;
;;  this adds a name to each of the rules in the given list
;;  The name is the concatenation of the given name and a number
;;  The name has to be unique to allow the left recursion check 
;;  to work in the chart parser.  But the name is still readable
;;  be mere mortals
;;
;;  This will probably not work if there are more than 1024 rules
;;  because of the way ints are represented.
;;
   (setq D-NUMBER 0)
   (cond
      ((equal (length rules) 1)   ;; be nice if only one rule
	 (ncons
	    (cons name (car rules)))
      )
      (t
	 (mapcar
	    #'(lambda (rule)
	       (setq D-NUMBER (add1 D-NUMBER))
	       (cons (concat name D-NUMBER) rule)
              )
            rules
         )
      )
   )
)

(defun D-SubsAliases (rulelist)
;;
;;   Thia function replaces all occurences of alises with their
;;   actuak values
;;
   (mapcar
      #'D-SubsAliasRule     ;; deal with each rule in turn
      rulelist
   )
)

(defun D-SubsAliasRule (rule)
;;
;;  Replaces all ocurrences of aliases with their values in a
;;  grammar rule
;;
   (cond
      ((null rule)
	 nil   ; end of list
      )
      (t
	 (cons
	    (D-CheckCategoryAndReturnIt;; check feature values
	       (D-SubsAliasCategory    ;; simple category
	          (car rule)
	       )
            )
	    (D-SubsAliasRule
	       (cdr rule)
            )
         )
      )
   )
)

(defun D-CheckCategoryAndReturnIt (category)
;;
;;   checks to see if the category is valid and returns it if it is 
;;   otherwise prints an error
;;
   (cond
      ((D-CheckCategory category)
	 category
      )
      (t      ;; invalid category in rule
	 (D-FindCurrentLine)
	 (error "invalid category in rule")
      )
   )
)

(defun D-ExpandRule (rule)
;;
;;   This expands a rule to remove all occurences of any variables.
;;   As variables are met in the rule they become bound to each
;;   possible value in the variable range, ie the variables
;;   are consistant within a rule 
;;   Returns a list of expanded rules
;;   note only category variables are expanded
;;
   (mapcar 
      #'car            ;; not the bindings
      (D-ExpandEachCategory
         rule 
         nil             ;; bindings
      )
   )
)

(defun D-ExpandEachCategory (catlist bindings)
;;
;;  expanded each category to list of categories containing no
;;  variables
;;
   (cond
      ((equal (length catlist) 1)
	 (mapcar
	    #'(lambda (expandedcat)
		  (cons
		     (ncons (car expandedcat))  ;; the expanded category
		     (cdr expandedcat)))        ;; the bindings
            (D-ExpandCategory (car catlist) nil) ;; no bindings
         )
      )
      (t
	 (let ( (explist (D-ExpandEachCategory (cdr catlist) bindings)) )
	    (mapcan 
	       #'(lambda (expansion)
		  (mapcar
		     #'(lambda (catandbind)
			 (cons
			    (cons (car catandbind) (car expansion))
			    (cdr catandbind) ;; bindings
                         )
                       )
                  (D-ExpandCategory (car catlist) (cdr expansion))))
               explist
            )
         )
      )
   )
)

(defun D-ExpandCategory (category bindings)
;;
;;   This expands a category remove all references to variables,
;;   using the variable bindings given or newly created ones as
;;   variables are found
;;   Returns a list of categories expanded from the original
;;
   (cond
      ((D-AliasP category)
	 (ncons
	    (cons category bindings))
      )
      ((D-DeclaredVariableP category)    ;; if variable on its own
	 (let ( (value (D-GetBindings category bindings)) )
	    (cond
	       ((and (eq value 'D-UNBOUND) 
		     (not (eq (DK-category) (D-GetDeclVarRange category))))
		  (mapcar
		     #'(lambda (cat)
			   (cons
			      cat
			      (cons  ;; new bindings
			         (list category cat)
			         bindings)))
		     (D-GetDeclVarRange category)
                  )
               )
	       (t       ;; is already bound or a is feature value variable
		  (ncons
		     (cons value bindings)
                  )
               )
            )
         )
      )
      ((not (listp category))  ;; its an atom but not an alias or a variable
	 (D-FindCurrentLine)
	 (error (concat "Unknown symbol " category))
      )
      (t            ;; not a variable category, so check within it
	 (D-ExpandCat category bindings)
      )
   )
)
 
(defun D-ExpandCat (cat bindings)
;;
;;   expands a category returning a list of (newcat bindings)
;;
   (cond
      ((null cat)        ;; end of expansion
	 (ncons
	    (cons cat bindings))
      )
      ((D-DeclaredVariableP (car cat))   ;; is it a category variable
	 (let ( (value (D-GetBindings (car cat) bindings)) )
	    (cond
	       ((eq value 'D-UNBOUND)    ;; variable not bound
		  (mapcan
		     #'(lambda (val)
			(mapcar
			   #'(lambda (rest)
				(cons
				   (cons val (car rest))
				   (cdr rest)))
			   (D-ExpandCat
			      (cdr cat)
			      (cons  ;; new bindings
			         (list (car cat) val)
			         bindings))))
                     (D-GetDeclVarRange (car cat)))
               )
	       (t     ;; variable is already bound
		  (mapcar
		     #'(lambda (rest)
			  (cons
			     (cons value (car rest))
			     (cdr rest)))
		  (D-ExpandCat
	             (cdr cat) bindings))
               )
            )
         )
      )
      (t         ;; car is not a variable
        (mapcar
           #'(lambda (rest)
	        (cons
	           (cons (car cat) (car rest))
	           (cdr rest)))
           (D-ExpandCat (cdr cat) bindings))
      )
   )
)

(defun D-GetBindings (var bindings)
;;
;;   This gets the current binding of a variable 
;;
   (let ( (value (assq var bindings)) )
      (cond
	 ((null value) 'D-UNBOUND)
	 (t
	    (cadr value)
         )
      )
   )
)

(defun D-VarBoundP (variable bindings)
;;
;;  returns non-nil if variable is bound, nil if unbound
;;
   (assoc variable bindings)
)

