# include <stdio.h>				/* kwic.c (rev3.7) */
# include <ctype.h>
# include <signal.h>

usage()			/* print usage and synopsis of options */
{
	puts("Key Word In Context concordance program\t\t(rev3.7)");
	puts("Usage: kwic [-kn -m -wS -fn -r -ln -pn -ic -cn -dF + -] filename(s)");
	puts("-kn: keyword is n characters long (defaults to 15)");
	puts("-m : keywords not mapped from upper to lower case");
	puts("-wS: write string S onto id field (use quotes around blanks)");
	puts("-fn: filename (up to n characters) written onto id field");
	puts("-r : reset linenumber to 1 at beginning of every file");
	puts("-ln: line numbering begins with line n (instead of 1)");
	puts("-pn: page numbering begins with page n (instead of 1)");
	puts("-ic: page incrementer is character c (defaults to =)");
	puts("-cn: context is n characters long (defaults to 50)");
	puts("-d : define punctuation set according to file F");
	puts("+  : the + character indicates cedilla or umlaut");
	puts("-  : read text from standard input (terminal or pipe)");
	puts("Kwic will linenumber until first page indicator.");
	exit(1);
}

int kwlen = 15;		/* lefthand keyword length */
int kwmap = 1;		/* toggle mapping of keyword to lcase */
int wrtid = 0;		/* toggle write onto id field */
char *idfld;		/* string to write onto id field */
int wrtfnm = 0;		/* toggle writing of filename */
long lineno = 1;	/* count line numbers */
int resetno = 0;	/* reset lineno for new file */
char pgincr = '=';	/* page incrementing character */
int pageno = 0;		/* toggle and count page numbers */
int cntxt = 50;		/* context width around keyword */
int plusm = 0;		/* toggle plus sign as accent mark */
char punct[BUFSIZ/4] 	= ",.;:-?!\"()[]{}" ;
char zerowid[BUFSIZ/4]	= "+" ;

main(argc, argv)	/* Key Word In Context concordance program */
int argc;
char *argv[];
{
	FILE *fp, *fopen();
	int i;

	if (argc == 1)
		usage();

	for (i = 1; i < argc; i++)
	{
		if (*argv[i] == '-')
			getflag(argv[i]);
		else if (*argv[i] == '+')
			plusm = 1;
		else if ((fp = fopen(argv[i], "r")) != NULL)
		{
			kwic(fp, argv[i]);
			fclose(fp);
			if (resetno)
				lineno = 1;
		}
		else
		{
			fprintf(stderr,
			"Kwic cannot access the file: %s\n", argv[i]);
			continue;
		}
	}
	exit(0);
}

getflag(f)		/* parses command line to set options */
char *f;
{
	char *pfile;
	long atol();

	f++;
	switch(*f++)
	{
		case 'k':
			kwlen = atoi(f);
			break;
		case 'm':
			kwmap = 0;
			break;
		case 'w':
			wrtid = 1;
			idfld = f;
			break;
		case 'f':
			wrtfnm = atoi(f);
			break;
		case 'r':
			resetno = 1;
			break;
		case 'l':
			lineno = atol(f);
			break;
		case 'p':
			pageno = atoi(f);
			break;
		case 'i':
			pgincr = *f;
			break;
		case 'c':
			cntxt = atoi(f);
			break;
		case 'd':
			pfile = f;
			getpunct(pfile);
			break;
		case NULL:
			rd_stdin();
			break;
		default:
			fprintf(stderr,
			"Invalid kwic flag: -%s\n", --f);
			exit(1);
			break;
	}
}

getpunct(pfile)		/* read user's punctuation from pfile */
char *pfile;
{
	FILE *pfp, *fopen();
	char s[BUFSIZ/4], *strcpy();

	if ((pfp = fopen(pfile, "r")) == NULL)
	{
		fprintf(stderr,
		"Kwic cannot access Punctfile: %s\n", pfile);
		exit(1);
	}
	if (fgets(s, BUFSIZ/4, pfp))
	{
		strcpy(punct, s);
		punct[strlen(punct)-1] = NULL;
	}
	if (fgets(s, BUFSIZ/4, pfp))
	{
		strcpy(zerowid, s);
		zerowid[strlen(zerowid)-1] = NULL;
		plusm = 1;
	}
}

char *tempfile;		/* file storage for seeking stdin */

rd_stdin()		/* create tempfile with standard input */
{
	FILE *tfp, *fopen();
	char s[BUFSIZ], *mktemp();
	int catch();

	tempfile = "/tmp/KwicXXXXX";
	mktemp(tempfile);
	tfp = fopen(tempfile, "w");
	if (signal(SIGINT, SIG_IGN) != SIG_IGN)
		signal(SIGINT, catch);
	while (fgets(s, BUFSIZ, stdin))
		fputs(s, tfp);
	fclose(tfp);
	tfp = fopen(tempfile, "r");
	kwic(tfp, "Stdin");
	unlink(tempfile);
}

catch()			/* remove tempfile in case of interrupt */
{
	unlink(tempfile);
	fprintf(stderr, "\nInterrupt\n");
	exit(1);
}

kwic(fp, fname)		/* prints kwic entry for each word */
FILE *fp;
char *fname;
{
	char word[BUFSIZ], str[BUFSIZ];
	long pos, ftell();
	register int wdlen, i;

	while (wdlen = getword(word, fp))
	{
		pr_keywd(word);		/* lefthand keyword */
		pr_idfld(fname);	/* identificaton field */

		pos = ftell(fp);		/* where in text? */
		if (pos < cntxt/2 + wdlen + 2)	/* if at beginning */
		{
			fseek(fp, -(pos), 1);
			for (i = 0; i < cntxt/2 - pos + wdlen + 2; i++)
				putchar(' ');
			load_str(str, (int)(pos - wdlen - 1), 1, fp);
		}
		else			/* if past beginning of text */
		{
			fseek(fp, (long) -(cntxt/2 + wdlen + 2), 1);
			load_str(str, (cntxt/2 + 1), 0, fp);
		}
		printf("%s|", str);	/* print out first half */

		end_entry(fp);		/* print out second half */

		fseek(fp, pos, 0);	/* back to original location */
		putchar('\n');		/* end concordance entry */
	}
}

pr_keywd(word)		/* print keyword, adjusting for backspaces */
char word[];
{
	char *cp, *index();
	register int i, kwbsno = 0;

	cp = word;

	for (i = 0; i < kwlen; i++)
	{
		if (*cp)
		{
			if (*cp == '\b')
				kwbsno += 2;
			if (plusm && index(zerowid, *cp))
				kwbsno++;
			putchar(*cp++);
		}
		else
			putchar(' ');
	}
	while (kwbsno-- > 0)
		putchar(' ');
	putchar('|');
}


pr_idfld(fname)		/* print specified id fields and numbering */
char *fname;
{
	char *wfile;
	int i;

	if (wrtid)
		printf("%s ", idfld);
	if (wrtfnm)
	{
		wfile = fname;
		for (i = 0; i < wrtfnm; i++)
			if (*wfile)
				putchar(*wfile++);
			else
				putchar(' ');
		putchar(' ');
	}
	if (pageno)
		printf("%3d,%2ld|", pageno, lineno);
	else
		printf("%6ld|", lineno);
}

load_str(str, max, prnt, fp)	/* load first half onto string */
char str[];
int max, prnt;
FILE *fp;
{
	int i, bsno = 0;
	int c;

	for (i = 0; i < max; i++)
	{
		c = getc(fp);
		if (!prnt)
		{
			str[i] = ' ';
			prnt = isspace(c);
		}
		else	/* if (prnt) */
		{
			if (c == '\n')
				str[i] = '/';
			else if (c == '\t')
				str[i] = ' ';
			else
				str[i] = c;

			if (c == '\b')
				bsno += 2;
			if (plusm && index(zerowid, c))
				bsno++;
		}
	}
	str[i] = NULL;
	while (bsno-- > 0)	/* adjust for backspaces */
		putchar(' ');
}

end_entry(fp)		/* read and write to end of entry */
FILE *fp;
{
	register int i, c;

	for (i = 0; i < cntxt/2 - 2; i++)
	{
		c = getc(fp);
		if (c == '\n')
			c = '/';
		if (c == '\t')
			c = ' ';
		if (c == EOF)
			return;
		putchar(c);
	}
	while (!isspace(c = getc(fp)) && c != EOF)
		putchar(c);
}

getword(word, fp)	/* drives program through text word by word */
char word[];
FILE *fp;
{
	static char nl = 0;
	register int wlen = 1;

	if (nl)		/* increments lineno at beginning of line */
	{
		nl = 0;
		lineno++;
	}
	while ((*word = getc(fp)) && isskip(*word) && *word != EOF)
		if (*word == '\n')	/* skip to text */
			lineno++;

	if (*word == pgincr)
	{
		pageno++;	/* pgincr must begin word */
		lineno = 0;
	}
	if (*word == EOF)
		return(NULL);
	if (kwmap)
		if (isupper(*word))
			*word = tolower(*word);

	while ((*++word = getc(fp)) && !isskip(*word) && *word != EOF)
	{				/* get next word */
		if (kwmap)
			if (isupper(*word))
				*word = tolower(*word);
		wlen++;
	}
	if (*word == '\n')	/* set nl at end of line */
		nl = 1;
	*word = NULL;

	return(wlen);
}

isskip(c)		/* function to evaluate punctuation */
char c;
{
	char *ptr;

	if (isspace(c))
		return(1);
	for (ptr = punct; *ptr != c && *ptr != NULL; ptr++)
		;
	if (*ptr == NULL)
		return(0);
	else
		return(1);
}
