# include <stdio.h>			/* freq.c (rev3.7) */
# include <ctype.h>

struct tnode		/* binary tree for word and count */
{
	char *word;
	int count;
	struct tnode *left;
	struct tnode *right;
};

char punctuation[BUFSIZ] = ",.;:-?!\"()[]{}" ;

long int total = 0;		/* total number of words */
long int different = 0;		/* number of different words */
char numfreq = 0;		/* toggle for numerical freq */
char nomap = 0;			/* do not map to lower case */

usage()			/* print proper usage and exit */
{
	puts("Usage: freq [-n -m -dF -] filename(s)\t\t(rev3.7)");
	puts("-n: list words in numerical order of frequency");
	puts("-m: disable mapping of words to lower case");
	puts("-d: define punctuation set according to file F");
	puts("- : read standard input instead of files");
	exit(1);
}

main(argc, argv)	/* tabulate word frequencies of a text */
int argc;
char *argv[];
{
	FILE *fopen(), *fp;
	struct tnode *root, *tree();
	char word[BUFSIZ];
	int i;

	if (argc == 1)
		usage();
	root = NULL;			/* initialize tree */
	for (i = 1; i < argc; i++)
	{
		if (*argv[i] == '-')
			getflag(argv[i]);
		else if ((fp = fopen(argv[i], "r")) != NULL)
		{
			while (getword(word, fp))
			{
				++total;
				root = tree(root, word);
			}
			fclose(fp);
		}
		else  /* attempt to open file failed */
		{
			fprintf(stderr,
			"Freq cannot access the file: %s\n", argv[i]);
			exit(1);
		}
	}
	if (numfreq)			/* print results */
		treesort(root);
	else
		treeprint(root, stdout);

	printf("------------------------------\n");
	printf("%5ld  Total number of words\n", total);
	printf("%5ld  Different words used\n", different);
	exit(0);
}

getflag(f)		/* parses command line to set options */
char *f;
{
	char *pfile, word[BUFSIZ];
	struct tnode *root, *tree();

	f++;
	switch(*f++)
	{
		case 'n':
			numfreq = 1;
			break;
		case 'm':
			nomap = 1;
			break;
		case 'd':
			pfile = f;
			getpunct(pfile);
			break;
		case NULL:
			root = NULL;
			while (getword(word, stdin))
			{
				++total;
				root = tree(root, word);
			}
			if (numfreq)
				treesort(root);
			else
				treeprint(root, stdout);
			break;
		default:
			fprintf(stderr,
			"Invalid freq flag: -%s\n", --f);
			exit(1);
			break;
	}
}

getpunct(pfile)		/* read user's punctuation from pfile */
char *pfile;
{
	FILE *pfp, *fopen();
	char s[BUFSIZ], *strcpy();

	if ((pfp = fopen(pfile, "r")) == NULL)
	{
		fprintf(stderr,
		"Freq cannot access Pfile: %s\n", pfile);
		exit(1);
	}
	else
		while (fgets(s, BUFSIZ, pfp))
			strcpy(punctuation, s);
}

getword(word, fp)	/* drives program through text word by word */
char word[];
FILE *fp;
{
	while ((*word = getc(fp)) && isskip(*word) && *word != EOF)
		;
	if (*word == EOF)
		return(0);
	if (!nomap && isupper(*word))
		*word = tolower(*word);

	while ((*++word = getc(fp)) && !isskip(*word) && *word !=EOF)
	{
		if (!nomap && isupper(*word))
			*word = tolower(*word);
	}
	*word = NULL;
	return(1);
}

isskip(c)		/* function to evaluate punctuation */
char c;
{
	char *ptr;

	if (isspace(c))
		return(1);
	for (ptr = punctuation; *ptr != c && *ptr != NULL; ptr++)
		;
	if (*ptr == NULL)
		return(0);
	else
		return(1);
}

struct tnode *tree(p, w)	/* build tree beginning at root */
struct tnode *p;
char *w;
{
	struct tnode *talloc();
	char *strsave();
	int cond;

	if (p == NULL)
	{
		p = talloc();
		p->word = strsave(w);
		p->count = 1;
		p->left = p->right = NULL;
	}
	else if ((cond = strcmp(w, p->word)) == 0)
		p->count++;
	else if (cond < 0)
		p->left = tree(p->left, w);
	else /* if cond > 0 */
		p->right = tree(p->right, w);
	return(p);
}

treesort(p)		/* sort contents of binary tree and print */
struct tnode *p;
{
	FILE *pfp, *popen();

	pfp = popen("sort +0rn -1 +1", "w");
	if (p != NULL)
		treeprint(p, pfp);
	pclose(pfp);
}

treeprint(p, fp)	/* write tree onto fp file stream */
struct tnode *p;
FILE *fp;
{
	if (p != NULL)
	{
		treeprint(p->left, fp);
		fprintf(fp, "%5d  %s\n", p->count, p->word);
		++different;
		treeprint(p->right, fp);
	}
}

struct tnode *talloc()		/* core allocator for tree */
{
	struct tnode *p;
	char *malloc();

	if ((p = ((struct tnode *)malloc(sizeof(struct tnode)))) != NULL)
		;  /* will return */
	else /* if (p == NULL) */
		overflow();
	return(p);
}

char *strsave(s)	/* allocate space for string of text */
char *s;
{
	char *p, *malloc();

	if ((p = malloc((unsigned)(strlen(s)+1))) != NULL)
		strcpy(p, s);
	else /* if (p == NULL) */
		overflow();
	return(p);
}

overflow()		/* exit gracefully in case of core overflow */
{
	fprintf(stderr,
	"Freq: no more core available (maximum on PDP 11/70 is 64K bytes).\n");
	fprintf(stderr,
	"You might try: dissolve filename(s) | sort | uniq -c\n");
	exit(1);
}
