(in-package cl-lib)

;;; ****************************************************************
;;; String I/O Streams  ********************************************
;;; ****************************************************************
;;;
;;; This is the string I/O Streams package written May 1994 by 
;;;   Bradford W. Miller
;;;   miller@cs.rochester.edu
;;;   University of Rochester, Department of Computer Science
;;;   610 CS Building, Comp Sci Dept., U. Rochester, Rochester NY 14627-0226
;;;   716-275-1118
;;; I will be glad to respond to bug reports or feature requests.
;;;
;;; This version was NOT obtained from the directory
;;; /afs/cs.cmu.edu/user/mkant/Public/Lisp-Utilities/initializations.lisp
;;; via anonymous ftp from a.gp.cs.cmu.edu. (you got it in cl-lib).
;;;
;;; Bug reports, improvements, and feature requests should be sent
;;; to miller@cs.rochester.edu. Ports to other lisps are also welcome.
;;;
;;; Copyright (C) 1994 by Bradford W. Miller, miller@cs.rochester.edu 
;;;                       and the Trustees of the University of Rochester
;;; All rights reserved.
;;; Right of use & redistribution is granted for non-commercial use as 
;;; per the terms of the GNU LIBRARY GENERAL PUBLIC LICENCE version 2 which is
;;; incorporated here by reference. Contact the author for commercial 
;;; use or distribution arrangements.
;;;
;;; This library is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU Library General Public License for more details.
;;;
;;; You should have received a copy of the GNU Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;

;; this depends on the allegro process and stream stuff, but should be simple to port.

;;; A common need for streams is to act as a delay, and for IPC. This package implements a simple single-writer 
;;; multiple-reader stream for characters. The producer and each consumer is expected to be in their own process,
;;; that is, have a unique binding of mp:*current-process*.

(defclass-x string-io-stream (stream:fundamental-input-stream stream:fundamental-output-stream stream:fundamental-character-stream)
  ((buffer :initform (make-array '(20) :fill-pointer 0 :adjustable t :element-type 'character :initial-element #\space) :type vector :accessor sios-buffer)
   (objects :initform nil :type list :accessor sios-objects)
   (proc-pointer-alist :initform nil :type alist :accessor sios-proc-pointer-alist)
   (writer-proc :initform mp:*current-process* :reader sios-writer-proc :initarg :writer-proc)
   (eof-p :initform nil :type symbol :accessor sios-eof-p))
  )

(defun get-posn (stream)
  (cond
   ((verify-writer stream nil)
    (fill-pointer (sios-buffer stream)))
   ((cdr (assoc MP:*CURRENT-PROCESS* (sios-proc-pointer-alist stream))))
   ;; new reader
   (t
    (update-alist mp:*current-process* 0 (sios-proc-pointer-alist stream))
    0)))

(defsetf get-posn (stream) (new-posn)
  `(setf (cdr (assoc MP:*CURRENT-PROCESS* (sios-proc-pointer-alist ,stream))) ,new-posn))

(defun verify-writer (stream error-p)
  (cond
   ((equal mp:*current-process* (sios-writer-proc stream)))
   (error-p
    (error "Attempt to write to string-io-stream by a reader proc"))))

(defmethod stream:stream-read-char ((stream string-io-stream))
  (let ((posn (get-posn stream)))
    (labels ((input-available (wait-fn)
               (declare (ignore wait-fn))
               (or (sios-eof-p stream)
                   (< posn (fill-pointer (sios-buffer stream))))))
      (while-not (input-available nil)
        (mp:wait-for-input-available 
         stream 
         :whostate "Stream Char Wait"
         :wait-function #'input-available)))
    (if (>= posn (fill-pointer (sios-buffer stream))) ; must have been at :eof, else we'd still be waiting.
        :eof
      (prog1 (aref (sios-buffer stream) posn)
        (incf (get-posn stream))))))

(defmethod stream:stream-unread-char ((stream string-io-stream) char)
  (let ((posn (1- (get-posn stream))))
    (unless (eql (aref (sios-buffer stream) posn) char)
      (error "Attempt to unread char ~C, but last character read was ~C." char (aref (sios-buffer stream) posn)))
    (setf (get-posn stream) posn)))

(defmethod stream:stream-read-char-no-hang ((stream string-io-stream))
  (if (or (sios-eof-p stream)
          (stream:stream-listen stream))
      (stream:stream-read-char stream)))

(defmethod stream:stream-listen ((stream string-io-stream))
  (<= (get-posn stream) (fill-pointer (sios-buffer stream))))

(defmethod stream:stream-write-char ((stream string-io-stream) character)
  (verify-writer stream t)
  (vector-push-extend character (sios-buffer stream)))

(defmethod stream:stream-terpri ((stream string-io-stream))
  (stream:stream-write-char stream #\newline))

(defmethod stream:stream-line-column ((stream string-io-stream))
  (let* ((posn (get-posn stream))
         (rel-posn 0)
         (rel-ptr (1- posn)))
    (cond
     ((zerop posn)
      0)
     (t
      (while-not (zerop rel-ptr)
        (when (eql (aref (sios-buffer stream) rel-ptr) #\newline)
          (return-from stream:stream-line-column rel-posn))
        (incf rel-posn)
        (decf rel-ptr))
      rel-posn))))

(defmethod stream:stream-start-line-p ((stream string-io-stream))
  (zerop (stream:stream-line-column stream)))

(defmethod stream:stream-write-string ((stream string-io-stream) string &optional start end)
  (unless start
    (setq start 0))
  (unless end
    (setq end (length string)))
  (verify-writer stream t)
  (let* ((posn (fill-pointer (sios-buffer stream)))
         (new-fill (- (+ posn end) start))
         (temp start)
         (needed-increase (- new-fill (array-dimension (sios-buffer stream) 0))))
    (when (plusp needed-increase)
      (adjust-array (sios-buffer stream) (list new-fill)
                    :element-type 'character))
    (while (< posn new-fill)
      (setf (elt (sios-buffer stream) posn) (elt string temp))
      (incf posn)
      (incf temp))
    (setf (fill-pointer (sios-buffer stream)) posn))
  string)

(defmethod stream:stream-read-line ((stream string-io-stream))
  (let* ((posn (get-posn stream))
         next-newline)
    (labels ((input-available (wait-fn)
               (declare (ignore wait-fn))
               (or (sios-eof-p stream)
                   (setq next-newline (position #\newline (sios-buffer stream) :start posn)))))
      (while-not (input-available nil)
        (mp:wait-for-input-available 
         stream 
         :whostate "Stream Line Wait"
         :wait-function #'input-available)))
    (cond
     (next-newline
      (incf next-newline)
      (setf (get-posn stream) next-newline)
      (values (subseq (sios-buffer stream) posn next-newline) nil))
     (t
      (setf (get-posn stream) (fill-pointer (sios-buffer stream)))
      (values (subseq (sios-buffer stream) posn) t)))))

;; eventually want to specialize these to make protocol more efficient.

;;(defmethod stream:stream-read-sequence ((stream string-io-stream) sequence &optional start end)
;;  )

;;(defmethod stream:stream-write-sequence ((stream string-io-stream) sequence &optional start end)
;;  )

(defmethod close ((stream string-io-stream) &key abort)
  (declare (ignore abort))
  (when (verify-writer stream nil)
    (setf (sios-eof-p stream) t)
    (setf (sios-writer-proc stream) (list :closed (sios-writer-proc stream))))
  (call-next-method))

