;;; -*- Mode: LISP; Syntax: Ansi-common-lisp; Package: CL-LIB; Base: 10 -*-
(in-package cl-lib)

;;; ****************************************************************
;;; Triangular Matrices ********************************************
;;; ****************************************************************
;;;
;;; This is the triangular matrices package written January 1994 by 
;;;   Bradford W. Miller
;;;   miller@cs.rochester.edu
;;;   University of Rochester, Department of Computer Science
;;;   610 CS Building, Comp Sci Dept., U. Rochester, Rochester NY 14627-0226
;;;   716-275-1118
;;; I will be glad to respond to bug reports or feature requests.
;;;
;;; This version was NOT obtained from the directory
;;; /afs/cs.cmu.edu/user/mkant/Public/Lisp-Utilities/initializations.lisp
;;; via anonymous ftp from a.gp.cs.cmu.edu. (you got it in cl-lib).
;;;
;;; Bug reports, improvements, and feature requests should be sent
;;; to miller@cs.rochester.edu. Ports to other lisps are also welcome.
;;;
;;; Copyright (C) 1994 by Bradford W. Miller, miller@cs.rochester.edu 
;;;                       and the Trustees of the University of Rochester
;;; All rights reserved.
;;; Right of use & redistribution is granted for non-commercial use as 
;;; per the terms of the GNU LIBRARY GENERAL PUBLIC LICENCE version 2 which is
;;; incorporated here by reference. Contact the author for commercial 
;;; use or distribution arrangements.
;;;
;;; This library is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU Library General Public License for more details.
;;;
;;; You should have received a copy of the GNU Library General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
;;;

;;; ********************************
;;; Motivation *********************
;;; ********************************

;;; Twofold. 1) using only the upper or lower portion of a multi-d matrix wastes 1/2 the space. That's a minor
;;; consideration. 2) increasing the size of a matrix by 1 (e.g. using adjust-array) potentially copies the entire 
;;; array to a new location, which is O(n**axes) with n being the length of the axis (triangular arrays are "square")
;;; The following  algorithm is O(n**(axes-1)). 
;;; That is, for a two-d array it's O(n) instead of O(n**2). Time to access or write 
;;; an element is still O(1) <for an existing array>, but the constant factor is larger than normal arrays, because 
;;; an extra level of indirection is involved.
;;;
;;; Based on an algorithm I came up with on 12/29/93 while taking a shower.

;;; ********************************
;;; Description ********************
;;; ********************************

;;; Instead of relying on adjust-array, we have increment-array to add a new row (column) to a upper/lower triangular matrix
;;; We also supply access (defsetf'd) functions to access triangular arrays.

;;; The basic idea is this.

;;; an axis of a triangular array is a vector, each element of which is another vector. For a "square" upper triangular matrix, the
;;; 0th entry will have length 1, the next entry length 2, etc. I.e. the size is the column j +1.

;;; to add a new column, then, we can just add a new slot to the axis vector (potentially copying it, an O(n) operation). and cons
;;; up a new column vector, which will be length n, another O(n) operation. Accessing is a two-step operation, and row-major-aref
;;; won't work with triangular arrays because they aren't linear in memory! We supply a copy function, however.

;;; The package supports 2d arrays, though the algorithm can be extended to any n-dimensions, just not this implementation.
;;; For upper dimensions, we need to store the extra dimension vectors in the tarray structure.

;;; Visualization 
;;; each string xxx represents a vector (array) in the upper triangle, each string yyy is a vector (array)
;;; in the lower triangle. The lengths show
;;; connectivity is illustrated by length of the string. 0,0 is in the upper triangle. 2,0 and 2,1 are in the 
;;; same vector of the lower triangle.

;;;    0    1    2    3    4    5 x-axis
;;;
;;; 0  x    x    x    x    x    x
;;;         x    x    x    x    x
;;; 1  y    x    x    x    x    x
;;;              x    x    x    x
;;; 2  yyyyyy    x    x    x    x
;;;                   x    x    x
;;; 3  yyyyyyyyyyy    x    x    x
;;;                        x    x
;;; 4  yyyyyyyyyyyyyyyy    x    x
;;;                             x
;;; 5  yyyyyyyyyyyyyyyyyyyyy    x
;;; y axis

;;; the algorithm (but not this code) works just as well in n-d. For 3-d, we would add planes (2d arrays) on each side of our
;;; starting cube each time we increment the axis maximum. We also need to add another index vector. Viewed from the "front",
;;; i.e. x/y axis, our planes would line up like the above diagram. From the top, we'd see:

;;;    0    1    2    3    4    5 z-axis
;;;
;;; 0  x    z    z    z    z    z
;;;              z    z    z    z
;;; 1  xxxxxx    z    z    z    z
;;;                   z    z    z
;;; 2  xxxxxxxxxxx    z    z    z
;;;                        z    z
;;; 3  xxxxxxxxxxxxxxxx    z    z
;;;                             z
;;; 4  xxxxxxxxxxxxxxxxxxxxx    z
;;;                             
;;; 5  xxxxxxxxxxxxxxxxxxxxxxxxxx
;;; x axis

;;; so the main extension to this implementation for higher order N would be to 
;;;  a) change the tarray defstruct to make the "upper" and "lower" vector slots a list of vectors, one for each dimension.
;;;  b) change make-tarray to build each of the d arrays instead of just 2
;;;  c) change the logic in taref to correctly figure out which (d-1) array a point is located in, and find it via the vectors in a).
;;;  d) change the subroutines called in b) to correctly build multi-d arrays of the right size.

;;; so, the notion of "upper" and "lower" only works with 2d arrays, for n-d arrays, there are n senses of where a point is located.
;;; it obviously isn't practical to export all of them as separately creatable "slices" of an n-d array (though if an appropriate automatic
;;; naming scheme were arrived at, it might be possible). Also so generifying all the code below makes the 2d case slower, and that's
;;; the only case I care about right now :-).

;;; Using this code with >2 dimensions will work, but you'll only get the "pyramid"/ upper-d object corresponding to an upper triangle of
;;; an array, i.e. it won't fill in the rest of the matrix with the other slices. e.g. for a (3 3 3) triangular matix,
;;; you'll have slots (0,0,0) but not (0,0,1), and (1,1,1), but not (1,1,2) etc. Think of it as a pyramid. (It's not what you want, of
;;; course, but see above comments on extending this implementation for the full algorithm.

;;; ********************************
;;; HOW TO USE *********************
;;; ********************************

;;; As cltl/2 array like as possible, but displacement and fill-pointers are not supported.
;;; the type may be specialized or general (i.e. you get to supply an element type). All triangular matrices are "adjustable" but only
;;; via increment-array.
;;; Type is tarray; for the full (upper+lower) version, upper-tarray for i,j i>=j. lower-tarray
;;; for i,j i<j. Note that lower triangular+upper triangualr + full triangular, since only upper-triangular arrays allow i=j.

;;; make-tarray dimensions &key :element-type :initial-element :initial-contents 
;;; the supported arguments have the same meaning as in CLtL/2's definition of make-array. Note that this will construct
;;; a pair of triangular arrays, upper and lower, but they will have the nice updating properties described above.
;;; Attempting to create a 1 dimensional triangular array will just create a vector. There is no advantage over using make-array or vector,
;;; and supplying :adjustable t.

;;; make-upper-tarray dimensions &key :element-type :initial-element :initial-contents 
;;; see above, upper triangle only.

;;; make-lower-tarray dimensions &key :element-type :initial-element :initial-contents 
;;; see above, lower triangle only.

;;; taref tarray &rest subscripts
;;; like aref, but works on triangular arrays. It **is an error** to attempt to access the lower-triangle of an upper triangular array, and
;;; vice-versa. A "full" triangular array (from make-tarray) has both, so it works.

;;; if you aren't sure if your subscripts are valid arguments to taref for a (upper/lower) triangular array, use
;;; tarry-in-bounds-p to check them.

;;; the following tarry-<fn> are just like array-<fn> in CLtL2.
;;; tarray-element-type
;;; tarray-rank 
;;; tarray-dimension
;;; tarray-dimensions
;;; tarray-total-size (note, the total size of the triangle, i.e. about 1/2 from the product of the dimensions for upper or lower tas.)
;;; tarray-in-bounds-p
;;; 
;;; increment-tarray tarray :initial-element :initial-contents
;;; like adjust-array, but will only add 1 to (all of) the array dimensions. If triangular array 
;;; is upper or lower only, then the right thing happens.

;;; The following to the obvious
;;; upper-tarray-p
;;; lower-tarray-p
;;; full-tarray-p
;;; tarray-p
;;; copy-tarray (takes an optional argument for the copier of the elements, defaults to #'identity)

;;; bits, general array adjustment, row-major-aref etc. are not currently supported.

;;; Note: I only needed this for 2d arrays, it should work for nd
;;; arrays, (and trivially tested) but it's possible that it's broken.


;;; representation issues
(defstruct (tarray (:constructor make-tarray-struct) (:copier nil) (:print-function print-tarray))
  ;; exported slots
  (element-type 't)
  (rank 0 :type fixnum)
  (dimensions nil :type list)
  
  ;; internal only
  (upper-vector nil :type (or null vector))
  (lower-vector nil :type (or null vector))
  )
  

(defun make-tarray (dimensions &key (element-type 't) (initial-element nil iep) (initial-contents nil icp))
  "Create a full triangular array (both upper and lower). Supported options like make-array."
  (progfoo (make-upper-tarray-i dimensions element-type initial-element iep initial-contents icp nil)
    (make-lower-tarray-i dimensions element-type initial-element iep initial-contents icp foo)))

(defun make-upper-tarray (dimensions &key (element-type 't) (initial-element nil iep) (initial-contents nil icp))
  "Make only an upper triangular array. subscripts may be equal, for any subscript ordering (1 .. i .. j .. n)
the value of subscript i must be >= j."
  (make-upper-tarray-i dimensions element-type initial-element iep initial-contents icp nil))

(defun make-lower-tarray (dimensions &key (element-type 't) (initial-element nil iep) (initial-contents nil icp))
  "Make only a lower triangular array. subscripts may NOT be equal, for any subscript ordering (1 .. i .. j .. n)
the value of subscript i must be < j."
  (make-lower-tarray-i dimensions element-type initial-element iep initial-contents icp nil))

(defun taref (tarray &rest subscripts)
  "Like aref on tarrays."
  (if (<= (first subscripts) (second subscripts))
      (taref-upper-tarray tarray subscripts)
    (taref-lower-tarray tarray subscripts)))

(defsetf taref (tarray &rest subscripts) (new-value)
  `(if (<= ,(first subscripts) ,(second subscripts))
       (setf (taref-upper-tarray ,tarray ,@subscripts) ,new-value)
     (setf (taref-lower-tarray ,tarray ,@subscripts) ,new-value)))

(defun tarray-dimension (tarray axis-number)
  "Return the dimension of a particular axis of the tarray. Note that all axes must be equal, so one is as good as another."
  (nth axis-number (tarray-dimensions tarray)))

(defun tarray-total-size (tarray)
  "Return the total size of the passed tarray, including uninitialized cells."
  (cond
   ((full-tarray-p tarray)
    (reduce #'* (tarray-dimensions tarray)))
   ((upper-tarray-p tarray)
    (reduce #'+ (mapcar #'array-total-size (tarray-upper-vector tarray))))
   (t
    (reduce #'+ (mapcar #'array-total-size (tarray-lower-vector tarray))))))

(defun tarray-in-bounds-p (tarray &rest subscripts)
  (and (= (length subscripts) (tarray-rank tarray))
       (every #'< subscripts (tarray-dimensions tarray))
       (cond
        ((full-tarray-p tarray))
        ((upper-tarray-p tarray)
         (let ((temp (car subscripts)))
           (every #'(lambda (sub) (if (<= temp sub) (setq temp sub))) (cdr subscripts))))
        (t                      ; lower tarray.
         (let ((temp (car subscripts)))
           (every #'(lambda (sub) (if (> temp sub) (setq temp sub))) (cdr subscripts)))))))

(defun upper-tarray-p (thingo)
  "True iff thingo is an upper (not full) triangular array."
  (and (tarray-p thingo)
       (not (null (tarray-upper-vector thingo)))
       (null (tarray-lower-vector thingo))))

(defun lower-tarray-p (thingo)
  "True iff thingo is a lower (not full) triangular array."
  (and (tarray-p thingo)
       (not (null (tarray-lower-vector thingo)))
       (null (tarray-upper-vector thingo))))

(defun full-tarray-p (thingo)
  "True iff thingo is a full triangualr array."
  (and (tarray-p thingo)
       (not (null (tarray-upper-vector thingo)))
       (not (null (tarray-lower-vector thingo)))))

(defun increment-tarray (tarray &key (initial-element nil iep) (initial-contents nil icp))
  "Add 1 to the size of all axes of the passed tarray, and initialize as per adjust-array."
  (when (tarray-lower-vector tarray)
    (vector-push-extend (make-array (cdr (tarray-dimensions tarray))
                                    :element-type (tarray-element-type tarray)
                                    (if iep :initial-element :ignore) initial-element
                                    (if icp :initial-contents :ignore) initial-contents
                                    :allow-other-keys t)
                        (tarray-lower-vector tarray))
    (assert (eql (fill-pointer (tarray-lower-vector tarray)) (1+ (car (tarray-dimensions tarray)))) () "Bad increment"))
  (mapl #'(lambda (el) (setf (car el) (incf (car el))))
        (tarray-dimensions tarray))

  (when (tarray-upper-vector tarray)
    (vector-push-extend (make-array (cdr (tarray-dimensions tarray))
                                    :element-type (tarray-element-type tarray)
                                    (if iep :initial-element :ignore) initial-element
                                    (if icp :initial-contents :ignore) initial-contents
                                    :allow-other-keys t)
                        (tarray-upper-vector tarray))
    (assert (eql (fill-pointer (tarray-upper-vector tarray)) (car (tarray-dimensions tarray))) () "Bad increment")))

(defun copy-tarray (tarray &optional (element-copier #'identity))
  (progfoo (make-tarray-struct :element-type (tarray-element-type tarray)
                               :rank (tarray-rank tarray)
                               :dimensions (copy-list (tarray-dimensions tarray)))
    (when (tarray-upper-vector tarray)
      (let ((vector (setf (tarray-upper-vector foo) (make-tarray-vector (tarray-dimensions tarray)))))
        (dotimes (i (car (tarray-dimensions tarray)))
          (setf (aref vector i) (copy-array (aref (tarray-upper-vector tarray) i) element-copier)))))
    (when (tarray-lower-vector tarray)
      (let ((vector (setf (tarray-lower-vector foo) (make-tarray-vector (tarray-dimensions tarray)))))
        (dotimes (i (car (tarray-dimensions tarray)))
          (setf (aref vector i) (copy-array (aref (tarray-lower-vector tarray) i) element-copier)))))))
               
;;; internal functions
                                                    
(defun make-tarray-vector (dimensions)
  (make-array (car dimensions)
              :adjustable t
              :fill-pointer t
              :element-type (list* 'array (make-list (list-length (cdr dimensions)) :initial-element '*))))
              

(defun make-upper-tarray-i (dimensions element-type initial-element iep initial-contents icp reuse-tarray)
  (unless reuse-tarray
    (setq reuse-tarray (make-tarray-struct :element-type element-type :rank (list-length dimensions) :dimensions (copy-list dimensions))))
  
  (let ((vector (setf (tarray-upper-vector reuse-tarray) (make-tarray-vector dimensions))))
    (dotimes (i (car dimensions))
      (setf (aref vector i)
        (make-array (make-list (list-length (cdr dimensions)) :initial-element (1+ i))
                    :element-type element-type
                    (if iep :initial-element :ignore) initial-element
                    (if icp :initial-contents :ignore) initial-contents
                    :allow-other-keys t))))
  reuse-tarray)

(defun make-lower-tarray-i (dimensions element-type initial-element iep initial-contents icp reuse-tarray)
  (unless reuse-tarray
    (setq reuse-tarray (make-tarray-struct :element-type element-type :rank (list-length dimensions) :dimensions (copy-list dimensions))))
  
  (let ((vector (setf (tarray-lower-vector reuse-tarray) (make-tarray-vector dimensions))))
    (dotimes (i (car dimensions))
      (setf (aref vector i)
        (make-array (make-list (list-length (cdr dimensions)) :initial-element i)
                    :element-type element-type
                    (if iep :initial-element :ignore) initial-element
                    (if icp :initial-contents :ignore) initial-contents
                    :allow-other-keys t))))
  reuse-tarray)

(defun taref-upper-tarray (tarray subscripts)
  (apply #'aref (aref (tarray-upper-vector tarray) (cadr subscripts)) (car subscripts) (cddr subscripts)))

(defun taref-lower-tarray (tarray subscripts)
  (apply #'aref (aref (tarray-lower-vector tarray) (car subscripts)) (cdr subscripts)))

(defsetf taref-upper-tarray (tarray &rest subscripts) (new-value)
  `(setf (aref (aref (tarray-upper-vector ,tarray) ,(cadr subscripts)) ,(car subscripts) ,@(cddr subscripts)) ,new-value))

(defsetf taref-lower-tarray (tarray &rest subscripts) (new-value)
  `(setf (aref (aref (tarray-lower-vector ,tarray) ,(car subscripts)) ,@(cdr subscripts)) ,new-value))


(defun print-tarray (tarray stream depth)
  (cond
   ((and *print-level* (> depth *print-level*))
    (format stream "#"))
   (t
    (print-unreadable-object (tarray stream :identity t)
      (format stream "~A of rank ~D" (cond ((upper-tarray-p tarray)
                                            "Upper Tarray")
                                           ((full-tarray-p tarray)
                                            "Full Tarray")
                                           ((lower-tarray-p tarray)
                                            "Lower Tarray")
                                           (t
                                            "Illegal Tarray"))
              (tarray-rank tarray))))))
    
