;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? data structure defintions for KNOWBEL

 

(provide 'prove-structs)



(def-cond-struct (time-point (:conc-name point-))
  start end precedes preceded-by same-as interval
  )


(def-cond-struct (time-int (:conc-name time-))
  start end not-shared
)


(defmacro belief-time-start (time)
  `(mlet ((time ,time)) (if (not (consp time)) time (car time)))
  )

(defmacro belief-time-end (time)
  `(mlet ((time ,time))
     (cond ((not (consp time)) :+)
	   ((cdr time) (cadr time))
	   (t :+))
     )
  )



(defmacro make-belief-time (&key (start '(get-universal-time)) (end :+ end-set) constraint)
  (cond (constraint `(list ,start ,end ,constraint))
	(end-set `(list ,start ,end))
	(t start))
  )

(defmacro belief-time-constraint (time)
  `(mlet ((time ,time))
     (cond ((not (consp time)) :during)
	   ((cddr time) (caddr time))
	   (t :during))
     )
  )

(defvar *next-time-context* 0)


(def-cond-struct (time-context)
		 number
		 includes)

(def-cond-struct (belief-context)
		 context
		 belief
		 )

(defmacro next-time-context (&rest includes)
  `(let ((c (make-time-context :number *next-time-context*
			       :includes (logior (expt 2 *next-time-context*)
						 ,@(doloop (i includes)
						     :collect `(let ((i ,i))
								 (cond ((null i) 0)
								       ((time-context-p i) (time-context-number i))
								       ((belief-context-p i)
									(time-context-number (belif-context-context i)))
								       (t i))))))))
     (++ *next-time-context*)
     c)
  )

(defmacro new-belief-context (&key context includes (belief '(std-belief)))
  `(make-belief-context :context ,(if context context `(next-time-context ,@includes))
			:belief (if (null ,belief) (error "Null belief") ,belief))
  )


(defmacro get-context-context (context)
  `(let ((context ,context))
     (cond ((belief-context-p context) (belief-context-context context))
	   (t nil))
     )
  )


(defmacro get-context-belief (context)
  `(let ((context ,context))
     (cond ((belief-context-p context) (belief-context-belief context))
	   (t context))
     )
  )

(defmacro context-includes (super incl)
  `(logbitp (time-context-number ,incl) (time-context-includes ,super))
  )



(def-cond-struct (cnf-var (:conc-name cnf-var-))
  name input
)

(defmacro clause-wild-card nil `'*)

(defmacro clause-wild-card-p (expr) `(eq ,expr (clause-wild-card)))



(def-cond-struct (clause-var (:conc-name var-))
  number name owner type meta-type
)

(def-cond-struct (clause (:conc-name clause-))
  number expression n-vars belief-time vars props
  )


(defmacro clause-time-context (clause)
  `(let ((b (clause-belief-time ,clause)))
     (get-context-context b)
     )
  )


(defmacro clause-get-prop (tok ind)
  `(getf (clause-props ,tok) ,ind)
  )

(defmacro get-clause-prop (tok ind)
  `(getf (clause-props ,tok) ,ind)
  )


(def-cond-struct (kb (:conc-name kb-))
   clause-number tokens strings theories integrity-constraints deductive-rules
   root-object all-time ic-object ded-object meta-attr attribute-class)


(def-cond-struct (theory (:conc-name theory-))
  name clauses pos-indices neg-indices includes kb horn)


(defmacro theory-clause-number (item)
  `(kb-clause-number (theory-kb ,item))
  )

(defmacro theory-tokens (item)
  `(kb-tokens (theory-kb ,item))
  )

(defmacro theory-strings (item)
  `(kb-strings (theory-kb ,item))
  )

(defmacro theory-integrity-constraints (item)
  `(kb-integrity-constraints (theory-kb ,item))
  )

(defmacro theory-deductive-rules (item)
  `(kb-deductive-rules (theory-kb ,item))
  )

(defmacro theory-attribute-class (item)
  `(kb-attribute-class (theory-kb ,item))
  )

(defmacro theory-meta-attr (item)
  `(kb-meta-attr (theory-kb ,item))
  )

(defmacro theory-root-object (item)
  `(kb-root-object (theory-kb ,item))
  )

(defmacro theory-all-time (item)
  `(kb-all-time (theory-kb ,item))
  )

(defmacro theory-ic-object (item)
  `(kb-ic-object (theory-kb ,item))
  )

(defmacro theory-ded-object (item)
  `(kb-ded-object (theory-kb ,item))
  )
   


(def-cond-struct (active-clause (:conc-name ac-))
  clause generator collect justifications)


(defmacro ac-belief (ac)
  `(clause-belief-time (ac-clause ,ac))
  )


(defmacro ac-time-context (ac)
  `(let ((b (ac-belief ,ac)))
     (get-context-context b)
     )
  )


(def-cond-struct (inference-state (:conc-name istate-))
  active-theory theory-last horn search-style theory-list root-theory belief current-clause sigma sigma-size
  context
  )


(defmacro istate-depth-p (state)
  `(eq (istate-search-style ,state) :depth)
  )

(defmacro istate-best-p (state)
  `(eq (istate-search-style ,state) :best)
  )

(defmacro istate-breadth-p (state)
  `(eq (istate-search-style ,state) :breadth)
  )


(defmacro istate-set-search-style (state style)
  `(case ,style
     ((:breadth :breadth-first :b) (setf (istate-search-style ,state) :breadth))
     ((:best :best-first) (setf (istate-search-style ,state) :best))
     (t (setf (istate-search-style ,state) :depth))
     )
  )

(defmacro istate-get-sigma (state rows cols)
  `(let ((s ,state) (r ,rows))
     (when (or (null (istate-sigma s)) (< (istate-sigma-size s) r))
       (setf (istate-sigma s) (make-array (list r ,cols)))
       (setf (istate-sigma-size s) r)
       )
     (istate-sigma s))
  )

(def-cond-struct (meta-type (:conc-name meta-type-))
  type)


(def-cond-struct (kb-description (:conc-name desc-))
  types parents attrs) 


(def-cond-struct (kb-token (:conc-name tok-))
  name inst-of attrs props)

#+clos(defmethod print-object ((tok kb-token) stream)
	(break)
	(print-unreadable-object (tok stream :type t)
	  (princ (tok-name tok) stream)
	  )
	)


(defmacro tok-get-prop (tok ind)
  `(getf (tok-props ,tok) ,ind)
  )


(defmacro tok-make-class (tok)
  `(let ((tok ,tok))
     (when (null (tok-get-prop tok :class))
       (setf (tok-get-prop tok :class) (list nil nil nil)))
     )
  )


(defmacro tok-parents (tok)
  `(car (tok-get-prop ,tok :class))
  )

(defmacro tok-children (tok)
  `(cadr (tok-get-prop ,tok :class))
  )

(defmacro tok-instances (tok)
  `(caddr (tok-get-prop ,tok :class))
  )

(defmacro tok-description (tok)
  `(tok-get-prop ,tok :description)
  )

(defmacro attr-tok-prop (tok)
  `(mlet ((tok ,tok)) (and (kb-attr-p tok) tok))
  )



(defmacro attr-token-class-label (attr-prop)
  `(attr-label ,attr-prop)
  )


(defmacro tok-inst-level (tok)
  `(tok-get-prop ,tok :inst-level)
  )

(defmacro tok-prim (tok)
  `(tok-get-prop ,tok :prim)
  )



(defmacro attr-label (attr)
  `(tok-name ,attr)
  )

(defmacro attr-token (attr)
  attr)

(defmacro attr-value (attr)
  `(car (tok-get-prop ,attr :attr))
  )

(defmacro attr-history (attr)
  `(cadr (tok-get-prop ,attr :attr))
  )

(defmacro attr-belief (attr)
  `(caddr (tok-get-prop ,attr :attr))
  )

(defmacro attr-from (attr)
  `(cadddr (tok-get-prop ,attr :attr))
  )

(defmacro kb-attr-p (obj)
  `(mlet ((obj ,obj))
     (and (kb-token-p obj) (tok-get-prop obj :attr)))
  )

(defmacro make-kb-attr (&key from label token value history belief name inst-of attrs props)
  (let ((from-var (gensym "FROM"))
	(tok-expr (if token token `(make-kb-token :name ,label))))
    `(mlet ((,from-var ,from)
	    (token ,tok-expr))
       (if (null (tok-get-prop token :attr))
	 (setf (tok-get-prop token :attr) (list ,value ,history ,belief ,from-var))
	 (progn
	   ,@(if value `((setf (attr-value token) ,value)))
	   ,@(if history `((setf (attr-history token) ,history)))
	   ,@(if belief `((setf (attr-belief token) belief))))
	 )
       ,@(if (and label (not token)) `((setf (tok-name token) ,label)))
       ,@(if name `((setf (tok-name token) ,name)))
       ,@(if inst-of `((setf (tok-inst-of token) ,inst-of)))
       ,@(if attrs `((setf (tok-attrs token) ,attrs)))
       ,@(if props `((setf (tok-props token) ,props)))
       token
       ))
  )







(def-cond-struct (token-description (:conc-name tok-desc-))
  instance-of attrs)





(def-cond-struct (attr-gen (:conc-name attr-gen-))
  prop-class prop parents instance)



(def-cond-struct (search-generator (:conc-name sg-) (:type vector))
  clause-ptr theory-ptr lisp-state meta-state match-position
  )



(def-cond-struct (kb-prop-object (:conc-name prop-))
  type src dest belief-time history-time)

(defmacro prop-object-p (v &optional class)
  (if (null class)
    `(kb-prop-object-p ,v)
    (let ((var (gensym "VAR")))
      `(let ((,var ,v))
	 (and (kb-prop-object-p ,var)
	      ,@(case class
		  (:inst `((eq (prop-type ,var) :inst)))
		  (:isa `((eq (prop-type ,var) :isa)))
		  (t (error "~&prop-object-p requires class :isa, :inst, nil, got '~a'~%" class)))
	      ))
      ))
  )

(defmacro make-prop-object (&rest args)
  `(make-kb-prop-object ,@args)
  )

(defmacro make-isa-link (&rest args)
  `(make-kb-prop-object :type :isa ,@args)
  )

(defmacro make-inst-link (&rest args)
  `(make-kb-prop-object :type :inst ,@args)
  )



(defmacro prop-belief (prop)
  `(prop-belief-time ,prop)
  )


(defmacro prop-history (prop)
  `(prop-history-time ,prop)
  )



(defmacro abstract-prop-belief (prop)
  `(mlet ((prop ,prop))
     (cond ((prop-object-p prop) (prop-belief prop))
	   ((kb-attr-p prop) (attr-belief prop))
	   ))
  )

(defmacro set-abstract-prop-belief (prop val)
  `(mlet ((prop ,prop))
     (cond ((prop-object-p prop) (setf (prop-belief prop) ,val))
	   ((kb-attr-p prop) (setf (attr-belief prop) ,val))
	   ))
  )


(defmacro abstract-prop-history (prop)
  `(mlet ((prop ,prop))
     (cond ((prop-object-p prop) (prop-history prop))
	   ((kb-attr-p prop) (attr-history prop))
	   ((time-int-p prop) prop)
	   ))
  )


(defmacro set-abstract-prop-history (prop val)
  `(mlet ((prop ,prop))
     (cond ((prop-object-p prop) (setf (prop-history prop) ,val))
	   ((kb-attr-p prop) (setf (attr-history prop) ,val))
	   ))
  )


(defvar *number-type* nil)
(defvar *time-interval-type* nil)
(defvar *all-time* nil)
(defvar *theory* nil)
(defvar *string-type* nil)




(defmacro time-force-string (item)
  `(mlet ((item ,item))(cond ((numberp item) (format nil "~d" item))
			    ((stringp item) item)
			    ((symbolp item) (string item))
			    (t (format nil "~a" item))))
  )


(defmacro some-conventional-time (time)
  `(mlet ((time ,time))
     (let ((try2 (time-start time)))
	   (if (numberp try2)
	       try2
	     (let ((try3 (time-end time)))
	       (if (numberp try3)
		   try3
		 (get-universal-time)
		 )
	       ))))
  )



(defmacro point-link-source (x) `(caddr ,x))
(defmacro point-link-dest (x) `(cadr ,x))
(defmacro point-link-belief (x) `(car ,x))


(defmacro time-conventional-p (x)
  `(mlet ((int ,x))
     (and (not (time-point-p (time-start int)))
	  (not (time-point-p (time-end int))))
     )
  )


(defmacro time-max (p1 p2)
  `(mlet ((p1 ,p1) (p2 ,p2))
     (cond ((null p1) p2)
	   ((null p2) p1)
	   ((eq p1 :+) :+)
	   ((eq p2 :+) :+)
	   ((eq p1 :-) p2)
	   ((eq p2 :-) p1)
	   (t (max p1 p2)))
     )
  )

(defmacro time-min (p1 p2)
  `(mlet ((p1 ,p1) (p2 ,p2))
     (cond ((null p1) p2)
	   ((null p2) p1)
	   ((eq p1 :+) p2)
	   ((eq p2 :+) p1)
	   ((eq p1 :-) :-)
	   ((eq p2 :-) :-)
	   (t (min p1 p2)))
     )
  )


(defmacro time> (p1 p2)
  `(mlet ((p1 ,p1) (p2 ,p2))
     (cond ((eq p1 :+) (not (eq p2 :+)))
	   ((eq p2 :+) nil)
	   ((eq p1 :-) nil)
	   ((eq p2 :-) t)
	   (t (> p1 p2)))
     )
  )



(defmacro defresolve (symbol args &body body)
  `(let ((resolve-fn #'(lambda ,args ,@body)))
     ,@(if (listp symbol)
	 (doloop (s symbol) :collect `(setf (get ',s 'lisp-fn) resolve-fn))
	 `((setf (get ',symbol 'lisp-fn) resolve-fn))
	 )
     )
  )

(defmacro defresolve-s (symbol args &body body)
  `(let ((resolve-fn (list #'(lambda ,args ,@body))))
     ,@(if (listp symbol)
	 (doloop (s symbol) :collect `(setf (get ',s 'lisp-fn) resolve-fn))
	 `((setf (get ',symbol 'lisp-fn) resolve-fn))
	 )
     )
  )




(defmacro defnamed-resolve (symbol name args &body body)
  `(progn
     (defun ,name ,args ,@body)
     (setf (get ',symbol 'lisp-fn) #',name)
     )
  )


(defmacro removef (thing place)
  `(setf ,place (remove ,thing ,place))
  )


(defmacro long-desc (&rest body)
  `(let ((*description-number* nil))
     ,@body)
  )



   
		

(defmacro tell (expr &optional (theory '*theory*) default-history (belief '(std-belief)) return-props)
  (let ((evar (gensym "EXPR")))
    `(let ((,evar ,expr))
       (multiple-value-bind (token errors props)
	   (real-tell ,evar :theory ,theory :default-history ,default-history :belief ,belief)
	 ,(if return-props
	    `(if (car errors) (error "~%~%tell failed ~a~%~%" (car errors)) props)
	    `(if (car errors) (list 'error (car ,evar) (car errors)) token)
	    )
	 ))
    )
  )




(defmacro var-num (var root1 &optional root2)
  (declare (ignore root2))
  (let ((vv (gensym "VAR"))
	(rv (gensym "ROOT1")))
    `(let ((,vv ,var)
	   (,rv ,root1))
       (if (eq (var-owner ,vv) ,rv)
	 (var-number ,vv)
	 (+ (clause-n-vars ,rv) (var-number ,vv))))
    )
  )
