;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? a macro package for implementing generators

;;;    

(eval-when (load compile eval)

  (provide 'ndprog)
  (require 'tools)

)

(comment (ndprog  state (v1 v2 v3)
	   (nd-for v1 cont1 exp1 		      ;;; nd-for and nd-if expand inline so that continuation can jump in
	     (nd-for v2 cont2 exp2
	       (nd-for v3 exp3
		 (nd-if (test)
		   (success)
		   )
		 )
	       )
	     )
	   )
	 )


;; nd macros are expected to return two values: the body and a list of the nd-vars in their scope

(defmacro defndmacro (name args &body body)
  `(eval-when (load compile eval)
     (setf (get ',name :ndmacro)
       #'(lambda ,args ,@body)
       )
     )
  )


(defndmacro fail (split-points split-tag arg)
  `((setq nd-cont (svref nd-cont 2))
    (if nd-cont
      (go ,split-tag)
      (return nil)))
  )

(defndmacro success (split-points split-tag val)
  (let ((tag (gensym "nd-success")))
    (tconc split-points tag)
    `((setf (svref nd-cont 1) ',tag)
      (return (cons ,(car val) nd-cont))
      ,tag)
    )
  )


(defndmacro nd-for (split-points split-tag arg)
  (dlet* (((nd-var nd-cont nd-gen . body) arg))
    (let ((vars (list nd-var nd-cont))
	  (loop-tag1 (gensym "nd-for-loop1"))
	  (loop-exit (gensym "nd-for-exit")))
      (values `((setf ,nd-cont :init)
		,loop-tag1
		(setf ,nd-cont ,nd-gen)
		(when (not ,nd-cont)
		  (go ,loop-exit)
		  )
		(setf ,nd-var (car ,nd-cont))
		(setf ,nd-cont (cdr ,nd-cont))
		,@body
		(go ,loop-tag1)
		,loop-exit)
	      vars)
      )
    )
  )


(defndmacro nd-progn (split-points split-tag arg)
  arg
  )


(defndmacro nd-loop (split-points split-tag arg)
  (dlet* ((((nd-var nd-expr tail-fn) . body) arg))
    (let* ((tail-var (gensym "nd-tail-var" ))
	   (vars (list nd-var tail-var))
	   (loop-tag1 (gensym "nd-loop-loop1"))
	   (loop-exit (gensym "nd-loop-exit")))
      (values `((setf ,tail-var ,nd-expr)
		,loop-tag1
		(when (not ,tail-var)
		  (go ,loop-exit)
		  )
		(setf ,nd-var (car ,tail-var))
		,@body
		(setf ,tail-var (,(or tail-fn 'cdr) ,tail-var))
		(go ,loop-tag1)
		,loop-exit)
	      vars)
      )
    )
  )


(defndmacro nd-while (split-points split-tag arg)
  (let ((loop-tag (gensym "nd-while-loop"))
	(cont-tag (gensym "nd-while-cont")))
    (values `(,cont-tag
	      (when (not ,(car arg)) (go ,loop-tag))
	      ,@(cdr arg)
	      (go ,cont-tag)
	      ,loop-tag)
	    nil)
    )
  )


(defndmacro nd-if (split-points split-tag arg)
  (dlet* (((expr t-expr nil-expr) arg))
    (let ((tag1 (gensym "nd-if-exit"))
	  (tag2 (gensym "nd-if-else")))
      (values `((when (not ,expr) (go ,tag2))
		,t-expr
		(go ,tag1)
		,tag2
		,@(if nil-expr `(,nil-expr) nil)
		,tag1)))
    )
  )


(defun nd-subst-vars (expr vars)
  (cond ((null expr) nil)
	((consp expr) (cons (nd-subst-vars (car expr) vars)
			    (nd-subst-vars (cdr expr) vars)))
	(t (let ((tail (assoc expr vars)))
	     (if tail
	       `(svref nd-cont ,(cdr tail))
	       expr))))
  )


(defun nd-add-newvars (new vars vnum)
  (cond ((null new) vars)
	(t (let ((n (car vnum)))
	     (setf (car vnum) (+ 1 n))
	     (cons (cons (car new) n)
		   (nd-add-newvars (cdr new) vars vnum))
	     )))
  )
	   


(defun process-nd-body (body vars vnum split-points split-tag)
  (cond ((null body) nil)
	((not (consp body)) nil)
	((not (consp (car body))) (cons (car body) (process-nd-body (cdr body) vars vnum split-points split-tag)))
	((and (symbolp (caar body))
	      (get (caar body) :ndmacro))
	 (multiple-value-bind (new-body new-vars) (funcall (get (caar body) :ndmacro) split-points split-tag (cdar body))
	   (append (process-nd-body new-body (nd-add-newvars new-vars vars vnum) vnum split-points split-tag)
		   (process-nd-body (cdr body) vars vnum split-points split-tag)
		   )))
	(t (cons (nd-subst-vars (car body) vars)
		 (process-nd-body (cdr body) vars vnum split-points split-tag)))
	)
  )


(defmacro ndprog (cont-expr ndvars &rest body)
  (let* ((vnum (list 3))
	 (split-points (tconc))
	 (start-tag (gensym "nd-body"))
	 (split-tag (gensym "nd-split"))
	 (processed-body (process-nd-body body (nd-add-newvars ndvars nil vnum) vnum split-points split-tag)))
    `(prog ((nd-cont ,cont-expr))
       (cond ((null nd-cont) (return nil))
	     ((eq nd-cont :init)
	      (setq nd-cont (make-array ,(car vnum)))
	      (setf (svref nd-cont 0) 'ndstate)
	      (go ,start-tag)))
       ,split-tag
       (case (svref nd-cont 1)
	 ,@(doloop (split (car split-points))
	     :collect `(,split (go ,split)))
	 (t (return nil)))
       ,start-tag
       ,@processed-body
       )
    )
  )


(defmacro ndstate-p (x)
  `(let ((x ,x)) (and (vectorp x) (eq 'ndstate (svref x 0))))
  )
