;;;
;;;   KNOWBEL knowledge representation system
;;;    
;;;    author: Bryan M. Kramer
;;;    
;;;    
;;; Copyright (c) 1990, 1991 University of Toronto, Toronto, ON
;;;
;;; Permission is granted to any individual or institution to use, copy,
;;; modify, and distribute this software, provided that this complete
;;; copyright and permission notice is maintained, intact, in all copies and
;;; supporting documentation.
;;;
;;; The University of Toronto provides this software "as is" without
;;; express or implied warranty.
;;;

;;;    
;;;    


;? functions for the KNOWBEL object module: TELL and INIT-KB
;;;    
;; kb 
;;	11/5/90 mts - added verbose option for loadkb and loadkb3; this should be changed to
;;		use keywords instead.




(eval-when (load compile eval)
  
  (require 'prove)



(defmacro do-with-time ((var props history belief) &body body)
  `(doloop (,var ,props)
    :when (and (belief-satisfies ,belief (prop-belief ,var))
	       (time-intersect-p ,history (prop-history ,var) ,belief))
     ,@body)
  )



(defmacro find-attr (class label history belief)
  `(mlet ((class ,class)
	  (label ,label)
	  (belief ,belief)
	  (history ,history))
	 (dofind (attr-prop (lookup-index-cdr (tok-attrs class) :all))
		 (progn (and (eq (attr-label attr-prop) label)
			     (belief-satisfies belief (attr-belief attr-prop))
			     (time-intersect-p history (attr-history attr-prop) belief)
			     attr-prop))
		 )
	 )
  )
)



(defvar *built-in-type-names* nil)
(defvar *built-in-type-alist* nil)

(defvar *built-in-type-symbols*
    '(integer proposition string number real symbol boolean sexpr list clause time time-interval clause-set theory description))


					; (inst x y)
					; (isa x y)
					; (memb obj label val)
;; almost everything is a string

(defun lookup-proposition (expr theory belief errors)
  (case (car expr)
    ((:instance-of instance-of inst :inst)
     (let ((obj (lookup-type (cadr expr) theory))
	   (class (lookup-type (caddr expr) theory)))
       (cond ((or (null class) (null obj))
	      (when errors (tconc errors `(undefined symbol in time expr ,expr)))
	      nil)
	     (t (type-instance obj class nil belief)))))
    ((:isa :is-a isa is-a)
     (let ((class1 (lookup-type (cadr expr) theory))
	   (class2 (lookup-type (caddr expr) theory)))
       (cond ((or (null class1) (null class2))
	      (when errors (tconc errors `(undefined symbol in time expr ,expr)))
	      nil)
	     (t (type-isa class1 class2 nil belief)))))
    ((:attr attr member :member :memb memb)
     (let ((obj (lookup-type (cadr expr) theory)) 
	   (val (lookup-type (cadddr expr) theory)))
       (cond ((or (null obj) (null val))
	      (when errors (tconc errors `(undefined symbol in time expr ,expr)))
	      nil)
	     (t (find-token-attr (find-string (caddr expr) theory) obj nil belief)))))  ;; need a new attr function that
					; accepts null history and checks value
    (t
     (when errors (tconc errors `(unknown proposition type ,expr)))
     nil)
  ))


(defmacro tell-prop-get-token (prop)
  `(let ((prop ,prop))
     (cond ((kb-token-p prop) prop)
	   ((listp prop) (car prop))
	   ((prop-object-p prop) (prop-dest prop)))
     )
  )


(defun find-token-level (token history belief)
  (let ((level (tok-inst-level token)))
    (if level
	level
      (dofind (parent-prop (tok-parents token))
	      (and (belief-satisfies belief (prop-belief parent-prop))
		   (time-intersect-p history (prop-history parent-prop) belief)
		   (find-token-level (prop-dest parent-prop) history belief)
		   )
	      )))
  )


(defun find-set-level (token-set history belief)
					; returns nil for token, a level if consistent, :error otherwise
  (and token-set
       (do ((tokenp (cdr token-set) (cdr tokenp))
	    (result (find-token-level (tell-prop-get-token (car token-set)) history belief)))
	   ((or (null tokenp) (eq result :error)) result)
	 (let ((t-l (find-token-level (tell-prop-get-token (car tokenp)) history belief)))
	   (cond ((eq t-l :+) nil)
		 ((eq result :+)
		  (if (null t-l) (setq result :error)))
		 ((not (eql t-l result)) (setq result :error)))
	   )))
  )
		   

; complete-clause
; complete a clause with history times
;
;     (not (instance-of $x class))   ---> type of $x becomes class (temporarily discarded)
;     (member-of <object> <value> <label>) ---> (attr <object> <label> <value> <token> <history>)
;     (instance-of <obj> <class>) ---> (instance-of <obj> <class> <history>)
;     common history var added where its not
;     all typed variables result in the addition of (not (instance-of $x class <history>)) when not explicit and not
;                 subject of attr

 
(defmacro parse-type (name &optional (th '*theory*))
  `(mlet ((name ,name))
     (cond ((kb-token-p name) name)
	   ((symbolp name) (lookup-type1 (string-downcase name) ,th))
	   ((stringp name) (lookup-string name ,th))
	   ((numberp name) name)
	   (t name))
     )
  )


(defun tell-ic-dominated-var-p (var clause time-var)
  (doloop (atom clause)
   :find (and (eq (car atom) 'not)
	      (and (eq (car (cadr atom)) 'attr) ; (attr object label class-label value prop time)
		   (eq (cadr (cadr atom)) var)
		   (eq (nth 6 (cadr atom)) time-var)
		   (var-type var)))
    )
  )





(defun find-class-attr (label prop-list history belief)
  (doloop (prop prop-list)
   :when (and (belief-satisfies belief (prop-belief prop))
	      (time-intersect-p history (prop-history prop) belief))
   :find (or (doloop (attr-prop (lookup-index-cdr (tok-attrs (prop-dest prop)) :all))
	      :when (and (eq (attr-label attr-prop) label)
			 (belief-satisfies belief (attr-belief attr-prop))
			 (time-intersect-p history (attr-history attr-prop) belief))
	      :return attr-prop)
	     (find-class-attr label (tok-parents (prop-dest prop)) history belief))
    )
  )

(comment
  (do-with-time (prop prop-list history belief)
   :find (or
	  (dofind (attr-prop (lookup-index-cdr (tok-attrs (prop-dest prop)) :all))
		  (progn
		    (and (eq (attr-label attr-prop) label)
			 (belief-satisfies belief (attr-belief attr-prop))
			 (time-intersect-p history (attr-history attr-prop) belief)
			 attr-prop))
		  )
	  (find-class-attr label (tok-parents (prop-dest prop)) history belief))
    )
  )


(defun find-token-attr (label token history belief)
  (or
   (dofind (attr-prop (lookup-index-cdr (tok-attrs token) :all))
	   (progn (and (eq (attr-label attr-prop) label)
		       (belief-satisfies belief (attr-belief attr-prop))
		       (time-intersect-p history (attr-history attr-prop) belief)
		       attr-prop))
	   )
   (find-class-attr label (tok-parents token) history belief))
  )

	


(defun built-in-type (attr-list theory belief)
  (dofind (attr attr-list)
	  (cond ((weak-type-isa (attr-token attr) (theory-ic-object theory) belief)
		 :integrity-constraint)
		((weak-type-isa (attr-token attr) (theory-ded-object theory) belief)
		 :deductive-rule)
		((and (attr-value attr) (kb-token-p (attr-value attr)))
		 (let ((x (assoc (tok-name (attr-value attr)) *built-in-type-alist*)))
		   (if x (cdr x))))
	  ))
  )


(defun dominated-var-p (var clause time-var)
  (doloop (atom clause)
   :find (and (eq (car atom) 'not)
	      (or (and (eq (car (cadr atom)) 'attr) ; (attr object label class-label value prop time)
		       (eq (cadr (cadr atom)) var)
		       (eq (nth 6 (cadr atom)) time-var)
		       (var-type var))
		  (and (eq (car (cadr atom)) 'instance-of) ; (instance-of object class time)
		       (eq (cadr (cadr atom)) var)
		       (eq (nth 3 (cadr atom)) time-var)
		       (not (clause-var-p (nth 2 (cadr atom))))
		       (nth 2 (cadr atom)))
		  ))
    )
  )


(defun telos-collect-vars (clause vars)
  (cond ((null clause) nil)
	((clause-var-p clause) (tconc-unique vars clause))
	((consp clause)
	 (telos-collect-vars (car clause) vars)
	 (telos-collect-vars (cdr clause) vars))
	)
  )


(defun telos-parse-member (atom default-time theory)
					; (member-of value object label [time]) --> (attr object label class-label value prop time)
  (cond ((= (length atom) 4)
	 `(attr ,(nth 2 atom) * ,(type-name (nth 3 atom) theory) ,(nth 1 atom) * ,default-time))
	((= (length atom) 5)
	 `(attr ,(nth 2 atom) * ,(type-name (nth 3 atom) theory) ,(nth 1 atom) *
		,(if (cnf-var-p (nth 4 atom))
		   (progn (setf (cnf-var-input (nth 4 atom)) "$x(-,+)") (nth 4 atom))
		   (make-cnf-var :name (gensym "time") :input (format nil "$x(~a)" (nth 4 atom))))))
	(t atom)
	)
  )


(defun telos-parse-attr (atom default-time theory belief)
					; (attr object label value class-label prop h_t b_t )
					;      --> (attr object label class-label value prop h_t)
  (cond ((= (length atom) 8)
	 (dlet* (((attr object label value class-label prop h_t b_t ) atom)
		 (object (if (cnf-var-p object)
			  object
			   (find-or-create (type-name object theory) (theory-root-object theory) theory (theory-all-time theory) belief)))
		 (label1
		  (cond ((or (cnf-var-p label) (eq label '*)) label)
			(t (type-name label theory) label)))
		 (real-label (if (and (stringp label1) (string-equal label1 "_")) '|$.label| label1))
		 (class-label (if (and (not (eq class-label '*)) (not (cnf-var-p class-label)))
				(type-name class-label theory)
			       class-label))
		 (h_t (if (not (cnf-var-p h_t))
			(make-cnf-var :name (gensym "time") :input (format nil "$x(~a)"  h_t))
			(progn (setf (cnf-var-input h_t) "$x(-,+)") h_t)))
		 )
	 `(attr ,object ,real-label ,class-label ,value ,prop ,h_t)
	 ))
	(t atom)
	)
  )


(defun telos-parse-inst (atom default-time neg theory belief)
					; (instance-of object class) --> (instance-of object class time)
					; (instance-of object class h_t b_t) --> (instance-of object class time_var)
  (dlet* (((object class h_t) (cdr atom))
	  (h_t (if (not h_t)
		default-time
		 (if (not (cnf-var-p h_t))
		   (make-cnf-var :name (gensym "time") :input (format nil "$x(~a)"  h_t))
		   (progn (setf (cnf-var-input h_t) "$x(-,+)") h_t)))))
    (when (and neg (cnf-var-p object) (not (cnf-var-p class)))
      (setf (cnf-var-input object) (find-or-create class (theory-root-object theory) theory (theory-all-time theory) belief))
      )
    `(instance-of ,object ,class ,h_t)
    )
  )


  

(defun telos-parse (clause theory belief)
  (doloop (atom clause)
    :vars ((default-time '|$t(-,+)|))
    :collect
    (cond ((and (eq (car atom) 'not) (eq (car (cadr atom)) 'member-of))
	   `(not ,(telos-parse-member (cadr atom) default-time theory)))
	  ((eq (car atom) 'member-of) (telos-parse-member atom default-time theory))
	  ((and (eq (car atom) 'not) (eq (car (cadr atom)) 'instance-of))
	   `(not ,(telos-parse-inst (cadr atom) default-time t theory belief)))
	  ((eq (car atom) 'instance-of) (telos-parse-inst atom default-time nil theory belief))
	  ((and (eq (car atom) 'not) (eq (car (cadr atom)) 'attr))
	   `(not ,(telos-parse-attr (cadr atom) default-time theory belief)))
	  (t atom))
    )
  )



(defun telos-stash (expr theory belief &optional name props errors)
  (doloop (cnf-expr (cnf expr))
   :vars ((common-vars (tconc nil))
	  (count 0 (+ 1 count)))
   :collect
    (stash-b (telos-parse cnf-expr theory belief) belief theory t common-vars (when name (cons name count)) props errors)
    )
  )





(defmacro add-error (&rest args)
  `(progn
     (tconc errors (list ,@args))
     nil
     )
  )


(defvar *tell-error-break* nil)

(defmacro tell-break (&rest set)
  (if set
    `(setq *tell-error-break* ,@set)
    `(setq *tell-error-break* (not *tell-error-break*))
    )
  )


(defmacro tell-error (&rest args)
  `(ifn *tell-error-break*
     (error "Error in TELL: '~a'~%" (list ,@args))
     else
     (tconc errors (list ,@args))
     (throw 'tell nil)
     )
  )


(defun add-clause-kb-theory (c clause-class theory)
  (let* ((name (if (stringp clause-class)
		 (intern (string-upcase clause-class))
		 clause-class))
	 (stash-theory (theory-find-theory theory name)))
    (if (null stash-theory)
      (setq stash-theory (new-theory (theory-kb theory) name))
      )
    (stash c stash-theory)
    (setf (clause-get-prop c :clause-class) name)
    )
  )


(defun parse-explict-clause (clause-expr clause-class type theory history belief errors props type-attrs label)
  (let* ((stash-theory (if! (eq type :integrity-constraint)
			 (theory-integrity-constraints theory)
			elseif (eq type :deductive-rule)
			 (theory-deductive-rules theory)
			else
			theory))
	 (clauses (telos-stash clause-expr stash-theory belief nil nil errors)))
    (doloop (c clauses)
      (tconc props c)
      (if clause-class
	(add-clause-kb-theory c clause-class theory ))
      (if! (eq type :integrity-constraint)
	(compute-ic-relevance c (list (theory-deductive-rules theory)) belief)
       elseif (eq type :deductive-rule)
	(compute-dr-ic-relevance c (list (theory-deductive-rules theory)) (theory-integrity-constraints theory) belief))
      (if (and label (stringp label) (not (string-equal label ""))) (setf (clause-number c) label)))
    clauses)
  )


(defun parse-value (expr theory history belief errors props type-attrs label)
  (let ((type (built-in-type type-attrs theory belief)))
    (cond ((eq type 'string)
	   (find-string (cond ((stringp expr) expr)
			      ((numberp expr) (format nil "~d" expr))
			      ((symbolp expr) (string expr))
			      (t (format nil "~s" expr))) theory))
	  ((and (eq type 'list) (listp expr)) expr)
	  ((and (consp expr) (symbolp (car expr)) (string-equal (string (car expr)) "clause"))
	   (parse-explict-clause (cadr expr) (caddr expr) type theory history belief errors props type-attrs label))
	  ((eq type :integrity-constraint)
	   (parse-explict-clause expr nil type theory history belief errors props type-attrs label))
	  ((eq type :deductive-rule)
	   (parse-explict-clause expr nil type theory history belief errors props type-attrs label))
	  ((eq type 'theory)
	   (let* ((th (new-theory (theory-kb theory) (gensym)))
		  (clauses (telos-stash expr th belief nil nil errors)))
	     (mapc #'(lambda (c) 
		       (if (and label (stringp label) (not (string-equal label ""))) (setf (clause-number c) label)))
		   clauses)
	     (tconc props th)
	     th))
	  ((eq type 'boolean) (not (null expr)))
	  ((eq type 'real) (if (numberp expr) expr (add-error  'bad-real-number expr)))
	  ((numberp expr) expr)
	  ((stringp expr) (find-string expr theory))
	  ((consp expr)
	   (case (car expr)
	     ((forall exists and or => <= <== ==> not) (let ((clauses (telos-stash expr theory belief nil nil errors)))
							 (mapc #'(lambda (c) (tconc props c)) clauses)
							 clauses))
	     ((before after) nil)
	     (t expr)))
	  ((symbolp expr)
	   (let ((tok (lookup-type expr theory)))
	     (or tok
		 (let* ((first-type (lookup-type (or (attr-value (car type-attrs)) 'proposition) theory))
			(t-val (and first-type (add-prove-is (type-name expr theory) first-type theory history belief))))
		   (when (null first-type)
		     (tconc errors `(bad-type-for-attr ,expr ,(car type-attrs))))
		   (mapc #'(lambda (attr-prop) (if (and (attr-value attr-prop) (kb-token-p (attr-value attr-prop)))
						 (assert-inst expr (attr-value attr-prop) history belief nil theory)
						 (tconc errors `(bad-type-for-attr ,expr ,attr-prop))))
			 (cdr type-attrs))
		   (tconc props t-val)
		   (tconc errors (list 'warning-creating t-val))
		   t-val)
		 (tconc errors `(undefined-object ,expr))
		 )))
	  (t expr))
    )
  )



(defun parse-time-work (item default obj theory errors belief)
  (cond ((null item) default)
	((time-int-p item) item)
	((member item '(:alltime :alltime all-time all-time)) (theory-all-time theory))
	((numberp item)			; as in 1984
	 (let ((str (format nil "~d" item)))
	   (make-time-int :start (parse-time-point-year str (some-conventional-time default) t)
			  :end (parse-time-point-year str (some-conventional-time default) nil))))
	((or (symbolp item) (stringp item))
	 (make-time-int :start (time-convert-str (string item) (some-conventional-time default) t)
			:end (time-convert-str (string item) (some-conventional-time default) nil)))
	((and (listp item) (null (cddr item)) (not (keywordp (car item)))
	      (not (consp (car item))) (not (consp (cadr item))))
	 (make-time-int :start (time-convert-str (time-force-string (car item)) (some-conventional-time default) t)
			:end (time-convert-str (time-force-string (cadr item)) (some-conventional-time default) nil))
	 )
	((consp item)
	 (or (time-create-relative item theory belief obj errors)
	     default))
	(t (when errors (tconc errors `((couldnt parse time expr ,item)))
		 default)))
  )
	   
(defun parse-time (item default obj theory errors belief)
  (let ((interval (parse-time-work item default obj theory errors belief)))
    (if (or (null interval) (null (time-start interval)) (null (time-end interval)))
      (progn
	(when errors (tconc errors `(bad time expr item --- using default))
	      default))
      interval))
  )





(defun most-specialized-attrs (class label history belief result)
  (let ((match (find-attr class label history belief)))
    (cond (match (tconc result match))
	  (t (do-with-time (prop (tok-parents class) history belief)
	       (most-specialized-attrs (prop-dest prop) label history belief result)
	       )))
    )
  )


;;; functions to check that an attribute fits in with the is-a parents of an object

(defun tell-check-attr-isa1 (class label token val history belief)
					; maybe should return all mismatches
					; should check token instance of ?
  (let ((match (find-attr class label history belief)))
    (cond (match
	   (when (not (type-isa val (attr-value match) history belief))
	     match))
	  (t (do-with-time (prop (tok-parents class) history belief)
			   :find (tell-check-attr-isa1 (prop-dest prop) label token val history belief)
			   )))
    )
  )


(defun tell-check-attr-isa (object attr errors history belief props theory)
  (let ((result (tconc nil)))
    (do-with-time (prop (tok-parents object) (attr-history attr) (attr-belief attr))
      (most-specialized-attrs (prop-dest prop) (attr-label attr) (attr-history attr) (attr-belief attr) result)
      )
    (doloop (parent-attr (car result))
      (when (not (type-isa (attr-value attr) (attr-value parent-attr) (attr-history attr) (attr-belief attr)))
	(tconc errors (list 'attr-isa-failure attr parent-attr)))
      (assert-isa attr parent-attr history belief props theory)
      )
    )
  )


(defun tell-index-attr-parents (props new-attr token)
  (doloop (prop props)
    (when (kb-attr-p (prop-dest prop))
      (add-to-index (tok-attrs token) ((attr-label (prop-dest prop))) new-attr t)
      )
    (tell-index-attr-parents (tok-parents (prop-dest prop)) new-attr token)
    ) 
  )

(defun tell-index-attr (new-attr token)
  (add-to-index (tok-attrs token) (:all) new-attr t)
  (doloop (prop (tok-inst-of new-attr))
    (when (kb-attr-p (prop-dest prop))
      (add-to-index (tok-attrs token) ((attr-label (prop-dest prop))) new-attr t)
      )
    (tell-index-attr-parents (tok-parents (prop-dest prop)) new-attr token)
    )
  )



(defun tell-special-rep (token attr-prop spec value theory history belief errors)
  (let ((defn (tok-get-prop attr-prop :special-rep)))
    (apply (caddr defn) token (car defn) value history belief (cdddr defn))
    )
  )



(defun tell-with-group (name token group errors props find-token theory def-history belief)
  (let ((group-labels (mapcar #'(lambda (label)
				  (find-string (string-downcase (string label)) theory))
			      (if (consp (car group)) (car group) (list (car group))))))
    (do ((attrp (cdr group) (cdr attrp)))
	((null attrp))
      (let* ((label (find-string (string-downcase (caar attrp)) theory))
	     (u-label (if (equal label "_") (make-string 0) label))
	     (val (cadar attrp))
	     (new-attr (make-kb-attr :from token))
	     (history (parse-time (when (and (caddar attrp) (not (eq (caddar attrp) :with))) (caddar attrp))
				  def-history new-attr theory errors belief))
	     (attr-props (doloop (class-label group-labels)
			  :vars ((attr nil (find-class-attr class-label (tok-inst-of token) history belief)))
			   (when (not attr)
			     (tell-error 'could-not-identify-attr-class class-label history))
			  :collect attr))
	     (prop-props (or (member 'with (cddar attrp))(member :with (cddar attrp)))))
	(if attr-props
	  (let ((t-val (parse-value val theory history belief errors props attr-props u-label)))
	    (if (and (null (cdr attr-props))
		     (tok-get-prop (car attr-props) :special-rep))
	      (tell-special-rep token (car attr-props) (car attrp) t-val theory history belief errors)
	      (progn
		(setf (attr-label new-attr) u-label)
		(setf (attr-value new-attr) t-val)
		(setf (attr-history new-attr) history)
		(setf (attr-belief new-attr) belief)
		(doloop (type attr-props)
		  (assert-inst new-attr (attr-token type) history belief props theory)
		  )
		(tell-index-attr new-attr token)
		(tconc props new-attr)
		(tell-check-attr-isa token new-attr errors history belief props theory)
		)
	      )
	    (when prop-props
	      (let ((new-props (tconc nil)))
		(doloop (group (cdr prop-props))
		  (tell-with-group u-label new-attr group errors new-props find-token theory def-history belief)
		  )
		(tconc props (cons new-attr (car new-props))))
	      ))
	  )
	)
      )
    )
  )


(defun tell-assure-level-root (token type-level history belief errors props theory)
  (when (and (numberp type-level) (> type-level 1))
    (let ((root (case type-level
		  (2 (lookup-type 'token theory))
		  (3 (lookup-type 's-class theory))
		  (4 (lookup-type 'm1-class theory))
		  (t (tell-error 'type-level-not-implemented type-level)))))
      (if (not (type-isa token root history belief))
	  (assert-isa token root history belief props theory)
	)))
  )



(defun tell-check-attr-inst (token props errors)
  (doloop (prop props) :when (kb-attr-p prop)
	  (let ((no-type t))
	    (do-with-time (type-prop (tok-inst-of (attr-token prop)) (attr-history prop) (attr-belief prop))
			  (if (attr-tok-prop (prop-dest type-prop))
			      (let ((type (attr-value (attr-tok-prop (prop-dest type-prop)))))
				(setq no-type nil)
				(when (and type
					   (not (type-instance (attr-value prop)
							       type
							       (attr-history prop) (attr-belief prop))))
				  (tconc errors
					 (list 'attr-value-violates-type-constraint
					       prop type (attr-tok-prop (prop-dest type-prop))))
				  ))
			    (tconc errors
				   (list 'attr-class-has-no-type prop (attr-tok-prop (prop-dest type-prop)))))
			  )
	    (when no-type
	      (tconc errors (list 'attr-has-no-type prop))))
	  )
  )


(defmacro tell-active-stash (theory clause active-theory)
  `(mlet ((clause ,clause))
     ,@(if theory `((if (and ,theory (theory-horn ,theory)) (setf (clause-get-prop clause :ic-horn) t))))
     (active-stash clause ,active-theory))
  )



(defun tell-check-attr-ic (ic object prop theory active-theory history belief justs)
  (when (and (not (get-clause-prop (car ic) :integrity-check-ignore))
	     (get-clause-prop (car ic) :ic-relevance))
    (let* ((clause (car ic))
	   (atom (cdr ic))
	   (sigma (make-array (list (clause-n-vars clause) 2))))
      (doloop (prop-type-prop (tok-inst-of (attr-token prop)))
	(reset-sigma sigma clause)
       :vars (ac)
       :when (dlet* (((a-object a-label a-class a-value a-prop a-time) (cdr (cadr (car atom)))))	       
	       (and (unify-work a-object object sigma clause nil belief) ; (attr object label label-class value prop time)
		    (unify-work a-label (attr-label prop) sigma clause nil belief)
		    (unify-work a-class (attr-token-class-label (prop-dest prop-type-prop)) sigma clause nil belief)
		    (unify-work a-value (attr-value prop) sigma clause nil belief)
		    (or (not a-prop) (unify-work a-prop (attr-token prop) sigma clause nil belief))
		    (or (not a-time)
			(and (clause-var-p a-time)
			     (time-int-p (var-type a-time))
			     (let ((h (time-intersect history (var-type a-time) belief)))
			       (setf (aref sigma (var-number a-time) 1) h)
			       h)))))
	(setq ac (tell-active-stash theory (build-resolvent clause atom nil nil sigma belief t) active-theory))
	(setf (ac-justifications ac) (list* prop clause justs))
	)
      )
    )
  )


(defun tell-collect-attr-ics (object prop theories active-theory history belief)
  (doloop (theory theories)
    (let ((tok-index (lookup-index (theory-neg-indices theory) 'attr :_attr_arg_1_token object))
	  (type-indices (lookup-index (theory-neg-indices theory) 'attr :_attr_arg_1_var)))
      (doloop (ic (cdr tok-index))
	(tell-check-attr-ic ic object prop theory active-theory history belief '(tok-index)))
      (doloop (index (cdr type-indices))
       :when (weak-type-instance object (car index) belief)
	(doloop (ic (cdr index))
	  (tell-check-attr-ic ic object prop theory active-theory history belief '(var-index)))
	)
      )
    )
  )
  


(defun tell-collect-isa-ics (object prop theory active-theory history belief)
  )





(defun tell-check-inst-ic (ic object prop theory active-theory history belief just)
  (when (and (not (get-clause-prop (car ic) :integrity-check-ignore))
	     (get-clause-prop (car ic) :ic-relevance))
    (let* ((clause (car ic))
	   (atom (cdr ic))
	   (sigma (make-array (list (clause-n-vars clause) 2))))
      (reset-sigma sigma clause)
      (dlet* (ac
	      ((a-object a-class a-time) (cdr (cadr (car atom)))))
	(when 
	    (and (unify-work a-object object sigma clause nil belief)
		 (unify-work a-class (prop-dest prop) sigma clause nil belief)
		 (or (not a-time)
		     (and (clause-var-p a-time)
			  (let ((h (time-intersect history (var-type a-time) belief)))
			    (setf (aref sigma (var-number a-time) 1) h)
			    h))))
	  (setq ac (tell-active-stash theory (build-resolvent clause atom nil nil sigma belief t) active-theory))
	  (setf (ac-justifications ac) (list* (list :inst object (prop-dest prop)) clause just)))
	))
    )
  )  


(defun tell-collect-inst-ics (object prop-list theory-list active-theory history belief)
					;should be checking up is-a parents of prop-dest
  (doloop (prop prop-list)
    (tell-collect-inst-ics object (tok-parents (prop-dest prop)) theory-list active-theory history belief)
    (doloop (theory theory-list)
      (let ((tok-index (lookup-index (theory-neg-indices theory) 'instance-of :_inst_arg_1_token object))
	    (var-indices (lookup-index (theory-neg-indices theory) 'instance-of :_inst_arg_1_var)))
	(doloop (ic (cdr tok-index))
	  (tell-check-inst-ic ic object prop theory active-theory history belief nil))
	(doloop (index (cdr var-indices))
	 :when (weak-type-instance object (car index) belief)
	  (doloop (ic (cdr index))
	    (tell-check-inst-ic ic object prop theory active-theory history belief nil))
	  )
	)
      )
    )
  )

(defun tell-collect-clause-ics (clause theory-list active-theory history belief)
  (when (and (clause-get-prop clause :ic-relevance)
	     (not (get-clause-prop clause :integrity-check-ignore)))
    (let ((ac (tell-active-stash nil clause active-theory)))
      (setf (ac-justifications ac) (list (list :new-ic clause)))
      ac)
    )
  )


(defun tell-collect-ics (object props theory active-theory history belief)
  (doloop (prop props)
    :vars ((theory-list (traverse-theory (list (theory-deductive-rules theory) (theory-integrity-constraints theory) theory))))
    (cond ((kb-attr-p prop)
	   (tell-collect-attr-ics object prop theory-list active-theory history belief))
	  ((prop-object-p prop :isa)
	   (tell-collect-isa-ics object prop theory-list active-theory history belief))
	  ((prop-object-p prop :inst)
	   (tell-collect-inst-ics (prop-src prop) (list prop) theory-list active-theory history belief))
	  ((clause-p prop)
	   (tell-collect-clause-ics prop theory-list active-theory history belief))
	  (t nil))
    )
  )


(defun tell-build-class-list (input old-props theory default-history belief errors)
  (normal-class-list
   (doloop (spec input)
    :collect (cond ((consp spec)
		    (let ((class (lookup-type (car spec) theory))
			  (h (if (null (cdr spec))
			       default-history
			       (parse-time (cadr spec) default-history nil theory errors belief))))
		      (ifn (or (null class) (null h))
			(tell-error (list spec 'bad-class-or-interval))
			else
			(list class h))))
		   (t (let ((class (lookup-type spec theory)))
			(ifn (null class)
			  (tell-error (list spec 'bad-class))
			  else
			  class))))
     )
   old-props
   default-history
   belief)
  )

(defun safe-add-child (class prop)
  (tok-make-class class)
  (push prop (tok-children class))
  )

(defun tell-update-parents (token parents history belief props theory)
					; remove redundant props from existing parent classes
  (tok-make-class token)
  (doloop (prop (tok-parents token))
   :when (not (member prop parents))
    (removef prop (tok-children (prop-dest prop)))
    )
  (setf (tok-parents token) nil)
  (doloop (parent parents)
    (cond ((prop-object-p parent)
	   (push parent (tok-parents token)))
	  (t (let ((prop (cond ((kb-token-p parent)
				(make-isa-link :src token :dest parent :history-time history :belief-time belief))
			       ((listp parent)
				(make-isa-link :src token :dest (car parent) :history-time (cadr parent) :belief-time belief)))))
	       (push prop (tok-parents token))
	       (safe-add-child (prop-dest prop) prop)
	       (when props (tconc props prop))
	       )))
    )
  ;; inherit tok prim
  (when (null (tok-prim token))
    (doloop (prop (tok-parents token))
     :for fn := (tok-prim (prop-dest prop))
     :when fn
      (setf (tok-prim token) fn)
     :return nil)
    )
  )



(defun safe-add-instance (class prop)
  (tok-make-class class)
  (push prop (tok-instances class))
  )

(defun tell-update-types (token types history belief props theory)
					; remove redundant props from existing parent classes
  (doloop (prop (tok-inst-of token))
   :when (not (member prop types))
    (removef prop (tok-instances (prop-dest prop)))
    )
  (setf (tok-inst-of token) nil)
  (doloop (type types)
    (cond ((prop-object-p type)
	   (push type (tok-inst-of token)))
	  (t (let ((prop (cond ((kb-token-p type)
				(make-inst-link :src token :dest type :history-time history :belief-time belief))
			       ((listp type)
				(make-inst-link :src token :dest (car type) :history-time (cadr type) :belief-time belief)))))
	       (push prop (tok-inst-of token))
	       (safe-add-instance (prop-dest prop) prop)
	       (when props (tconc props prop))
	       )))
    )
  (if (null (tok-inst-of token)) (error "~%Token somehow got no types:'~a'~%" (tok-name token)))
  )
   

(defvar *ic-checking* t)

(defun real-tell (expr &key (theory *theory*) default-history (belief (std-belief)) context forward)
  (declare (special *ic-checking*))
  (let ((history (or default-history (default-history)))
	(errors (tconc nil))
	(props (tconc nil))
	(token nil))
    (flet ((find-token (pair)
	     (let ((type (lookup-type (if (consp pair) (car pair) pair) theory)))
	       (if (null type)
		 (tell-error 'undefined pair)
		 type))))
      (catch 'tell
	(let ((name (car expr))
	      (isa (or (getf (cdr expr) 'is-a) (getf (cdr expr) 'isa)))
	      (inst (or (getf (cdr expr) 'instance-of) (getf (cdr expr) 'instanceof) (getf (cdr expr) 'instance)))
	      (with (getf (cdr expr) 'with)))
	  (ifn (kb-token-p name)
	    (setq token name)
	    (setq name (tok-name token))
	   else
	    (setq token (lookup-type name theory))
	    )
	  (let* ((types (tell-build-class-list inst (if token (tok-inst-of token)) theory default-history belief errors))
		 (parents (tell-build-class-list isa (if token (tok-parents token)) theory default-history belief errors))
		 (type-level (find-set-level types history belief))
		 (parent-level (find-set-level parents history belief)))
	    (when (null types) (error "~%Token has no types: ~s~%" name))
	    (when (and (null token) (null types)) (tell-error 'need-types-for-new-token))
	    (when (eq type-level :error) (tell-error 'bad-type-level))
	    (when (eq parent-level :error) (tell-error 'bad-parent-level))
	    (when (and parents (eq type-level nil)) (tell-error 'token-may-not-have-isa-parents))
	    (when (and parents (not (or (eq type-level :+)
					(and (numberp type-level)
					     (numberp parent-level)
					     (eq (- type-level 1) parent-level)))))
	      (tell-error 'isa-level-and-parent-level-incompatible type-level parent-level))
	    (when (null token)
	      (setq token (new-token (type-name name theory) theory)))
	    (when inst (tell-update-types token types history belief props theory))
	    (when isa (tell-update-parents token parents history belief props theory))
	    (tell-assure-level-root token type-level history belief errors props theory)
	    (mapc #'(lambda (with-group)
		      (tell-with-group (string name) token with-group errors props #'find-token theory history belief))
		  with)
	    (tell-check-attr-inst token (car props) errors)
	    (when *ic-checking*
	      (let ((ac (make-theory :kb (theory-kb theory))) check)
		(tell-collect-ics token (car props) theory ac history belief)
		(do-trace
		 (dbgn-print 'props (car props))
		 (format t "~%THEORY~%")
		 (print-theory ac))
		(setq check (backward-theory-inference ac (list theory (theory-integrity-constraints theory)) belief))
		(when check
		  (tell-error 'integrity-constraint-failure (collect-explanation check))
		  ))
	      )
	    )	  
	  )
	)
      )
    (time-context-reset)
    (values token errors (car props))
    )
  )


(defun kb-print-attr (attr &optional (stream t))
  (format stream "<~a>(~{ ~a~^~} ): "
	    (normal-type (attr-token attr))
	    (mapcar #'(lambda (type-prop) (tok-name (prop-dest type-prop)))
		    (tok-inst-of (attr-token attr)))
	    )
  (if (and (attr-value attr) (listp (attr-value attr)))
    (format stream "~{~%- ~5T~a~}~%" (normal-clause (attr-value attr)))
    (format stream "~a" (normal-clause (attr-value attr)))
    )
  (format stream " [~a][~a]~%"
	    (normal-clause (attr-history attr))
	    (normal-belief (attr-belief attr))
	    )
  )


(defun kb-print-token (token &optional (stream t))
  (format stream "TOKEN ~a~%" (tok-name token))
  (format stream "INSTANCE OF~13T~{ ~a~^~}~%"
	  (mapcar #'(lambda (type-prop) (list (tok-name (prop-dest type-prop)) (normal-clause (prop-history type-prop))))
		  (tok-inst-of token)))
  (format stream "ISA~13T~{ ~a~^~}~%"
	  (mapcar #'(lambda (type-prop) (list (tok-name (prop-dest type-prop)) (normal-clause (prop-history type-prop))))
		  (tok-parents token)))
  (when (attr-tok-prop token)
    (format stream "COMPONENTS  ")
    (kb-print-attr (attr-tok-prop token)))
  (when (tok-attrs token) (format stream "WITH~%"))
  (do ((ap (lookup-index-cdr (tok-attrs token) :all) (cdr ap)))
      ((null ap))
    (kb-print-attr (car ap) stream)
    )
  )


(defun print-kb (&optional (theory *theory*)(stream t))
  (maphash #'(lambda (key val)
	       (if (kb-token-p val)
		   (kb-print-token val)
		 (dbgn-print '???? val)
		 )
	       (terpri)
	       )
	   (theory-tokens theory)
	   )
  )

(defun read-kb3 (name verbose &optional (theory *theory*) default-history (belief (std-belief)))
  (declare (special *errors*))
  (let ((history (if default-history
		   (if (time-int-p default-history)
		     default-history
		     (parse-time default-history (default-history) default-history theory nil belief)
		     )
		   (default-history))))
    (when verbose (dbgn-print 'read-history history))
    (if (open name :direction :probe)
      (with-open-file (input-stream name :direction :input)
	(do ((x nil)
	     (obj nil)
	     (fl (file-length input-stream)))
	    ((eq x 'done))
	  (setq x (read input-stream nil 'done))
	  (when (listp x)
	    (if verbose (print x))
	    (cond ((eq (car x) 'time)
		   (setq history (parse-time (cadr x) history nil theory nil belief)))
		  ((eq (car x) 'belief)
		   (setq belief (get-belief-time (if (cdr x) (cadr x) "now")
						 (if (cddr x) (caddr x) "+")
						 nil)))
		  ((eq (car x) 'clause))
		  ((eq (car x) 'eval) (doloop (form (cdr x)) (eval form)))
		  (t (setf obj (tell (if (eq (car x) 'defobject)
				       (cdr x)
				       x)
				     theory history belief))
		     (if verbose (format t "~%~a~%~%" (normal-clause obj)))
                     (when (and (consp obj) (eq (car obj) 'error)) (push obj *errors*))
		     ))
	    )
	  (if verbose (format t "~%~10,2f%" (/ (* 100.0 (file-position input-stream)) fl)))
	  )
	)
      t)
    )
  )



(defun tell-attr (token label val attr-token th history belief)
  (let* ((attr-token (lookup-type attr-token th))
	 (attr-prop (make-kb-attr :from (lookup-type token th)
				  :token attr-token
				  :value val
				  :history history
				  :belief belief)))
    (tell-index-attr attr-prop (lookup-type token th))
    )
  )


(defun init-prim-types (theory)  
  (setf (tok-prim (lookup-type 'string theory)) #'stringp)
  (setf (tok-prim (lookup-type 'number theory)) #'numberp)
  (setf (tok-prim (lookup-type 'time theory)) #'(lambda (token) (when (numberp token) t)))
  (setf (tok-prim (lookup-type 'description theory)) #'(lambda (token) (when (kb-description-p token) t)))
  (setf (tok-prim (lookup-type 'integer theory)) #'integerp)
  (setf (tok-prim (lookup-type 'list theory)) #'listp)
  (setf (tok-prim (lookup-type 'real theory)) #'realp)
  (setf (tok-prim (lookup-type 'symbol theory)) #'symbolp)
  (setf (tok-prim (lookup-type 'sexpr theory)) #'(lambda (token) t))
  (setf (tok-prim (lookup-type 'clause theory)) #'clause-p)
  (setf (tok-prim (lookup-type 'clause-set theory)) #'(lambda (token)
						       (when (not (dofind (clause token) (not (clause-p clause)))) t)))
  (setf (tok-prim (lookup-type 'time-interval theory)) #'time-int-p)
  (setf (tok-prim (lookup-type 'boolean theory)) #'(lambda (token) (when (or (eq token t) (null token)) t)))
  (setf (tok-prim (lookup-type 'theory theory)) #'theory-p)
  )


(defun init-kb ()
  (let* ((belief-alltime (make-belief-time :start :- :end :+))
	 (kb (new-kb :clause-number 0))
	 (theory (new-theory kb 'root))
	 (*ic-checking* nil)
	 theory-int theory-ded)
    (setq *next-time-context* 0)
    (flet ((add-inst (tok-name class-name &optional level)
	     (progn (add-prove-is (type-name tok-name theory) class-name theory *all-time* belief-alltime)
		    (if level
		      (setf (tok-inst-level (lookup-type tok-name theory)) level))))
	   (add-type (tok-name class-name) 
	     (let ((tok (lookup-type tok-name theory))
		   (type (lookup-type class-name theory)))
	       (assert-inst tok type *all-time* belief-alltime nil theory)))
	   (add-parent (tok-name class-name) 
	     (let ((tok (lookup-type tok-name theory))
		   (type (lookup-type class-name theory)))
	       (assert-isa tok type *all-time* belief-alltime nil theory)))
	   (add-attr (token label val attr-token)  
	     (tell-attr token label val attr-token theory  *all-time* belief-alltime)
	     )
	   (init-tell (expr) (tell expr theory *all-time* belief-alltime))
	   )
					; we have to eliminate these global vars ... they are theory dependent
      (setq *theory* theory)
      (setq *all-time* (make-time-int :start :- :end :+ ))
      (new-token (type-name 'proposition theory) theory)
      (setq *root-type* (lookup-type 'proposition theory ))
      (setf (tok-inst-level *root-type*) :+)
      (setf (theory-root-object theory) *root-type*)
      (setf (kb-all-time kb) *all-time*)
      (setf (kb-meta-attr kb) (find-string "meta-rule" theory))
      (setq theory-int (new-theory kb 'integrity-constraints))
      (setq theory-ded (new-theory kb 'deductive-rules :horn t))
      (includes theory theory-ded)

      (doloop (x *built-in-type-symbols*)
	(push (type-name x theory) *built-in-type-names*)
	(push (cons (type-name x theory) x) *built-in-type-alist*)
	)
      (add-inst (find-string "attribute" theory) "proposition" :+)
      (add-inst (find-string "attributeclass" theory) "proposition" :+)
      (setf (kb-attribute-class kb) (lookup-type "attribute" theory))
      (add-inst "individual" "proposition")
      (add-inst "individualclass" "proposition")
      (add-inst "class" "proposition" :+)
      (add-inst "omegaclass" "proposition" :+)
      
      (add-type "attributeclass" "attributeclass")
      (add-type "attribute" "attributeclass")

      (add-type "individual" "individualclass")
      (add-type "proposition" "individualclass")
      (add-type "omegaclass" "omegaclass")
      (add-type "class" "class")

      (add-parent "attribute" "proposition")
      (add-parent "attributeclass" "attribute")
      (add-parent "attribute" "class")
      (add-parent "individual" "proposition")
      (add-parent "individualclass" "individual")
      (add-parent "class" "proposition")
      (add-parent "individualclass" "class")
      (add-parent "omegaclass" "individualclass")

      (add-inst "m2-class" "class" 4)
      (add-inst "m1-class" "class" 3)
      (add-inst "s-class" "class" 2)
      (add-inst "token" "class" 1)
      
      (add-type "m1-class" "m2-class")
      (add-type "s-class" "m1-class")
      (add-type "token" "s-class")

      (add-parent "m2-class" "class")
      (add-parent "m1-class" "class")
      (add-parent "s-class" "class")
      (add-parent 'token 'proposition)

      (mapc #'(lambda (name) (when (not (equal name "proposition"))
			       (add-inst name 's-class)(add-parent name 'token)))
	    *built-in-type-names*)
      
      (setq *time-interval-type* (lookup-type 'time-interval theory))
      
      (setq *number-type* (lookup-type 'number theory))
      (setq *string-type* (lookup-type 'string theory))

      
      
      (add-inst (find-string "necessary" theory) "attributeclass")
      (add-inst (find-string "single" theory) "attributeclass")
      (add-inst (find-string "integrity-constraint" theory) "attributeclass")
      (add-inst (find-string "deductive-rule" theory) "attributeclass")
      (add-inst (find-string "input" theory) "attributeclass")
      (add-inst (find-string "output" theory) "attributeclass")
      (add-inst (find-string "meta-rule" theory) "attributeclass")
      (add-inst (find-string "lisp-rep" theory) "attributeclass")
      (add-inst (find-string "search-method" theory) "attributeclass")
      (add-inst (find-string "precondition" theory) "attributeclass")

      (setf (theory-integrity-constraints theory) theory-int)
      (setf (theory-ic-object theory) (lookup-type 'integrity-constraint theory))
      (setf (theory-deductive-rules theory) theory-ded)
      (setf (theory-ded-object theory) (lookup-type 'deductive-rule theory))
      
      (add-parent (find-string "necessary" theory) "attribute")
      (add-parent (find-string "single" theory) "attribute")
      (add-parent (find-string "integrity-constraint" theory) "attribute")
      (add-parent (find-string "deductive-rule" theory) "attribute")
      (add-parent (find-string "input" theory) "attribute")
      (add-parent (find-string "output" theory) "attribute")
      (add-parent (find-string "lisp-rep" theory) "attribute")
      (add-parent (find-string "meta-rule" theory) "attribute")
      (add-parent (find-string "search-method" theory) "attribute")
      (add-parent (find-string "precondition" theory) "attribute")

      (add-attr "class" "attribute" (lookup-type-error 'class theory) "attribute")
      (add-attr "class" "necessary" (lookup-type-error 'class theory) "necessary")
      (add-attr "class" "single" (lookup-type-error 'class theory) "single")
      (add-attr "class" "integrity-constraint" (lookup-type-error 'clause-set theory) "integrity-constraint")
      (add-attr "class" "deductive-rule" (lookup-type-error 'clause-set theory) "deductive-rule")
      (add-attr "class" "input" (lookup-type-error 'proposition theory) "input")
      (add-attr "class" "output" (lookup-type-error 'proposition theory) "output")
      (add-attr "class" "meta-rule" (lookup-type-error 'theory theory) "meta-rule")
      (add-attr "class" "lisp-rep" (lookup-type-error 'proposition theory) "lisp-rep")
      (add-attr "class" "search-method" (lookup-type-error 'proposition theory) "search-method")
      (add-attr "class" "precondtion" (lookup-type-error 'theory theory) "precondition")
      
      (telos-stash '(<= (multiple-valued |$p1([single])| |$p-class(single)| |$t(-,+)|)
		     (attr $x $l $cl $v1 $p1 $t)
		     (attr $x $l2 $cl $v2 |$p2([single])| $t)
		     (instance-of $p2 $p-class $t)
		     (not (= $v1 $v2)))       
		   theory belief-alltime 'multiple-value (list :integrity-check-ignore t))
      
      (telos-stash '(forall ($x |$class(class)| $label $label-class $v |$pclass(necessary)| |$t(-,+)|)
		     (or
		      (not (necessary-check $x $class $t))
		      (not (attr $class $label $label-class $v $pclass $t))
		      (provable (exists |$p([necessary])|
				 (and (attr $x $xlabel $xlabel-class $xv $p $t)
				      (instance-of $p $pclass $t))
				 ))))
		   theory belief-alltime 'necessary-check (list :integrity-check-ignore t))
      
      (init-tell `(,(lookup-type "attribute" theory)
		   with ((attribute (meta-rule theory)))))
      (init-tell `(,(lookup-type "attribute" theory)
		   with ((attribute (lisp-rep proposition)))))
      
      (init-tell `(,(lookup-type "single" theory)
		   with ((integrity-constraint
			  (single-constraint (=> (instance-of |$p1([single])| |$p-class(single)| |$t(-,+)|)
						 (not (multiple-valued $p1 $p-class $t))))))))
      (init-tell `(,(lookup-type "necessary" theory)
		   with ((integrity-constraint
			  (necessary-constraint (=> (instance-of $x |$class(class)| |$t(-,+)|)
						    (necessary-check $x $class $t)))))))
      
      (comment init-tell `(,(lookup-type "precondition" theory)
			   with ((integrity-constraint
				  (precond-constraint
				   (=> (attr $class $l "precondition" $v |$p(precondition)| |$t(-,+)|)
				       (instance-of $object $class $t)
				       (exists |$htime(-,+)| (and (overlaps $htime $t)
								  (holds $v $object $htime)))))))
			   ))
      
      (time-context-reset)
      (init-prim-types theory)
      theory)
    )
  )




(defun print-isa (token &optional (level 0) (theory *theory*))
  (let ((token (lookup-type token theory)))
    (format t "~vt~a~%" level (normal-clause token))
    (mapc #'(lambda (x) (print-isa (prop-src x) (+ level 5) theory)) (tok-children token))
    t
    )
  )




(defun print-inst (token &optional (level 0) (theory *theory*)(stream t))
  (let ((token (lookup-type token theory)))
    (format stream "~vt~a~%~vt   [" level (normal-clause token) level)
    (mapc #'(lambda (x) (format stream " ~a" (tok-name (prop-src x)))) (tok-instances token))
    (format stream " ]~%")
    (mapc #'(lambda (x) (print-inst (prop-src x) (+ level 5) theory) stream) (tok-children token))
    t
    )
  )

(defvar *errors* nil)

(defmacro loadkb (kbname &optional (init t) (verbose nil) default-history)
  `(mlet* ((kbname ,kbname)
	   (kbstring (if (symbolp kbname)
		       (string-downcase (force-string kbname))
		       (force-string kbname)))
	   (kbpath (merge-pathnames kbstring  ',(merge-pathnames "foo.kb"))))
	  (declare (special th))
	  (if ,verbose (print kbpath))
	  (ifn ,init (setq th (init-kb)) (setq *errors* nil))
	  (read-kb3 kbpath ,verbose *theory* ,default-history)
	  )
  )



(defmacro print-ic (&optional (clauses nil) (theory '*theory*))
  `(print-theory (theory-integrity-constraints ,theory) :clauses ,clauses)
  )


(defmacro print-dr (&optional (clauses nil) (theory '*theory*))
  `(print-theory (theory-deductive-rules ,theory) :clauses ,clauses)
  )


(defmacro tok (&rest names)
  `(doloop (name ',names)
       (if (lookup-type name)
	 (kb-print-token (lookup-type name))
	 (format t "~&Object ~a not found" name)
	 )
       (terpri))
  )




(defun print-index (a-list)
  (doloop (bucket a-list)
    (dbgn-print 'key (car bucket))
    (if (and (consp (cdr bucket)) (consp (cadr bucket)) (not (consp (caadr bucket))))
      (print-index (cdr bucket))
      (pprint (normal-clause (cdr bucket)))
      )
    )
  )


(defun compare-attr-label (props class-label history belief)
  (do-with-time (prop props history belief)
   :when (or (and (attr-tok-prop (prop-dest prop))
		  (eq class-label (attr-label (attr-tok-prop (prop-dest prop)))))
	     (compare-attr-label (tok-parents (prop-dest prop)) class-label history belief))
   :return t)
  )


(defun special-token-attr-list (token class-label history belief state)
  nil)


(defun gav (token class-label h b result)
  (doloop (attr (lookup-index-cdr (tok-attrs token) class-label))
   :when (and (belief-satisfies b (attr-belief attr))
	      (time-intersect-p h (attr-history attr) b)
	      (compare-attr-label (tok-inst-of (attr-token attr)) class-label h b))
    (tconc result attr)
    )
  (do-with-time (prop (tok-parents token) h b)
    (gav (prop-dest prop) class-label h b result)
    )
  )


(defun get-attr-value (token class-label history belief &optional state)
  (if (kb-token-p token)
    (let ((result (tconc nil)))
      (gav token class-label history belief result)
      (car result)
      )  
    (if (special-token-p token)
      (special-token-attr-list token class-label history belief state)
      )
    )
  )


(defmacro defobject (name &rest desc)
  `(tell '(,name ,@desc) *theory* (theory-all-time *theory*))
  )
  