;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         user-methods.l
; Description:  Contains the code for user defined methods.
; Author:       Eric G. Muehle
; Created:      9-Jul-86
; Package:      FROBS
; RCS $Header: user-methods.l,v 2.4 89/01/12 10:25:17 kessler Exp $
;
; (c) Copyright 1986, 1987, University of Utah, all rights reserved.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)

;;; Extract the documentation string, and declarations from a body.  Returns
;;; the body, declarations, and doc-string.
(defun frobs-parse-body (body doc-allowed-p)
  (let ((documentation nil)
	(declarations  nil))
    (loop
     (let ((first (if (and (listp (car body))
			   (eq (caar body) 'declare))
		       (car body)
		       (macroexpand-1 (car body) nil))))
	(cond ((and doc-allowed-p (stringp first) (cdr body))
	       (setq documentation first)
	       (pop body))
	      ((and (consp first) (eq (car first) 'declare))
	       (setq declarations (append declarations (cdr first)))
	       (pop body))
	      (t  (return nil)))))
    (values body declarations documentation)))

;;; Transforms the code into the correct format.  
(defun transform (code priv in-mod name)
  (multiple-value-bind (body decs doc) 
    (frobs-parse-body code t)
    ;; make the decs list so we can splice it in
    (when decs 
      (push 'declare decs)
      (setf decs (list decs)))
    ;; make the doc a list for splicing
    (when doc
      (setf doc (list doc)))
    ;; make sure that we can have access to the private methods if
    ;; we are defining a public method inside a module.
    (cond ((and in-mod (not priv))
	   `(,@doc
	     ,@decs
	     (let ((@@mod@@ *module*))
	       (unwind-protect
		 (progn
		   (setq *module* (get-type $self))
		   ,@body)
		 (setq *module* @@mod@@)))))
	  (priv
	   `(,@doc
	     ,@decs
	     (cond ((eq *module* (get-type $self))
		    ,@body)
		   (t (undefined-method-error ',name (get-type $self))))))
	  (t `(,@doc
	       ,@decs
	       ,@body)))))


;;; Will retroactively inherit a method to all children that
;;; do not already own this method
(defun retro-inherit (class-frob method-name private show in-mod)
  (let ((tuple (list (class-frob-name class-frob) method-name)))
    (mapcar #'(lambda (x) 
		(retro-aux x method-name tuple private show in-mod))
	    (all-class-instances class-frob))))


;;; Collects those methods that have the same name but are not owned
;;; by the current class.  If the method is owned by the current class
;;; then ok? is set to nil and all retro inheritance is stopped for this
;;; class
(defmacro retro-slots (ct fn class name type result set? val ok)
  `(when ,ok
     (let ((index ,ct))
       (dolist (tuple (,fn ,class))
	 (when (eq (method-name2 tuple) ,name)
	   (when (eq (method-context2 tuple)(class-frob-name class))
	     (setf ,ok nil))
	   (push (list tuple ',type index) ,result)
	   (setf ,set? ',val))
	 (incf index)))))
  

;;; Checks each slot and all methods owned by this class and recreates the 
;;; appropriate method for it.
(defun retro-aux (class name tuple private show in-mod)
  (let* ((owner (class-frob-name class))
	(open   (length (class-frob-open class)))
	(mv     (+ open (length (class-frob-mv class))))
	(priv   (+ mv (length (class-frob-private class))))
	(ok?    t)
	tuple-list set slots?)
    (retro-slots 0 class-frob-open class name open tuple-list set nil ok?)
    (retro-slots 0 class-frob-set-open class name set-open tuple-list set t ok?)
    (retro-slots open class-frob-mv class name mv tuple-list set nil ok?)
    (retro-slots open class-frob-set-mv class name set-mv tuple-list set t ok?)
    (retro-slots mv class-frob-private class name private tuple-list set nil ok?)
    (retro-slots mv class-frob-set-generic class name set-private tuple-list set t ok?)
    (retro-slots priv class-frob-generic class name generic tuple-list set nil ok?)
    (retro-slots priv class-frob-set-generic class name set-generic tuple-list set t ok?)
    ;; determine if the any of the methods are slots
    (setf slots? tuple-list)
    ;; if the method is defined over this class then we can stop now
    (if (member (list owner name)(class-frob-methods class) :test #'equal)
      (setf ok? nil))
    (when ok?
      (dolist (m (class-frob-methods class))
	(when (eq (method-name2 m) name)
	  (pushnew (list m 'meth 'meth) tuple-list :test #'equal))))
    (when ok?
      (pushnew tuple (class-frob-methods class) :test #'equal)
      (when private
	(pushnew tuple (class-frob-private-methods class) :test #'equal))
      (when in-mod
	(pushnew tuple (class-frob-in-mod-methods class) :test #'equal))
      (when show
	(pushnew tuple (class-frob-show-methods class) :test #'equal))
      (cond ((null tuple-list)
	     (push tuple tuple-list))
	    (t (pushnew (list tuple 'meth 'meth) tuple-list :test #'equal))))
    (cond ((not ok?) nil)
          ((null (cdr tuple-list))
	   `(progn
	      (add-tuple-to-the-right-lists 
	       ',tuple ',name (get-class-frob ',owner) ',private
	       ',show ',in-mod)
	      ,(car (make-methods (list tuple) owner))))
          (set
	   `(progn
	      (add-tuple-to-the-right-lists 
	       ',tuple ',name (get-class-frob ',owner) ',private
	       ',show ',in-mod)
	      ,(car (make-mixed-set tuple-list owner 
				    (class-frob-parent class)))))
          (slots?
	   `(progn
	      (add-tuple-to-the-right-lists 
	       ',tuple ',name (get-class-frob ',owner) ',private
	       ',show ',in-mod)
	      ,(car (make-mixed-get tuple-list owner 
				    (class-frob-parent class)))))
          (t
	   (setf tuple-list (mapcar #'car tuple-list))
	   `(progn
	      (add-tuple-to-the-right-lists 
	       ',tuple ',name (get-class-frob ',owner) ',private
	       ',show ',in-mod)
	      ,(car (make-context-methods tuple-list owner 
					  (class-frob-parent class))))))))

;;; creates a method name 
(defun create-meth-name (class name)
  (intern (format nil "~S$$~S" class name) (symbol-package class)))

;;; Defines the user method
(defmacro def-method ((class name &rest priv-show) param &rest code)
  (setf class (if (symbolp class)
		(get-class-frob class)
		(get-class-name class)))
  (let* ((class-name (class-frob-name class))
	 (aux-fn     (create-meth-name class-name name))
	 (in-mod     (or (eq *module* class-name)
			 (and (member :in-module priv-show)
			      (funcall (gethash class-name *key*) nil))))
	 (private    (member :private priv-show))
	 (show       (and (member :show priv-show)(null param)))
	 (only       (member :only priv-show))
	 (tuple      (list class-name name))
	 inherit)
    ;; add the $self parameter to the parameter list
    (push '$self param)
    ;; create the update function list
    ;; transform the code into the proper access functions
    (setf code (transform code private in-mod name))
    (add-tuple-to-the-right-lists tuple name class private show in-mod)
    (unless only (setf inherit (retro-inherit class name private show in-mod)))
    ;; define the method function
    `(eval-when (load compile eval)
       (defun ,aux-fn ,param ,@code)
       (set '*new-frob* (gethash ',class-name *class-hash*))
       ,(make-one-method name class-name aux-fn 'method param :inherit nil)
       ,@inherit
       (add-tuple-to-the-right-lists 
	',tuple ',name *new-frob* ',private ',show ',in-mod)
       ',name)))

  
;;; Removes a tuple associated with a method name from some place
(defmacro remove-tuple-name (name place class)
  `(setf (,place ,class)
	 (delete ,name (,place ,class) :key #'method-name2)))


;;; Adds the method to the correct lists
(defun add-tuple-to-the-right-lists (tuple name class private show in-mod)
  ;; remove this method from any slot lists
  (remove-tuple-name name class-frob-open class)
  (remove-tuple-name name class-frob-set-open class)
  (remove-tuple-name name class-frob-mv class)
  (remove-tuple-name name class-frob-set-mv class)
  (remove-tuple-name name class-frob-generic class)
  (remove-tuple-name name class-frob-set-generic class)
  (remove-tuple-name name class-frob-private class)
  (remove-tuple-name name class-frob-set-private class)
  ;; lets remove all versions of this method from this frob
  (remove-tuple-name name class-frob-methods class)
  (remove-tuple-name name class-frob-private-methods class)
  (remove-tuple-name name class-frob-show-methods class)
  (remove-tuple-name name class-frob-in-mod-methods class)
  ;; now lets update the lists
  (push tuple (class-frob-methods class))
  (when private
    (push tuple (class-frob-private-methods class)))
  (when show
    (push tuple (class-frob-show-methods class)))
  (when in-mod
    (push tuple (class-frob-in-mod-methods class))))

;;; Should only be called within the scope of a def-method
(defmacro call-method ((class method) &rest args)
  (let ((fn (get-generic-fn* method (get-class-frob class))))
    (unless fn
      (error "Undefined method ~S for class ~S in CALL-METHOD~%"
	     method class))
    (if (integerp (cdr fn))
      `(with-frob ,class (funcall ',method $self ,@args))
      `(with-frob ,class (funcall ',(create-meth-name class method)
				  $self ,@args)))))

;;; Should only be called within the scope of a def-method
(defmacro apply-method ((class method) &rest args)
  (let ((fn (get-generic-fn* method (get-class-frob class))))
    (unless fn
      (error "Undefined method ~S for class ~S in APPLY-METHOD~%"
             method class))
    (if (integerp (cdr fn))
      `(with-frob ,class (funcall ',method $self ,@args))
      `(with-frob ,class (apply ',(create-meth-name class method)
				$self ,@(butlast args) ,(car (last args)))))))

;;; Should only be called within the scope of a def-method
(defmacro call-method-in-context ((class method) &rest args)
  (let ((fn (get-generic-fn* method (get-class-frob class))))
    (unless fn
      (error "Undefined method ~S for class ~S in CALL-METHOD-IN-CONTEXT~%"
	     method class))
    (if (integerp (cdr fn))
	`(funcall ',method $self ,@args)
	`(funcall ',(create-meth-name class method)
		  $self ,@args))))

;;; Should only be called within the scope of a def-method
(defmacro apply-method-in-context ((class method) &rest args)
  (let ((fn (get-generic-fn* method (get-class-frob class))))
    (unless fn
      (error "Undefined method ~S for class ~S in APPLY-METHOD-IN-CONTEXT~%"
             method class))
    (if (integerp (cdr fn))
	`(funcall ',method $self ,@args)
	`(apply ',(create-meth-name class method)
		$self ,@(butlast args) ,(car (last args))))))

;;; Runs a previous version of this method
(defmacro run-super (name &rest args)
  `(let (fn)
     (dolist (p (class-frob-parent (get-class-frob (frob-type $self))))
       (when (setf fn (get-generic-fn* ',name p))
	 (return
	  (if (integerp (cdr fn))
	      (funcall ',name $self ,@args)
	      (funcall (car fn) $self ,@args)))))))

;; End of file.
