;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; File:         the-rules.l
; Description:  Forward chaining rules for frobs.
; Author:       Eric Muehle, with modification by Eric Eide
; Created:      22-Jul-87
; Package:      FROBS
; RCS $Header: /u/misc/pass/lisp/tools/frobs/RCS/the-rules.l,v 2.13 1993/08/17 22:50:40 eeide Exp $
;
; (c) Copyright 1987, University of Utah, all rights reserved
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Copyright (c) 1987 Eric G. Muehle and Utah Portable AI Support 
;;; Systems Project.  This program may be used, copied, modified, and 
;;; redistributed freely for noncommercial purposes, so long as this 
;;; notice remains intact and the program is redistributed in its 
;;; entirity.  Any commercial use of the software or derivative thereof
;;; requires a redistribution license from: Utah PASS Project 3190 M.E.B 
;;; Department of Computer Science University of Utah Salt Lake City, UT 
;;; 84112

(in-package 'frobs)

(defvar *special-clauses* '(evalp eval known not bind) "contains the special clauses")

(defvar *conflict* nil)

(defvar *tuple-queue* nil)

(defvar *temp-queue* nil)

(defvar *depth* 0)

(defvar *queue* 0)

(defvar *rule-index* (cons 1 2))

(defvar *rules-running* nil)

(defvar *back-of-queue* nil "points to the end of the queue")

(defvar *par-mode* nil)

(defvar *rules*
	(make-hash-table :size 100 :rehash-size 100
			 :rehash-threshold 80 :test #'equal)
	"contains the (slot . class) -> rulename associations")

(defvar *redefine*
	(make-hash-table :size 100 :rehash-size 100
			 :rehash-threshold 80)
	"contains rulename -> (slot . class) associations")

(defvar *rule-value*
	(make-hash-table :size 100 :rehash-size 100
			 :rehash-threshold 80)
	"contains rulename -> rule-value associations")

(defvar *text-order* nil)

(defvar *traced-rule-names* nil "list of rule names traced when fired")
(defvar *trace-all-rules* nil "if non-NIL, all rules are traced when fired")

;;; Predicate that returns T if a symbol is a var of the form ?...
(defun varp (thing)
  (when (symbolp thing)
    (char= (elt (symbol-name thing) 0) #\?)))

;;;; Accessor functions.  A clause is a 3 element list of the form:
;;;; (slot frob var).

;;; Returns the frob from a clause
(defun get-f (cl)
  (second cl))

;;; Returns the slot from a clause
(defun get-slot (cl)
  (car cl))

;;; Returns the variable or constant value from a clause
(defun get-v (cl)
  (third cl))

;;; Returns the rule value for a given @@rule
(defun rule-value (rule)
  (gethash rule *rule-value*))

;;; Sets the value for a @@rule
(defun set-rule-value (rule value)
  (setf (gethash rule *rule-value*) value))


;;; User interface for returning a rule value
(defun get-rule-value (rulename)
  (rule-value (make-@@-name rulename)))

;;; User interface for changing the rule value of a rule.
(defun change-rule-value (rulename value)
  (let ((new-name (make-@@-name rulename))
	rule-list)
    (unless (equal value (rule-value new-name))
      (set-rule-value new-name value)
      (dolist (tuple (gethash new-name *redefine*))
	(setf rule-list (gethash tuple *rules*))
	(setf rule-list (delete new-name rule-list :count 1))
	;; insertion sort
	(setf (gethash tuple *rules*) 
	      (insert-sort new-name value rule-list))))))

;;; Returns the slot type for a given slot in a clause
(defun rule-slot-type (cl type)
  (cond ((varp (get-f cl))
	 (second (assoc (get-f cl) type)))
	((eq 'get-class-frob (car (get-f cl)))
	 (second (second (get-f cl))))
	(t (get-type (get-frob (second (second (get-f cl))))))))

;;;; Each rule has 2 functions:  an @@fn and a premise/conc function.
;;;; The @@fn creates tuples and puts them on the tuple Q, while the
;;;; prem/conc function is used when checking the validity of a tuple
;;;;  and is used to assert the conc.

;;;; Each firing clause in a premise is translated into a when clause in
;;;; both functions.  A when clause in a @@fn determines if the rule
;;;; could be true, and installs the correct rule number information
;;;; on the tuple.  A when clause in a prem/conc function binds the firing
;;;; firing clause and sets the state for the rest of the premise evaluation.

;;; Will return a set of premises each with a unique firing cl
(defun get-firing-cls (prem)
  (let (temp)
    (dolist (cl prem)
      (when (and (not (member (get-slot cl) *special-clauses*))
		 (not (varp (get-slot cl))))
	(push cl temp)))
    (reverse temp)))

;;; Creates the new name for the tuple making functions
(defun make-@@-name (rulename)
  (intern (concatenate 'string "@@" (symbol-name rulename))
	  (symbol-package rulename)))

;;; Makes an @@fn for a rule.  It defines a function of 4 parameters:
;;; $frob -> the frob that was just asserted, $slot -> the slot that
;;; was just asserted on, $class -> the class of $frob and $value ->
;;; the value that was just asserted.
(defun make-@@-rule (name new-name prem type mv)
  `(defun ,new-name ($frob $slot $class $value)
     ,(when (zerop (length type))
	`(declare (ignore $class)))
     (let (%result
	   ,@(mapcar #'car type))
       ,(make-@@-cond prem type mv name))))
  
;;; Creates the cond statement of all of the when clauses for the @@fn.
(defun make-@@-cond (prem type mv name)
  `(cond
    ,@(mapcar 
       #'(lambda (x)
	   (make-@@-one-when prem x type mv name))

       (similar-clause prem type))))

;;; If more than one clause can fire at the same time, we need to adjust
;;; the premise formation.  This function collects all similar clauses
;;; and puts them into equivalent groups.  
(defun similar-clause (prem type)
  (let (result temp (temp-prem prem) cl)
    (loop 
      (setf cl (pop temp-prem))
      (unless cl (return nil))
      (setf temp (similar-clause-aux cl temp-prem type nil))
      (dolist (r temp)
	(setf temp-prem (remove r temp-prem :test #'equal :count 1)))
      (cond (temp 
	     (rplacd (last temp)(list cl))
	     (push temp result))
	    (t (push cl result))))
    (nreverse result)))

;;; Collects all similar clauses into a list out of the remaining premise
;;; clauses.
(defun similar-clause-aux (clause prem type collection)
  (cond ((null prem) collection)
	((and (eq (get-slot clause)
		  (get-slot (car prem)))
	      (eq (rule-slot-type clause type)
		  (rule-slot-type (car prem) type)))
	 (similar-clause-aux clause (cdr prem) type 
			     (cons (car prem) collection)))
	(t (similar-clause-aux clause (cdr prem) type collection))))

;;; calculates the rule-num for a given clause
(defun calc-rule-num (cl prem)
  (- (length prem)
     (length (member cl prem :test #'equal))))

;;; Makes one cond clause for an @@ rule-fn
(defun make-@@-one-when (prem clause type mv name)
  (cond ((listp (car clause))  ; clause is a colection of similar clauses
	 `((and (eq $slot ',(get-slot (car clause)))
                (member ',(rule-slot-type (car clause) type)
                        (inherits-from (get-class-frob $class))))
	   ,@(reverse
	      (mapcar #'(lambda (x) 
			  (make-@@-similar-clause x type name prem mv))
		      clause))))
	((varp (get-f clause))
	 `((and (eq $slot ',(get-slot clause))
                (member ',(rule-slot-type clause type)
                        (inherits-from (get-class-frob $class)))
		,(get-val-part clause mv))
	    (setf ,(get-f clause) $frob)
	    ,(make-@@-dolists clause prem type mv)
	    (funcall *conflict* 
		     (list ',name $frob 
			   ,(calc-rule-num clause prem) %result))))
	(t 
	 `((and (eq $slot ',(get-slot clause))
		     (equal ,(get-f clause) $frob)
		     ,(get-val-part clause mv))
	    ,(make-@@-dolists clause prem type mv)
	    (funcall *conflict* 
		     (list ',name $frob 
			   ,(calc-rule-num clause prem) %result))))))

;;; Creates one when form for a similar clause
(defun make-@@-similar-clause (cl type name prem mv)
  `(when ,(get-val-part cl mv)
     (setf %result nil)
     ,(make-@@-dolists cl prem type mv)
     (funcall *conflict* 
	      (list ',name $frob 
		    ,(calc-rule-num cl prem) %result))))

;;; Returns the value checking form for the cond clauses in the @@fn's.
(defun get-val-part (cl mv)
  (cond ((and (member (get-slot cl) mv)
	      (varp (get-v cl)))
	 `(,(get-slot cl) $frob))
	((member (get-slot cl) mv)
	 `(equal $value ',(get-v cl)))
	((varp (get-v cl))
	 `(not (eq $value *undefined*)))
	(t `(equal $value ',(get-v cl)))))

;;; Creates the dolist form for the when clause.
(defun make-@@-dolists (clause prem type mv)
  (let (forms)
    ;; if the clause is fired from an mv slot then we need to push the 
    ;; value on the tuple.
    (when (member (get-slot clause) mv)
      (setf forms '($value)))
    ;; set up the prem in the correct order
    (setf prem (remove clause prem :test #'equal :count 1))
    ;; ok now lets build those dolists
    (dolist-dispatch (car prem)(cdr prem) type mv 
		     (list (get-f clause)) (list (get-v clause)) forms)))

;;; Makes the dolist form.  Basically we need to iterate over every
;;; unique frob variable, and every multiple value slot.  We then
;;; push the instances or values onto a new tuple.  A new tuple is made 
;;; each possible combination of frobs/values.
(defun dolist-dispatch (cl rest type mv fbound sbound forms)
  (cond ((null cl)
	 `(push (list ,@(reverse forms)) %result))
	((and (varp (get-f cl))
	      (not (member (get-f cl) fbound)))
	 `(dolist (,(get-f cl)
                   (all-instances (get-class-frob ',(rule-slot-type cl type))))
	    ,(dolist-dispatch cl
			      rest
			      type
			      mv 
			      (cons (get-f cl) fbound)
			      sbound
			      (cons (get-f cl) forms))))
	;; mv slot var
	((and (varp (get-v cl))
	      (member (get-slot cl) mv)
	      (not (member (get-slot cl) sbound)))
	 `(dolist (,(get-v cl) (,(get-slot cl) ,(get-f cl)))
	    ,(dolist-dispatch (car rest) 
			      (cdr rest) 
			      type
			      mv 
			      ;; the value could be a frob
			      (cons (get-v cl) fbound)
			      (cons (get-v cl) sbound)
			      (cons (get-v cl) forms))))
	(t (dolist-dispatch (car rest)(cdr rest) type mv fbound sbound forms))))


;;;; Need to make the prem/conc function.  It contains 2 parts.  The first part
;;;; binds the firing premise and sets up the state for the rest of the premise 
;;;; to be evaluated in.  The second part strips off values from the tuples
;;;; and binds them into the premise.  If the premise is true, then fire
;;;; the conclusion code.
;;;; The prem conc rules look like:
;;;; (defun rule-name ($frob $rule-num $tuple)
;;;;   (let (local + type vars)
;;;;     (case $rule-num
;;;;        initialization code)
;;;;     ;; do the premise
;;;;     (when ...
;;;;       (when 
;;;;          (do-the-conc)))))

;;; Creates the prem/conc rule function.
(defun make-rule-fn (name prem fire-prem conc type local mv)
  `(defun ,name ($frob $rule-num $tuple)
     ,(cond ((zerop (+ (length type)(length mv)))
	     `(declare (ignore $frob)(ignore $tuple)))
	    ((zerop (length type))
	     `(declare (ignore $frob))))
     (let ,(make-rule-let type local)
       ,(make-rule-case name fire-prem mv)
       ,(make-rule-prem (car prem) (cdr prem)
			(make-rule-conc name conc type local) mv nil))))

;;; Creates the let binding for the variables in the function.
(defun make-rule-let (type local)
  (nconc (mapcar #'(lambda (x) (list x ''unbound)) local)
	 (mapcar #'(lambda (x) (list (car x) ''unbound)) type)))

;;; Creates the case statement for the prem/conc function.
(defun make-rule-case (name prem mv)
  (let ((index -1)
	cases frob slot val)
    (dolist (cl prem)
      (setf frob (get-f cl))
      (setf slot (get-slot cl))
      (setf val (get-v cl))
      (incf index)
      (cond ((and (varp frob)
		  (varp val)
		  (member slot mv))
	     (push `(,index (setf ,frob $frob)
			    (setf ,val (pop $tuple))
			    (unless (member ',val (,slot ,frob) :test #'equal)
			      (return-from ,name)))
		   cases))
	    ((and (varp frob)
		  (member slot mv))
	     (push `(,index (setf ,frob $frob)
			    (unless (member ',val (,slot ,frob) :test #'equal)
			      (return-from ,name)))
		   cases))
	    ((varp frob)
	     (push `(,index (setf ,frob $frob)) cases))
	    (t
	     (push `(,index nil) cases))))
    `(case $rule-num ,@(nreverse cases))))

(defun make-rule-conc (name conc type local)
  (let ((type-and-local-vars (nconc (mapcar #'(lambda (x) (car x)) type)
				    local)))
    `(progn
      (when (traced-rule-p ',name)
	(print-traced-rule-firing ',name ',type-and-local-vars
				  (list ,@type-and-local-vars)))
      ,conc)))

;;; all of the above functions assume that that the non firing clauses have
;;; been removed from the premise

;;; Creates the nested when forms to evaluate the premise.  This function
;;; should be given the full premise.
(defun make-rule-prem (cl rest conc mv bound)
  (cond ((null cl) `(progn (when *par-mode* (run-rules)) ,conc))
	;; sepcial clauses
	((member (get-slot cl) *special-clauses*)
	 (make-special-clause cl rest conc mv bound))
	;; unbound frob var
	((and (varp (get-f cl))
	      (not (member (get-f cl) bound)))
	 `(progn (when (eq 'unbound ,(get-f cl))
		   (setf ,(get-f cl) (pop $tuple)))
		 ,(make-rule-prem cl rest conc mv (cons (get-f cl) bound))))
	;; unbound mv var
	((and (varp (get-v cl))
	      (member (get-slot cl) mv)
	      (not (member (get-v cl) bound)))
	 `(progn (when (eq 'unbound ,(get-v cl))
		   (setf ,(get-v cl)(pop $tuple)))
		 ,(make-rule-prem (car rest)(cdr rest) conc 
				  mv (cons (get-v cl) bound))))
	;; unbound var
	((and (varp (get-v cl))
	      (not (member (get-v cl) bound)))
	 `(when 
	    (not (eq *undefined* 
		     (setf ,(get-v cl) 
			   ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))))
	    ,(make-rule-prem (car rest)(cdr rest) conc mv (cons (get-v cl) bound))))
	;; bound mv var
	((and (member (get-slot cl) mv)
	      (varp (get-v cl)))
	 `(when (member ,(get-v cl) 
			,(make-rule-slot-lookup (get-slot cl)(get-f cl)) 
			:test #'equal)
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	;; mv constant
	((member (get-slot cl) mv)
	 `(when (member ',(get-v cl) 
			,(make-rule-slot-lookup (get-slot cl)(get-f cl)) 
			:test #'equal)
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	;; bound var
	((varp (get-v cl))
	 `(when (equal ,(get-v cl) ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	;; constant
	((not (varp (get-v cl)))
	 `(when (equal ',(get-v cl) ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	;; anything else is an error
	(t
	 (error "Bad rule premise in~%~S" cl))))

;;; Creates the special clause bindings
(defun make-special-clause (cl rest conc mv bound)
  (cond ((eq (get-slot cl) 'eval)
	 `(progn 
	    ,(second cl)
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	((eq (get-slot cl) 'evalp)
	 `(when ,(second cl)
	    ,(make-rule-prem (car rest)(cdr rest) conc mv bound)))
	((eq (get-slot cl) 'bind)
	 (if (member (second cl) bound)
	   `(when (equal ,(second cl) ,(third cl))
	      ,(make-rule-prem (car rest)(cdr rest) conc mv bound))
	   `(progn 
	      (setf ,(second cl) ,(third cl))
	      ,(make-rule-prem (car rest)(cdr rest) conc mv (cons (second cl) bound)))))
	((eq (get-slot cl) 'known)
	 (make-known cl rest conc mv bound))
	((eq (get-slot cl) 'not)
	 (make-not (second cl) rest conc mv bound))))

;;; The default ask user fn for single valued slots
(defun known-ask-fn (frob slot)
  (format t "What is the value for ~S in ~S ?~%" slot frob)
  (assert-val frob slot (read)))

;;; The default ask user fn for multiple valued slots
(defun known-ask-mv-fn (frob slot)
  (format t 
   "What are the values for the MV slot ~S in ~S ? ~%(result should be a list)~%"
   slot frob)
  (assert-vals frob slot (read)))

;;; Makes the known special clause
;;; (known (foo ?foo ?x) (ask-foo-fn ?foo ?x))
(defun make-known (cl rest conc mv bound)
  (let ((frob (get-f (second cl)))
	(slot (get-slot (second cl)))
	(val  (get-v (second cl)))
	(fn   (third cl)))
    (unless fn 
      (if (member slot mv)
	(setf fn `(known-ask-mv-fn ,frob ',slot))
	(setf fn `(known-ask-fn ,frob ',slot))))
    (unless (listp fn) (error "Bad ask fn in KNOWN ~S" cl))
    ;; its an error to have a KNOWN with an unbound frob var
    (when (and (varp frob)
	       (not (member frob bound)))
      (error "Illegal KNOWN clause, the var ~S must be bound in ~S." frob cl))
    ;; its an error to have a KNOWN clause with a bound var for now
    (cond ((or (and (varp val)
		    (member val bound))
	       (not (varp val)))
	   (error "Illegal KNOWN clause, ~S must be an unbound var in ~S." val cl))
	  (t
	   (if (member slot mv)
	     `(progn
		(unless (,slot ,frob)
		  ,fn)
		(setf ,val (car (,slot ,frob)))
		,(make-rule-prem (car rest)(cdr rest) conc mv (cons val bound)))
	     `(progn
		(when (equal *undefined* (,slot ,frob))
		  ,fn)
		(setf ,val (,slot ,frob))
		,(make-rule-prem (car rest)(cdr rest) conc mv (cons val bound))))))))


  
;;; Makes the not special clause
(defun make-not (cl rest conc mv bound)
  (when (and (varp (get-f cl))(not (member (get-f cl) bound)))
    (error "Illegal NOT clause, the ~S var must be bound in ~S." 
	   (get-f cl) (list 'not cl)))
  ;; if the val var is bound then we check for unbound
  (cond ((and (varp (get-v cl))
	      (member (get-v cl) bound))
	 (if (member (get-slot cl) mv)
	   `(unless 
	     (member ,(get-v cl)
		     ,(make-rule-slot-lookup (get-slot cl)(get-f cl))
		     :test #'equal)
	     ,(make-rule-prem (car rest)(cdr rest) conc mv bound))
	   `(unless (equal ,(get-v cl) ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))
	      ,(make-rule-prem (car rest)(cdr rest) conc mv bound))))
	((not (varp (get-v cl)))
	 (if (member (get-slot cl) mv)
	   `(unless 
	     (member ',(get-v cl)
		     ,(make-rule-slot-lookup (get-slot cl)(get-f cl))
		     :test #'equal)
	     ,(make-rule-prem (car rest)(cdr rest) conc mv bound))
	   `(unless (equal ',(get-v cl) ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))
		    ,(make-rule-prem (car rest)(cdr rest) conc mv bound))))
	(t
	 (if (member (get-slot cl) mv)
	   `(unless ,(make-rule-slot-lookup (get-slot cl)(get-f cl))
	      ,(make-rule-prem (car rest)(cdr rest) conc mv bound))
	   `(when (eq *undefined* ,(make-rule-slot-lookup (get-slot cl)(get-f cl)))
	      ,(make-rule-prem (car rest)(cdr rest) conc mv bound))))))

;;; Makes the slot call.  If the slot is a var then a funcall is returned,
;;; otherwise a invocation on the slot is made.
(defun make-rule-slot-lookup (slot frob)
  (if (varp slot)
    `(funcall ,slot ,frob)
    `(,slot ,frob)))

;;; The macro that expands into the 2 fucntions and installs the rule into
;;; the approriate places.
;;;
;;; ENE, August 1993: This macro used to be named DEF-RULE, but now it is the
;;; "internal" macro that is called after the syntax/sanity of the rule has
;;; been checked.  The new DEF-RULE macro is defined later on in this file.
;;;
(defmacro def-rule-internal (name &key type mv local prem conc (value 0))
  (let ((fire-prem (get-firing-cls prem))
	(new-name    (make-@@-name name)))
    `(eval-when (load compile eval)
       ,(make-@@-rule name new-name fire-prem type mv)
       ,(make-rule-fn name prem fire-prem conc type local mv)
       (install-rule ',new-name ',fire-prem ',type ',value)
       ',name)))

;;; Installs the rule on the approriate hash tables.  NAME => @@name
(defun install-rule (name fire-prem type value)
  (let (tuple rule-list)
    (pushnew name *text-order*)
    ;; remove all of the old links to it
    (remove-rule name)
    ;; set the rule value
    (set-rule-value name value)
    (dolist (cl fire-prem)
      (dolist (child (cons (rule-slot-type cl type)
                           (mapcar #'class-name 
                                   (all-class-instances 
                                    (get-class-frob
                                     (rule-slot-type cl type))))))
        (setf tuple (cons (get-slot cl) child))
        (setf rule-list (gethash tuple *rules*))
        (unless (member name rule-list)
          ;; insertion sort
	  (setf (gethash tuple *rules*) 
		(insert-sort name value rule-list))
	  (push tuple (gethash name *redefine*)))))))
      
;;; Removes any old definitions and links to this rule.
(defun remove-rule (new-name)
  (let ((tuple-list (gethash new-name *redefine*)))
    (when tuple-list
      ;; remove old links
      (dolist (tuple tuple-list)
	(setf (gethash tuple *rules*)
	      (delete new-name (gethash tuple *rules*) :count 1)))
      (remhash new-name *redefine*)
      (remhash new-name *rule-value*))))

;;; Sorting predicate for rule values
(defun sort-rule (v1 v2 r1 r2)
  (cond ((eq v1 :best) t)
	((eq v2 :worst) t)
	((eq v1 :worst) nil)
	((eq v2 :best) nil)
	((> v1 v2))
	((= v1 v2)
	 (member r1 (member r2 *text-order*)))
	(t nil)))

;;; Insert sort routine for adding @@rules to the rule-list
(defun insert-sort (new-name @@val rule-list)
  (cond ((null rule-list)
	 (list new-name))
	((not (sort-rule @@val (rule-value (car rule-list)) new-name (car rule-list)))
	 (cons new-name rule-list))
	(t (insert-aux new-name @@val (cdr rule-list) rule-list)
	   rule-list)))

(defun insert-aux (new-name @@val rule-list back)
  (cond ((null rule-list)
	 (rplacd back (list new-name)))
	((not (sort-rule @@val (rule-value (car rule-list)) new-name (car rule-list)))
	 (rplacd back (cons new-name rule-list)))
	(t (insert-aux new-name @@val (cdr rule-list) rule-list))))


;;; Removes a rule from the system
(defun kill-rule (name)
  (setf *traced-rule-names* (delete name *traced-rule-names* :count 1))
  (setf name (make-@@-name name))
  (setf *text-order* (delete name *text-order* :count 1))
  (remove-rule name))

(defun check-rules (frob slot class-type value)
  (unless (class-frob-p frob)
    (rplaca *rule-index* slot)
    (rplacd *rule-index* class-type)
    (dolist (r (if (eq *queue* 'breadth)
		 (reverse (gethash *rule-index* *rules*))
		 (gethash *rule-index* *rules*)))
      (funcall r frob slot class-type value))))


;;; Macro that simulates a while loop control structure
(defmacro while (test &body forms)
  (let ((jump (gensym)))
    `(prog ()
       ,jump
       (when ,test ,@forms (go ,jump)))))

;;; Predicate that compares tuples based on the first 3 fields
(defun tuple-test (a b)
  (and (eq (car a)(car b))
       (eq (cadr a)(cadr b))
       (eql (third a)(third b))))

;;; Runs the rules
(defun run-rules ()
  (unless *rules-running*
    (unwind-protect
      (progn
	(setf *rules-running* t)
	(let (stuff tuples)
	  (cond (*par-mode*
		 (prog ()
		   front
		   (when (zerop *depth*)
		     (setf *tuple-queue* *temp-queue*)
		     (setf *temp-queue* nil))
		   (incf *depth*)
		   (while *tuple-queue*
		     (setf stuff (pop *tuple-queue*))
		     (setf tuples (fourth stuff))
		     (when tuples
		       (setf tuples (car tuples))
		       (setf (fourth stuff) (cdr (fourth stuff)))
		       (when (fourth stuff)
			 (push stuff *tuple-queue*))
		       (funcall (car stuff)(second stuff)(third stuff) tuples)
		       ))
		   (decf *depth*)
		   (when (and (zerop *depth*) *temp-queue*)
		     (go front))))
		(t
		 (while *tuple-queue*
		   (setf stuff (car *tuple-queue*))
		   (setf tuples (fourth stuff))
		   (pop (fourth stuff))
		   (unless (fourth stuff) (pop *tuple-queue*))
		   (when tuples
		     (funcall (car stuff)(second stuff)(third stuff)(car tuples))
		     ))))))
      (setf *rules-running* nil))))

;;; Atomic macro.  Useful in conclusions of FC rules.
(defmacro atomic (&body forms)
  `(progn ,@forms))
#|
  `(unwind-protect
     (progn
       (setf *rules-running* t)
       ,@forms)
     (setf *rules-running* nil)))
|#

(defmacro not-atomic (&body forms)
  `(unwind-protect
     (progn
       (setf *rules-running* nil)
       ,@forms)
     (setf *rules-running* t)))

;;; The depth first function for running rules.  New frob-rule tuples
;;; of the form (rule frob rule-num (list of tuples)) are pushed on the front
;;; of the *tuple-queue*
(defun depth-first (tuple)
  (when *tuple-queue*
    (setf *tuple-queue* 
	  (delete tuple *tuple-queue* :test #'tuple-test :count 1)))
  (push tuple *tuple-queue*))

;;; The breadth first function for running rules.  New frob-rule tuples
;;; of the form (rule frob rule-num (list of tuples)) are pushed on the back
;;; of the *tuple-queue*
(defun breadth-first (tuple)
  (cond (*tuple-queue*
	 (dolist (tupe (cdr *tuple-queue*))
	   (when (tuple-test tuple tupe)
	     (rplacd tupe (cdr tuple))
	     (return-from breadth-first)))
	 (rplacd *back-of-queue* (list tuple))
	 (pop *back-of-queue*))
	(t
	 (push tuple *tuple-queue*)
	 (setf *back-of-queue* *tuple-queue*))))

;;; Parallel depth-first
(defun par-depth-first (tuple)
  (setf *temp-queue* 
	(delete tuple *temp-queue* :test #'tuple-test :count 1))
  (push tuple *temp-queue*))

(defun par-breadth-first (tuple)
  (cond (*temp-queue*
	 (dolist (tupe (cdr *temp-queue*))
	   (when (tuple-test tuple tupe)
	     (rplacd tupe (cdr tuple))
	     (return-from par-breadth-first)))
	 (rplacd *back-of-queue* (list tuple))
	 (pop *back-of-queue*))
	(t
	 (push tuple *temp-queue*)
	 (setf *back-of-queue* *temp-queue*))))

;;; Resets the rule system to its virgin state.
(defun reset-rules ()
  (clrhash *redefine*)
  (clrhash *rule-value*)
  (clrhash *rules*)
  (setf *rules-running* nil)
  (setf *par-mode* nil)
  (setf *depth* 0)
  (set-conflict-resolution-strategy 'depth-first)
  (setf *text-order* nil)
  (setf *traced-rule-names* nil)
  (setf *temp-queue* nil)
  (setf *tuple-queue* nil))

;;; Sets the current conflict resolution strategy
(defun set-conflict-resolution-strategy (fn)
  (if (member fn '(par-depth-first par-breadth-first))
    (setf *par-mode* t)
    (setf *par-mode* nil))
  (if (member fn '(breadth-first par-breadth-first))
    (setf *queue* 'breadth)
    (setf *queue* 'depth))
  (setf *conflict* (symbol-function fn)))

;;; The default
(set-conflict-resolution-strategy 'depth-first)

;;; Clears the rule queues and stops all future FC chaining
(defun stop-fc ()
  (setf *rules-running* nil)
  (setf *depth* 0)
  (setf *tuple-queue* nil)
  (setf *temp-queue* nil))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Rule tracing facilities, added by Eric Eide.
;;;;
;;;; If a rule is "traced," an informative message is printed whenever the rule
;;;; fires.  This message contains the name of the rule and the values of all
;;;; the rule's type and local variables.
;;;;
;;;; The list of traced rule names is kept in *traced-rule-names*.  Named are
;;;; added to this list by trace-rules and removed by untrace-rules.  As an
;;;; additional feature, the user can trace EVERY rule by setting the flag
;;;; *trace-all-rules* to a non-NIL value.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun traced-rule-p (name)
  (or *trace-all-rules* (and *traced-rule-names* ;; If NIL, don't call member.
			     (member name *traced-rule-names*))))

(defun print-traced-rule-firing (name type-and-local-names type-and-local-vals)
  (let (#+:UCL (lisp::*verbose-gc* nil))
    (format *trace-output* ";;; Rule ~S fires:~%" name)
    (do ((names type-and-local-names (cdr names))
	 (vals type-and-local-vals (cdr vals)))
	((or (null names) (null vals)))
      (format *trace-output* ";;;   ~S = ~S~%" (car names) (car vals)))))

(defmacro trace-rules (&rest names)
  `(trace-rules-aux ',names))

(defun trace-rules-aux (names)
  (when *trace-all-rules*
    (warn "~S is true, so all Frobs rules are traced." '*trace-all-rules*)
    (warn "~S will be traced even when ~S is set to NIL."
	  (if names names *traced-rule-names*) '*trace-all-rules*))
  (if names
      (do ((name-list names (cdr name-list)))
	  ((null name-list) names)
	;; Here we use *text-order* as a convenient list of all the rules that
	;; have been defined.
	(if (member (make-@@-name (car name-list)) *text-order*)
	    (pushnew (car name-list) *traced-rule-names*)
	    (warn "~S is not a Frobs rule; not traced."
		  (car name-list))))
      (copy-list *traced-rule-names*)))

(defmacro untrace-rules (&rest names)
  `(untrace-rules-aux ',names))

(defun untrace-rules-aux (names)
  (when *trace-all-rules*
    (warn "~S is true, so all Frobs rules are traced." '*trace-all-rules*)
    (warn "~A will be traced until ~S is set to NIL."
	  (if names names "All Frobs rules") '*trace-all-rules*))
  (if names
      ;; Untrace specific rules.
      (dolist (name names names)
	(setf *traced-rule-names* (delete name *traced-rule-names* :count 1)))
      ;; Untrace all traced rules.
      (let ((result *traced-rule-names*))
	(setf *traced-rule-names* nil)
	result)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;
;;;; Rule-checking facilities, written and added by Eric Eide.
;;;;
;;;; The original version of DEF-RULE (above, now named DEF-RULE-INTERNAL) does
;;;; almost no error checking.  The version defined below does a considerable
;;;; amount of compile-time syntax and sanity checking, which can help prevent
;;;; many potentially obscure bugs.
;;;;
;;;; (This new DEF-RULE was first used for the 1993 Summer Computing Institute,
;;;; and was added to the real Frobs sources in August 1993.)
;;;;
;;;; Still left to do:
;;;;
;;;;   + Do better checking of the :MV slot and the use of multiply-valued
;;;;	 slots.
;;;;   + Make sure that the KNOWN and NOT clause cases are really handled
;;;;	 correctly.
;;;;   + Look into whether or not is is OK to assume that all :TYPE variables
;;;;	 are initially bound.  The premise-scanning code pretends that all of
;;;;	 the :TYPE variables are bound before any of the clauses are evaluated,
;;;;	 but this is not how Frobs actually works internally.  The question is
;;;;	 whether or not our technique is equivalent to the real implementation,
;;;;	 at least for all the cases in which Frobs produces correct code!
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconstant *rule-error-indent* "       ")
(defconstant *rule-warn-indent* "         ")

;;;
;;; *BOUND-VARIABLES* and *REFERENCED-VARIABLES* are used to keep track of the
;;; rule variables that have been bound and referenced, respectively.
;;;

(defvar *bound-variables* nil)
(defvar *referenced-variables* nil)

;;;
;;; Some simple auxiliary functions.
;;;

(defun proper-listp (the-list)
  (and (listp the-list)
       (null (cdr (last the-list)))))

(defun rule-error (name format-string &rest format-args)
  (error "In rule ~S,~%~A~?."
	 name *rule-error-indent* format-string format-args))

(defun rule-warn (name format-string &rest format-args)
  (warn "In rule ~S,~%~A~?."
	name *rule-warn-indent* format-string format-args))

(defun clause-error (name clause format-string &rest format-args)
  (error "In rule ~S,~%~Ain premise clause ~S,~%~A~?."
	 name
	 *rule-error-indent* clause
	 *rule-error-indent* format-string format-args))

;;;
;;; The new DEF-RULE.
;;;

(defmacro def-rule (name
		    &key type local mv
		    (prem nil prem-provided) (conc nil conc-provided)
		    (value 0)
		    &whole whole)
  (unless (symbolp name)
    (error "Rule name must be a symbol."))
  
  (let ((types-and-locals nil))
    ;; Check the syntax of the :TYPE list.  Each element must be a two element
    ;; list.  The first element must be a variable and the second must be the
    ;; name of a Frobs class.
    (unless (proper-listp type)
      (rule-error name ":TYPE must be a list"))
    (dolist (a-type type)
      (unless (and (proper-listp a-type) (= (length a-type) 2))
	(rule-error name "in :TYPE, ~S is not a valid (VAR TYPE) pair" a-type))
      (unless (varp (first a-type))
	(rule-error name "in :TYPE, ~S is not a variable" (first a-type)))
      (unless (get-class-frob (second a-type))
	(rule-error name "in :TYPE, ~S is not the name of a Frob class"
		    (second a-type))
	)
      (when (assoc (first a-type) types-and-locals)
	(rule-error name "in :TYPE, ~S is listed more than once"
		    (first a-type)))
      (push (cons (first a-type) (get-class-frob (second a-type)))
	    types-and-locals)
      )
    
    ;; Check the syntax of the :LOCAL list.  Every element must a variable.
    (unless (proper-listp local)
      (rule-error name ":LOCAL must be a list"))
    (dolist (a-local local)
      (unless (varp a-local)
	(rule-error name "in :LOCAL, ~S is not a variable" a-local))
      (when (assoc a-local types-and-locals)
	(rule-error name
		    "~S is listed more than once as a :TYPE or :LOCAL variable"
		    a-local))
      (push (cons a-local nil) types-and-locals))
    
    ;; Check the syntax of the :MV list.  For now, the checking is weak -- we
    ;; only require that every element of the list be a symbol.  We should
    ;; really check that the named slots exists and are multiple-valued slots.
    (unless (proper-listp mv)
      (rule-error name ":MV must be a list"))
    (dolist (a-mv mv)
      (unless (symbolp a-mv)
	(rule-error name "in :MV, ~S is not a slot name" a-mv))
      )
    
    ;; Check for the existence of the :PREM and :CONC.
    (unless prem-provided
      (rule-warn name "there is no :PREM"))
    (unless (proper-listp prem)
      (rule-error name ":PREM is not a list"))
    (when (and prem-provided (null prem))
      (rule-warn name "there are no premise clauses"))
    (unless conc-provided
      (rule-warn name "there is no :CONC"))
    
    ;; Check the rule value.
    (unless (or (and (numberp value) (not (complexp value)))
		(eq value :best)
		(eq value :worst))
      (rule-error name "~S is not a valid rule :VALUE" value))
    
    ;; Check the clauses in the rule premise.  Report any variables that are
    ;; not bound or referenced by the premise clauses.  NOTE that we start off
    ;; with all of the :TYPE variables being bound, although this isn't really
    ;; how Frobs rules work internally.
    (let ((*bound-variables* (mapcar #'first type))
	  (*referenced-variables* nil))
      (def-rule-check-premise-clauses name prem types-and-locals)
      (dolist (variable-info types-and-locals)
	(let ((bound-p (member (car variable-info) *bound-variables*))
	      (referenced-p (member (car variable-info)
				    *referenced-variables*)))
	  (unless (and bound-p referenced-p)
	    (rule-warn name "variable ~S is ~A by the premise"
		       (car variable-info)
		       (cond (bound-p
			      ;; In order for a variable to appear bound but
			      ;; not referenced, it must be a :TYPE variable
			      ;; that started off "bound."  But in this case
			      ;; Frobs will actually not bind the variable,
			      ;; either.  I consider this to be a Frobs bug.
			      "not referenced")
			     (referenced-p
			      ;; The only way to get here is through a NOT
			      ;; clause.  Other referenced-but-not-bound cases
			      ;; will produce an error before we get this far.
			      "not bound")
			     (T
			      ;; An unused :LOCAL variable.
			      "neither bound nor referenced")))
	    )
	  ))
      ))
  
  ;; If we made it through all that, then we can finally invoke the original
  ;; DEF-RULE macro.
  `(def-rule-internal ,@(cdr whole))
  )

;;;
;;; Check all of the clauses in the premise.
;;;

(defun def-rule-check-premise-clauses (name prem types-and-locals)
  (dolist (clause prem)
    (unless (and (proper-listp clause)
		 (symbolp (first clause)))
      (rule-error name "~S is not a valid premise clause" clause))
    (case (first clause)
      (bind
       ;; This is a special BIND clause.
       (unless (= (length clause) 3)
	 (rule-error name "~S is not a valid BIND clause" clause))
       (let ((variable (second clause)))
	 (unless (varp variable)
	   (clause-error name clause "~S is not a variable" variable))
	 (unless (assoc variable types-and-locals)
	   (clause-error
	    name clause
	    "~S was not listed in the :TYPE or :LOCAL variable lists"
	    variable))
	 (pushnew variable *bound-variables*)
	 (pushnew variable *referenced-variables*)
	 ))
      
      ((eval evalp)
       ;; This is a special EVAL or EVALP clause.
       (unless (= (length clause) 2)
	 (clause-error name clause
		       "only one form is allowed in an EVAL or EVALP clause")
	 ))
      
      (known
       ;; This is a special KNOWN clause.
       (let ((clause-length (length clause)))
	 (unless (or (= clause-length 2) (= clause-length 3))
	   (rule-error name "~S is not a valid KNOWN clause" clause))
	 (let ((known-clause (second clause))
	       (known-ask (third clause)))
	   ;; Check the clause within the KNOWN clause.
	   (unless (and (proper-listp known-clause)
			(= (length known-clause) 3)
			(let ((known-slot (first known-clause)))
			  (and (symbolp known-slot)
			       ;; Within KNOWN, the SLOT can't be a variable.
			       ;; (Should this test be moved down into the
			       ;; function DEF-RULE-CHECK-SLOT-CLAUSE?)
			       (not (varp known-slot))
			       (not (member known-slot *special-clauses*)))
			  ))
	     (clause-error name clause
			   "~S is not a valid firing clause for KNOWN"
			   known-clause))
	   (def-rule-check-slot-clause name known-clause types-and-locals
				       'known)
	   ;; Check the ASK function.
	   (unless (or (= clause-length 2)
		       (and known-ask
			    (proper-listp known-ask)))
	     (clause-error name clause
			   "~S is not a valid ask function" known-ask))
	   )))
      
      (not
       ;; This is a special NOT clause.  The Frobs user guide says that NOT can
       ;; accept any kind of premise clause, but I believe that NOT really only
       ;; works with slot-examining clauses.
       (unless (= (length clause) 2)
	 (clause-error name clause "NOT takes exactly one firing clause"))
       (let ((not-clause (second clause)))
	 ;; Check the clause within the NOT clause.
	 (unless (and (proper-listp not-clause)
		      (= (length not-clause) 3)
		      (let ((not-slot (first not-clause)))
			(and (symbolp not-slot)
			     (not (member not-slot *special-clauses*)))
			))
	   (clause-error name clause
			 "~S is not a valid slot-examining clause for NOT"
			 not-clause))
	 (def-rule-check-slot-clause name not-clause types-and-locals 'not)
	 ))
      
      (T
       ;; Otherwise, this clause is a regular slot-examining clause.
       (def-rule-check-slot-clause name clause types-and-locals nil))
      ))
  )

;;;
;;; Check a single slot-accessing clause of the form (<slot> <frob> <value>).
;;;

(defun def-rule-check-slot-clause (name clause types-and-locals context)
  ;; Upon entry to this function, we know that CLAUSE is a proper list and that
  ;; its first element is a symbol.
  (unless (= (length clause) 3)
    (rule-error name "~S is not a valid premise clause" clause))
  (let* ((slot (first clause))
	 (frob (second clause))
	 (frob-class nil)
	 (value (third clause)))
    
    ;; Check the SLOT...
    (when (varp slot)
      (let ((slot-info (assoc slot types-and-locals)))
	(unless slot-info
	  (clause-error name clause
			"~S was not listed in the :LOCAL variable list"
			slot))
	(unless (null (cdr slot-info))
	  (clause-error name clause "~S was listed in :TYPE, not :LOCAL" slot))
	;; SLOT variables must be bound before they can be used.
	(unless (member slot *bound-variables*)
	  (clause-error name clause
			"~S has not been bound by a previous clause"
			slot))
	;; SLOT must already be in the *BOUND-VARIABLES* list...
	(pushnew slot *referenced-variables*)
	))
    
    ;; Check the FROB...
    (cond ((varp frob)
	   (let ((var-info (assoc frob types-and-locals)))
	     (if var-info
		 (setf frob-class (cdr var-info))
		 (clause-error
		  name clause
		  "~S was not listed in the :TYPE or :LOCAL variable lists"
		  frob)))
	   ;; I'm pretty sure that for our purposes, the FROB variable must be
	   ;; bound at this point (i.e., be a :TYPE variable that was "bound"
	   ;; automatically, or be a previously-bound :LOCAL variable).  This
	   ;; condition may be too strict, however.
	   (unless (member frob *bound-variables*)
	     (clause-error name clause
			   "~S has not been bound by a previous clause"
			   frob))
	   ;; FROB must already be in the *BOUND-VARIABLES* list...
	   (pushnew frob *referenced-variables*)
	   )
	  ((and (proper-listp frob)
		(eq (first frob) 'get-frob))
	   ;; The Frob is a named constant instance Frob.  {FOO} is read as
	   ;; (GET-FROB (QUOTE FOO)).
	   (let ((constant-frob (get-frob (second (second frob)))))
	     (unless constant-frob
	       (clause-error name clause
			     "the named instance Frob does not exist"))
	     (setf frob-class (first (class-parent constant-frob)))
	     ))
	  ((and (proper-listp frob)
		(eq (first frob) 'get-class-frob))
	   ;; The Frob is a named constant class Frob.  {CLASS FOO} is read as
	   ;; (GET-CLASS-FROB (QUOTE FOO)).
	   (let ((constant-frob (get-class-frob (second (second frob)))))
	     (unless constant-frob
	       (clause-error name clause
			     "the named class Frob does not exist"))
	     (setf frob-class constant-frob)
	     ))
	  (T
	   (clause-error name clause
			 "~S is neither a variable nor a constant Frob"
			 frob)))
    
    ;; And check the VALUE...
    (when (varp value)
      (unless (assoc value types-and-locals)
	(clause-error name clause
		      "~S was not listed in the :TYPE or :LOCAL variable lists"
		      value))
      ;; Special cases for clauses within KNOWN and NOT clauses.
      (when (and (eq context 'known)
		 (member value *bound-variables*))
	;; For KNOWN, the value variable must not already be bound.
	(clause-error name clause
		      "~S must not be bound before its use in the KNOWN clause"
		      value))
      (unless (eq context 'not)
	;; If the value is a variable, the NOT clause does not bind it.
	(pushnew value *bound-variables*))
      (pushnew value *referenced-variables*))
    
    ;; Another special case for KNOWN.
    (when (and (eq context 'known)
	       (not (varp value)))
      (clause-error name clause "the value must be an unbound variable"))
    
    (when frob-class
      ;; We were able to determine the class of the Frob in this clause.  Check
      ;; that the slot name is okay.
      (unless (varp slot)
	(unless (slot? frob-class slot)
	  (clause-error name clause "~S is not a slot in ~S" slot frob-class)
	  )))
    ))

;; End of file.

