/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include "wissensbank.h"

Wissensbank::Wissensbank()
{
  _anwendung  = nil;
  _controller = nil;
}

Wissensbank::~Wissensbank()
{
  if(_anwendung != nil)
    delete _anwendung;
  if(_controller != nil)
    delete _controller;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::anwendung(Anwendung*)
 * Parameter : Anwendung* - neue Anwendung
 * Zweck : setzt neue Anwendung.
 *------------------------------------------------------------------------------
 */
void Wissensbank::anwendung(Anwendung* a)
{
  if(_anwendung != nil)
    *_anwendung = *a;
  else
    _anwendung = new Anwendung(*a);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::controller(FuzController*)
 * Parameter : FuzController* - neuer Controller
 * Zweck : setzt neuen Controller.
 *------------------------------------------------------------------------------
 */
void Wissensbank::controller(FuzController* controller)
{
  if(_controller != nil)
    delete _controller;
  _controller = controller;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::mach_fehler_var()
 * Rueckgabewert : Fehler-Variablen.
 * Zweck : erzeugt einen neuen LingVarSatz und verwendet diese fuer (als)
 *         die Fehler-Variablen. Dieser LingVarSatz enthaelt als ling. Variablen
 *         alle Eingabe-Variablen der Steuer-Variablen und eine Fehler-Variable.
 *         Die Variablen sind aber alle leer, enthalten also keine Fuzzy-Sets.
 *------------------------------------------------------------------------------
 */
LingVarSatz* Wissensbank::mach_fehler_var()
{
  // alle Eingabe-Variablen uebernehmen
  int anzahl = _steuer_var.anz_lingvars(EingabeVar);
  for(int i = 0; i < anzahl; i++) {
    LingVar* eingabe_var = _steuer_var.hol_lingvar(EingabeVar, i);
    if(_fehler_var.hol_lingvar(eingabe_var->name()) == nil) {
      LingVar var(eingabe_var->name());
      var.typ(EingabeVar);
      var.min(eingabe_var->min());
      var.max(eingabe_var->max());
      _fehler_var.add_lingvar(&var);
    }
  }

  // Fehler-Variable hinzufuegen
  if(_fehler_var.hol_lingvar(FehlerVarName) == nil) {
    LingVar fehler(String(FehlerVarName));
    fehler.typ(AusgabeVar);
    fehler.min(-1);
    fehler.max(1);
    _fehler_var.add_lingvar(&fehler);
  }

  return &_fehler_var;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::steuer_var_ok(Ausgabe*)
 * Parameter : Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> Steuer-Variablen sind in Ordnung
 *                 false <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::steuer_var_ok(Ausgabe* ausgabe_feld)
{
  return _teste_lingvarsatz(&_steuer_var, Txt(221), ausgabe_feld);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::fehler_var_ok(Ausgabe*)
 * Parameter : Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> Fehler-Variablen sind in Ordnung
 *                 false <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::fehler_var_ok(Ausgabe* ausgabe_feld)
{
  // Test, ob Fehler-Var einen gueltigen LingVarSatz bilden
  if(!_teste_lingvarsatz(&_fehler_var, Txt(222), ausgabe_feld))
    return false;

  // Test, ob eine Variable : 'fehler' vorhanden ist
  LingVar* fehler = _fehler_var.hol_lingvar(FehlerVarName);
  if(fehler == nil) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, Txt(149), FehlerVarName, Txt(150));
    return false;
  }

  // die natuerlich eine Ausgabe-Var sein muss
  if(fehler->typ() != AusgabeVar) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, Txt(117));
    return false;
  }

  // Test, ob Fehler-Var und Steuer-Var kompatibel sind
  int anzahl = _fehler_var.anz_lingvars();

  // Steuer-Var und Fehler-Var muessen gleiche Anzahl Variablen haben, weil:
  //    Der Controller, der Fehler berechnet, bekommt dieselben Messwerte wie
  //    der, der die Steuerwerte berechnet.
  //    nebenbei : Dies ist auch der Grund, dass die Variablen in der geleichen
  //               Reihenfolge stehen muessen.
  if(anzahl != _steuer_var.anz_lingvars()) {
    ausgabe_feld->ausgeben(Fehler, Txt(151));
    return false;
  }

  for(int i = 0; i < anzahl; i++) {
    String var_name = _fehler_var.hol_lingvar(i)->name();
    if(var_name != FehlerVarName)
      if(_steuer_var.hol_lingvar(var_name) == nil) {
        if(ausgabe_feld != nil)
          ausgabe_feld->ausgeben(Fehler, Txt(151));
        return false;
      }
  }

  // immer noch da ? => ok
  return true;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::anwendung_ok(Ausgabe*)
 * Parameter : Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> Anwendung ist in Ordnung
 *                 false <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::anwendung_ok(Ausgabe* ausgabe_feld)
{
  boolean ok = _anwendung != nil;

  for(LingVarTyp typ = EingabeVar; ok && typ <= AusgabeVar; typ++) {
    int anzahl = _anwendung->anzahl(typ);
    for(int i = 0; i < _anwendung->anzahl(typ); i++)
      if(_steuer_var.hol_lingvar(_anwendung->var(typ, i)) == nil)
        ok = false;
  }
  if(!ok && ausgabe_feld != nil)
    ausgabe_feld->ausgeben(Fehler, Txt(142));

  return ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::steuer_regeln_ok(Ausgabe*)
 * Parameter : Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> Steuer-Regeln sind in Ordnung
 *                 false <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::steuer_regeln_ok(Ausgabe* ausgabe_feld)
{
  if(_kompatibel(&_steuer_var, &_steuer_regeln))
    return true;

  if(ausgabe_feld != nil)
    ausgabe_feld->ausgeben(Fehler, Txt(152));
  return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::fehler_regeln_ok(Ausgabe*)
 * Parameter : Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> Fehler-Regeln sind in Ordnung
 *                 false <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::fehler_regeln_ok(Ausgabe* ausgabe_feld)
{
  if(_kompatibel(&_fehler_var, &_fehler_regeln))
    return true;

  if(ausgabe_feld != nil)
    ausgabe_feld->ausgeben(Fehler, Txt(153));
  return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Wissensbank::konsistent(boolean, Ausgabe*)
 * Parameter : boolean  - fuer neuronalen Controller ?
 *             Ausgabe* - Feld fuer evtl. Fehlermeldungen
 * Rueckgabewert : true  <=> alles in Ordnung
 *                 false <=> sonst
 * Zweck : Ueberprueft ob die gesamte Wissensbank fuer einen gegebenen
 *         Controller konsistent definiert ist.
 *------------------------------------------------------------------------------
 */
boolean Wissensbank::konsistent(boolean neuronal, Ausgabe* ausgabe_feld)
{
  if(!steuer_var_ok(ausgabe_feld))
    return false;

  if(neuronal && !fehler_var_ok(ausgabe_feld))
    return false;

  if(!anwendung_ok(ausgabe_feld))
    return false;

  if(!steuer_regeln_ok(ausgabe_feld))
    return false;

  if(neuronal && !fehler_regeln_ok(ausgabe_feld))
    return false;

/*
 *    fuer neuronalen Controller jetzt noch die Fuzzy-Sets testen :
 *    Fuer Steuer-Var :
 *      -) in der Praemisse sind hoechstens 2 Intervalle (also Linien) erlaubt;
 *         also in der Regel 3-Ecke
 *      -) in der Konklusion ist nur ein Intervall erlaubt;
 *         also nur monotone Fuzzy-Sets
 *    Fuer Fehler-Var :
 *      -) alles erlaubt
 */

  if(neuronal) {
    int anzahl_var = _steuer_var.anz_lingvars();
    for(int s_var_nr = 0; s_var_nr < anzahl_var; s_var_nr++) {
      LingVar* var   = _steuer_var.hol_lingvar(s_var_nr);
      LingVarTyp typ = var->typ();

      int anzahl_fuz = var->anz_fuzzysets();
      for(int fuz_nr = 0; fuz_nr < anzahl_fuz; fuz_nr++) {
        FuzzySet* fuz = var->hol_fuzzyset(fuz_nr);

        if(typ == EingabeVar) {
          if(fuz->anz_intervalle() > 2) {
            if(ausgabe_feld != nil) {
              ausgabe_feld->ausgeben(Info, Txt(154), fuz->name().string());
              ausgabe_feld->ausgeben(Info, Txt(155));
              ausgabe_feld->ausgeben(Info, LeerStr);
            }
            return false;
          }
        } else
          if(fuz->anz_intervalle() > 1) {
            if(ausgabe_feld != nil) {
              ausgabe_feld->ausgeben(Info, Txt(154), fuz->name().string());
              ausgabe_feld->ausgeben(Info, Txt(156));
              ausgabe_feld->ausgeben(Info, LeerStr);
            }
            return false;
          }
      }
    }

     // Fehler-Variable sortieren; d.h. die Eingabe-Variablen muessen in der
    //  gleichen Reihenfolge stehen, wie in den Steuer-Variablen
    anzahl_var = _steuer_var.anz_lingvars(EingabeVar);
    for(s_var_nr = 0; s_var_nr < anzahl_var; s_var_nr++) {
      int f_var_nr = _fehler_var.var_nr(
                       _steuer_var.hol_lingvar(EingabeVar, s_var_nr)->name()
                     );
      if(f_var_nr != s_var_nr)
        _fehler_var.tausche(f_var_nr, s_var_nr);
    }
  }

  // ansonsten : alles klar
  return true;
}

/*
 *----------------------------- lokale Funktionen ------------------------------
 */

boolean Wissensbank::_teste_lingvarsatz(
          LingVarSatz* lingvarsatz, char* offset_str, Ausgabe* ausgabe_feld
        )
{
  if(lingvarsatz->anz_lingvars() == 0) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, offset_str, Txt(72));
    return false;
  }

  if(lingvarsatz->anz_lingvars(EingabeVar) == 0) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, offset_str, Txt(68));
    return false;
  }

  if(lingvarsatz->anz_lingvars(AusgabeVar) == 0) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, offset_str, Txt(69));
    return false;
  }

  int anzahl = lingvarsatz->anz_lingvars();
  for(int i = 0; i < anzahl; i++)
    if(lingvarsatz->hol_lingvar(i)->anz_fuzzysets() == 0) {
      if(ausgabe_feld != nil)
        ausgabe_feld->ausgeben(
          Fehler,
          offset_str,
          Txt(70),
          lingvarsatz->hol_lingvar(i)->name()
        );
      return false;
    }

  return true;
}

boolean Wissensbank::_kompatibel(LingVarSatz* lv_satz, RegelBasis* rg_basis)
{
  // alle Regel durchgehen und pruefen, ob die darin vorkommenden LingVars
  // in dem LingVarSatz enthalten sind

  int anzahl_regeln = rg_basis->anzahl_regeln();
  for(int regel_nr = 0; regel_nr < anzahl_regeln; regel_nr++) {
    Regel* regel = rg_basis->hol_regel(regel_nr);
    for(RegelAusdruck a = Praemisse; a <= Konklusion; a++) {
      int anzahl_bed = regel->groesse(a);
      for(int bed_nr = 0; bed_nr < anzahl_bed; bed_nr++) {
        LingVar* var = lv_satz->hol_lingvar(
                         regel->bedingung(a, bed_nr)->lingvar()
                       );

        // ist entspr. ling. Variable im LingVarSatz enthalten ?
       if(var == nil)
         return false;

       // ist entspr. Fuzzy-Set in der ling. Variablen vorhanden ?
        if(var->hol_fuzzyset(regel->bedingung(a, bed_nr)->fuzzyset()) == nil)
          return false;
      }
    }
  }
  return true;
}
