/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/





#include "global.h"
#include "regel.h"

implementPtrList(RegelListe, Regel)

/*
 *------------------------------------------------------------------------------
 *--------------------- Definition der Klasse : 'Bedingung' --------------------
 *------------------------------------------------------------------------------
 */

Bedingung::Bedingung()
{
  _lingvar_name  = LeerStr;
  _fuzzyset_name = LeerStr;
}

Bedingung::Bedingung(String lingvar_name, String fuzzyset_name)
{
  _lingvar_name  = lingvar_name;
  _fuzzyset_name = fuzzyset_name;
}

Bedingung::Bedingung(const Bedingung& bedingung) { *this = bedingung; }

Bedingung& Bedingung::operator=(const Bedingung& bedingung)
{
  _lingvar_name  = bedingung.lingvar();
  _fuzzyset_name = bedingung.fuzzyset();
  return *this;
}

boolean Bedingung::operator!=(const Bedingung& bedingung) const
{
  if(_lingvar_name != bedingung.lingvar())
    return true;
  if(_fuzzyset_name != bedingung.fuzzyset())
    return true;
  return false;
}

boolean Bedingung::operator==(const Bedingung& bedingung) const
{
  return !(*this != bedingung);
}

/*
 *------------------------------------------------------------------------------
 *---------------------- Definition der Klasse : 'Regel' -----------------------
 *------------------------------------------------------------------------------
 */

Regel::Regel(int groesse_praemisse, int groesse_konklusion)
{
  _gr_praemisse  = groesse_praemisse;
  _praemisse     = new Bedingung[_gr_praemisse];

  _gr_konklusion = groesse_konklusion;
  _konklusion    = new Bedingung[_gr_konklusion];
}


Regel::Regel(const Regel& regel)
{
  _gr_praemisse = regel.groesse(Praemisse);
  _praemisse    = new Bedingung[_gr_praemisse];
  for(int i = 0;  i < _gr_praemisse; i++)
    _praemisse[i] = *regel.bedingung(Praemisse, i);

  _gr_konklusion = regel.groesse(Konklusion);
  _konklusion    = new Bedingung[_gr_konklusion];
  for(i = 0;  i < _gr_konklusion; i++)
    _konklusion[i] = *regel.bedingung(Konklusion, i);
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::bedingung(RegelAusdruck, int)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Position im 'RegelAusdruck'
 * Rueckgabewert : Durch die Parameter spezifizierte Bedingung.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
Bedingung* Regel::bedingung(RegelAusdruck ausdruck, int pos) const
{
  if(ausdruck == Praemisse)
    return &_praemisse[pos];
  else
    return &_konklusion[pos];
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::bedingung(RegelAusdruck, String var, String fuzzy)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             int           - Position im 'RegelAusdruck'
 *             String var    - Name der ling. Variablen
 *             String fuzzy  - Name des Fuzzy-Sets
 * Zweck : Initialisiert die durch die Parameter spezifizierte Bedingung mit
 *         'var' und 'var'.
 *------------------------------------------------------------------------------
 */
void Regel::bedingung(RegelAusdruck ausdruck, int pos, String var, String fuzzy)
{
  if(pos >= 0 && pos < _gr_praemisse) {
    if(ausdruck == Praemisse)
      _praemisse[pos] = Bedingung(var, fuzzy);
    else
      _konklusion[pos] = Bedingung(var, fuzzy);
  }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::enthalten(RegelAusdruck ausdruck, Bedingung* bed)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             Bedingung*    - beliebige Bedingung
 * Rueckgabewert : true  <=> Bedingung ist im RegelAusdruck enthalten.
 *                 false <=> sonst.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Regel::enthalten(RegelAusdruck ausdruck, Bedingung* bed) const
{
  int anzahl = groesse(ausdruck);
  for(int i = 0; i < anzahl; i++)
    if(*bed == *bedingung(ausdruck, i))
      return true;
  return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : enthalten(RegelAusdruck ausdruck, String lingvar)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             String        - Name einer ling. Variablen
 * Rueckgabewert : true  <=> Variable ist im RegelAusdruck enthalten.
 *                 false <=> sonst.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean Regel::enthalten(RegelAusdruck ausdruck, String lingvar) const
{
  int anzahl = groesse(ausdruck);
  for(int i = 0; i < anzahl; i++)
    if(lingvar == bedingung(ausdruck, i)->lingvar())
      return true;
  return false;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::groesse(RegelAusdruck ausdruck)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 * Rueckgabewert : Anzahl der Bedingungen im RegelAusdruck.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
int Regel::groesse(RegelAusdruck ausdruck) const
{
  if(ausdruck == Praemisse)
    return _gr_praemisse;
  else
    return _gr_konklusion;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::fuzzyset(RegelAusdruck ausdruck, String lingvar)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             String        - Name einer ling. Variablen
 * Rueckgabewert : Name eines Fuzzy-Sets <=> ling. Variable in RegelAusdruck
 *                                           enthalten
 *                 LeerStr               <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String Regel::fuzzyset(RegelAusdruck ausdruck, String lingvar)
{
  int anzahl = groesse(ausdruck);
  for(int i = 0; i < anzahl; i++)
    if(bedingung(ausdruck, i)->lingvar() == lingvar)
      return bedingung(ausdruck, i)->fuzzyset();
  return LeerStr;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Regel::fuzzyset(RegelAusdruck ausdruck, String lingvar)
 * Parameter : RegelAusdruck - Praemisse oder Konklusion
 *             String        - Name eines Fuzzy-Sets
 * Rueckgabewert : Name einer ling. Variablen <=> Fuzzy-Set in RegelAusdruck
 *                                                enthalten
 *                 LeerStr                    <=> sonst
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
String Regel::lingvar(RegelAusdruck ausdruck, String fuzzyset)
{
  int anzahl = groesse(ausdruck);
  for(int i = 0; i < anzahl; i++)
    if(bedingung(ausdruck, i)->fuzzyset() == fuzzyset)
      return bedingung(ausdruck, i)->lingvar();
  return LeerStr;
}


Regel& Regel::operator=(const Regel& regel)
{
  if(_gr_praemisse != regel.groesse(Praemisse)) {
    delete[] _praemisse;
    _gr_praemisse = regel.groesse(Praemisse);
    _praemisse    = new Bedingung[_gr_praemisse];
  }
  for(int i = 0;  i < _gr_praemisse; i++)
    _praemisse[i] = *regel.bedingung(Praemisse, i);

  if(_gr_konklusion != regel.groesse(Konklusion)) {
    delete[] _konklusion;
    _gr_konklusion = regel.groesse(Konklusion);
    _konklusion    = new Bedingung[_gr_konklusion];
  }
  for(i = 0;  i < _gr_konklusion; i++)
    _konklusion[i] = *regel.bedingung(Konklusion, i);

  return *this;
}

boolean Regel::operator==(const Regel& regel)
{
  for(RegelAusdruck ausdruck = Praemisse; ausdruck <= Konklusion; ausdruck++) {
    if(groesse(ausdruck) != regel.groesse(ausdruck))
      return false;
    for(int i = 0; i < groesse(ausdruck); i++)
      if(!regel.enthalten(ausdruck, bedingung(ausdruck, i)))
        return false;
  }
  return true;
}

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : 'RegelBasis' ---------------------
 *------------------------------------------------------------------------------
 */

RegelBasis::~RegelBasis() { while(del_regel(0)); }


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelBasis::del_regel(int)
 * Parameter : int - Nr. einer Regel
 * Rueckgabewert : true  <=> Regel gefunden
 *                 false <=> sondt
 * Zweck : Loescht die gewuenschte Regel aus RegelBasis und liefert den Erfolg
 *         zurueck.
 *------------------------------------------------------------------------------
 */
boolean RegelBasis::del_regel(int nr)
{
  if(nr >= 0 && nr < anzahl_regeln()) {
    delete _regel_liste.item(nr);
    _regel_liste.remove(nr);
    return true;
  } else
    return false;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelBasis::del_regel(Regel*)
 * alles genauso wie in 'RegelBasis::del_regel(int)', nur dass die Regel hier
 * direkt angegeben wird.
 *------------------------------------------------------------------------------
 */
boolean RegelBasis::del_regel(Regel* regel)
{
  boolean gefunden = false;
  int anzahl = anzahl_regeln();
  for(int i = 0; !gefunden && i < anzahl; i++)
    if(*_regel_liste.item(i) == *regel) {
      delete _regel_liste.item(i);
      _regel_liste.remove(i);
      gefunden = true;
    }
  return gefunden;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelBasis::add_regel(Regel*)
 * Parameter : Regel* - neue Regel
 * Zweck : fuegt neue Regel in RegelBasis ein.
 * Anmerkung : Falls eine Regel mit gleicher Praemisse und gleichen ling.
 *             Variablen in der Konklusion enthalten ist, wird diese durch die
 *             neue Regel ersetzt. Es wird jedoch gesamte Konklusion geprueft,
 *             d.h., es ist dennoch moeglich, dass Regeln mit gleicher
 *             Praemisse und einer (oder auch mehreren) gemeinsamen Variablen
 *             in der Konklusion in der RegelBasis existieren.
 *             (siehe auch : RegelBasis::such(RegelAusdruck, int&))
 *             Beispiel 1 :
 *                  alte Regel :  IF A = a AND B = b THEN C = c AND D = d
 *                  neue Regel :  IF A = a AND B = b THEN C = x AND D = y
 *             -) in diesem Fall wird die alte Regel durch die neue ersetzt
 *             Beispiel 2 :
 *                  alte Regel :  IF A = a AND B = b THEN C = c AND D = d
 *                  neue Regel :  IF A = a AND B = b THEN C = x
 *             -) in diesem Fall bleibt die alte Regel unveraendert und die
 *                neue wird zur RegelBasis hinzugefuegt.
 *------------------------------------------------------------------------------
 */
void RegelBasis::add_regel(Regel* regel)
{
  int nr;
  if(regel != nil) {
    Regel* neuer = new Regel(*regel);

    // testen, ob schon Regel mit gleicher Praemisse existiert
   if(such(neuer, nr)) {
      del_regel(nr);
      _regel_liste.insert(nr, neuer);
    } else
      _regel_liste.append(neuer);
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : RegelBasis::such(RegelAusdruck, Regel*, int&)
 * Parameter : Regel* - irgendeine Regel
 *             int&   - Position, an der Regel gefunden wurde
 * Rueckgabewert : gibt an, ob Regel gefunden wurde
 * Zweck : sucht in der RegelBasis eine Regel mit gleicher Praemisse und
 *         gleichen ling. Variablen in der Konklusion. Der Parameter 'int&'
 *         wird mit der ensprechenden Position initialisiert. Zurueckgeliefert
 *         wird der Erfolg der Operation.
 *------------------------------------------------------------------------------
 */
boolean RegelBasis::such(Regel* regel, int& nr)
{
  boolean ok, gefunden = false;
  int anzahl = anzahl_regeln();
  for(int i = 0; !gefunden && i < anzahl; i++)
    if(regel->groesse(Praemisse) == hol_regel(i)->groesse(Praemisse)) {
      ok = true;

      // Praemisse ueberpruefen
      for(int j = 0; ok && j < regel->groesse(Praemisse); j++)
        if(!hol_regel(i)->enthalten(Praemisse, regel->bedingung(Praemisse, j)))
          ok = false;

      // Konklusion ueberpruefen
      if(ok) {
        if(regel->groesse(Konklusion) == hol_regel(i)->groesse(Konklusion)) {
          for(int j = 0; ok && j < regel->groesse(Konklusion); j++)
            if(!hol_regel(i)->enthalten(
                  Konklusion, regel->bedingung(Konklusion, j)->lingvar()
                )
            )
              ok = false;
        } else
          ok = false;
      }

      if(ok) {
        gefunden = true;
        nr = i;
      }
    }
  return gefunden;
}


RegelBasis& RegelBasis::operator=(const RegelBasis& regel_basis)
{
  while(anzahl_regeln() > 0)
    del_regel(0);
  int anzahl = regel_basis.anzahl_regeln();
  for(int i = 0;  i < anzahl; i++)
    add_regel(regel_basis.hol_regel(i));
  return *this;
}
