/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <IV-look/kit.h>
#include <IV-look/button.h>
#include <OS/string.h>
#include <InterViews/glyph.h>
#include <InterViews/border.h>
#include <InterViews/font.h>
#include <InterViews/session.h>

#include "global.h"
#include "koosystem.h"
#include "texte.h"
#include "lingvar_ed.h"

#define YMIN 0
#define YMAX 1
#define MAXNAME 10


/*
 *------------------------------------------------------------------------------
 * Klasse : FuzzySetGrafik
 * Zweck  : Stellt ein Fuzzy-Set grafisch in einem Koordinatensystem dar.
 *------------------------------------------------------------------------------
 */
class FuzzySetGrafik : public KooSystem
{
  public : FuzzySetGrafik(FuzzyTyp von_x, FuzzyTyp bis_x,
                          FuzzyTyp von_y, FuzzyTyp bis_y);
           void setz_fuzzyset(FuzzySet* fs) { _fuzzyset = fs; };
           virtual void draw(Canvas*, const Allocation&) const;
           void gummiband(boolean, FuzzyTyp x1, FuzzyTyp y1,
                                   FuzzyTyp x2, FuzzyTyp y2);
           void markiere_linie(int nr);
  private: void _zeichne_fkt() const;
           int _markiert;      // Nummer einer Linie, die markiert werden soll
           FuzzySet* _fuzzyset;
           boolean _gummi_anzeigen;
           FuzzyTyp _merk_x1, _merk_x2;
           FuzzyTyp _merk_y1, _merk_y2;
};

FuzzySetGrafik::FuzzySetGrafik(
                  FuzzyTyp v_x, FuzzyTyp b_x, FuzzyTyp v_y, FuzzyTyp b_y
                ) : KooSystem(v_x, b_x, v_y, b_y)
{
  _gummi_anzeigen = false;
  _markiert       = -1;        // keine Linie hervorheben
  _fuzzyset       = nil;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetGrafik::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 * Zweck : zeichnet die Grafik im zugewiesenen Bildschirmbereich.
 *------------------------------------------------------------------------------
 */
void FuzzySetGrafik::draw(Canvas* c, const Allocation& a) const
{
  KooSystem::draw(c, a);
  _zeichne_fkt();
  if(_gummi_anzeigen && _fuzzyset != nil)
    linie(_merk_x1, _merk_y1, _merk_x2, _merk_y2);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetGrafik::gummiband(
 *                     boolean, FuzzyTyp, FuzzyTyp, FuzzyTyp, FuzzyTyp
 *                   )
 * Parameter : boolean         - setzen oder loeschen
 *             FuzzyTyp x1, y1 - Koordinaten des ersten Punktes
 *             FuzzyTyp x2, y2 - Koordinaten des 2ten Punktes
 * Zweck : zeichnet im Koordinatensystem eine Linie. Dabei handelt es sich
 *         nicht um den Teil eines Fuzzy-Sets. Diese Linie kann dazu verwendet
 *         werden, wie ein Gummiband, einen festen Punkt und den Mauszeiger
 *         miteinander zu verbinden.
 *------------------------------------------------------------------------------
 */
void FuzzySetGrafik::gummiband(
       boolean ein, FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2
     )
{
  _gummi_anzeigen = ein;
  _merk_x1        = x1;
  _merk_y1        = y1;
  _merk_x2        = x2;
  _merk_y2        = y2;
  aktualisiere();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetGrafik::markiere_linie(int)
 * Parameter : int - Nummer eines Fuzzy-Sets
 * Zweck : Markiert das Fuzzy-Set mit der angegebenen Nummer.
 *------------------------------------------------------------------------------
 */
void FuzzySetGrafik::markiere_linie(int nr)
{
  _markiert = nr;
  aktualisiere();
}

void FuzzySetGrafik::_zeichne_fkt() const
{
  FsIntervall* intervall;
  if(_fuzzyset != nil)
    for(int i = 0; i < _fuzzyset->anz_intervalle(); i++) {
      FuzzyTyp x1, x2, y1, y2;
      intervall = _fuzzyset->hol_intervall(i);
      intervall->links(x1, y1);
      intervall->rechts(x2, y2);
      linie(x1, y1, x2, y2);
      if(_markiert == i)
        zeig_linie(x1, y1, x2, y2);
    }
}


/*
 *------------------------------------------------------------------------------
 * Klasse : GrafFktEd
 * Zweck  : Faengt Mausbewegungen und Tastendruecke inerhalb eines
 *          Grafikbereiches ab und fuehrt ensprechende Callbacks aus. Der
 *          Grafikbereich ist in diesem Fall eine FuzzySetGrafik, so dass
 *          Fuzzy-Sets grafisch definiert werden koennen. Die 4 Gruende fuer den
 *          Aufruf der Callback-Funktion,
 *             - NeueKoo     : neue Mausposition,
 *             - LinkeTaste  : linke Maustaste gedrueckt,
 *             - MittelTaste : mittlere Maustaste gedrueckt,
 *             - RechteTaste : rechte Maustaste gedrueckt,
 *          werden als GrafikAktion definiert.
 *          Es werden die Koordinaten von 2 Punkten gespeichert und zwar sind
 *          dies die Koordinaten der aktuellen Mausposition und die Koordinaten 
 *          des letzten Tastendrucks.
 *------------------------------------------------------------------------------
 */
enum GrafikAktion { NeueKoo, LinkeTaste, MittelTaste, RechteTaste };

class GrafFktEd : public InputHandler
{
  public : GrafFktEd(Style*, FuzzySetGrafik* grafik, Action* callback);
           ~GrafFktEd();
           virtual void move(const Event&);
           virtual void press(const Event&);
           int punkt() { return _punkt; };
           GrafikAktion aktion() { return _aktion; };
           void koordinaten(int pkt, FuzzyTyp& x, FuzzyTyp& y);
           FuzzyTyp x_auswahl() { return _x_auswahl; };
           boolean auswahl_mod() { return _auswahl; };
  private: FuzzySetGrafik* _grafik;
           GrafikAktion _aktion;
           Action*_callback;
           int _punkt;
           FuzzyTyp _pkt1_x, _pkt2_x;
           FuzzyTyp _pkt1_y, _pkt2_y;
           boolean _auswahl;
           FuzzyTyp _x_auswahl;
};

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafFktEd::koordinaten(int, FuzzyTyp&, FuzzyTyp&)
 * Parameter : int         - erster oder zweiter Punkt ?
 *             FuzzyTyp& x - x-Koordinate
 *             FuzzyTyp& y - y-Koordinate
 * Zweck : Die x/y - Parameter werden mit Koordinaten des gewuenschten Punktes
 *         initialisiert.
 *------------------------------------------------------------------------------
 */
inline void GrafFktEd::koordinaten(int pkt, FuzzyTyp& x, FuzzyTyp& y)
{
  x = (pkt == 1) ? _pkt1_x : _pkt2_x;
  y = (pkt == 1) ? _pkt1_y : _pkt2_y;
}

GrafFktEd::GrafFktEd(Style* style, FuzzySetGrafik* grafik, Action* callback)
  : InputHandler(grafik, style)
{
  _auswahl  = false;
  _punkt    = 1;
  _grafik   = grafik;
  _callback = callback;
  Resource::ref(callback);
}

GrafFktEd::~GrafFktEd() { Resource::unref(_callback); }

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafFktEd::move(Event&)
 * Parameter : Event& - Ereignis
 * Zweck : Eine Mausbewegung wird zur Kenntnis genommen.
 *------------------------------------------------------------------------------
 */
void GrafFktEd::move(const Event& e)
{
  FuzzyTyp x = _grafik->xkoo2wert(e.pointer_x());
  FuzzyTyp y = _grafik->ykoo2wert(e.pointer_y());
  if(_auswahl)
    _x_auswahl = x;
  if(_punkt == 1) {   // Koordinaten immer aktualisieren
    _pkt1_x = x;
    _pkt1_y = y;
  } else {
    if(!_auswahl) {
      //_grafik->gummiband(false, _pkt1_x, _pkt1_y, _pkt2_x, _pkt2_y);
      _grafik->gummiband(true, _pkt1_x, _pkt1_y, x, y);
    }
    _pkt2_x = x;
    _pkt2_y = y;
  }
  _aktion = NeueKoo;
  if(_callback != nil)
    _callback->execute();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : GrafFktEd::press(Event&)
 * Parameter : Event& - Ereignis
 * Zweck : Ein Tastendruck wird zur Kenntnis genommen.
 *------------------------------------------------------------------------------
 */
void GrafFktEd::press(const Event& e)
{
  FuzzyTyp x = _grafik->xkoo2wert(e.pointer_x());
  FuzzyTyp y = _grafik->ykoo2wert(e.pointer_y());
  if(e.pointer_button() == Event::right) {   // Linie auswaehlen
    _auswahl   = !_auswahl;
    _x_auswahl = x;
    _aktion = RechteTaste;
    if(_callback != nil)
      _callback->execute();
  } else {                                 // andernfalls neue Linie setzen
    if(_auswahl) {
      _auswahl = false;
      _aktion = RechteTaste;
      if(_callback != nil)
        _callback->execute();
    }
    if(_punkt == 2) {
      _grafik->gummiband(false, _pkt1_x, _pkt1_y, _pkt2_x, _pkt2_y);
      if(e.pointer_button() == Event::middle)
        _aktion = MittelTaste;
      else
        _aktion = LinkeTaste;
      if(_callback != nil)
        _callback->execute();
    }
    _punkt   = _punkt % 2 + 1;
    if(_punkt == 1) {
      _pkt1_x = x;
      _pkt1_y = y;
    } else {
      _pkt2_x = x;
      _pkt2_y = y;
    }
    _aktion = NeueKoo;
    if(_callback != nil)
      _callback->execute();
  }
}


/*
 *------------------------------------------------------------------------------
 * Klasse : LingVarAuswahl
 * Zweck  : Bietet die Moeglichkeit zur Definition eines Satzes linguistischer
 *          Variablen. Und zwar koennen der Name, der Definitionbereich und
 *          der Typ der einzelnen Variablen angegeben werden.
 *------------------------------------------------------------------------------
 */
class LingVarAuswahl
{
  public : LingVarAuswahl(
             NfcAusgabe*, LingVarSatz* lingvar_satz, boolean steuer_var
           );
           virtual void setz_lingvar();
           virtual void aktua_auswahl(LingVar*);
           virtual boolean neuer_bereich(FuzzyTyp von, FuzzyTyp bis);
           virtual boolean neuer_typ(LingVarTyp typ);
           boolean typ_erlaubt(LingVarTyp typ, FuzzySet*, boolean steuer_var);
           LingVar* lingvar();
           boolean steuer_var();
           void geaendert();
           Glyph* lingvar_glyph();            // liefert Glyph dieser Klasse

  // Zustandsvariablen
  private: LingVarSatz* _lingvar_satz;
           LingVar _lva_lingvar;              // aktuelle ling. Variable
           CopyString _merk_name;             // Name der ling. Variablen
           boolean _gesichert;                // wurden Aenderungen gesichert ?
           boolean _steuer_var;               // handelt es sich um SteuerVar ?

  // Grafikelemente
  private: NfcStrBrowser* _var_tafel;         // Anzeigetafel fuer Variablen
           AuswahlButton* _typ_auswahl;       // Toggle-Button fue Var.-Typ
           EingabeFeld *_namens_ed,           // Eingabefeld fuer Namen
                       *_min_ed, *_max_ed;    // Eingabefelder fuer Def.bereich
           NfcAusgabe* _ausgabe_feld;
           Glyph* _glyph;
           WidgetKit* _kit;
           NfcKit* _nfc_kit;

  // lokale Funktionen
  private: void _init_var_tafel();            // initialisiert Anzeigetafel
           void _aktua_var();                 // aktualisiert aktuelle Var.
           void _lingvar_auswahl();           // Callback fuer Anzeigetafel
           void _neuer_name();                // Callback fuer Namens-Editor
           void _loesch_var();                // Callback fuer 'loesch'-Button
           void _reset();                     // Callback fuer 'neu'-Button
};

declareActionCallback(LingVarAuswahl);
implementActionCallback(LingVarAuswahl);
#define LvCb(PROC) ActionCallback(LingVarAuswahl)(this, &LingVarAuswahl::PROC)

LingVarAuswahl::LingVarAuswahl(
  NfcAusgabe* ausgabe, LingVarSatz* lingvar_satz, boolean steuer_var
)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();

  _ausgabe_feld = ausgabe;
  _gesichert    = true;
  _steuer_var   = steuer_var;
  _lingvar_satz = lingvar_satz;
  _kit          = WidgetKit::instance();
  _nfc_kit      = NfcKit::instance();

  // Auswahltafel erzeugen
  Action* lv_auswahl_ptr = new LvCb(_lingvar_auswahl);
  _var_tafel = new NfcStrBrowser(MAXNAME, 3, lv_auswahl_ptr);
  _init_var_tafel();

  // Eingabefeld fuer Namen erzeugen
  Action* aktua_name_ptr = new LvCb(_neuer_name);
  _namens_ed = new EingabeFeld(16, LeerStr, aktua_name_ptr);

  // Eingabefelder fuer Definitionsbereich erzeugen
  InputHandler* def_bereich = new InputHandler(nil, wkit->style());

  Action* aktua_var_ptr = new LvCb(_aktua_var);
  _min_ed = new EingabeFeld(7, _lva_lingvar.min(), aktua_var_ptr);
  def_bereich->append_input_handler(_min_ed->input_handler());
  _max_ed = new EingabeFeld(7, _lva_lingvar.max(), aktua_var_ptr);
  def_bereich->append_input_handler(_max_ed->input_handler());
  def_bereich->focus(_min_ed->input_handler());

  def_bereich->body(
    lkit->hbox(
      _nfc_kit->text_davor(
        Txt(26),
        lkit->hbox(lkit->hglue(), _min_ed),
        false
      ), _nfc_kit->text_davor(Txt(27), _max_ed, false)
    )
  );

  // Auswahlbutton fuer Variablen-Typ erzeugen
  _typ_auswahl = new AuswahlButton(_kit, aktua_var_ptr);
  _typ_auswahl->zur_auswahl(Txt(22));
  _typ_auswahl->zur_auswahl(Txt(23));

  // Buttons erzeugen
  Action* loesch_cb_ptr = new LvCb(_loesch_var);
  Action* setz_cb_ptr   = new LvCb(setz_lingvar);
  Action* reset_cb_ptr  = new LvCb(_reset);
  Glyph* loesch_bt      = _kit->push_button(Txt(1), loesch_cb_ptr);
  Glyph* setz_bt        = _kit->push_button(Txt(2), setz_cb_ptr);
  Glyph* neu_bt         = _kit->push_button(Txt(3), reset_cb_ptr);

  // gesamten Glyph erzeugen
  Coord rand = 10;
  _glyph = new Border(
             lkit->margin(
               lkit->hbox(
                 _nfc_kit->text_drauf(Txt(24), _var_tafel, true),
                 lkit->hglue(rand, 0, 0),
                 lkit->vbox(
                   lkit->vcenter(lkit->vglue(rand, 0, 0)),
                   _nfc_kit->text_davor(
                     Txt(25),
                     lkit->hbox(lkit->hglue(), _namens_ed),
                     false
                   ), lkit->vglue(rand),
                   def_bereich,
                   lkit->vglue(rand),
                   _nfc_kit->text_davor(
                     Txt(28),
                     lkit->hbox(lkit->hglue(), _typ_auswahl),
                     false
                   ), lkit->vglue(rand),
                   lkit->hbox(
                     loesch_bt, lkit->hglue(),
                     setz_bt,   lkit->hglue(),
                     neu_bt
                   ), lkit->vglue(rand, 0, 0)
                 )
               ), 10, 10
             ), _kit->foreground(), 1
           );

}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::aktua_auswahl(LingVar*)
 * Parameter : LingVar* - eine ling. Variable
 * Zweck : Aktualisiert die Aus/ bzw. Eingabefelder mit den Werten der angegeben
 *         Variablen.
 *------------------------------------------------------------------------------
 */
void LingVarAuswahl::aktua_auswahl(LingVar* neuer)
{
  if(neuer != nil) {
    _lva_lingvar = *neuer;
    _gesichert   = true;
   } else {
    _lva_lingvar = LingVar();
    _gesichert   = false;
   }
  _namens_ed->text(_lva_lingvar.name());
  String typ_str = (_lva_lingvar.typ() == EingabeVar) ? Txt(22) : Txt(23);
  _typ_auswahl->auswahl(typ_str);
  _min_ed->setz_wert(_lva_lingvar.min());
  _max_ed->setz_wert(_lva_lingvar.max());
  _merk_name = _lva_lingvar.name();
  if(_merk_name != LeerStr)
    _ausgabe_feld->ausgeben(Meldung, Txt(34), _lva_lingvar.name(), Txt(35));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::setz_lingvar()
 * Zweck : Nimmt die Variable, die gerade in Bearbeitung ist, in den
 *         'LingVarSatz' auf, falls diese noch nicht gesichert wurde.
 *------------------------------------------------------------------------------
 */
void LingVarAuswahl::setz_lingvar()
{
  if(_lva_lingvar.name() == LeerStr)
    _ausgabe_feld->ausgeben(Warnung, Txt(11)); // keine Var. ausgew.
  else
    if(_lingvar_satz != nil && !_gesichert) {
      _gesichert = true;
      boolean tafel_aktualisieren = false;
      if(_lingvar_satz->hol_lingvar(_lva_lingvar.name()) == nil)
         tafel_aktualisieren = true;
      _lingvar_satz->add_lingvar(&_lva_lingvar);
      _ausgabe_feld->ausgeben(Meldung, Txt(34), _lva_lingvar.name(), Txt(38));
      if(tafel_aktualisieren)
        _init_var_tafel();
    }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::neuer_bereich(FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp von - unterer Bereich
 *             FuzzyTyp bis - oberer Bereich
 * Zweck : Nimmt zur Kenntnis, dass neuer Definitionbereich eingegeben wurde und
 *         prueft, ob dieser fuer die vorhandenen Fuzzy-Sets nicht zu klein ist.
 *------------------------------------------------------------------------------
 */
boolean LingVarAuswahl::neuer_bereich(FuzzyTyp von, FuzzyTyp bis)
{
  if(_lva_lingvar.min(von) && _lva_lingvar.max(bis)) {
     _gesichert = false;
     return true;
   } else {  // Definitionsbereich ist zu klein
     _min_ed->setz_wert(_lva_lingvar.min());  // alten Bereich anzeigen
     _max_ed->setz_wert(_lva_lingvar.max());
     _ausgabe_feld->ausgeben(Fehler, Txt(29));
     return false;
   }
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::neuer_typ(LingVarTyp)
 * Parameter : LingVarTyp - neuer Typ
 * Zweck : Nimmt zur Kenntnis, dass neuer Typ eingegeben wurde und prueft, ob
 *         dieser mit den vorhandenen Fuzzy-Sets zu vereinbaren ist.
 *------------------------------------------------------------------------------
 */
boolean LingVarAuswahl::neuer_typ(LingVarTyp typ)
{
  boolean ok = true;
  int anzahl = _lva_lingvar.anz_fuzzysets();
  for(int i = 0; ok && i < anzahl; i++)
    ok = typ_erlaubt(typ, _lva_lingvar.hol_fuzzyset(i), _steuer_var);
  if(ok) {
    _lva_lingvar.typ(typ);
    _gesichert = false;
  } else
    _ausgabe_feld->ausgeben(Fehler, Txt(32));
  return ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::typ_erlaubt(LingVarTyp, FuzzySet*, boolean)
 * Parameter : LingVarTyp - Typ der Variablen
 *             FuzzySet*  - zu ueberpruefendes Fuzzy-Set
 *             boolean    - werden Steuer-Variablen definiert ?
 * Rueckgabewert : true <=> erlaubt
 * Zweck : Ueberprueft, ob das Fuzzy-Set fuer den entsprechenden Variablen-Typ
 *         erlaubt ist.
 *------------------------------------------------------------------------------
 */
boolean LingVarAuswahl::typ_erlaubt(
          LingVarTyp typ, FuzzySet* fuzzyset, boolean steuer_var
        )
{
  if(!steuer_var || typ == EingabeVar)
    return true;
  return fuzzyset->anz_intervalle() <= 1;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarAuswahl::lingvar(),
 *                   LingVarAuswahl::geaendert(),
 *                   LingVarAuswahl::lingvar_glyph()
 * Zweck : Funktionen, um auf lokale Daten zuzugreifen.
 *------------------------------------------------------------------------------
 */

LingVar* LingVarAuswahl::lingvar() { return &_lva_lingvar; }

boolean LingVarAuswahl::steuer_var() { return _steuer_var; }

void LingVarAuswahl::geaendert() { _gesichert = false; }

Glyph* LingVarAuswahl::lingvar_glyph() { return _glyph; }

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

void LingVarAuswahl::_init_var_tafel()
{
  while(_var_tafel->anzahl() > 0)
    _var_tafel->loeschen(0);
  for(int i = 0; i < _lingvar_satz->anz_lingvars(); i++)
    _var_tafel->zur_auswahl(_lingvar_satz->hol_lingvar(i)->name());
  _var_tafel->aktualisiere();
}

void LingVarAuswahl::_aktua_var()
{
  // Testen, ob neuer Definitionbereich eingegeben wurde
  FuzzyTyp min, max;
  if(_min_ed->hol_wert(min) && _max_ed->hol_wert(max)) {
    if(min != _lva_lingvar.min() || max != _lva_lingvar.max())
      neuer_bereich(min, max);
  } else
    _ausgabe_feld->ausgeben(Fehler, Txt(31));

  // Testen, ob der Variablentyp geaendert wurde
  LingVarTyp typ;
  typ = (_typ_auswahl->auswahl() == Txt(22))? EingabeVar : AusgabeVar;
  if(typ != _lva_lingvar.typ())
    if(!neuer_typ(typ)) {          // evtl. alten Typ wieder anzeigen
      if(typ == EingabeVar)
        _typ_auswahl->auswahl(Txt(23));
      else
        _typ_auswahl->auswahl(Txt(22));
    }
}

void LingVarAuswahl::_lingvar_auswahl()
{
  String auswahl = _var_tafel->ausgewaehlt();
  if(auswahl != _lva_lingvar.name()) {
    if(_lva_lingvar.name() != LeerStr)
      setz_lingvar();
    aktua_auswahl(_lingvar_satz->hol_lingvar(auswahl));
 }
}

void LingVarAuswahl::_neuer_name()
{
  if(_lingvar_satz != nil) {
    String name = _namens_ed->text();
    if(name == LeerStr) {
      setz_lingvar();
      _var_tafel->markieren(-1);
      aktua_auswahl(nil);
    } else
      if(name != _lva_lingvar.name()) {
        LingVar* lingvar = _lingvar_satz->hol_lingvar(name);
        if(lingvar == nil) {
          setz_lingvar();
          _lva_lingvar.name(name);
          _gesichert = false;
          setz_lingvar();
          _var_tafel->markieren(name);
          aktua_auswahl(_lingvar_satz->hol_lingvar(name));
        } else {
          _var_tafel->markieren(name);
          aktua_auswahl(lingvar);
        }
      }
  } else
    aktua_auswahl(nil);
}

void LingVarAuswahl::_loesch_var()
{
  if(_lingvar_satz != nil) {
    if(_lingvar_satz->hol_lingvar(_lva_lingvar.name()) != nil) {
      _lingvar_satz->del_lingvar(_lva_lingvar.name());
      _ausgabe_feld->ausgeben(Meldung, Txt(34), _lva_lingvar.name(), Txt(36));
      _init_var_tafel();
      aktua_auswahl(nil);
    } else  // Variable nicht gefunden
      _ausgabe_feld->ausgeben(Meldung, Txt(37));
  }
}

void LingVarAuswahl::_reset()
{
  if(_lingvar_satz != nil)
    aktua_auswahl(_lingvar_satz->hol_lingvar(_merk_name));
  else
    aktua_auswahl(nil);
}


/*
 *------------------------------------------------------------------------------
 * Klasse : FuzzySetAuswahl
 * Zweck  : Bietet die Moeglichkeit zur Eingabe von Fuzzy-Sets. Dabei koennen
 *          sowohl der Name als auch die eigentliche Zugehoerigkeitsfunktion
 *          eingegeben werden.
 *------------------------------------------------------------------------------
 */
class FuzzySetAuswahl : public LingVarAuswahl
{
  public : FuzzySetAuswahl(NfcAusgabe*, LingVarSatz*, boolean steuer_var);
           virtual void aktua_auswahl(LingVar*);
           virtual void setz_lingvar();
           virtual boolean neuer_bereich(FuzzyTyp von, FuzzyTyp bis);
           virtual boolean neuer_typ(LingVarTyp);

           Glyph* fuzzyset_glyph();
           Glyph* grafik_glyph();

  // Zustandsvariablen
  private: FuzzySet _fuzzyset;           // aktuelles Fuzzyset
           CopyString _merk_name;        // Name des alten Fuzzysets
           boolean _gesichert,           // true, wenn alles gespeichert
                   _geaendert;           // true, wenn Variable geandert wurde

  // lokale Funktionen
  private: void _init_fuz_tafel();
           void _fuz_auswahl();
           void _neues_fuzzyset(String name, boolean sichern);
           void _aktua_koo();            // aktuelle Koordinaten anzeigen
           void _setz_intervall(boolean ext = false);
           void _setz_intervall_cb();    // Callback fuer Eingabefeld y2
           void _neuer_name();           // Callback fuer Eingabe eines Namens
           void _loeschen_cb();          // Callback fuer Loesch - Button
           void _setz_fuzzyset();        // Callback fuer Setz - Button
           void _reset_cb();             // Callback fuer Neu - Button
           void _grafik_cb();            // Callback fuer GrafFktEd
           void _elastizitaet();         // Callback fuer Breite
           void _neue_lingvar();         // neue ling. Variable setzen
           void _neu_skala();            // neue Bereichsgrenzen der ling. Var.
           boolean _bereit();

  // Grafikelemente
  private: FuzzySetGrafik* _grafik;      // grafische Darstellung des Fuzzysets
           GrafFktEd* _grafik_ed;        // fuer Mauseingaben in Grafik
           NfcStrBrowser* _fuz_tafel;    // Anzeige der definierten Fuzzysets
           EingabeFeld *_namens_ed,      // verschiedene Eingabefelder
                       *_x1_ed, *_y1_ed,
                       *_x2_ed, *_y2_ed,
                       *_dehn_ed, *_schr_ed;
           NfcAusgabe* _ausgabe_feld;
           int _akt_linie;               // Nummer einer evtl. markierten Linie
           Glyph* _glyph;
           WidgetKit* _kit;
           NfcKit* _nfc_kit;
};

declareActionCallback(FuzzySetAuswahl);
implementActionCallback(FuzzySetAuswahl);
#define FsCb(PROC) ActionCallback(FuzzySetAuswahl)(this,&FuzzySetAuswahl::PROC)

FuzzySetAuswahl::FuzzySetAuswahl(
  NfcAusgabe* ausgabe, LingVarSatz* lingvars, boolean steuer_var
) : LingVarAuswahl(ausgabe, lingvars, steuer_var)
{
  Coord rand = 10;

  LayoutKit* lkit = LayoutKit::instance();
  _ausgabe_feld   = ausgabe;

  // einfache Callbacks
  Action* fuz_cb_ptr    = new FsCb(_fuz_auswahl);
  Action* setz_koo_ptr  = new FsCb(_aktua_koo);
  Action* loesch_cb_ptr = new FsCb(_loeschen_cb);
  Action* setz_cb_ptr   = new FsCb(_setz_fuzzyset);
  Action* neu_cb_ptr    = new FsCb(_reset_cb);
  Action* name_cb_ptr   = new FsCb(_neuer_name);
  Action* y2_cb_ptr     = new FsCb(_setz_intervall_cb);
  Action* grafik_cb_ptr = new FsCb(_grafik_cb);
  Action* elaste_cb_ptr = new FsCb(_elastizitaet);

  LingVar def;
  _kit       = WidgetKit::instance();
  _nfc_kit   = NfcKit::instance();
  _gesichert = true;
  _geaendert = false;
  _akt_linie = -1;
  _fuz_tafel = new NfcStrBrowser(MAXNAME, 3, fuz_cb_ptr);

  // Eingabefeld fuer Namen erzeugen
  _namens_ed = new EingabeFeld(16, LeerStr, name_cb_ptr);

  // Eingabefelder fuer Dehnbarkeit erzeugen
  InputHandler* dehn_bereich = new InputHandler(nil, _kit->style());

  _schr_ed = new EingabeFeld(7, _fuzzyset.schrumpf(), elaste_cb_ptr);
  _dehn_ed = new EingabeFeld(7, _fuzzyset.dehn(), elaste_cb_ptr);

  if(steuer_var) {
    dehn_bereich->append_input_handler(_schr_ed->input_handler());
    dehn_bereich->append_input_handler(_dehn_ed->input_handler());

    dehn_bereich->focus(_schr_ed->input_handler());

    dehn_bereich->body(
      lkit->hbox(
        _nfc_kit->text_davor(
          Txt(228),
          lkit->hbox(lkit->hglue(), _schr_ed),
          false
        ), _nfc_kit->text_davor(Txt(229), _dehn_ed, false)
      )
    );
 }

  // Eingabefelder fuer die Linien erzeugen
  _x1_ed = new EingabeFeld(9,  "0", nil);
  _y1_ed = new EingabeFeld(9,  "0", nil);
  _x2_ed = new EingabeFeld(9,  "0", nil);
  _y2_ed = new EingabeFeld(9,  "0", y2_cb_ptr);

  _grafik    = new FuzzySetGrafik(def.min(), def.max(), YMIN, YMAX);
  _grafik_ed = new GrafFktEd(_kit->style(), _grafik, grafik_cb_ptr);

  Glyph* loesch_bt = _kit->push_button(Txt(1), loesch_cb_ptr);
  Glyph* setz_bt   = _kit->push_button(Txt(2), setz_cb_ptr);
  Glyph* neu_bt    = _kit->push_button(Txt(3), neu_cb_ptr);

  InputHandler* punkt_eingabe = new InputHandler(nil, _kit->style());
  punkt_eingabe->append_input_handler(_x1_ed->input_handler());
  punkt_eingabe->append_input_handler(_y1_ed->input_handler());
  punkt_eingabe->append_input_handler(_x2_ed->input_handler());
  punkt_eingabe->append_input_handler(_y2_ed->input_handler());
  punkt_eingabe->focus(_x1_ed->input_handler());
  punkt_eingabe->body(
    lkit->hbox(
      _nfc_kit->text_davor(
        Txt(6),
        lkit->vbox(
          lkit->vcenter(_nfc_kit->text_davor("x1 = ", _x1_ed, false)),
          lkit->vglue(5),
          _nfc_kit->text_davor("y1 = ", _y1_ed, false)
        ), false
      ),
      _nfc_kit->text_davor(
        Txt(7),
        lkit->vbox(
          lkit->vcenter(_nfc_kit->text_davor("x2 = ", _x2_ed, false)),
          lkit->vglue(5),
          _nfc_kit->text_davor("y2 = ", _y2_ed, false)
        ), false
      )
    )
  );

  _glyph = new Border(
             lkit->margin(
               lkit->hbox(
                 _nfc_kit->text_drauf(Txt(4), _fuz_tafel, true),
                 lkit->hglue(rand, 0, 0),
                 lkit->vbox(
                   lkit->vcenter(lkit->vglue(rand, 0, 0)),
                   _nfc_kit->text_davor(
                      Txt(5),
                      lkit->hbox(lkit->hglue(), _namens_ed),
                      false
                   ), lkit->vglue(rand, 0, 0),
                   dehn_bereich,
                   lkit->vglue(rand),
                   punkt_eingabe,
                   lkit->vglue(rand),
                   lkit->hbox(
                     loesch_bt, lkit->hglue(), setz_bt, lkit->hglue(), neu_bt
                   ), lkit->vglue(rand, 0, 0)
                 )
               ), 10, 10
             ), _kit->foreground()
           );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetAuswahl::aktua_auswahl(LingVar*)
 * Parameter : LingVar* - eine ling. Variable
 * Zweck : Aktualisiert die Aus/ bzw. Eingabefelder mit den Werten der angegeben
 *         Variablen.
 *------------------------------------------------------------------------------
 */
void FuzzySetAuswahl::aktua_auswahl(LingVar* neu_lingvar)
{
  LingVarAuswahl::aktua_auswahl(neu_lingvar);
  _neues_fuzzyset(LeerStr, false);
  neuer_bereich(lingvar()->min(), lingvar()->max());
  _init_fuz_tafel();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetAuswahl::setz_lingvar()
 * Zweck : Speichert das Fuzzy-Set, dass gerade in Bearbeitung ist in der
 *         aktuellen ling. Variablen ab.
 *------------------------------------------------------------------------------
 */
void FuzzySetAuswahl::setz_lingvar()
{
  _setz_fuzzyset();
  LingVarAuswahl::setz_lingvar();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetAuswahl::neuer_bereich(FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp von - unterer Bereich
 *             FuzzyTyp bis - oberer Bereich
 * Zweck : Nimmt zur Kenntnis, dass neuer Definitionbereich eingegeben wurde und
 *         prueft, ob dieser fuer das gerade in Bearbeitung befindliche 
 *         Fuzzy-Set nicht zu klein ist.
 *------------------------------------------------------------------------------
 */
boolean FuzzySetAuswahl::neuer_bereich(FuzzyTyp von, FuzzyTyp bis)
{
  boolean ok = true;
  int anzahl = _fuzzyset.anz_intervalle();
  for(int i = 0; i < anzahl; i++) {
    FsIntervall intervall = *_fuzzyset.hol_intervall(i);
    if(intervall.links() < von || intervall.rechts() > bis)
      ok = false;
  }
  if(!ok) {   // Definitionsbereich ist zu klein fuer aktuelles Fuzzy-Set
    _ausgabe_feld->ausgeben(Fehler, Txt(30));
    return false;
  }

  if(LingVarAuswahl::neuer_bereich(von, bis)) {
    _grafik->neue_skala(Dimension_X, von, bis);
    _grafik->aktualisiere();
    return true;
  } else
    return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion :  FuzzySetAuswahl::neuer_typ(LingVarTyp)
 * Parameter : LingVarTyp - neuer Typ
 * Zweck : Nimmt zur Kenntnis, dass neuer Typ eingegeben wurde und prueft, ob
 *         dieser mit dem gerade in Bearbeitung befindlichen Fuzzy-Set zu
 *         vereinbaren ist.
 *------------------------------------------------------------------------------
 */
boolean FuzzySetAuswahl::neuer_typ(LingVarTyp typ)
{
  if(typ_erlaubt(typ, &_fuzzyset, steuer_var()))
    return LingVarAuswahl::neuer_typ(typ);
  _ausgabe_feld->ausgeben(Fehler, Txt(33));
  return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : FuzzySetAuswahl::fuzzyset_glyph(),
 *                   FuzzySetAuswahl::grafik_glyph()
 * Zweck : Funktionen, um auf lokale Daten zuzugreifen.
 *------------------------------------------------------------------------------
 */

Glyph* FuzzySetAuswahl::fuzzyset_glyph() { return _glyph; }

Glyph* FuzzySetAuswahl::grafik_glyph() { return _grafik_ed; };

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

void FuzzySetAuswahl::_init_fuz_tafel()
{
  if(lingvar() != nil) {
    while(_fuz_tafel->anzahl() > 0)
      _fuz_tafel->loeschen(0);
    for(int i = 0; i < lingvar()->anz_fuzzysets(); i++)
      _fuz_tafel->zur_auswahl(lingvar()->hol_fuzzyset(i)->name());
    _fuz_tafel->aktualisiere();
  }
}

void FuzzySetAuswahl::_fuz_auswahl()
{
  _neues_fuzzyset(_fuz_tafel->ausgewaehlt(), true);
}

void FuzzySetAuswahl::_neues_fuzzyset(String name, boolean sichern)
{
  boolean gueltig = true;

  // wenn neues Fuzzy-Set gesetzt werden soll, aber keine ling. Variable
  // gesetzt ist : Fehlermeldung ("keine Var. ausgewaehlt")
  if(lingvar()->name() == LeerStr)
    if(name != LeerStr)
      gueltig = false;

  if(!gueltig)
    _ausgabe_feld->ausgeben(Fehler, Txt(11));
  else { 
    if(sichern)
      _setz_fuzzyset();
    FuzzySet* neuer = lingvar()->hol_fuzzyset(name);
    if(neuer == nil) {

        // neues Fuzzy-Set erzeugen
        _fuzzyset  = FuzzySet(name);
        _gesichert = false;           // damit auf jeden Fall gesichert wird
        _setz_fuzzyset();
        neuer = lingvar()->hol_fuzzyset(name); // wg. Default-Elast.
      }

    if(neuer != nil) {
      _fuzzyset = *neuer;
      _fuz_tafel->markieren(name);
    }
    _namens_ed->text(_fuzzyset.name());
    _schr_ed->setz_wert(_fuzzyset.schrumpf());
    _dehn_ed->setz_wert(_fuzzyset.dehn());
    _grafik->setz_fuzzyset(&_fuzzyset);
    if(_grafik_ed->auswahl_mod()) {
      _akt_linie = -1;
      _grafik->markiere_linie(-1);   // keine Linie soll markiert sein
    }
    _grafik->aktualisiere();
    _gesichert = true;
    _merk_name = _fuzzyset.name();
    if(_merk_name != LeerStr)
       _ausgabe_feld->ausgeben(Meldung, Txt(8), _fuzzyset.name(), Txt(12));
  }
}

void FuzzySetAuswahl::_grafik_cb()
{
  GrafikAktion aktion = _grafik_ed->aktion();
  switch(aktion) {
    case NeueKoo     : _aktua_koo();           break;
    case LinkeTaste  : _setz_intervall(false); break;
    case MittelTaste : _setz_intervall(true);  break;
    case RechteTaste :
      if(_grafik_ed->auswahl_mod())
        _ausgabe_feld->ausgeben(Meldung, Txt(18));
      else {
        _ausgabe_feld->ausgeben(Meldung, Txt(19));
        _akt_linie = -1;
        _grafik->markiere_linie(-1);         // Markierung loeschen
      }
      _aktua_koo();
      break;
    default: break;
  }
}

void FuzzySetAuswahl::_aktua_koo()
{
  if(_grafik_ed->auswahl_mod()) {  // ausgewaehlte Linie markieren
    FuzzyTyp x = _grafik_ed->x_auswahl();
    boolean fertig = false;
    for(int i = 0; i < _fuzzyset.anz_intervalle() && !fertig; i++) {
      FsIntervall* intervall = _fuzzyset.hol_intervall(i);
      if(_akt_linie != i && intervall->enthalten(x)) {
        FuzzyTyp x1, x2;
        FuzzyTyp y1, y2;
        intervall->links(x1, y1);
        intervall->rechts(x2, y2);
        _x1_ed->setz_wert(x1);
        _y1_ed->setz_wert(y1);
        _x2_ed->setz_wert(x2);
        _y2_ed->setz_wert(y2);
        _grafik->markiere_linie(i);
        _akt_linie = i;
        fertig = true;
      }
    }
  } else {   // aktuelle Mausposition anzeigen
    FuzzyTyp x, y;
    int pkt = _grafik_ed->punkt();
    _grafik_ed->koordinaten(pkt, x, y);
    if(pkt == 1) {
      _x1_ed->setz_wert(x);
      _y1_ed->setz_wert(y);
    } else {
      _x2_ed->setz_wert(x);
      _y2_ed->setz_wert(y);
    }
  }
}

// Callback fuer y2-Eingabefeld
void FuzzySetAuswahl::_setz_intervall_cb() { _setz_intervall(false); }

void FuzzySetAuswahl::_setz_intervall(boolean extend)
{
  FuzzyTyp x1, x2;
  FuzzyTyp y1, y2;
  if(_bereit()) {
    if(_x1_ed->hol_wert(x1) && _y1_ed->hol_wert(y1)) {
      if(_x2_ed->hol_wert(x2) && _y2_ed->hol_wert(y2)) {
        FsIntervall intervall(x1, y1, x2, y2);
        boolean ok = true;

        // der Teil der Linie, der aus Def.bereich ragt, wird abgeschnitten
        if(intervall.links() < lingvar()->min()) {
          if(intervall.rechts() > lingvar()->min())
            intervall.setz_links(lingvar()->min());
          else
            ok = false;
        }
        if(intervall.rechts() > lingvar()->max()) {
          if(intervall.links() < lingvar()->max())
            intervall.setz_rechts(lingvar()->max());
          else
            ok = false;
        }
        if(ok) {
          if(extend)
            intervall.ausdehnen(lingvar()->min(), lingvar()->max(), YMIN, YMAX);
          _fuzzyset.add_intervall(&intervall);
          _grafik->setz_fuzzyset(&_fuzzyset);
          _grafik->aktualisiere();
          _gesichert = false;
        } else                          // Linie nicht im Definitionsbereich
          _ausgabe_feld->ausgeben(Fehler, Txt(14));
      } else                            // Fehler bei Angabe des 2. Punktes
        _ausgabe_feld->ausgeben(Fehler, Txt(15));
    } else                              // Fehler bei Angabe des 1. Punktes
      _ausgabe_feld->ausgeben(Fehler, Txt(16));
  } else                                // Kein Fuzzy-Set ausgewaehlt
    _ausgabe_feld->ausgeben(Warnung, Txt(17));
}

void FuzzySetAuswahl::_neuer_name()
{
  if(lingvar()->name() == LeerStr)
    _ausgabe_feld->ausgeben(Fehler, Txt(11)); // keine Var. ausgewaehlt
  else {
    _neues_fuzzyset(_namens_ed->text(), true);
    _gesichert = false;
  }
}

void FuzzySetAuswahl::_elastizitaet()
{
  FuzzyTyp schr, dehn;
  if(_schr_ed->hol_wert(schr) && _dehn_ed->hol_wert(dehn)) {
    _fuzzyset.elastizitaet(schr, dehn);
    _gesichert = false;
  } else
    _ausgabe_feld->ausgeben(Fehler, Txt(230));
}

void FuzzySetAuswahl::_loeschen_cb()
{
  if(lingvar() != nil) {
    if(lingvar()->hol_fuzzyset(_fuzzyset.name()) != nil) {
      _geaendert = true;
      lingvar()->del_fuzzyset(_fuzzyset.name());
      geaendert();
      _ausgabe_feld->ausgeben(Meldung, Txt(8), _fuzzyset.name(), Txt(20));
      _init_fuz_tafel();
      _neues_fuzzyset(LeerStr, false);
    } else
      _ausgabe_feld->ausgeben(Meldung, Txt(21));
  }
}

void FuzzySetAuswahl::_setz_fuzzyset()
{
  if(_bereit() && !_gesichert) {
    _geaendert = true;

    // abfragen, ob schon ein Fuzzyset mit demselben Namen existiert
    boolean aktua_tafel = (lingvar()->hol_fuzzyset(_fuzzyset.name()) == nil);

    // Fuzzyset eintragen
    if(lingvar()->add_fuzzyset(&_fuzzyset)) {
      _ausgabe_feld->ausgeben(Meldung, Txt(8), _fuzzyset.name(), Txt(9));
      geaendert();
    } else   // sollte nicht vorkommen
      _ausgabe_feld->ausgeben(Warnung, Txt(8), _fuzzyset.name(), Txt(10));

    // falls noetig, Auswahltafel aktualisieren
    if(aktua_tafel)
      _init_fuz_tafel();
  }
}

void FuzzySetAuswahl::_reset_cb() { _neues_fuzzyset(_merk_name, false); }

boolean FuzzySetAuswahl::_bereit()
{
  if(lingvar()->name() == LeerStr)
    return false;
  if(_fuzzyset.name() == LeerStr)
    return false;
  return true;
}


/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : LingVarEd -----------------------
 *------------------------------------------------------------------------------
 */

MenueStruktur LvEd_menue[] = {
  { 66, nil },                   // "beenden"
  { 67, nil },                   // "abbrechen"
  { -1 }
};

MenueStruktur LvEd_menue_bar[] = {
  { 65, nil, LvEd_menue, 3 },    // "fertig"
  { -1 }
};

declareActionCallback(LingVarEd);
implementActionCallback(LingVarEd);

LingVarEd::LingVarEd(
  const LingVarSatz* lingvarsatz,
  boolean steuer_var,
  Action* ende_cb,
  Action* abbruch_cb
) : NfcDialogFenster(nil)
{
  _ausgabe_feld = new NfcAusgabe(new AusgabeFeld(30, 10));

  _lingvarsatz = *lingvarsatz;
  _fuz_editor  = new FuzzySetAuswahl(_ausgabe_feld, &_lingvarsatz, steuer_var);

  WidgetKit* kit  = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nfc_kit = NfcKit::instance();

  // Ende-Callback merken
  _ende_cb = ende_cb;
  if(_ende_cb != nil)
    Resource::ref(_ende_cb);

  // Callbacks fuer das Menue installieren
  Action* cb = new ActionCallback(LingVarEd)(this, &LingVarEd::_ende);
  LvEd_menue[0].cb = cb;
  LvEd_menue[1].cb = abbruch_cb;
  Menu* menue_balken = nfc_kit->menue(LvEd_menue_bar, kit, lkit);

  Coord rand = 10;
  body(lkit->vbox(
         lkit->vcenter(
             lkit->hbox(
               lkit->variable_span(menue_balken, 0, 0),
               lkit->hglue()
             )
         ), lkit->vglue(rand, 0, 0),
         lkit->hbox(
           lkit->hglue(rand, 0, 0),
           lkit->variable_span_dimension(
             lkit->vbox(
               lkit->vcenter(_fuz_editor->lingvar_glyph()),
               lkit->vglue(rand, 0, 0),
               _fuz_editor->fuzzyset_glyph(),
               lkit->vglue(rand, 0, 0),
               nfc_kit->text_drauf(Txt(40), _ausgabe_feld, true)
             ), Dimension_X, 0, 0
           ), lkit->hglue(rand, 0, 0),
           lkit->vcenter(
             new Border(
               lkit->margin(_fuz_editor->grafik_glyph(), 1, 1),
               kit->foreground(), 1
             ), 1
           ), lkit->hglue(rand, 0, 0)
         ), lkit->vglue(rand, 0, 0)
       )
  );
}

LingVarEd::~LingVarEd()
{
  delete _fuz_editor;
  if(_ende_cb != nil)
    Resource::unref(_ende_cb);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : LingVarEd::lingvar()
 * Zweck : liefert eingegeben LingVarSatz und sichert vorher aktuelle ling.
 *         Variable.
 *------------------------------------------------------------------------------
 */
LingVarSatz* LingVarEd::lingvar()
{
  _fuz_editor->setz_lingvar();
  return &_lingvarsatz;
}

void LingVarEd::_ende()
{
  _fuz_editor->setz_lingvar();
  if(_ende_cb != nil)
    _ende_cb->execute();
}
