/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#ifndef lingvar_h
#define lingvar_h

#include <OS/string.h>
#include <OS/list.h>

#include "texte.h"
#include "global.h"

// Definition eines Typs, der fuer interne Berechnungen des Fuzzy-Controllers
// verwendet wird. Damit sind auch alle Mess- und Steuerwerte von diesem Typ.
typedef float FuzzyTyp;

/*
 * wegen dauernder Compiler-Warnings wird die Zeile
 *           'enum LingVarTyp { EingabeVar, AusgabeVar };'
 * durch folgende Konstruktion ersetzt :
 * (wichtig : fuer die Werte, die bei 'define' benutzt werden, muss gelten :
 *            EingabeVar = AusgabeVar - 1  )
 */
typedef int LingVarTyp;
#define EingabeVar 0
#define AusgabeVar 1

/*
 *------------------------------------------------------------------------------
 * Klasse : FsIntervall
 * Zweck  : Definiert ein Intervall einer Funktion. Das Intervall wird durch
 *          zwei Eckpunkte dargestellt und verlaeuft dazwischen liniear.
 *------------------------------------------------------------------------------
 */
class FsIntervall
{
  public : FsIntervall();
           FsIntervall(FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2,
                       boolean links_drin = true, boolean rechts_drin = true);

           void setz_links(FuzzyTyp x, boolean drin = true);
           void setz_rechts(FuzzyTyp x, boolean drin = true);
           FuzzyTyp links() { return _x1; };
           FuzzyTyp rechts() { return _x2; };
           void links(FuzzyTyp& x, FuzzyTyp& y)  { x = _x1; y = _y1; };
           void rechts(FuzzyTyp& x, FuzzyTyp& y) { x = _x2; y = _y2; };

           boolean leer() { return _x1 == _x2 && !_links_drin; };
           boolean enthalten(FuzzyTyp x);

           FuzzyTyp wert(FuzzyTyp x);
           boolean invertiere(FuzzyTyp y, FuzzyTyp& x);
           void breiter(
                  FuzzyTyp grad,
                  FuzzyTyp min,    FuzzyTyp max,
                  FuzzyTyp min_br, FuzzyTyp max_br
                );
           void ausdehnen(
                  FuzzyTyp x_min, FuzzyTyp x_max, FuzzyTyp y_min, FuzzyTyp y_max
                );
  private: FuzzyTyp _xwert(FuzzyTyp y);
           FuzzyTyp _ywert(FuzzyTyp x);
           FuzzyTyp _x1, _x2;
           FuzzyTyp _y1, _y2;
           boolean _links_drin,
                   _rechts_drin;
};

declarePtrList(IntervallListe, FsIntervall)

/*
 *------------------------------------------------------------------------------
 * Klasse : FuzzySet
 * Zweck  : Definition eines Fuzzy-Sets. Das Fuzzy-Set wird durch eine Liste
 *          'FsIntervall'en und einen Namen dargestellt.
 *------------------------------------------------------------------------------
 */
class FuzzySet
{
  public : FuzzySet();
           FuzzySet(String name);
           FuzzySet(const FuzzySet& fs);
           ~FuzzySet();

           String name() const { return _name; };
           void name(String name) { _name = name; };

           FuzzyTyp wert(FuzzyTyp x);
           FuzzyTyp invertiere(FuzzyTyp y);
           void justiere(FuzzyTyp grad);

           FsIntervall* hol_intervall(int nr) const;
           void del_intervall(int nr);
           void add_intervall(FsIntervall* intervall);
           void add_intervall(FuzzyTyp x1, FuzzyTyp y1,
                              FuzzyTyp x2, FuzzyTyp y2,
                              boolean li_drin = true, boolean re_drin = true);
           int anz_intervalle() const;

           void setz_grenzen(FuzzyTyp min, FuzzyTyp max);

           FuzzyTyp schrumpf() { return _schrumpf; };
           FuzzyTyp dehn() { return _dehn; };
           void elastizitaet(FuzzyTyp schrumpf, FuzzyTyp dehn);

           FuzzySet& operator=(const FuzzySet&);
  private: IntervallListe _intervall_liste;
           CopyString _name;
           void _loeschen(FuzzyTyp x1, FuzzyTyp x2);
           FuzzyTyp _min, _max;          // Definitionbereich
           FuzzyTyp _schrumpf, _dehn;    // minimale bzw. maximale Breite
};

inline FsIntervall* FuzzySet::hol_intervall(int nr) const {
  return _intervall_liste.item(nr);
}

inline void FuzzySet::del_intervall(int nr) {
  delete _intervall_liste.item(nr);
  _intervall_liste.remove(nr);
}

inline int FuzzySet::anz_intervalle() const {
  return _intervall_liste.count();
}

declarePtrList(FuzzySetListe, FuzzySet)

/*
 *------------------------------------------------------------------------------
 * Klasse : LingVar
 * Zweck  : Definition einer linguistischen Variablen. Eine linguistische
 *          Variable wird modelliert durch einen Name, einen Definitionsbereich,
 *          einen Typ und eine Liste von Fuzzy-Sets.
 *------------------------------------------------------------------------------
 */

class LingVar
{
  public : LingVar();
           LingVar(String name);
           LingVar(const LingVar& lv);
           ~LingVar();

           String name() const { return _name; };
           void name(String name) { _name = name; };

           FuzzyTyp min() const { return _von; };
           FuzzyTyp max() const { return _bis; };
           boolean min(FuzzyTyp min);
           boolean max(FuzzyTyp max);

           LingVarTyp typ() const { return _typ; };
           void typ(LingVarTyp typ);

           int fuz_nr(String name);
           FuzzySet* hol_fuzzyset(int nr) const;
           FuzzySet* hol_fuzzyset(String name) const;
           void del_fuzzyset(int nr);
           void del_fuzzyset(String name);
           boolean add_fuzzyset(FuzzySet* fuzzyset);
           int anz_fuzzysets() const;

           LingVar& operator=(const LingVar&);
  private: FuzzySetListe _fuzzyset_liste;
           CopyString _name;
           LingVarTyp _typ;
           FuzzyTyp _von, _bis;
};

inline LingVar::LingVar(const LingVar& lingvar) { *this = lingvar; }

inline FuzzySet* LingVar::hol_fuzzyset(int nr) const {
  return _fuzzyset_liste.item(nr);
}

inline int LingVar::anz_fuzzysets() const {
  return _fuzzyset_liste.count();
}

declarePtrList(LingVarListe, LingVar)

/*
 *  ----------------- Deklaration der Klasse : 'LingVarSatz' -------------------
 */

/*
 *------------------------------------------------------------------------------
 * Klasse : LingVarSatz
 * Zweck  : Definiert eine Menge von ling. Variablen; diese werden in Form
 *          einer Liste gespeichert.
 *------------------------------------------------------------------------------
 */
class LingVarSatz
{
  public : LingVarSatz() {};
           LingVarSatz(const LingVarSatz& lvs);
           ~LingVarSatz();

           int var_nr(LingVarTyp, String name) const;
           int var_nr(String name) const;
           LingVar* hol_lingvar(int nr) const;
           LingVar* hol_lingvar(LingVarTyp typ, int nr) const;
           LingVar* hol_lingvar(String name) const;

           void del_lingvar(int nr);
           void del_lingvar(String name);
           void add_lingvar(LingVar* lingvar);

           int anz_lingvars() const;
           int anz_lingvars(LingVarTyp) const;

           void tausche(int nr1, int nr2);

           LingVarSatz& operator=(const LingVarSatz&);
  private: LingVarListe _lingvar_liste;
};

inline LingVarSatz::LingVarSatz(const LingVarSatz& lingvarsatz) {
  *this = lingvarsatz;
}

inline LingVar* LingVarSatz::hol_lingvar(int nr) const {
  return _lingvar_liste.item(nr);
}

inline void LingVarSatz::del_lingvar(int nr) {
  delete _lingvar_liste.item(nr);
  _lingvar_liste.remove(nr);
}

inline int LingVarSatz::anz_lingvars() const {
  return _lingvar_liste.count();
}

#endif
