/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#include <math.h>
#include <InterViews/style.h>
#include <IV-look/button.h>
#include <InterViews/glyph.h>
#include <InterViews/window.h>
#include <InterViews/font.h>
#include <OS/list.h>
#include <OS/math.h>

#include "global.h"
#include "koosystem.h"

#define MAXSTRING  20

// Defaultgroesse des Koordinatensystems
#define DEF_BREITE 600.0
#define DEF_HOEHE  200.0
#define MAX_BREITE 900.0
#define MAX_HOEHE  900.0
#define MIN_BREITE 50.0
#define MIN_HOEHE  50.0

/*
 *------------------------------------------------------------------------------
 *------------------- Definition der Klasse : 'KooKreuz' -----------------------
 *------------------------------------------------------------------------------
 */

KooKreuz::KooKreuz()
{
  _kit = WidgetKit::instance();
  FontBoundingBox font_box;
  _kit->font()->font_bbox(font_box);

  _canvas       = nil;
  _chr_b        = font_box.left_bearing() + font_box.right_bearing();
  _chr_h        = font_box.ascent() + font_box.descent();
  _laenge_pfeil = 12.0;
  _strich       = 3.0;
  _max_str      = 5;
  _rand_y       = _laenge_pfeil / 2.0 + _chr_h;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::request(Requisition&)
 * Parameter : Requisition& - Definiert Anforderung fuer Bildschirmplatz
 * Zweck : fordert Bildschirmplatz von ausreichender Groesse an.
 *------------------------------------------------------------------------------
 */
void KooKreuz::request(Requisition& req) const
{
  Requirement rx(DEF_BREITE,MAX_BREITE-DEF_BREITE,DEF_BREITE-MIN_BREITE,0);
  Requirement ry(DEF_HOEHE, MAX_HOEHE -DEF_HOEHE, DEF_HOEHE -MIN_HOEHE, 0);
  req.require(Dimension_X, rx);
  req.require(Dimension_Y, ry);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::allocate(Canvas*, const Allocation&, Extension&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 *             Extension&  - gibt an, wieviel vom Bildschirmplatz belegt werden
 *                           wird.
 * Zweck : Initialisiert Variablen in Abhaengigkeit vom zugewiesenen
 *         Bildschirmplatz.
 *------------------------------------------------------------------------------
 */
void KooKreuz::allocate(Canvas* c, const Allocation& a, Extension& e)
{
  _canvas = c;
  _links  = a.left();
  _rechts = a.right();
  _oben   = a.top();
  _unten  = a.bottom();
  _init();
  e.set(c, a);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 * Zweck : zeichnet das Koordinatenkreuz im zugewiesenen Bildschirmbereich.
 *------------------------------------------------------------------------------
 */
void KooKreuz::draw(Canvas* c, const Allocation&) const
{
  Coord abz_x = _x_koo(0),
        abz_y = _y_koo(_y0),
        ord_x = _x_koo(_x0),
        ord_y = _y_koo(0),
        x_max = _rechts,
        y_max = _oben;

  // Zeichnen der Abzisse
  c->line(abz_x, abz_y, x_max, abz_y, _kit->foreground(), nil);
  c->new_path();                                  // Pfeil nach rechts
  c->move_to(x_max                    , abz_y);
  c->line_to(x_max - _laenge_pfeil    , abz_y + _laenge_pfeil / 2.0);
  c->line_to(x_max - _laenge_pfeil / 2, abz_y);
  c->line_to(x_max - _laenge_pfeil    , abz_y - _laenge_pfeil / 2.0);
  c->line_to(x_max                    , abz_y);
  c->close_path();
  c->fill(_kit->foreground());

  if(_anz_einh_abz > 0)
    for(int ex = 0; ex <= _anz_einh_abz; ex++)    // Einheiten abtragen
      c->line(
        _x_koo(ex), abz_y - _strich,
        _x_koo(ex), abz_y,
        _kit->foreground(), nil
      );

  // Zeichnen der Ordinate

  c->line(ord_x, ord_y, ord_x, y_max, _kit->foreground(), nil);
  c->new_path();    // Pfeil nach oben
  c->move_to(ord_x                      , y_max);
  c->line_to(ord_x + _laenge_pfeil / 2.0, y_max - _laenge_pfeil);
  c->line_to(ord_x                      , y_max - _laenge_pfeil / 2.0);
  c->line_to(ord_x - _laenge_pfeil / 2.0, y_max - _laenge_pfeil);
  c->line_to(ord_x                      , y_max);
  c->close_path();
  c->fill(_kit->foreground());

  if(_anz_einh_ord > 0)
    for(int ey = 0; ey <= _anz_einh_ord; ey++)    // Einheiten abtragen
      c->line(
        ord_x - _strich, _y_koo(ey),
        ord_x          , _y_koo(ey),
        _kit->foreground(), nil
      );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::x_einh(Coord)
 * Parameter : Coord - Bildschirm-Koordinate auf x-Achse
 * Rueckgabewert : KooKreuz-Einheit fuer x-Achse
 * Zweck : wandelt Bildschirm-Koordinate in KooKreuz-Einheit um.
 *------------------------------------------------------------------------------
 */
float KooKreuz::x_einh(Coord x)
{
  Coord x_offset = x - _x_koo(0);
  if(x_offset <= 0)
    return 0;
  if(x_offset >= _laenge_abz)
    return _anz_einh_abz;
  if(_laenge_abz != 0 && _anz_einh_abz != 0)
    return x_offset / (_laenge_abz / _anz_einh_abz);
  else
    return 0;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::y_einh(Coord)
 * Parameter : Coord - Bildschirm-Koordinate auf y-Achse
 * Rueckgabewert : KooKreuz-Einheit fuer y-Achse
 * Zweck : wandelt Bildschirm-Koordinate in KooKreuz-Einheit um.
 *------------------------------------------------------------------------------
 */
float KooKreuz::y_einh(Coord y)
{
  Coord y_offset = y - _y_koo(0);
  if(y_offset <= 0)
    return 0;
  if(y_offset >= _laenge_ord)
    return _anz_einh_ord;
  if(_laenge_ord != 0 && _anz_einh_ord != 0)
    return y_offset / (_laenge_ord / _anz_einh_ord);
  else
    return 0;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::skaliere(DimensionName, int, int)
 * Parameter : DimensionName        - welche Achse ?
 *             int anzahl_einheiten - Anzahl der Einheiten
 *             int null_position    - an welcher Einheit ist der Ursprung ?
 * Zweck : skaliert gegebene Achse mit gegebener Anzahl von Einheiten.
 *------------------------------------------------------------------------------
 */
void KooKreuz::skaliere(DimensionName dim, int anzahl_einheiten, int null_pos)
{
  if(dim == Dimension_X) {
    _anz_einh_abz = anzahl_einheiten;
    _x0           = null_pos;
    if(_x0 == 0)
      _rand_x = Max(_laenge_pfeil / 2, _max_str * _chr_b);
    else
      _rand_x = Max(_laenge_pfeil / 2, _max_str * _chr_b / 2);
  } else {
    _anz_einh_ord = anzahl_einheiten;
    _y0           = null_pos;
  }
  _init();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::beschriften(DimensionName, float, char* , int)
 * Parameter : DimensionName - welche Achse ?
 *             float         - Einheit
 *             char*         - Beschriftung
 *             int           - Anzahl der Zeichen in der Beschriftung
 * Zweck : Beschriftet die Achse an der angegebenen Einheit mit gegebener
 *         Zeichenkette.
 *------------------------------------------------------------------------------
 */
void KooKreuz::beschriften(
       DimensionName dim, float einheit, char* str, int max
     ) const
{
  Coord x_pos, y_pos;
  if(dim == Dimension_X) {
    x_pos = _x_koo(einheit) - Min(max, _max_str) * _chr_b / 2;
    y_pos = _y_koo(_y0) - _chr_h - _strich;
  } else {
    x_pos = _x_koo(_x0) - Min(max, _max_str) * _chr_b - _strich;
    y_pos = _y_koo(einheit) - _chr_h / 2;
  }
  if(_canvas != nil)
    for(int i = 0; i < max && i < _max_str; i++)
      _canvas->character(
        _kit->font(),
        (long) (str[i]),
        _chr_b,
        _kit->foreground(),
        x_pos + i * _chr_b,
        y_pos
      );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::linie(float, float, float, float)
 * Parameter : float x1, y1, x2, y2 - Endpunkte der Linie
 * Zweck : zeichnet Linie von Endpunkt zu Endpunkt.
 *------------------------------------------------------------------------------
 */
void KooKreuz::linie(float x1, float y1, float x2, float y2) const
{
  if(_canvas != nil)
    _canvas->line(
      _x_koo(x1), _y_koo(y1), _x_koo(x2), _y_koo(y2), _kit->foreground(), nil
    );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::zeig_linie(float, float, float, float)
 * Parameter : float x1, y1, x2, y2 - Endpunkte der Linie
 * Zweck : markiert gegebene Punkte.
 *------------------------------------------------------------------------------
 */
void KooKreuz::zeig_linie(float x1, float y1, float x2, float y2) const
{
  _markierung(_x_koo(x1), _y_koo(y1));
  _markierung(_x_koo(x2), _y_koo(y2));
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooKreuz::aktualisiere()
 * Zweck : zeichnet das Koordinatenkreuz neu.
 *------------------------------------------------------------------------------
 */
void KooKreuz::aktualisiere()
{
  if(_canvas != nil)
    _canvas->damage(_links, _unten, _rechts, _oben);
}

/*
 *---------------------------- lokale Funktionen -------------------------------
 */

void KooKreuz::_init()
{
  if(_anz_einh_abz == 0)
    _laenge_abz = 0;
  else
    _laenge_abz = _rechts - _links - _rand_x - _laenge_pfeil;
  if(_anz_einh_ord == 0)
    _laenge_ord = 0;
  else
    _laenge_ord = _oben - _unten - _rand_y - _laenge_pfeil;
}

void KooKreuz::_markierung(Coord x, Coord y) const
{
  Coord rad = 5;
  if(_canvas != nil) {
    _canvas->new_path();    // Kugel zeichnen
    _canvas->move_to(x - rad, y);
    _canvas->curve_to(x + rad, y, x - rad, y - rad, x + rad, y - rad);
    _canvas->curve_to(x - rad, y, x + rad, y + rad, x - rad, y + rad);
    _canvas->close_path();
    _canvas->fill(_kit->foreground());
  }
}

/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : 'KooSystem' ---------------------
 *------------------------------------------------------------------------------
 */

KooSystem::KooSystem(
             FuzzyTyp von_x, FuzzyTyp bis_x, FuzzyTyp von_y, FuzzyTyp bis_y
           )
{
  _min_anz_einh = 4;
  neue_skala(Dimension_X, von_x, bis_x);
  neue_skala(Dimension_Y, von_y, bis_y);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::neue_skala(DimensionName, FuzzyTyp, FuzzyTyp)
 * Parameter : DimensionName - welche Achse ?
 *             FuzzyTyp von  - unterer Wert
 *             FuzzyTyp bis  - oberere Wert
 * Zweck : Skaliert Achse mit gegebenem Bereich.
 *------------------------------------------------------------------------------
 */
void KooSystem::neue_skala(DimensionName dim, FuzzyTyp von, FuzzyTyp bis)
{
  int null_pos, anzahl_einheiten;
  float& max_wert = (dim == Dimension_X) ? _max_wert_x : _max_wert_y;

  if(von > bis) {
    FuzzyTyp merk = bis;
    bis = von;
    von = merk;
  }
  _skaliere(dim, von, bis, null_pos);
  _runden(dim, max_wert, anzahl_einheiten);
  KooKreuz::skaliere(dim, anzahl_einheiten, null_pos);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::draw(Canvas*, Allocation&)
 * Parameter : Canvas*     - gesamter Bildschirmplatz
 *             Allocation& - zugewiesener Anteil des Bildschirmplatzes
 * Zweck : zeichnet das Koordinatensystem im zugewiesenen Bildschirmbereich.
 *------------------------------------------------------------------------------
 */
void KooSystem::draw(Canvas* c, const Allocation& a) const
{
  KooKreuz::draw(c, a);
  _beschriften(Dimension_X);
  _beschriften(Dimension_Y);
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::linie(FuzzyTyp, FuzzyTyp, FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp x1, y1, x2, y2 - Endpunkte der Linie.
 * Zweck : zeichnet Linie von Endpunkt zu Endpunkt.
 *------------------------------------------------------------------------------
 */
void KooSystem::linie(FuzzyTyp x1, FuzzyTyp y1, FuzzyTyp x2, FuzzyTyp y2) const
{
  if(drin(x1, y1) && drin(x2, y2))
    KooKreuz::linie(
      _einheit(Dimension_X, x1), _einheit(Dimension_Y, y1),
      _einheit(Dimension_X, x2), _einheit(Dimension_Y, y2)
    );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::zeig_linie(
 *                     FuzzyTyp, FuzzyTyp, FuzzyTyp, FuzzyTyp
 *                   )
 * Parameter : FuzzyTyp x1, y1, x2, y2 - Endpunkte der Linie.
 * Zweck : markiert gegebene Punkte.
 *------------------------------------------------------------------------------
 */
void KooSystem::zeig_linie(FuzzyTyp x1, FuzzyTyp y1,
                           FuzzyTyp x2, FuzzyTyp y2) const
{
  if(drin(x1, y1) && drin(x2, y2))
    KooKreuz::zeig_linie(
      _einheit(Dimension_X, x1), _einheit(Dimension_Y, y1),
      _einheit(Dimension_X, x2), _einheit(Dimension_Y, y2)
    );
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::drin(FuzzyTyp, FuzzyTyp)
 * Parameter : FuzzyTyp x - x-Wert
 *             FuzzyTyp y - y-Wert
 * Rueckgabewert : true  <=> gegebener Punkt befindet sich innerhalb des
 *                           definierten Bereichs des Koordinatensystems.
 *                 false <=> sonst,
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
boolean KooSystem::drin(FuzzyTyp x, FuzzyTyp y) const
{
  FuzzyTyp ca_x = Abs(x / 100);  // kleinere Ungenauigkeiten ..
  FuzzyTyp ca_y = Abs(y / 100);  // werden in Kauf genommen (Rundungsfehler)
  boolean x_ok = (x >= _min_wert_x - ca_x) && (x <= _max_wert_x + ca_x);
  boolean y_ok = (y >= _min_wert_y - ca_y) && (y <= _max_wert_y + ca_y);
  return x_ok && y_ok;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::xkoo2wert(Coord)
 * Parameter : Coord - Bildschirkoordinate
 * Rueckgabewert : x-Achsen-Einheit, welche der Bildschirkoordinate entspricht.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
FuzzyTyp KooSystem::xkoo2wert(Coord x)
{
  return _min_wert_x + KooKreuz::x_einh(x) * _einheit_x;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : KooSystem::ykoo2wert(Coord)
 * Parameter : Coord - Bildschirkoordinate
 * Rueckgabewert : y-Achsen-Einheit, welche der Bildschirkoordinate entspricht.
 * Zweck : siehe Rueckgabewert.
 *------------------------------------------------------------------------------
 */
FuzzyTyp KooSystem::ykoo2wert(Coord y)
{
  return _min_wert_y + KooKreuz::y_einh(y) * _einheit_y;
}

/*
 *------------------------- lokale Funktionen ----------------------------------
 */

void KooSystem::_skaliere(DimensionName dim,
                          FuzzyTyp von, FuzzyTyp bis,
                          int& p0)
{
  float& einh = (dim == Dimension_X) ? _einheit_x  : _einheit_y;
  float& min  = (dim == Dimension_X) ? _min_wert_x : _min_wert_y;
  float& max  = (dim == Dimension_X) ? _max_wert_x : _max_wert_y;
  float rund;

  einh = _such_einheit(Max(Abs(bis), Abs(von)));
  if(_darstellbar(von , einh, rund))
    min = rund;
  else
    min = ((int)(von / einh) - 1) * einh;
  if(_darstellbar(bis, einh, rund))
    max = rund;
  else
    max = ((int)(bis / einh) + 1) * einh;

  if(!_runden(dim, 0, p0)) {
    if(bis < 0)
      p0 = _runden(dim, max - min, p0);
    else
      p0 = _runden(dim, 0, p0);
  }
}

float KooSystem::_such_einheit(FuzzyTyp bereich)
{
  float intervall = (float) bereich;
  if(intervall == 0)
    return 0;
  double e = 1;
  if(intervall > 1)
    while(Math::round(intervall / e)  > 11)
      e *= 10;
  else
    while(Math::round(intervall / e)  < 9)
      e /= 10;
  while(intervall / e < _min_anz_einh)
    e /= 2;
  return e;
}

float KooSystem::_einheit(DimensionName dim, FuzzyTyp wert) const
{
  if(dim == Dimension_X)
    return (wert - _min_wert_x) / _einheit_x;
  else
    return (wert - _min_wert_y) / _einheit_y;
}

boolean KooSystem::_runden(DimensionName dim, float wert, int& erg) const
{
  float einheit = (dim == Dimension_X) ? _einheit_x  : _einheit_y;
  float start   = (dim == Dimension_X) ? _min_wert_x : _min_wert_y;
  float stop    = (dim == Dimension_X) ? _max_wert_x : _max_wert_y;

  erg = 0;
  if(wert > stop || wert < start)
    return false;
  while(start + erg * einheit < wert - einheit / 2)
    erg++;
  return true;
}

boolean KooSystem::_darstellbar(float wert, float einheit, float& rund) const
{
  rund = wert;
  if(wert == 0)
    return true;

  float ungenauigkeit = Abs(wert) / 50;
  float a;

  for(int i = 0; (a = Abs(wert) - i * Abs(einheit)) > ungenauigkeit; i++);

  if(wert > 0)
    rund = i * einheit;
  else
    rund = -1 * i * einheit;
  return a > 0 - ungenauigkeit;
}

void KooSystem::_beschriften(DimensionName dim) const
{
  char str[MAXSTRING];
  int laenge, einheit;
  float min = (dim == Dimension_X) ? _min_wert_x : _min_wert_y;
  float max = (dim == Dimension_X) ? _max_wert_x : _max_wert_y;

  // kleinsten Wert ausgeben
  if(min != 0) {
    hol_str(min, str, laenge);
    _runden(dim, min, einheit);
    KooKreuz::beschriften(dim, einheit, str, laenge);
  }

  // groessten Wert ausgeben
  if(max != 0) {
    hol_str(max, str, laenge);
    _runden(dim, max, einheit);
    KooKreuz::beschriften(dim, einheit, str, laenge);
  }
}
