/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/




#ifndef controller_h
#define controller_h

#include "lingvar.h"
#include "regel.h"
#include "anwendung.h"

/*
 *------------------------------------------------------------------------------
 * Klasse : Protokoll
 * Zweck  : Deklariert die Funktionen, die das Protokoll erstellen. Diese
 *          Klasse dient dabei als Elternklasse fuer verschiedene moegliche
 *          Protokolle.
 *------------------------------------------------------------------------------
 */
class Protokoll
{
  public :
           // Funktionen zim Initialisieren des Protokolls mit linguistischen
           // Variablen und Regeln.
           virtual void init(const LingVarSatz*) = 0;
           virtual void init(const RegelBasis*) = 0;

           // wird aufgerufen, wenn sich Fuzzy-Mengen geaedert haben
           virtual void aktua_lingvarsatz() = 0;

           // gibt die Nummer der aktivsten Regel an
           virtual void zeig_regel(int nr) = 0;

           // liefert die Werte der Ein\Ausgabevariablen und des Fehlers
           virtual void zeig_eingabe_var(int var_nr, FuzzyTyp wert) = 0;
           virtual void zeig_ausgabe_var(int var_nr, FuzzyTyp wert) = 0;
           virtual void zeig_fehler(FuzzyTyp wert) = 0;

           // informiert ueber Fehlschlagen der Regelung
           virtual void fehlschlag() = 0;
};


/*
 *------------------------------------------------------------------------------
 * Klasse : FuzController
 * Zweck  : Allgemeiner Fuzzy-Controller, dessen Eigenschaften im Wesentlichen
 *          durch die virtuellen Elementfunktionen :
 *                - FuzzyTyp _t_norm(FuzzyTyp, FuzzyTyp),
 *                - FuzzyTyp _defuz(FuzzyTyp, FuzzySet*, FuzzyTyp&),
 *                - FuzzyTyp* berechne_steuerwerte(FuzzyTyp*)
 *          bestimmt werden. (siehe auch c-file)
 *------------------------------------------------------------------------------
 */
class EingabeSchicht;
class RegelKnoten;

class FuzController
{
  public : FuzController(const LingVarSatz*, const RegelBasis*);
           virtual ~FuzController();

           virtual FuzzyTyp* berechne_steuerwerte(FuzzyTyp* eingaben);

           virtual void reset() {};
           virtual void lern_faktor(float) {};
           virtual void protokoll(Protokoll*) {};

           LingVarSatz* lingvarsatz();

  protected : virtual FuzzyTyp _t_norm(FuzzyTyp, FuzzyTyp);
              virtual FuzzyTyp _defuz(
                                 FuzzyTyp wert, FuzzySet*, FuzzyTyp& gewicht
                               );
              virtual void _init_regelbasis(const RegelBasis*);
              void _erzeuge_regelbasis();
              RegelKnoten** _regeln;
              int _anz_eingabe_var,
                  _anz_ausgabe_var,
                  _anz_regeln,
                  _merk_anz_regeln;
              LingVarSatz* _lingvarsatz;
              Protokoll* _protokoll;
              EingabeSchicht* _eingabe;
              FuzzyTyp* _ausgabe;
              FuzzyTyp* _treffer_summe;

};

inline FuzzyTyp FuzController::_t_norm(FuzzyTyp a, FuzzyTyp b)
{
  // Min - Funktion
  return (a < b) ? a : b;
}


/*
 *------------------------------------------------------------------------------
 * Klasse : NeuroFuzController
 * Zweck  : Neuronaler Fuzzy-Controller. Die Berechnung der Steuerwerte wird
 *          im Wesentlichen durch die Elternklasse 'FuzController' durchge-
 *          fuehrt. Der neuronale Fuzzy-Controller enthaelt ausserdem einen
 *          FuzController zur Berechnung des Systemfehlers.
 *------------------------------------------------------------------------------
 */
class NeuroFuzController : public FuzController
{
  public : NeuroFuzController(
             const LingVarSatz* steuer_var, const LingVarSatz* fehler_var,
             const RegelBasis* steuer_regeln, const RegelBasis* fehler_regeln
           );
           virtual ~NeuroFuzController();
           virtual FuzzyTyp* berechne_steuerwerte(FuzzyTyp* eingaben);

           virtual void reset();
           virtual void lern_faktor(float);
           virtual void protokoll(Protokoll*);

  protected: virtual void _init_regelbasis(const RegelBasis*);
             void _optimieren(FuzzyTyp fehler);
             void _protokollieren(
                     FuzzyTyp* ein, FuzzyTyp* aus, FuzzyTyp fehler
                  );
             FuzController* _fehler_ctrl;
             Protokoll* _protokoll;
             float _lern_faktor;           // fuer Optimierung der Fuzzy-Sets
             boolean _erster_durchlauf;

};


/*
 *------------------------------------------------------------------------------
 * Struktur : RgLernStruktur
 * Zweck    : Definiert, wie Regeln erlernt werden sollen (genauer : wann sie
 *            beim Lernen geloescht werden sollen).
 *------------------------------------------------------------------------------
 */
struct RgLernStruktur
{
  RgLernStruktur();

  int schritte_phase1,        // Anzahl der Lernschritte
      schritte_phase2;

  boolean nach_prozent;       // nach Prozentanteil loeschen
  int prozent;

  boolean nach_dauer;         // nach bestimmter Dauer ohne Treffer loeschen
  int dauer;

  FuzzyTyp min_treffergrad;   // ab wann ein Treffer gezaehlt wird

  boolean fuz_trainieren;
};


/*
 *------------------------------------------------------------------------------
 * Klasse : RgLrnNeuroFuzCtrl
 * Zweck  : Regeln erlernender neuronaler Fuzzy-Controller.  (Wow !!)
 *          Die Steuer-Regeln werden in 2 Phasen erlernt. Danach werden
 *          die Fuzzy-Sets optimiert und der 'RgLrnNeuroFuzCtrl' erscheint
 *          nach aussen nur noch als 'NeuroFuzController'.
 *------------------------------------------------------------------------------
 */
class RgLrnNeuroFuzCtrl : public NeuroFuzController
{
  public : RgLrnNeuroFuzCtrl(
             const LingVarSatz* steuer_var, const LingVarSatz* fehler_var,
             const RegelBasis* fehler_regeln,
             RegelBasis* steuer_regeln,
             RgLernStruktur*
           );
           virtual ~RgLrnNeuroFuzCtrl();
           virtual FuzzyTyp* berechne_steuerwerte(FuzzyTyp* eingaben);
           virtual void protokoll(Protokoll*);

  // lokale Funktionen
  private: void _regeln_lern_phase1(FuzzyTyp fehler);
           void _regeln_lern_phase2(FuzzyTyp fehler);
           boolean _regel_unsinnig(int regel_nr, FuzzyTyp fehler);
           FuzzyTyp* _rnd_steuerwerte(FuzzyTyp* eingaben);
           void _loesch_regel(int nr);
           void _erzeuge_regelbasis(RegelBasis*);
           boolean _gleiche_gruppe(int rg_nr1, int rg_nr2);

  // lokale Variable
  private: FuzzyTyp* _bereich;
           RgLernStruktur _lern_strukt;
           RegelBasis* _regelbasis;
           float _rnd_grenze;
};

#endif
