/* NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/



#include <string.h>
#include <stdlib.h>
#include <osfcn.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <InterViews/border.h>
#include <InterViews/rule.h>
#include <Dispatch/dispatcher.h>
#include <Dispatch/iocallback.h>

#include "anwendung.h"

extern "C" {
  extern int socketpair(int, int, int, int[2]);
  extern int vfork();
}

declareIOCallback(Anwendung)
implementIOCallback(Anwendung)

// Definition einer Konstanten, die vom Anwendungsprogramm ausgegeben werden
// kann, um den Controller zu 'reseten'; wenn z.B ein Pendel umkippt.
#define RESET 'r'

/*
 *------------------------------------------------------------------------------
 *-------------------- Definition der Klasse : Anwendung -----------------------
 *------------------------------------------------------------------------------
 */

Anwendung::Anwendung(const LingVarSatz* lingvarsatz)
{
  _lingvarsatz   = lingvarsatz;
  _eingabe_folge = nil;
  _ausgabe_folge = nil;
  _eingabe_werte = nil;
  _init();
}

Anwendung::Anwendung(const Anwendung& a)
{
  _eingabe_folge = nil;
  _ausgabe_folge = nil;
  _eingabe_werte = nil;
  *this = a;
}

Anwendung::~Anwendung()
{
  if(_eingabe_var->count() > 0)
    delete _eingabe_var;
  if(_ausgabe_var->count() > 0)
    delete _ausgabe_var;
  if(_quelle != nil)
    delete _quelle;
  if(_ziel != nil)
    delete _ziel;
  delete[] _eingabe_folge;
  delete[] _ausgabe_folge;
  delete[] _eingabe_werte;

  if(_nook_callback != nil)
    _nook_callback->execute();

  unlink();
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::starten(AusgabeFeld*)
 * Parameter       : AusgabeFeld* - Feld fuer evtl. Fehlermeldungen.
 * Zweck           : Startet ein angegebenes Anwendungsprogramm als eigenen
 *                   Prozess und setzt dessen Standart- Ein/Ausgabekanaele um.
 *------------------------------------------------------------------------------
 */

void Anwendung::starten(Ausgabe* ausgabe_feld)
{
  if(aktiv()) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Meldung, Txt(130));
    return;                   // Prg. laeuft schon
  }

  if(_kanal[0] != -1)
    close(_kanal[0]);         // alten Kanal schliessen
  if(socketpair(AF_UNIX, SOCK_STREAM, 0, _kanal) < 0) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Meldung, Txt(131));
    return;                   // Es konnte keine Pipe erzeugt werden
  }

  _merk_pid = vfork();
  if(_merk_pid == -1) {
    if(ausgabe_feld != nil)
      ausgabe_feld->ausgeben(Fehler, Txt(128));
    return;                   // Es konnte kein Prozess gestartet werden
  }

  if(_merk_pid == 0) {

    // Kind-Prozess
    close(_kanal[0]);
    dup2(_kanal[1], 0);
    dup2(_kanal[1], 1);
    execlp(_name.string(), _name.string(), (char*) 0);

    ausgabe_feld->ausgeben(Fehler, Txt(129));
    _exit(1);                 // Fehler beim Starten des Prg.'s
  } else {

    // Eltern-Prozess
    close(_kanal[1]);
    IOHandler* cb = new IOCallback(Anwendung)(this, &Anwendung::_lies_eingaben);
    Dispatcher::instance().link(_kanal[0], Dispatcher::ReadMask, cb);

    if(_quelle != nil)
      delete _quelle;
    if(_ziel != nil)
      delete _ziel;
    _quelle = new ifstream(_kanal[0]);
    _ziel   = new ofstream(_kanal[0]);
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::aktiv()
 * Rueckgabewert : true  <=> Anwendungsprogramm laeuft
 *                 false <=> sonst
 *------------------------------------------------------------------------------
 */

boolean Anwendung::aktiv()
{
  if(_merk_pid != -1 && waitpid(_merk_pid, nil, WNOHANG) == 0)
    return true;
  else
    return false;
}

/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::link(Action* ok_cb, Action* ok_cb)
 * Parameter       : ok_cb   - Callback, das aufgerufen wird, wenn Eingaben vom
 *                             Anwendungsprg. gelesen wurden.
 *                   nook_cb - wird bei Verbindungs-Stoerungen aufgerufen.
 * Zweck           : Initialisiert Anwendung mit entsprechenden Callbacks
 *------------------------------------------------------------------------------
 */

void Anwendung::link(Action* ok_cb, Action* nook_cb)
{
  _ok_callback = ok_cb;
  if(_ok_callback != nil)
    Resource::ref(_ok_callback);

  _nook_callback = nook_cb;
  if(_nook_callback != nil)
    Resource::ref(_nook_callback);
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::steuerwerte(FuzzyTyp*)
 * Parameter       : FuzzyTyp* - zu schreibende Steuerwerte.
 * Zweck           : Uebergibt dem Anwendungsprg. Steuerwerte.
 *------------------------------------------------------------------------------
 */

void Anwendung::steuerwerte(FuzzyTyp* steuer_array)
{
  if(_ziel != nil) {
    int anzahl = _lingvarsatz->anz_lingvars(AusgabeVar);
    for(int i = 0; i < anzahl; i++)
      *_ziel << steuer_array[_ausgabe_folge[i]] << '\n';
    _ziel->flush();
  }
}


Anwendung& Anwendung::operator=(const Anwendung& a)
{
  _lingvarsatz = a._lingvarsatz;
  _init();

  _name = a._name;

  // zuerst evtl. vorhandene Eintraege loeschen
  while(_eingabe_var->count() > 0)
    _eingabe_var->remove(0);

  // dann Eintraege kopieren
  long anzahl = a._eingabe_var->count();
  for(int i = 0; i < anzahl; i++)
    var(EingabeVar, a._eingabe_var->item_ref(i));

  // dasselbe fuer Ausgabe-Variable
  while(_ausgabe_var->count() > 0)
    _ausgabe_var->remove(0);

  anzahl = a._ausgabe_var->count();
  for(i = 0; i < anzahl; i++)
    var(AusgabeVar, a._ausgabe_var->item_ref(i));

  return *this;
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::loesch_var(LingVarTyp)
 * Parameter       : LingVarTyp - Typ der ling. Var.
 * Zweck           : loescht alle Eintraege des entsprechende Typs aus der Liste
 *                   der ling. Variablen.
 *------------------------------------------------------------------------------
 */

void Anwendung::loesch_var(LingVarTyp typ)
{
  StringList* l = (typ == EingabeVar) ? _eingabe_var : _ausgabe_var;
  while(l->count() > 0)
    l->remove(0);
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::var(LingVarTyp, String)
 * Parameter       : LingVarTyp - Typ der ling. Var.
 *                   String     - Name der ling. Var.
 * Zweck           : Fuegt den Namen der ling. Var. ans Ende der entspr. Liste
 *                   an. Durch die Listen wird die Reihenfolge fuer die
 *                   Ein/Ausgabe der ling. Var. festgelegt.
 *------------------------------------------------------------------------------
 */

void Anwendung::var(LingVarTyp typ, String name)
{
  if(_lingvarsatz->hol_lingvar(name) == nil)
    return;     // sollte nicht vorkommen

  int index;
  if(typ == EingabeVar) {
    index = (int)_eingabe_var->count();
    if(index < _lingvarsatz->anz_lingvars(EingabeVar))
      _eingabe_folge[index] = _lingvarsatz->var_nr(typ,name);
    _eingabe_var->append(name);
  } else {
    index = (int)_ausgabe_var->count();
    if(index < _lingvarsatz->anz_lingvars(AusgabeVar))
      _ausgabe_folge[index] = _lingvarsatz->var_nr(typ,name);
    _ausgabe_var->append(name);
  }
}


/*
 *------------------------------------------------------------------------------
 * Elementfunktion : Anwendung::var(LingVarTyp, int)
 * Parameter       : LingVarTyp - Typ der ling. Var.
 *                   int        - Nummer der ling. Var.
 * Rueckgabewert   : String     - Name der gesuchten ling. Var.
 * Zweck           : Bestimmt anhand der Nummer und des Typ Namen einer ling.
 *                   Var. Mit Hilfe dieser Funktion laesst sich zum Bsp. der
 *                   Name der 2.ten Eingabe-Variablen fuer eine Anwendung
 *                   feststellen.
 *------------------------------------------------------------------------------
 */

String Anwendung::var(LingVarTyp typ, int nr)
{
  StringList* l = (typ == EingabeVar) ? _eingabe_var : _ausgabe_var;
  if(nr < l->count())
    return l->item_ref(nr);
  else
    return LeerStr;
}

void Anwendung::_init()
{
  _ok_callback   = nil;
  _nook_callback = nil;

  if(_eingabe_folge != nil)
    delete[] _eingabe_folge;
  if(_ausgabe_folge != nil)
    delete[] _ausgabe_folge;
  if(_eingabe_werte != nil)
    delete[] _eingabe_werte;

  int anzahl_eingabe = _lingvarsatz->anz_lingvars(EingabeVar);
  int anzahl_ausgabe = _lingvarsatz->anz_lingvars(AusgabeVar);
  _eingabe_folge = new int[anzahl_eingabe];
  _ausgabe_folge = new int[anzahl_ausgabe];
  _eingabe_werte = new FuzzyTyp[anzahl_eingabe];

  for(int i = 0; i < anzahl_eingabe; _eingabe_folge[i++] = 0);
  for(i = 0; i < anzahl_ausgabe; _ausgabe_folge[i++] = 0);

  _name        = LeerStr;
  _eingabe_var = new StringList;
  _ausgabe_var = new StringList;
  _kanal[0]    = -1;       // noch kein Kanal geoeffnet
  _merk_pid    = -1;       // Anwendungsprg. laeuft noch nicht
  _quelle      = nil;
  _ziel        = nil;
}

int Anwendung::_lies_eingaben(int)
{
  if(_quelle == nil)
    return 0;

  char c;

  // returns's aus Eingabestrom entfernen
  while(_quelle->peek() == '\n')
    _quelle->read(&c, 1);

  // Testen, ob Anwendungsprg. terminiert wurde
  if(_quelle->peek() == EOF) {
    Dispatcher::instance().unlink(_kanal[0]);
    close(_kanal[0]);
    _kanal[0] = -1;
    delete _quelle;
    delete _ziel;
    _quelle = nil;
    _ziel   = nil;
    return 0;
  }

  // Testen, ob Steuerung fehlschlug (Bsp. Pendelsimulation : Pendel kippte um)
  // in diesem Fall wird 'nil' zurueckgeliefert
  if(_quelle->peek() == RESET) {
    *_quelle >> c;    // Zeichen aus Eingabestrom nehmen
    _mess_werte = nil;

    // Controller informieren
    if(_ok_callback != nil)
      _ok_callback->execute();

    return 1;         // 'return 1' heisst : es muessen!! noch Messwerte folgen
  }

  // Werte einlesen
  int anzahl = _lingvarsatz->anz_lingvars(EingabeVar);
  for(int i = 0; i < anzahl; i++)
    *_quelle >> _eingabe_werte[_eingabe_folge[i]];
  _mess_werte = _eingabe_werte;

  // bei Fehler : Schrott aus Eingabestrom nehmen
  if(!*_quelle) {

    _quelle->clear();
    *_quelle >> c;
    if(_nook_callback != nil)
      _nook_callback->execute();
  } else

    // Controller informieren
    if(_ok_callback != nil)
      _ok_callback->execute();

  return 0;
}
