/* PENDULUM: a graphical simulation of an inverted pendulum; part of 
   NEFCON-I: an interactive system for realization of a neural fuzzy controller

   Copyright (C) 1994 

   Institut fuer Betriebssysteme und Rechnerverbund, Technische Universitaet
   Braunschweig, Bueltenweg 74/75, 38106 Braunschweig, Germany, 
   hereby disclaims all copyright interests in the program NEFCON-I 
   written by Hermann-Josef Diekgerdes and Roland Stellmach.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *   Grafik-Modul fuer die Pendelsimulation
 */

#include <iostream.h>
#include <stdio.h>

#include <InterViews/image.h>
#include <InterViews/tiff.h>

#include <math.h>
#include <IV-look/kit.h>
#include <IV-look/telltale.h>
#include <InterViews/color.h>
#include <InterViews/canvas.h>
#include <InterViews/display.h>
#include <InterViews/action.h>
#include <InterViews/layout.h>
#include <InterViews/rule.h>
#include <InterViews/border.h>
#include <InterViews/session.h>
#include <Dispatch/dispatcher.h>
#include <stream.h>

#include "global.h"
#include "pendel_texte.h"
#include "pendel.h"

// Konstante zur Berechnung der Zufallswerte
#define MaxRndWinkel  30
#define MinRndWinkel -30
#define MaxRndGeschw  100
#define MinRndGeschw -100

#define RESET 'r'

 // maximal zulaessige Werte; sehr grosse Werte konnen in der Formel
//  nicht verabeitet werden
#define MAX_KRAFT     1000
#define MAX_WI_GESCHW 1000
#define MAX_WA_GESCHW 1000
#define MAX_POSITION  1000

 // Folgende Konstante gibt an, nach wieviel Durchlaeufen das Pendel neu
//  gezeichnet werden soll
#define ZEICHNEN_RATE 1

/*
 *----------- Deklaration und Definition der Klasse : PendelGrafik -------------
 */

#define DEF_BREITE 400.0
#define DEF_HOEHE  100.0
#define MAX_BREITE 1000.0
#define MAX_HOEHE  400.0
#define MIN_BREITE 200.0
#define MIN_HOEHE  70.0

class PendelGrafik : public Glyph
{
  public : PendelGrafik(float winkel, float lage);
           virtual void request(Requisition&) const;
           virtual void allocate(Canvas*, const Allocation&, Extension&);
           virtual void draw(Canvas*, const Allocation&) const;
           boolean setz_winkel(float);
           void fahren(float);
  private: void _drehen(Coord&, Coord&) const;
           Allocation _allocation;
           Coord _y_pos    ,       // Geometrie des Pendels
                 _radius   ,
                 _stab_h   ,
                 _stab_b   ,
                 _wagen_h  ,
                 _wagen_b  ,
                 _strecke_h,
                 _einheit  ;
           float _winkel;
           Coord _lage_wagen,      // Position von Pendel und Strecke
                 _lage_strecke;
           float _wagen_position;
};

PendelGrafik::PendelGrafik(float winkel, float lage)
{
  _winkel         = winkel;
  _lage_wagen     = lage;
  _lage_strecke   = 0;
  _wagen_position = 0;
}

void PendelGrafik::request(Requisition& req) const
{
  Requirement rx(DEF_BREITE,MAX_BREITE-DEF_BREITE,DEF_BREITE-MIN_BREITE,0.0);
  Requirement ry(DEF_HOEHE, MAX_HOEHE -DEF_HOEHE, DEF_HOEHE -MIN_HOEHE, 0.0);
  req.require(Dimension_X,rx);
  req.require(Dimension_Y,ry);
}

void PendelGrafik::allocate(Canvas* canv, const Allocation& all, Extension& e)
{
  Coord hoehe  = all.allotment(Dimension_Y).span();
  Coord breite = all.allotment(Dimension_X).span();

  _allocation   = all;
  _lage_wagen   = 0;
  _lage_strecke = 0;

  _radius    = 0.15 * Min(hoehe, breite / 2);
  _stab_h    = 4.0 * _radius;
  _stab_b    = 0.25 * _radius;
  _wagen_h   = _radius;
  _wagen_b   = 4.0 * _wagen_h;
  _strecke_h = _stab_b;
  _einheit   = 2.0 * _strecke_h;
  _y_pos     = (hoehe - _stab_h - _radius) / 2.0 + _wagen_h / 2.0;

  e.set(canv, all);
}

void PendelGrafik::draw(Canvas* c, const Allocation& a) const
{
  WidgetKit* wkit = WidgetKit::instance();

  const Color* vg = wkit->foreground();
  const Color* hg = wkit->background();

  Coord wagen_x;      // Position des Wagens
  Coord strecke_x,
        strecke_y;    // Position der Strecke
  Coord kugel_x,      // Koordinaten der Kugel
        kugel_y;
  Coord stab_xoff,    // zur Berechnung der Eckpunkte des Stabes
        stab_yoff;

  // Initialisierung der Koordinaten
  wagen_x   = _lage_wagen  * _einheit + a.left() +
              a.allotment(Dimension_X).span() / 2;
  strecke_x = _lage_strecke * _einheit + a.left();
  strecke_y = _y_pos + a.bottom();
  kugel_x   = 0;
  kugel_y   = _stab_h;
  _drehen(kugel_x, kugel_y);
  kugel_x   = 0;
  kugel_y   = _stab_h;
  _drehen(kugel_x, kugel_y);
  kugel_x  += wagen_x;
  kugel_y  += strecke_y;
  stab_xoff = _stab_b / 2;
  stab_yoff = 0;
  _drehen(stab_xoff, stab_yoff);

  // Strecke zeichnen
  if(_lage_strecke > 1)
    c->fill_rect(a.left()                    , strecke_y - _strecke_h / 2,
                 Max(strecke_x - _einheit, 0), strecke_y + _strecke_h / 2, vg);
  for(Coord x = strecke_x; x < a.right(); x += 2 * _einheit)
    c->fill_rect(x                           , strecke_y - _strecke_h / 2,
                 Min(x + _einheit, a.right()), strecke_y + _strecke_h / 2, vg);

  // Wagen zeichnen
  c->new_path();
  c->move_to(wagen_x - _wagen_b / 2, strecke_y - _wagen_h / 2);
  c->line_to(wagen_x - _wagen_b / 2 + _wagen_h / 2, strecke_y);
  c->line_to(wagen_x - _wagen_b / 2, strecke_y + _wagen_h / 2);
  c->line_to(wagen_x + _wagen_b / 2, strecke_y + _wagen_h / 2);
  c->line_to(wagen_x + _wagen_b / 2 - _wagen_h / 2, strecke_y);
  c->line_to(wagen_x + _wagen_b / 2, strecke_y - _wagen_h / 2);
  c->line_to(wagen_x - _wagen_b / 2, strecke_y - _wagen_h / 2);
  c->close_path();
  c->fill(vg);

  // Stab zeichnen
  c->new_path();
  c->move_to(wagen_x - stab_xoff, strecke_y - stab_yoff);
  c->line_to(kugel_x - stab_xoff, kugel_y   - stab_yoff);
  c->line_to(kugel_x + stab_xoff, kugel_y   + stab_yoff);
  c->line_to(wagen_x + stab_xoff, strecke_y + stab_yoff);
  c->line_to(wagen_x - stab_xoff, strecke_y - stab_yoff);
  c->close_path();
  c->fill(vg);

  // Kugel zeichnen
  c->new_path();
  c->move_to(kugel_x - _radius, kugel_y);
  c->curve_to(kugel_x + _radius, kugel_y, kugel_x - _radius, kugel_y - _radius,
              kugel_x + _radius, kugel_y - _radius);
  c->curve_to(kugel_x - _radius, kugel_y, kugel_x + _radius, kugel_y + _radius,
              kugel_x - _radius, kugel_y + _radius);
  c->close_path();
  c->fill(vg);
}

void PendelGrafik::fahren(float wohin)
{
  Coord breite = _allocation.allotment(Dimension_X).span();

  // Das Pendel nimmt als Einheit fuer die Position die Hoehe des Stabes
  wohin *= _stab_h;

  _lage_wagen -= wohin;
  if(Abs(_lage_wagen * _einheit) > breite / 2 - _stab_h - _radius) {
    _lage_wagen += wohin;
    _lage_strecke += wohin;

    while(_lage_strecke <= -2)
      _lage_strecke += 2;
    while(_lage_strecke >= -2)
      _lage_strecke -= 2;
  }
}

boolean PendelGrafik::setz_winkel(float w)
{
  // zuerst Testen, ob Pendel umgekippt ist
  float max = pi() / 2;
  if(w >= max) {
    _winkel = max;
    return false;
  } else
    if(w <= -max) {
      _winkel = -max;
      return false;
    }

  _winkel = w;
  return true;
}

void PendelGrafik::_drehen(Coord& x, Coord& y) const
{
  Coord merk_x = x;
  Coord merk_y = y;
  float winkel = _winkel;

  x = cos(winkel) * merk_x - sin(winkel) * merk_y;  // entspricht 2-dim.
  y = sin(winkel) * merk_x + cos(winkel) * merk_y;  // Rotations-Matrix
}

/*
 *----------------- Deklaration und Definition der Klasse InfoFeld -------------
 * gibt nach Anklicken des AusgabeFeldes eine Meldung in dieses aus
 */


class InfoFeld : public InputHandler
{
  public : InfoFeld(NfcAusgabe*, Style*);
           virtual void press(const Event&);
  private: NfcAusgabe* _ausgabefeld;
};

InfoFeld::InfoFeld(NfcAusgabe* ausgabe, Style* style)
 : InputHandler(ausgabe, style) { _ausgabefeld = ausgabe; }

void InfoFeld::press(const Event& e)
{
  if(e.pointer_button() == Event::right)
    _ausgabefeld->ausgeben(Info, "(c) 1993 Hermann-Josef Diekgerdes");
}

/*
 *              /\                 |_        |_        |_|_|_    |_|_|_
 *              |  \               |_        |_            |_      |_  |_
 *              \ |  \             |_        |_            |_      |_    |_
 *               \/   \            |_        |_            |_      |_     |_
 *               |      \_         |_|_|_|_|_|_   |_|_|_   |_      |_     |_
 *              /         \        |_        |_            |_      |_     |_
 *              |      /  o\       |_        |_            |_      |_    |_
 *             /      / \_ |       |_        |_            |_      |_  |_
 *             |/     \  |\\       |_        |_            |_    |_|_|_
 *             /|/    /  \ \|                        |_|_|_|_
 *              /|   /\   \
 *               /| /  \_  \
 *                |/     \_\
 *                /        \
 *
 *                                            MM
 *                              \\           /  OO___o
 *                               \\~~~~~~~~~~  _||||||
 *                                | ________  /
 *                               //\\      //\\
 *                              //  \\    //  ||
 *----------------------- Definition der Klasse : PendelFenster ----------------
 */

declareActionCallback(PendelFenster);
implementActionCallback(PendelFenster);

declareIOCallback(PendelFenster);
implementIOCallback(PendelFenster);

PendelFenster::PendelFenster(Style* style)
  : InputHandler(nil, style)
{
  WidgetKit* wkit = WidgetKit::instance();
  LayoutKit* lkit = LayoutKit::instance();
  NfcKit* nkit    = NfcKit::instance();
  Action* cb;

  _pend_sim      = new PendelSim;
  _pendel        = new PendelGrafik(0, 0);
  _grafik        = new Patch(_pendel);
  _aktiv         = 0;
  _zaehler       = 0;     // Hilfszaehler, falls ZEICHNEN_RATE > 0
  _anzahl_zyklen = 0;     // Zaehler fuer Anzahl der Zyklen
  _auto_start    = false;
  _rnd_winkel    = false;
  _sync_faktor   = 1;
  _toggle1       = true;
  _toggle2       = true;

  // Eingabefelder fuer Startwerte

  // Winkel-Eingabe
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_winkel);
  if(wkit->style()->value_is_on("mitWinkel"))
    _winkel_ed = new EingabeFeld(10, _pend_sim->winkel(), cb);
  else
    _winkel_ed = new EingabeFeld(10, Txt(29), cb);
  append_input_handler(_winkel_ed->input_handler());

  // Winkelgeschw.-Eingabe
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_wi_geschw);
  if(wkit->style()->value_is_on("mitWiGeschw"))
    _wi_geschw_ed = new EingabeFeld(10, _pend_sim->wi_geschw(), cb);
  else
    _wi_geschw_ed = new EingabeFeld(10, Txt(29), cb);
  append_input_handler(_wi_geschw_ed->input_handler());

  // Wagenposition-Eingabe
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_position);
  if(wkit->style()->value_is_on("mitPosition"))
    _position_ed = new EingabeFeld(10, _pend_sim->position(), cb);
   else
    _position_ed = new EingabeFeld(10, Txt(29), cb);
  append_input_handler(_position_ed->input_handler());

  // Wagengeschw.-Eingabe
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_wa_geschw);
  if(wkit->style()->value_is_on("mitWaGeschw"))
    _wa_geschw_ed = new EingabeFeld(10, _pend_sim->wa_geschw(), cb);
  else
    _wa_geschw_ed = new EingabeFeld(10, Txt(29), cb);
  append_input_handler(_wa_geschw_ed->input_handler());

  // Eingabe der Strecke
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_strecke);
  float laenge = 0;
  wkit->style()->find_attribute("LaengeStrecke", laenge);
  if(laenge != 0)
    _strecke_ed = new EingabeFeld(10, laenge, cb);
  else
    _strecke_ed = new EingabeFeld(10, Txt(29), cb);
  append_input_handler(_strecke_ed->input_handler());

  wkit->begin_style("NfcInfoStyle","NfcInfoFont");

  // Buttons fuer auto-start und rnd-winkel definieren
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_auto_start);
  Button* auto_start_bt = wkit->check_box(Txt(4), cb);
  auto_start_bt->state()->set(TelltaleState::is_chosen, _auto_start);
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_rnd_winkel);
  Button* rnd_wi_bt = wkit->check_box(Txt(5), cb);
  rnd_wi_bt->state()->set(TelltaleState::is_chosen, _rnd_winkel);

  // Zaehler-Feld und -Button erzeugen
  _zaehler_feld = new NfcLabel("0", wkit);
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_zaehler_reset);
  Button* zaehler_bt = wkit->push_button(Txt(6), cb);
  zaehler_bt->state()->set(TelltaleState::is_chosen, false);

  wkit->end_style();

  // Sync. Faktor-Eingabe
  cb = new ActionCallback(PendelFenster)(this,&PendelFenster::_setz_sync);
  _sync_ed = new EingabeFeld(5, _sync_faktor, cb);
  append_input_handler(_sync_ed->input_handler());

  _ausgabefeld = new NfcAusgabe(new AusgabeFeld(25, 4));

  Coord rand = 10;
  Glyph* start_werte = nkit->text_drauf(
                         Txt(7),
                         lkit->margin(
                           lkit->vbox(
                             nkit->text_davor(
                               Txt(8), _winkel_ed, false
                             ), lkit->vglue(rand),
                             nkit->text_davor(
                               Txt(9), _wi_geschw_ed, false
                             ), lkit->vglue(rand),
                             nkit->text_davor(
                               Txt(10), _position_ed, false
                             ), lkit->vglue(rand),
                             nkit->text_davor(
                               Txt(11), _wa_geschw_ed, false
                             )
                           ), 10, fil, 0, 10, fil, 0, 10, 0, 0, 10, 0, 0
                         ), true
                       );

  Glyph* feste_werte = nkit->text_drauf(
                         Txt(12),
                         lkit->margin(
                           lkit->vbox(
                             nkit->text_davor(
                               Txt(13), _strecke_ed, false
                             ), lkit->vglue(rand),
                             nkit->text_davor(
                               Txt(14), _sync_ed, false
                             ), lkit->vglue(rand),
                             lkit->hbox(
                               auto_start_bt,
                               lkit->hglue(rand),
                               rnd_wi_bt
                             )
                           ), 10, fil, 0, 10, fil, 0, 10, 0, 0, 10, 0, 0
                         ), true
                       );

  Glyph* zaehler = lkit->vbox(
                     lkit->vglue(0,0,0),
                     lkit->hbox(
                       lkit->hglue(),
                       lkit->vcenter(
                         nkit->text_davor(
                           Txt(15), lkit->hnatural(_zaehler_feld, 50), true
                         )
                       ), lkit->hglue(rand),
                       lkit->vcenter(zaehler_bt),
                       lkit->hglue()
                     ),
                     lkit->vglue(0,0,0)
                   );

  Glyph* ausgabe = nkit->text_drauf(
                     Txt(16),
                     new InfoFeld(_ausgabefeld, wkit->style()),
                     true
                   );

  Action* cb1 = new ActionCallback(PendelFenster)(this,&PendelFenster::_start);
  Action* cb2 = new ActionCallback(PendelFenster)(this,&PendelFenster::_stop);
  Action* cb3 = new ActionCallback(PendelFenster)(this,&PendelFenster::_ende);

  Glyph* buttons = lkit->hbox(
                     wkit->push_button(Txt(17), cb1), lkit->hglue(rand),
                     wkit->push_button(Txt(18), cb2), lkit->hglue(rand),
                     wkit->push_button(Txt(19), cb3)
                   );

  // Dispatcher initialisieren
  IOHandler* ptr =
     new IOCallback(PendelFenster)(this, &PendelFenster::_lies_steuer_wert);
  Dispatcher::instance().link(0, Dispatcher::ReadMask, ptr);

  body(
    lkit->margin(
      lkit->vbox(
        lkit->v_variable_span(
          lkit->hbox(
            lkit->vbox(start_werte, lkit->vglue(rand, 0, 0), buttons),
            lkit->hglue(rand, 0, 0),
            lkit->vbox(
              feste_werte,
              lkit->vglue(rand, 0, 0),
              zaehler,
              lkit->vglue(rand, 0, 0),
              ausgabe
            )
          ), 0, 0
        ),
        lkit->vglue(rand, 0, 0),
        new Border(_grafik, wkit->foreground())
      ), rand, rand
    )
  );
}

PendelFenster::~PendelFenster() { delete _pend_sim; }


/*
 * Initialisiert Winkel mit dem Wert aus dem Eingabefeld
 */
void PendelFenster::_setz_winkel()
{
  _winkel_ausgeben = _winkel_ed->text() != String(Txt(29));
  FuzzyTyp wert;
  if(_winkel_ed->hol_wert(wert)) {
    _pend_sim->winkel((float) grad2bog(wert));
    if(!_pendel->setz_winkel(_pend_sim->winkel()))
      _ausgabefeld->ausgeben(Fehler, Txt(20));
    _grafik->redraw();
  } else
    _pend_sim->winkel(0);
}

/*
 * Initialisiert Winkel mit zufaelligem Wert
 */
void PendelFenster::_zufaelliger_winkel()
{
  FuzzyTyp wert;
  if(_toggle1)
    wert = MaxRndWinkel * rnd();
  else
    wert = MinRndWinkel * rnd();
  _toggle1 = !_toggle1;

  _winkel_ed->setz_wert(wert);
  _pend_sim->winkel((float) grad2bog(wert));

  if(_strecke_begrenzt)
    if(_position_ed->text() != String(Txt(29))) {
      wert = _laenge_strecke * rnd() - _laenge_strecke / 2;
     _position_ed->setz_wert(wert);
    }
}

/*
 * Initialisiert Geschwindigkeit mit dem Wert aus dem Eingabefeld
 */
void PendelFenster::_setz_wi_geschw()
{
  _wi_geschw_ausgeben = _wi_geschw_ed->text() != String(Txt(29));
  FuzzyTyp wert;
  if(!_wi_geschw_ed->hol_wert(wert)) 
    wert = 0;
  if(wert > MAX_WI_GESCHW || wert < -MAX_WI_GESCHW) {
    wert = (wert > 0) ? MAX_WI_GESCHW : -MAX_WI_GESCHW;
      _ausgabefeld->ausgeben(Warnung, Txt(21));
  }
  _pend_sim->wi_geschw((float) grad2bog(wert));
}

/*
 * Initialisiert Geschwindigkeit mit zufaelligem Wert
 */
void PendelFenster::_zufaellige_geschw()
{
  FuzzyTyp wert;
  if(_toggle2)
     wert = MaxRndGeschw * rnd();
  else
    wert = MinRndGeschw * rnd();
  _toggle2 = !_toggle2;

  _wi_geschw_ed->setz_wert(wert);
  _pend_sim->wi_geschw((float) grad2bog(wert));
}

/*
 * Initialisiert Wagen-Position mit dem Wert aus dem Eingabefeld
 */
void PendelFenster::_setz_position()
{
  _position_ausgeben = _position_ed->text() != String(Txt(29));
  FuzzyTyp wert;
  if(!_position_ed->hol_wert(wert))
    wert = 0;
  if(wert > MAX_POSITION || wert < -MAX_POSITION) {
    wert = (wert > 0) ? MAX_POSITION : -MAX_POSITION;
      _ausgabefeld->ausgeben(Warnung, Txt(22));
  }
  _pend_sim->position((float) wert);
}

/*
 * Initialisiert Wagen-Geschwindigkeit mit dem Wert aus dem Eingabefeld
 */
void PendelFenster::_setz_wa_geschw()
{
  _wa_geschw_ausgeben = _wa_geschw_ed->text() != String(Txt(29));
  FuzzyTyp wert;
  if(!_wa_geschw_ed->hol_wert(wert))
    wert = 0;
  if(wert > MAX_WA_GESCHW || wert < -MAX_WA_GESCHW) {
    wert = (wert > 0) ? MAX_WA_GESCHW : -MAX_WA_GESCHW;
      _ausgabefeld->ausgeben(Warnung, Txt(23));
  }
  _pend_sim->wa_geschw((float) wert);
}

/*
 * Liest die Laenge der Strecke aus dem Eingabefeld
 */
void PendelFenster::_setz_strecke()
{
  _strecke_begrenzt = _strecke_ed->text() != String(Txt(29));
  if(_strecke_begrenzt)
    _strecke_ed->hol_wert(_laenge_strecke);
  else
    _laenge_strecke = -1;
}

/*
 * aktiviert/deaktiviert auto-start
 */
void PendelFenster::_setz_auto_start() { _auto_start = !_auto_start; }

/*
 * aktiviert/deaktiviert Zufallswinkel
 */
void PendelFenster::_setz_rnd_winkel() { _rnd_winkel = !_rnd_winkel; }

/*
 * setzt Synchronisationsfaktor
 */
void PendelFenster::_setz_sync()
{
  FuzzyTyp wert;
  if(_sync_ed->hol_wert(wert))
    _sync_faktor = Max(1, (int)wert);
}

int PendelFenster::_lies_steuer_wert(int)
{
  // returns's aus Eingabestrom entfernen
  char c;
  while(cin.peek() == '\n')
    cin.read(&c, 1);

  if(cin.peek() == EOF)
    Session::instance()->quit();

  float kraft;
  if(cin >> kraft) {

    if(_aktiv) {

      // zu grosse Werte sind nicht erlaubt -> erzeugen irgendwie eine
     //  Endlos-Schleife in der Formel
      if(kraft > MAX_KRAFT || kraft < -MAX_KRAFT) {
        kraft = (kraft > 0) ? MAX_KRAFT : -MAX_KRAFT;
        _ausgabefeld->ausgeben(Warnung, Txt(24));
      }

      boolean umgkippt = false;
      for(int i = 0; i < _sync_faktor && !umgkippt; i++) {
        if(i == 0)
          _pend_sim->simulation(kraft);
        else
          _pend_sim->simulation(0);

        if(!_pendel->setz_winkel(_pend_sim->winkel())) {
          _ausgabefeld->ausgeben(Meldung, Txt(25));
          umgkippt = true;
        }
        if(_strecke_begrenzt &&
           Abs(_pend_sim->position()) >= _laenge_strecke / 2
        ) {
          _ausgabefeld->ausgeben(Meldung, Txt(26));
          umgkippt = true;
        }

        if(umgkippt) {

          /*
           * Pendel ist umgekippt
           */
        
          // danach Controller 'reseten'
          cout << RESET << '\n';
          cout.flush();

          // evtl. Pendel mit Zufalls-Startwerten neu starten
          if(_auto_start) {
            if(_rnd_winkel)
              _zufaelliger_winkel();
            else
              _setz_winkel();
            _setz_wi_geschw();
            _setz_position();
            _setz_wa_geschw();
          } else {
            _grafik->canvas()->window()->display()->ring_bell(50);
            _aktiv = false;
          }

          // auf jeden Fall das Pendel aber neu zeichnen
          _zaehler = 0;
        }
      }

      if(_zaehler++ % ZEICHNEN_RATE == 0) {
        _pendel->fahren((_pend_sim->position() - _alte_pos) / 2);
        _alte_pos = _pend_sim->position();
        _grafik->redraw();
      }
      _schreib_messwerte();
    }
  } else {
    cin.clear();
    char wegdamit[2];
    cin >> wegdamit[0];
    wegdamit[1] = '\0';
    _ausgabefeld->ausgeben(Fehler, Txt(27), wegdamit);
  }
  return 0;
}

void PendelFenster::_schreib_messwerte()
{
  if(_winkel_ausgeben)
    cout << Min(90, bog2grad(_pend_sim->winkel())) << '\n';
  if(_wi_geschw_ausgeben)
    cout << bog2grad(_pend_sim->wi_geschw()) << '\n';
  if(_position_ausgeben)
    cout << _pend_sim->position() << '\n';
  if(_wa_geschw_ausgeben)
    cout << _pend_sim->wa_geschw() << '\n';
  cout.flush();

  sprintf(_buffer, "%i", ++_anzahl_zyklen);
  _zaehler_feld->text(_buffer);
  _zaehler_feld->aktualisiere();
}

void PendelFenster::_start()
{
  if(!_aktiv) {

    if(_rnd_winkel)
      _zufaelliger_winkel();
    _setz_winkel();
    _setz_wi_geschw();
    _setz_position();
    _setz_wa_geschw();
    _setz_strecke();

    // Merker setzen und Messwerte ausgeben
    _aktiv    = true;
    _alte_pos = _pend_sim->position();
    _schreib_messwerte();
    _ausgabefeld->ausgeben(Info, Txt(32));
    _ausgabefeld->ausgeben(Info, "");
  } else
    _ausgabefeld->ausgeben(Fehler, Txt(28));
}

void PendelFenster::_stop()
{
  if(_aktiv) {
    _aktiv = false;
    _ausgabefeld->ausgeben(Info, Txt(31));
    _ausgabefeld->ausgeben(Info, "");
  }
}

void PendelFenster::_ende()
{
  if(_aktiv)
    _ausgabefeld->ausgeben(Fehler, Txt(28));
  else
    Session::instance()->quit();
}

void PendelFenster::_zaehler_reset()
{
  _anzahl_zyklen = 0;
  _zaehler_feld->text("0");
  _zaehler_feld->aktualisiere();
}
