//---------------------------------------------------------------------------
//
// CONFINT.H
//
//    Confidence interval type declarations.
//
//    Based on A. Kaufmann, M. Gupta, "Introduction to Fuzzy Arithmetic",
//    Van Nostrand Reinhold, New York, 1991.
//
// Contents:
//
//    class ConfInt
//
//    inline double ConfInt::GetLower()
//    inline double ConfInt::GetUpper()
//    inline double Confint::GetSize()
//
//    inline int ConfInt::operator[]( double value)
//
//    inline ConfInt ConfInt::operator+()
//    inline ConfInt ConfInt::operator-()
//
//    inline ConfInt ConfInt::operator+( double opnd2)
//    inline ConfInt ConfInt::operator+( ConfInt opnd2)
//    inline ConfInt ConfInt::operator-( double opnd2)
//    inline ConfInt ConfInt::operator-( ConfInt opnd2)
//    inline ConfInt ConfInt::operator*( double opnd2)
//    inline ConfInt ConfInt::operator/( double opnd2)
//    inline ConfInt ConfInt::operator/( ConfInt opnd2)
//
//    inline ConfInt ConfInt::operator+=( double opnd2)
//    inline ConfInt ConfInt::operator+=( ConfInt opnd2)
//    inline ConfInt ConfInt::operator-=( double opnd2)
//    inline ConfInt ConfInt::operator-=( ConfInt opnd2)
//    inline ConfInt ConfInt::operator*=( double opnd2)
//    inline ConfInt ConfInt::operator/=( double opnd2)
//    inline ConfInt ConfInt::operator/=( ConfInt opnd2)
//
//    inline ConfInt operator+( double opnd1, ConfInt opnd2)
//    inline ConfInt operator-( double opnd1, ConfInt opnd2)
//    inline ConfInt operator*( double opnd1, ConfInt opnd2)
//
//    inline ostream& operator<<( ostream& smStream, ConfInt ciValue)
//
//    inline ConfInt min( ConfInt opnd1, ConfInt opnd2)
//    inline ConfInt max( ConfInt opnd1, ConfInt opnd2)
//
//    inline double ciDelta( ConfInt opnd1, ConfInt opnd2)
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------



#ifndef ___CONFINT_H
#define ___CONFINT_H

#include <iomanip.h>

#include <iostream.h>

#include <math.h>


//---------------------------------------------------------------------------
//
// class ConfInt
//
//    Confidence interval representation.
//
// Members:
//
//    double lower
//    double upper
//       lower and upper bounds of the confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------


class ConfInt 

{
      double lower;
      double upper;

   public:

      ConfInt() 
         :  lower( 0),
            upper( 0)
      {}

      ConfInt( double value)
         :  lower( value),
            upper( value)
      {}

      ConfInt( double lBound, double uBound);

      double GetLower();
      double GetUpper();
      double GetSize();

      int operator[]( double value);

      ConfInt operator+();
      ConfInt operator-();

      ConfInt operator+( double opnd2);
      ConfInt operator+( ConfInt opnd2);
      ConfInt operator-( double opnd2);
      ConfInt operator-( ConfInt opnd2);
      ConfInt operator*( double opnd2);
      ConfInt operator*( ConfInt opnd2);
      ConfInt operator/( double opnd2);
      ConfInt operator/( ConfInt opnd2);

      ConfInt operator+=( double opnd2);
      ConfInt operator+=( ConfInt opnd2);
      ConfInt operator-=( double opnd2);
      ConfInt operator-=( ConfInt opnd2);
      ConfInt operator*=( double opnd2);
      ConfInt operator*=( ConfInt opnd2);
      ConfInt operator/=( double opnd2);
      ConfInt operator/=( ConfInt opnd2);

      friend ConfInt operator+( double opnd1, ConfInt opnd2);
      friend ConfInt operator-( double opnd1, ConfInt opnd2);
      friend ConfInt operator*( double opnd1, ConfInt opnd2);
      friend ConfInt operator/( double opnd1, ConfInt opnd2);
   
      friend ostream& operator<<( ostream& smStream, ConfInt ciValue);

      friend ConfInt min( ConfInt opnd1, ConfInt opnd2);
      friend ConfInt max( ConfInt opnd1, ConfInt opnd2);

      friend ConfInt sin( ConfInt opnd);
      friend ConfInt cos( ConfInt opnd);

      friend double ciDelta( ConfInt opnd1, ConfInt opnd2);
};


//---------------------------------------------------------------------------
//
// inline double ConfInt::GetLower()
//
//    Return the lower bound of the confidence interval.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the requested lower bound
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline double ConfInt::GetLower()

{
   return lower;
}


//---------------------------------------------------------------------------
//
// inline double ConfInt::GetUpper()
//
//    Return the upper bound of the confidence interval.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the requested upper bound
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline double ConfInt::GetUpper()

{
   return upper;
}


//---------------------------------------------------------------------------
//
// inline double ConfInt::GetSize()
//
//    Return the size of the confidence interval.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the requested size
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 26.08.94)
//
//---------------------------------------------------------------------------

inline double ConfInt::GetSize()

{
   return ( upper - lower);
}


//---------------------------------------------------------------------------
//
// inline int ConfInt::operator[]( double value)
//
//    Overload the subscripting operator to test the belonging of a given
//    value to the confidence interval.
//
// Arguments:
//
//    double value
//       the value to be tested
//
// Return value:
//
//    the result of the test
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 27.08.94)
//
//---------------------------------------------------------------------------

inline int ConfInt::operator[]( double value)

{
   return ( value >= lower && value <= upper);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator+()
//
//    Apply the unary plus operator to a confidence interval.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator+()

{
   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator-()
//
//    Apply the unary minus operator to a confidence interval.
//
// Arguments:
//
//    none
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator-()

{
   return ConfInt( -upper, -lower);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator+( double opnd2)
//
//    Addiction between a confidence interval and a singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to add
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator+( double opnd2)

{
   return ConfInt( lower + opnd2, upper + opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator+( ConfInt opnd2)
//
//    Addiction between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to add
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator+( ConfInt opnd2)

{
   return ConfInt( lower + opnd2.lower, upper + opnd2.upper);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator-( double opnd2)
//
//    Subtraction between a confidence interval and a singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to subtract
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator-( double opnd2)

{
   return ConfInt( lower - opnd2, upper - opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator-( ConfInt opnd2)
//
//    Subtraction between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to subtract
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator-( ConfInt opnd2)

{
   return ConfInt( lower - opnd2.upper, upper - opnd2.lower);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator*( double opnd2)
//
//    Multiplication between a confidence interval and a singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to multiply for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator*( double opnd2)

{
   return ConfInt( opnd2 * (( opnd2 >= 0) ? lower : upper),
                   opnd2 * (( opnd2 >= 0) ? upper : lower));
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator/( double opnd2)
//
//    Division between a confidence interval and a singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to divide for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator/( double opnd2)

{
   return *this * ( 1 / opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator/( ConfInt opnd2)
//
//    Division between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to divide for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator/( ConfInt opnd2)

{
   return *this * ( 1 / opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator+=( double opnd2)
//
//    Addiction and assignment between a confidence interval and a
//    singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to add
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator+=( double opnd2)

{
   lower += opnd2;
   upper += opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator+=( ConfInt opnd2)
//
//    Addiction and assignment between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to add
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator+=( ConfInt opnd2)

{
   lower += opnd2.lower;
   upper += opnd2.upper;

   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator-=( double opnd2)
//
//    Subtraction and assignment between a confidence interval and a
//    singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to subtract
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator-=( double opnd2)

{
   lower -= opnd2;
   upper -= opnd2;

   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator-=( ConfInt opnd2)
//
//    Subtraction and assignment between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to subtract
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator-=( ConfInt opnd2)

{
   lower -= opnd2.upper;
   upper -= opnd2.lower;

   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator*=( double opnd2)
//
//    Multiplication and assignment between a confidence interval and a
//    singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to multiply for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator*=( double opnd2)

{
   double tmp = lower;
   lower = opnd2 * (( opnd2 >= 0) ? lower : upper);
   upper = opnd2 * (( opnd2 >= 0) ? upper : tmp);

   return *this;
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator/=( double opnd2)
//
//    Division and assignment between a confidence interval and a singleton.
//
// Arguments:
//
//    double opnd2
//       the singleton to divide for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator/=( double opnd2)

{
   return *this *= ( 1 / opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt ConfInt::operator/=( ConfInt opnd2)
//
//    Division and assignment between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd2
//       the confidence interval to divide for
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    updates the current confidence interval
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt ConfInt::operator/=( ConfInt opnd2)

{
   return *this *= ( 1 / opnd2);
}


//---------------------------------------------------------------------------
//
// inline ConfInt operator+( double opnd1, ConfInt opnd2)
//
//    Sum a singleton to a confidence interval.
//
// Arguments:
//
//    double opnd1
//    ConfInt opnd2
//       the operands
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt operator+( double opnd1, ConfInt opnd2)

{
   return opnd2 + opnd1;
}


//---------------------------------------------------------------------------
//
// inline ConfInt operator-( double opnd1, ConfInt opnd2)
//
//    Subtract from a singleton a confidence interval.
//
// Arguments:
//
//    double opnd1
//    ConfInt opnd2
//       the operands
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt operator-( double opnd1, ConfInt opnd2)

{
   return -opnd2 + opnd1;
}


//---------------------------------------------------------------------------
//
// inline ConfInt operator*( double opnd1, ConfInt opnd2)
//
//    Multiply a singleton for a confidence interval.
//
// Arguments:
//
//    double opnd1
//    ConfInt opnd2
//       the operands
//
// Return value:
//
//    the resulting confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt operator*( double opnd1, ConfInt opnd2)

{
   return opnd2 * opnd1;
}


//---------------------------------------------------------------------------
//
// inline ostream& operator<<( ostream& smStream, ConfInt ciValue)
//
//    Write a confidence interval into an output stream.
//
// Arguments:
//
//    ostream& smStream
//       reference to the used output stream
//
//    ConfInt ciValue
//       confidence interval to write
//
// Return value:
//
//    reference to the used output stream
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 02.03.94)
//
//---------------------------------------------------------------------------
 
inline ostream& operator<<( ostream& smStream, ConfInt ciValue)

{
   return smStream << "[ " << setprecision( 6) << ciValue.lower
                   << ", " << setprecision( 6) << ciValue.upper << "]";
}


//---------------------------------------------------------------------------
//
// inline ConfInt min( ConfInt opnd1, ConfInt opnd2)
//
//    Calculate the minimum confidence interval from two given ones.
//
// Arguments:
//
//    ConfInt opnd1
//    ConfInt opnd2
//       the operands
//
// Return value:
//
//    the computed minimum confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 12.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt min( ConfInt opnd1, ConfInt opnd2)

{
   return ConfInt( ( opnd1.lower <= opnd2.lower) ? opnd1.lower
                                                 : opnd2.lower,
                   ( opnd1.upper <= opnd2.upper) ? opnd1.upper
                                                 : opnd2.upper);
}


//---------------------------------------------------------------------------
//
// inline ConfInt max( ConfInt opnd1, ConfInt opnd2)
//
//    Calculate the maximum confidence interval from two given ones.
//
// Arguments:
//
//    ConfInt opnd1
//    ConfInt opnd2
//       the operands
//
// Return value:
//
//    the computed maximum confidence interval
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 12.03.94)
//
//---------------------------------------------------------------------------

inline ConfInt max( ConfInt opnd1, ConfInt opnd2)

{
   return ConfInt( ( opnd1.lower >= opnd2.lower) ? opnd1.lower
                                                 : opnd2.lower,
                   ( opnd1.upper >= opnd2.upper) ? opnd1.upper
                                                 : opnd2.upper);
}


//---------------------------------------------------------------------------
//
// inline double ciDelta( ConfInt opnd1, ConfInt opnd2)
//
//    Calculate the distance between two confidence intervals.
//
// Arguments:
//
//    ConfInt opnd1
//    ConfInt opnd2
//       the two confidence intervals to compare
//
// Return value:
//
//    the computed distance
//
// Side effects:
//
//    none
//
// Author: S. Deodato ( 19.04.94)
//
//---------------------------------------------------------------------------

inline double ciDelta( ConfInt opnd1, ConfInt opnd2)

{
   return ( fabs( opnd1.lower - opnd2.lower) +
            fabs( opnd1.upper - opnd2.upper));
}


#endif // ___CONFINT_H
