//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Moments.c,v 0.19 1994/01/28 18:54:04 cncl-adm Exp cncl-adm $
 *
 * CNClass: CNMoments --- Simple stat. evaluation: mean, variance, skewness etc.
 *
 * Derived from SIC - Simulation In C++
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include <math.h>
#include <values.h>

#include "Moments.h"



CNMoments::CNMoments(char *new_name)
    : n(0), sum(0), sqsum(0), cbsum(0), 
      min_value(MAXDOUBLE), max_value(-MAXDOUBLE)
{
    name = new char[strlen(new_name) + 1];
    strcpy(name, new_name);
}



void CNMoments::reset()
{
    n = 0;
    sum = sqsum = cbsum = 0;
    min_value = MAXDOUBLE;
    max_value = -MAXDOUBLE;
}



CNMoments::~CNMoments()
{
    delete name;
}



virtual double CNMoments::mean() const
{
    return n ? (sum / n) : 0. ;
}



double CNMoments::variance() const
{
    return n > 1 ? (sqsum - sum * sum / n) / (n - 1) : 0. ;
}



double CNMoments::relative_variance() const
{
    return n<=1 || sum==0. ? 0. : variance()*n*n/sum/sum ;
}



double CNMoments::M_2() const
{
    return n ? sqsum / n : 0. ;
}



double CNMoments::M_3() const
{
    return n ? cbsum / n : 0. ; 
}



double CNMoments::Z_3() const
{
    if (n < 3)
	return 0.;

    return (n*cbsum + sum*(2.0*sum*sum/n - 3.0*sqsum)) / (n - 1) / (n - 2);
}



double CNMoments::skewness() const
{
    if (n < 3)
	return 0.;
    double var = variance();
    if (var == 0)
	return 0.;
    return Z_3() / (var * sqrt(var));
}



double CNMoments::deviation() const
{
    return n>1 ? sqrt(variance()) : 0. ;
}



double CNMoments::relative_deviation() const
{
    if (n <= 1 || mean() == 0.)
	return 0.;
    else
	return sqrt(variance()) / mean() ;
}



/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNMoments::print(ostream &strm) const
{
    strm << "Evaluation " << name << " (CNMoments)" << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
}


// Debug output
void CNMoments::dump(ostream &strm) const
{
    strm << "CNMoments { " << endl;
    strm << "name = " << name << endl;
    strm << "sum = " << sum << "  sqsum = " << sqsum
	 << "  cbsum = " << cbsum << endl;
    strm << "n = " << n << "  min_value = " << min_value
	 << "  max_value = " << max_value << endl;
    strm << "Evaluation " << name << " (CNMoments)" << endl;
    strm << "Trials             : " << n                    << endl;
    strm << "Mean               : " << mean()               << endl;
    strm << "Variance           : " << variance()           << endl;
    strm << "Relative variance  : " << relative_variance()  << endl;
    strm << "Deviation          : " << deviation()          << endl;
    strm << "Relative deviation : " << relative_deviation() << endl;
    strm << "Skewness           : " << skewness()           << endl;
    strm << "Minimum            : " << min()                << endl;
    strm << "Maximum            : " << max()                << endl;
    strm << " }" << endl;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNMoments
static CNClass CNMoments_desc("CNMoments", "$Revision: 0.19 $", CNMoments::new_object);

// "Type" for type checking functions
CNClassDesc CN_MOMENTS = &CNMoments_desc;






