//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: RNG.h,v 0.19 1994/01/28 18:54:32 cncl-adm Exp cncl-adm $
 *
 * CNClass: CNRNG --- Abstract base class for random number generators
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *****************************************************************************
 * original Copyright:
 * -------------------
 * Copyright (C) 1988 Free Software Foundation
 *    written by Dirk Grunwald (grunwald@cs.uiuc.edu)
 * 
 * This file is part of the GNU C++ Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 **EHEADER********************************************************************/


#ifndef __RNG_h
#define __RNG_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>

#include <CNCL/Object.h>		// Base class

extern CNClassDesc CN_RNG;	// CNClass CNRNG description


#include <assert.h>
#include <math.h>
#include <stdio.h>



union PrivateCNRNGSingleType {	   	// used to access floats as unsigneds
    float s;
    unsigned long u;
};

union PrivateCNRNGDoubleType {	   	// used to access doubles as unsigneds
    double d;
    unsigned long u[2];
};



/*
 * The class CNRNG
 */

class CNRNG : public CNObject
{
public:
    //
    // Return a long-words word of random bits
    //
    virtual unsigned long as_long32() = 0;
    virtual void reset() = 0;
    virtual bool has_long32() = 0;
    //
    // Return a 31bit long word of random bits
    //
    unsigned long as_long();
    //
    // Return random bits converted to either a float or a double
    //
    float as_float();
    double as_double();

    // Private members
private:
    static PrivateCNRNGSingleType singleMantissa;	// mantissa bit vector
    static PrivateCNRNGDoubleType doubleMantissa;	// mantissa bit vector

    /***** Constructors ******************************************************/
public:
    CNRNG();				// Default constructor
    CNRNG(CNParam *param) {}			// CNParam constructor

    /***** Member functions required by CNCL *********************************/
public:
    virtual CNClassDesc class_desc() const	// CNClass description
    {
	return CN_RNG;
    };
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    {
	return desc == CN_RNG ? TRUE : CNObject::is_a(desc);
    };
        
    static CNRNG *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNRNG *)obj;
#   else
	return (CNRNG *)( !obj || obj->is_a(CN_RNG)
	       ? obj : fatal_type(obj->class_desc(), CN_RNG) );
#   endif
    }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const = 0;
    virtual void dump (ostream &strm = cout) const = 0;
};



inline unsigned long CNRNG::as_long()
{
    return as_long32() & 0x7fffffff;
}

inline float CNRNG::as_float()
{
    PrivateCNRNGSingleType result;
    result.s = 1.0;
    result.u |= (as_long32() & singleMantissa.u);
    result.s -= 1.0;
    assert( result.s < 1.0 && result.s >= 0);
    return( result.s );
}
	
inline double CNRNG::as_double()
{
    PrivateCNRNGDoubleType result;
    result.d = 1.0;
    result.u[0] |= (as_long32() & doubleMantissa.u[0]);
    result.u[1] |= (as_long32() & doubleMantissa.u[1]);
    result.d -= 1.0;
    assert( result.d < 1.0 && result.d >= 0);
    return( result.d );
}

#endif /**__RNG_h**/
