//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: String.c,v 0.19 1994/01/28 18:55:13 cncl-adm Exp cncl-adm $
 *
 * CNClass: CNString --- Character string
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include <string.h>
#include <ctype.h>

#include "String.h"



/*
 * Comparison operators
 */
bool operator <  (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) <  0; }	

bool operator >  (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) >  0; }	

bool operator >= (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) >= 0; }

bool operator <= (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) <= 0; }

bool operator == (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) == 0; }

bool operator != (const CNString& a, const CNString& b)
{   return strcmp(a.p, b.p) != 0; }



/***** CNString private functions *******************************************/
 
void CNString::init(unsigned l, unsigned extra)
{
    len = l;
    alloc = len + extra;
    p = new char[alloc + 1];
    p[len] = '\0';
}



/*
 * Copy from string of length l (l+1 == length + '\0')
 *
 * DOES NOT SET len !!!
 */
void CNString::cpyn(const char *s, unsigned l)
{
    unsigned n = l+1>alloc ? alloc : l+1;

    strncpy(p, s, n);
    p[n] = 0;
}

    
    
/***** CNString constructors ************************************************/

CNString::CNString(int extra)
{
    if ( extra < 0 ) extra = default_extra;
    init(0,extra);
}


CNString::CNString(const char* cs)
{ 
    unsigned l = strlen(cs);
    init(l);
    cpyn(cs, l);
}


CNString::CNString(const char* cs, int extra)
{
    if ( extra < 0 ) extra = default_extra;
    unsigned l = strlen(cs);
    init(l, extra);
    cpyn(cs, l);
}

CNString::CNString(const CNString& s)
{
    init(s.len);
    cpyn(s.p, s.len);
}

CNString::CNString(const CNString& s, int extra)
{
    if (extra < 0) extra = default_extra;
    init(s.len, extra);
    cpyn(s.p, s.len);
}

CNString::~CNString() 
{
    delete [] p;
}



/***** public functions ******************************************************/
 
void CNString::resize(unsigned new_capacity) 
/* sets capacity alloc of CNString to new_capacity */
{
    if ( new_capacity != alloc )
    {
	alloc = new_capacity;
	char* tmp = p;
	p = new char[alloc + 1];
	if(len > alloc)
	    len = alloc;
	cpyn(tmp, len);
	delete [] tmp;
    }
}    	


void CNString::to_lower()
/* sets all letters in CNString to lower case */
{
    for(int i=0; i<len; i++)
	if(isupper(p[i]))
	    p[i] = tolower(p[i]);
}



void CNString::to_upper()
/* sets all letters to upper case */
{
    for(int i=0; i<len; i++)
	if(islower(p[i]))
	    p[i] = toupper(p[i]);
}



void CNString::capitalize()
/* sets all first letters of the words (after a space) to upper case, all 
 * other letters will be lower case */
{
    int flag, i;
    
    for(i=0, flag=TRUE; i<len; i++)
    {
	if(isspace(p[i]))
	    flag = TRUE;
	else if(flag)
	{
	    flag = FALSE;
	    if(islower(p[i]))
		p[i] = toupper(p[i]);
	}
	else 
	{
	    if(isupper(p[i]))
		p[i] = tolower(p[i]);
	}
    }
}



void CNString::to_ascii()
/* sets CNString to ASCII */
{    
    for(int i=0; i<len; i++)
	p[i] = toascii(p[i]);
}



CNString CNString::after( int pos, int l)  const
/* returns the l characters after position pos (zero based) as a CNString; 
 * char at pos is first of CNString; default value l = 0 returns all 
 * characters until end of CNString */ 
{
    CNString r;

    if(l <= 0)
	l = len;
    
    if(pos>= 0 && pos<len)
    {
	int m = pos+l>len ? len : pos+l;
	int n = m - pos;
	
	r.resize(n + default_extra);
	
	for(int i=0; i<n; i++)
	    r.p[i] = p[pos+i];
	r.p[i] = 0;
	r.len  = n;
    }

    return r;
}



CNString CNString::before( int pos, int l) const
/* returns the l characters before position pos (zero based) as a CNString;
 * char at pos is last of CNString; default value l = 0 returns all characters 
 * until position zero */
{
    CNString r;

    if(l <= 0)
	l = len;
    
    if(pos>= 0 && pos<=len)
    {
	int m = pos;
	pos -= l;
	if(pos < 0)
	    pos = 0;
	int n = m - pos;
	
	r.resize(n + default_extra);
	
	for(int i=0; i<n; i++)
	    r.p[i] = p[pos+i];
	r.p[i] = 0;
	r.len  = n;
    }

    return r;
}



CNString& CNString::add(char c)
/* adds character c at the end of CNString this */
{
    if(len >= alloc)
    	resize(len + default_extra);
    p[len++] = c;
    p[len]   = '\0';
    return *this;
}



CNString& CNString::add(const char *cs)
/* adds characterstring cs at the end of CNString this */
{
    insert(cs, len);
    return *this;
}



/***** Delete chars *********************************************************/

CNString& CNString::del(int pos, int l)
/* deletes l characters beginning at position pos (zero based); default values
 * for pos is zero, for l is 1 ( e.g. del() will delete first char of this).
 * If l is set to 0 all characters beginning at pos will be deleted ( e.g. 
 * del(0,0) will return an empty CNString) */
{
    if(l <= 0)
	l = len;
    
    if(pos>=0 && pos<len)
    {
	int m = pos+l>len ? len : pos+l;
	int n = m - pos;

	while(m <= len)
	    p[pos++] = p[m++];
	
	len -= n;
	p[len] = 0;
    }

    return *this;
}



CNString& CNString::del(char c, int pos) 
/* deletes the first char c in this after pos (zero based) or nothing if c 
 * is not in this after pos. Default value for pos is zero. */
{
    pos = upsearch(c, pos);
    if(pos != ERROR)
	del(pos, 1);			/* Delete 1 char */

    return *this;
}


    
CNString& CNString::del(const CNString& s, int pos) 
/* deletes the first occurence of CNString s in CNString this after position pos 
 * (zero based) or nothing if s not in this after pos; default value for pos 
 * is zero */    
{
    pos = upsearch(s, pos);
    if(pos != ERROR)
	del(pos, s.len);

    return *this;
}



CNString& CNString::del(const char *s, int pos)
/* deletes the first occurence of cs in in CNString this after position pos
 * (zero based) or nothing if cs is not this after pos; default value for 
 * pos is zero */
{
    pos = upsearch(s, pos);
    if(pos != ERROR)
	del(pos, strlen(s));

    return *this;
}



/***** Insert chars *********************************************************/

CNString& CNString::insert ( char c, int pos)
/* inserts character c at position pos (zero based); default value for pos 
 * is zero */
{
    if ( pos >= 0  &&  pos <= len ) 
    {
	int i;
	
	if ( len >= alloc ) resize( len + default_extra );
	for(i=len; i>=pos; i--)
	    p[i+1] = p[i];
	len++;
	p[len] = 0;
	p[pos] = c;
    }
    return *this;
}



CNString& CNString::insert(const char* cs, int pos)
/* inserts char* cs at position pos (zero based); default value for pos is
 * zero */ 
{
    if ( pos >= 0  &&  pos <= len )
    {
	int l = strlen(cs);
	int i;
	
	resize ( l + len + default_extra );
	
	for(i=len; i>=pos; i--)
	    p[i+l] = p[i];
	len += l;
	p[len] = 0;
	for(i=0; i<l; i++)
	    p[pos+i] = cs[i];
    }
    return *this;
}

	 
CNString& CNString::insert (const CNString& s, int pos)
/* inserts CNString s at position pos (zero based); default value for pos is 
 * zero */

{
    return insert(s.p, pos);
}



/***** Replace chars ********************************************************/

CNString& CNString::replace(char c, int pos)
/* replaces the character at position pos (zero based) with c, default value
 * for pos is zero */
{
    if(pos>=0 && pos<len)
	p[pos] = c;
    return *this;
}


CNString& CNString::replace(char oldc, char newc, int pos)
/* replaces the first character oldc after position pos (zero based) with newc,
 * default value for pos is zero */
{
    pos = upsearch(oldc, pos);
    if(pos != ERROR)
	p[pos] = newc;

    return *this;
}


CNString& CNString::replace(const CNString& s, int pos, int l)
/* replaces the l characters, starting at position pos (zero based, default 
 * value zero), with CNString s; default value l = 0 will delete as many 
 * characters as the length of CNString s */    
{
    if(pos>=0 && pos<=len)
    {
	del(pos, l);
	insert(s, pos);
    }
    
    return *this;
}


CNString& CNString::replace(const CNString& olds,const CNString& news, int pos)
/* replaces the first occurence of CNString olds after position pos with 
 * CNString news, default value for pos is zero. If olds is not in this
 * nothing is changed */
{	    
    pos = upsearch(olds, pos);
    if(pos != ERROR)
    {
	del(pos, olds.len);
	insert(news, pos);
    }

    return *this;
}



/***** Search ***************************************************************/

/*
 * Search for a char or a string, starting at `pos' and working towards
 * the _start_ of the string. Return the index if found, else ERROR.
 */
int CNString::downsearch (char c, int pos) const  
{
    int i;

    if(pos >= 0 && pos < len)
    {
	for(i=pos; i>=0; i--)
	    if(p[i] == c)
		return i;
    }
    
    return ERROR;
}


int CNString::downsearch (const CNString& s, int pos) const 
{
    int i;

    if(pos >= 0 && pos < len)
    {
	for(i=pos; i>=0; i--)
	    if(matches(s, i))
		return i;
    }
    
    return ERROR;
}


int CNString::downsearch (const char *s, int pos) const 
{
    int i;

    if(pos >= 0 && pos < len)
    {
	for(i=pos; i>=0; i--)
	    if(matches(s, i))
		return i;
    }
    
    return ERROR;
}



/*
 * Search for a char or a string, starting at `pos' and working towards
 * the _end_ of the string. Return the index if found, else ERROR.
 */
int CNString::upsearch (char c, int pos) const 
{
    int i;

    if(pos >= 0)
    {
	for(i=pos; i<len; i++)
	    if(p[i] == c)
		return i;
    }
    
    return ERROR;
}


int CNString::upsearch (const CNString& s, int pos) const
{
    int i;
    
    if(pos >= 0)
    {
	for(i=pos; i<len; i++)
	    if(matches(s, i))
		return i;
    }
    
    return ERROR;
}


int CNString::upsearch(const char *s, int pos) const
{
    int i;
    
    if(pos >= 0)
    {
	for(i=pos; i<len; i++)
	    if(matches(s, i))
		return i;
    }
    
    return ERROR;
}



/*
 * Returns TRUE, if the argument string is to be found at `pos', else FALSE
 */
bool CNString::matches(const char *cs, int pos) const
{
    int l = strlen(cs);

    if(pos>=0 && pos<=len-l)
    {
	if(cs[0] == p[pos])
	    return strncmp(cs, p+pos, l) == 0;
	else
	    return FALSE;
    }
    
    return FALSE;
}


bool CNString::matches(const CNString& s, int pos) const
{   
    return matches(s.p, pos);
}



/*
 * Input string from IOStream
 */
void CNString::icopy( istream& strm )
{
    char tmp[512];
    strm.getline(tmp, sizeof(tmp));
    len = strlen(tmp);
    resize(len + default_extra);
    cpyn(tmp, len);
}



/***** operators *************************************************************/

char CNString::operator()(int i) const
{
    if(i<0 || i>=len)
	return '\0';
    return p[i];
}



char& CNString::operator[](int i)
{
    static char cc = '\0';
    if(i<=0 || i>=len)
	return cc;
    return p[i];
}



void CNString::operator=(const CNString& s)
{
    if (p != s.p)
    {
	delete [] p;
	init(s.len);
	cpyn(s.p, s.len);
    }
}


void CNString::operator=(const char *cs)
{
    if (p != cs)
    {
	delete [] p;
	unsigned l = strlen(cs);
	init(l);
	cpyn(cs, l);
    }
}


void CNString::operator=(char c)
{
    delete [] p;
    init(1);
    p[0] = c;
    p[1] = '\0';
}



CNString operator + (const CNString& s1, const CNString& s2)
{
    CNString t(s1);
    t.add(s2);
    return t;
}



CNString& CNString::operator += (const CNString& s) 
{
    add(s);
    return *this;
}

CNString& CNString::operator += (const char* cs)
{     
    add(cs);
    return *this;
}

	
 

/***** Default I/O member function for CNCL classes **************************/

// CNNormal output
void CNString::print(ostream &strm) const
{
    strm << p;
}

// Debug output
void CNString::dump(ostream &strm) const
{
    strm << "CNString { $Revision: 0.19 $  len=" << len << " alloc="
	 << alloc << " p=\"" << p << "\" }" << endl;
}

// input
istream &operator >> (istream& strm, CNString& s)
{
    s.icopy(strm);
    return strm;
}



/***** CNCL stuff for type information and exemplar objects ******************/

// Describing object for class CNString
static CNClass CNString_desc("CNString", "$Revision: 0.19 $",
			     CNString::new_object);

// "Type" for type checking functions
CNClassDesc CN_STRING = &CNString_desc;
