//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: FRuleBase.c,v 0.19 1994/01/28 18:55:53 cncl-adm Exp cncl-adm $
 *
 * Class: CNFRuleBase --- Rule base and Fuzzy inference engine
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "FRuleBase.h"

#include <minmax.h> // FIXME: libg++ only???



/*
 * Add rule to base
 */
void CNFRuleBase::add_rule(CNFRule *rule)
{
    int n = rules.get_size();
    
    rules.set_size(n + 1);
    rules[n] = rule;
}

void CNFRuleBase::add_rule(CNFRule &rule)
{
    add_rule(&rule);
}



/*
 * Add input variable to base
 */
void CNFRuleBase::add_in_var(CNFVar *var)
{
    int n = in_vars.get_size();
    
    in_vars.set_size(n + 1);
    in_vars[n] = var;
}

void CNFRuleBase::add_in_var(CNFVar &var)
{
    add_in_var(&var);
}



/*
 * Add output variable to base
 */
void CNFRuleBase::add_out_var(CNFVar *var)
{
    int n = out_vars.get_size();
    
    out_vars.set_size(n + 1);
    out_vars[n] = var;
}

void CNFRuleBase::add_out_var(CNFVar &var)
{
    add_out_var(&var);
}



/*
 * Get sizes of arrays
 */
int CNFRuleBase::get_n_rules() const
{
    return rules.get_size();
}

int CNFRuleBase::get_n_in_vars() const
{
    return in_vars.get_size();
}

int CNFRuleBase::get_n_out_vars() const
{
    return out_vars.get_size();
}



int CNFRuleBase::resolution() const
{
    return nsteps;
}



/*
 * Compute inference for variable var and value set, using match value
 * from rule LHS aggregation. Result is stored in an array fuzzy set.
 */
void CNFRuleBase::inference(CNFVar *var, CNFSet *set,
			  double match, CNFSetArray &res)
{
    double xmin, xmax, x, dx;
    int i, n;
    
    n    = res.get_n();
    xmin = var->xmin();
    xmax = var->xmax();
    dx   = (xmax - xmin) / n;

    for(i=0, x=xmin; i<n; i++, x+=dx)
	/**FIXME: this should use the inference operator**/
	res[i] = match * set->get_membership(x);
}



/*
 * Compute output fuzzy set for variable var
 */
void CNFRuleBase::evaluate (CNFVar *var, CNFSetArray &result)
{
    int i, j, k, m, n;
    double agg;
    CNFRule *rule;
    CNFClause *clause;
    CNFSetArray tmp(result.get_n(), var->xmin(), var->xmax());
    
    /*
     * Rules must be aggregated beforehand!
     */

    n = get_n_rules();
    
    /*
     * Search all rules
     */
    for(i=0; i<n; i++)
    {
	rule = CNFRule::cast_from_object( rules[i] );
	agg = rule->aggregate_value();
	if(agg > 0)				// Rule fired
	{
	    m = rule->get_n_rhs();
	    /*
	     * Search all variables in RHS of rule
	     */
	    for(j=0; j<m; j++)
	    {
		clause = rule->get_rhs(j);
		if(clause->var == var)		// Found variable
		{
		    inference(var, clause->value, agg, tmp);
		    /**FIXME: this should use the accumulation operator**/
		    for(k=0; k<result.get_n(); k++)
			result[k] = max(tmp[k], result[k]);
		}
	    }
	}
    }
}



/*
 * Aggregate all rules
 */
void CNFRuleBase::aggregate_all()
{
    for(int i=0; i<get_n_rules(); i++)
	CNFRule::cast_from_object(rules[i])->aggregate();
}



/*
 * Evaluate all variables
 */
void CNFRuleBase::evaluate_all()
{
    int i;
    CNFVar      *var;
    CNFSetArray *set;

    aggregate_all();
    
    for(i=0; i<get_n_out_vars(); i++)
    {
	var = CNFVar::cast_from_object(out_vars[i]);
	if(var->fuzzy_value())
	    delete var->fuzzy_value();
	set = new CNFSetArray( resolution(), var->xmin(), var->xmax() );
	evaluate(var, *set);
	var->fuzzy_value(set);
    }
}



/*
 * Defuzzify output variables using center_of_gravity()
 */
void CNFRuleBase::defuzzy_all()
{
    int i;
    CNFVar *var;
    CNFSet *set;

    for(i=0; i<get_n_out_vars(); i++)
    {
	var = CNFVar::cast_from_object(out_vars[i]);
	if(set = var->fuzzy_value())
	    var->value( set->center_of_gravity( var->xmin(), var->xmax() ) );
    }
}

    

/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNFRuleBase::print(ostream &strm) const
{
    int i;

    strm << "BEGIN " << name() << endl;
    
    strm << "// ----- INPUT -----" << endl;
    for(i=0; i<get_n_in_vars(); i++)
	strm << in_vars.get(i);
    strm << "// ----- OUTPUT -----" << endl;
    for(i=0; i<get_n_out_vars(); i++)
	strm << out_vars.get(i);
    strm << "// ----- RULES -----" << endl;
    for(i=0; i<get_n_rules(); i++)
	strm << rules.get(i);

    strm << "END" << endl;
}

// Debug output
void CNFRuleBase::dump(ostream &strm) const
{
    strm << "CNFRuleBase { $Revision: 0.19 $"
	 << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNFRuleBase
static CNClass CNFRuleBase_desc("CNFRuleBase", "$Revision: 0.19 $",
			  CNFRuleBase::new_object);

// "Type" for type checking functions
CNClassDesc CN_FRULEBASE = &CNFRuleBase_desc;






