// -*- C++ -*-

#include <iostream.h>

#include <CNCL/QueueFIFO.h>
#include <CNCL/EventScheduler.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Moments.h>

#include "Job.h"


enum { NJOBS=100000 };

enum { JOB, TIMER_G, TIMER_S };		// CNEvent types for M/M/1 simulation


class Server : public CNEventHandler
{
private:
    Job *job;		// Served job
    CNQueueFIFO queue;	// CNQueue
    CNRandom &rnd_b;	// Distribution of service time b
    CNMoments t_w, t_b;	// Evaluation tau_w, tau_b
    enum { WAITING, SERVING };
    
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();
    void eval_job(Job *job);

    Server(CNRandom &rnd) : rnd_b(rnd), job(NIL), t_w("tau_w"), t_b("tau_b")
    {
	state(WAITING);
    }
};



class Generator : public CNEventHandler
{
private:
    CNRandom &rnd_a;	// Distribution of arrival time a
    Server *server;	// Connected queue/server
    long n;
    
public:
    virtual void event_handler(const CNEvent *ev);

    Generator(CNRandom &rnd, Server *serv) : rnd_a(rnd), server(serv), n(0) {}
};



void Generator::event_handler(const CNEvent *ev)
{
    if(n == NJOBS)
	// Stop simulation
	return;

#ifdef DEBUG
    cout << "----- Generator -----" << endl;
    ev->dump(cout);
#endif    

    // Incoming event -> generate new Job
    send_now(new CNEvent(JOB, server, new Job));
    // CNRandom delay
    send_delay(new CNEvent(TIMER_G), rnd_a());
    n++;
}



void Server::event_handler(const CNEvent *ev)
{
#ifdef DEBUG
    cout << "----- Server -----" << endl;
    ev->dump(cout);
#endif    

    switch(state())
    {
    case SERVING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job, put into queue
	    {
		Job *job;
		job = (Job *)ev->object();
		job->in = now();
		queue.put(job);
	    }
	    break;
	    
	case TIMER_S:
	    // Timer event, service time run down
	    job->out = now();
	    // Evaluate job
	    eval_job(job);
	    delete job;
	    job = NIL;
	    // Get new job from queue
	    if(!queue.empty())
	    {
		job = (Job *)queue.get();
		job->start = now();
		// CNRandom service time
		send_delay(new CNEvent(TIMER_S), rnd_b());
		state(SERVING);
	    }
	    else
		state(WAITING);
	    break;
	    
	default:
	    error("mm1: ", "illegal event in state SERVING");
	    break;
	}

	break;
	
    case WAITING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job
	    job = (Job *)ev->object();
	    job->in    = now();
	    job->start = now();
	    // CNRandom service time
	    send_delay(new CNEvent(TIMER_S), rnd_b());
	    state(SERVING);
	    break;

	default:
	    error("mm1: ", "illegal event in state WAITING");
	    break;
	}
	
	break;
	
    default:
	error("mm1: ", "illegal state");
	break;
	
    }
}



void Server::eval_job(Job *job)
{
    t_w.put(job->start - job->in);
    t_b.put(job->out   - job->in);
}


void Server::print_results()
{
    cout << t_w << t_b;
}

 

main()
{
    CNRNG    *rng   = new CNFiboG;
    CNNegExp rnd_a(10, rng);
    CNNegExp rnd_b( 5, rng);

    Server         server(rnd_b);
    Generator      generator(rnd_a, &server);
    CNEventScheduler scheduler;
    
    scheduler.start(new CNEvent(TIMER_G, &generator, 0));

    server.print_results();
    
}
