//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id: Event.h,v 0.19 1994/01/28 18:55:23 cncl-adm Exp cncl-adm $
 *
 * CNClass: CNEvent --- Generic event
 *
 *****************************************************************************
 * Copyright (C) 1992/1993   Communication Networks
 *                           Aachen University of Technology
 *                           Kopernikusstr. 16
 *                           W-5100 Aachen
 *                           Germany
 *                           Email: mj@dfv.rwth-aachen.de (Martin Junius)
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#ifndef __Event_h
#define __Event_h


#include <CNCL/Class.h>
#include <CNCL/Param.h>
#include <CNCL/DLObject.h>		// Base class
#include <CNCL/SimTime.h>

#include <stddef.h>

extern CNClassDesc CN_EVENT;		// CNClass CNEvent description



class CNEventHandler;


typedef unsigned long CNEventID;		// Type of CNEvent::ev_id



/*
 * The class CNEvent
 */

class CNEvent : public CNDLObject
{
    friend class CNEventScheduler;
    friend class CNEventList;

  private:
    enum { default_max_events=100 };

  public:
    // Get/set maximum # of events. set_max_events() must be called before
    // any event is created.
    static unsigned long get_max_events();
    static void set_max_events(unsigned long n);
    
  public:
    // Get/set priority
    int priority()             const        { return ev_prio; }
    int get_priority()         const        { return ev_prio; }
    void priority(int prio)                 { ev_prio=prio;   }
    void set_priority(int prio)             { ev_prio=prio;   }

    // Get/set type
    int type()                 const        { return ev_type;   }
    int get_type()             const        { return ev_type;   }
    void type(int new_type)                 { ev_type=new_type; }  
    void set_type(int new_type)             { ev_type=new_type; }

    // Get/set scheduled time
    CNSimTime scheduled()      const        { return ev_scheduled; }
    CNSimTime get_scheduled()  const        { return ev_scheduled; }
    void scheduled(const CNSimTime new_scheduled)
                                            { ev_scheduled = new_scheduled; }
    void set_scheduled(const CNSimTime new_scheduled)
                                            { ev_scheduled = new_scheduled; }

    // Get issued time
    CNSimTime issued()         const        { return ev_issued; }
    CNSimTime get_issued()     const        { return ev_issued; }

    // Get/set to addressed event handler
    CNEventHandler *to()       const        { return ev_to; }
    CNEventHandler *get_to()   const        { return ev_to; }
    void to(CNEventHandler *new_to)         { ev_to=new_to; }
    void set_to(CNEventHandler *new_to)     { ev_to=new_to; }

    // Get/set to sending event handler
    CNEventHandler *from()     const        { return ev_from; }
    CNEventHandler *get_from() const        { return ev_from; }
    void from(CNEventHandler *new_from)     { ev_from=new_from; }
    void set_from(CNEventHandler *new_from) { ev_from=new_from; }

    // Get/set object
    CNObject *object()         const        { return ev_obj; }
    CNObject *get_object()     const        { return ev_obj; }
    void object(CNObject *obj)              { ev_obj=obj; }
    void set_object(CNObject *obj)          { ev_obj=obj; }

    // Get ID
    CNEventID id()             const        { return ev_id; }
    CNEventID get_id()         const        { return ev_id; }

    // Operator new/delete
    static void *operator new(size_t);
    static void  operator delete(void *);
    
    // Private members
  private:
    static CNEvent *pool;		// Memory area for events
    static unsigned long pool_size;	// Size of event pool
    static unsigned long next_free;	// Index of next free event
    // ev_id == 0 indicates that the event in the event pool is not used

    CNEventID ev_id;			// CNEvent ID code
    int ev_prio;			// Priority
    int ev_type;			// User defined event type
    CNSimTime ev_issued;			// Time event issued
    CNSimTime ev_scheduled;		// Time event scheduled for
    CNEventHandler *ev_to;		// Recipient of event
    CNEventHandler *ev_from;		// Sender of event
    CNObject *ev_obj;			// CNObject pointer

    static CNEventID id_sequencer;	// Sequencer for event IDs

    // Set issued time (used by CNEventScheduler)
    void issued(const CNSimTime t)     { ev_issued=t; }
    void set_issued(const CNSimTime t) { ev_issued=t; }

    /***** Constructors ******************************************************/
  public:
    CNEvent();
    CNEvent(CNParam *param);
    CNEvent(int new_type);
    CNEvent(int new_type, const CNSimTime t, int new_prio=0);
    CNEvent(int new_type, CNEventHandler *new_to, const CNSimTime t,
	    CNObject *new_object=NIL, int new_prio=0);
    CNEvent(int new_type, CNEventHandler *new_from, CNEventHandler *new_to,
	    const CNSimTime t, CNObject *new_object=NIL, int new_prio=0);
    CNEvent(int new_type, CNEventHandler *new_to,
	    CNObject *new_object=NIL, int new_prio=0);

private:
    ~CNEvent() {}

    /***** Member functions required by CNCL *********************************/
public:
    virtual CNClassDesc class_desc() const	// CNClass description
    {
	return CN_EVENT;
    };
            
    virtual bool is_a(CNClassDesc desc) const	// Type checking
    {
	return desc == CN_EVENT ? TRUE : CNDLObject::is_a(desc);
    };
        
    static CNEvent *cast_from_object(CNObject *obj) // Safe type cast
    {
#   ifdef NO_TYPE_CHECK
	return (CNEvent *)obj;
#   else
	return (CNEvent *)( !obj || obj->is_a(CN_EVENT)
	       ? obj : fatal_type(obj->class_desc(), CN_EVENT) );
#   endif
    }
    
    static CNObject *new_object(CNParam *param = NIL) // Object creation
    { return param ? new CNEvent(param) : new CNEvent; }
    
    // Print/debug output
    virtual void print(ostream &strm = cout) const;
    virtual void dump (ostream &strm = cout) const;
};


#endif /**__Event_h**/
