
#include "global.h"


/*
 * Structure to hold information for heuristics with variable ordering.
 */
struct info {
    int index, value;
};


/*
 *  For convexity heuristic:
 *  Count the strings of zeros between ones
 *  in the rows of a (0,1)-matrix R.
 */
int
variable_convexity( R, k )
	CONSTRAINT R;
	int	k;
{
	int	a, count, fp, lp;

	count = 0;
	for( a = 0; a < k; a++ ) {

		fp = 0;
		/*  Skip leading zeros  */
		while( fp < k && R [a][fp] == 0 )
			fp++;

		lp = k-1;
		/*  Skip trailing zeros  */
		while( lp >= 0 && R [a][lp] == 0 )
			lp--;

		/*  Skip to the beginning of this string of ones  */
		while( lp > 0 && R [a][lp-1] == 1 )
			lp--;

		while( fp < lp ) {
			count++;
			/*  Skip to the end of this string of ones  */
			while( fp < k && R [a][fp] == 1 )
				fp++;
			/*  Skip to the end of this string of zeros  */
			while( fp < k && R [a][fp] == 0 )
				fp++;
		}
	}

	return( count );
}


/*
 *  For degree heuristic:
 *  Determine whether a (0,1)-matrix R is a
 *  non-trivial constraint.
 */
int
variable_degree( R, k )
	CONSTRAINT R;
	int	k;
{
	int	a, b;

	for( a = 0; a < k; a++ )
	for( b = 0; b < k; b++ )
		if( R[a][b] == 0 )
			return( 1 );

	return(0);
}


/*
 *  For satisfiability heuristic:
 *  Count the number of zeros in a (0,1)-matrix R.
 */
int
variable_satisfy( R, k )
	CONSTRAINT R;
	int	k;
{
	int	a, b, count;

	count = 0;
	for( a = 0; a < k; a++ )
	for( b = 0; b < k; b++ )
		if( R[a][b] == 0 )
			count++;

	return( count );
}


/*
 *  Greedy largest first strategy:  Order the variables according to width
 *  with largest first.  For each iteration, perform the following steps:
 *  find the variable with the largest remaining heuristic value, swap this
 *  with the current position, and decrement each of the unchosen values by
 *  the appropriate amount.  The appropriate amount is kept in the "edge"
 *  matrix, which holds the heuristic value for each edge in the matrix.
 */
void
glf( n, edges, order )
	int n, edges[N][N];
	struct info order[N];
{
    int i, j, max;
    struct info temp;

    for (i = 1; i < n; i++) {
	/*
	 *  Find the maximum in order[i] .. order[n].
	 */
        max = i;
        for (j = i+1; j <= n; j++)
            if (order[j].value > order[max].value)
                max = j;
	/*
	 *  Swap so max is in position i.
	 */
        memcpy((char *)&temp, (char *)&order[i], sizeof(struct info));
        memcpy((char *)&order[i], (char *)&order[max], sizeof(struct info));
        memcpy((char *)&order[max], (char *)&temp, sizeof(struct info));
	/*
	 *  Adjust values to reflect that node i has been removed.
	 */
        for (j = i+1; j <= n; j++)
            order[j].value -= edges[order[j].index][order[i].index];
    }
}


/*
 *  Greedy smallest last strategy:  Order the variables according to width
 *  with smallest last.  For each iteration, perform the following steps:
 *  find the variable with the smallest remaining heuristic value, swap this
 *  with the current position, and decrement each of the unchosen values by
 *  the appropriate amount.  The appropriate amount is kept in the "edge"
 *  matrix, which holds the heuristic value for each edge in the matrix.
 */
void
gsl( n, edges, order )
	int n, edges[N][N];
	struct info order[N];
{
    int i, j, min;
    struct info temp;

    for (i = n; i > 1; i--) {
	/*
	 *  Find the minimum in order[1] .. order[i].
	 */
        min = 1;
	for( j = 2; j <= i; j++ )
	    if( order[j].value < order[min].value )
		min = j;
	/*
	 *  Swap so min is in position i.
	 */
        memcpy((char *)&temp, (char *)&order[i], sizeof(struct info));
        memcpy((char *)&order[i], (char *)&order[min], sizeof(struct info));
        memcpy((char *)&order[min], (char *)&temp, sizeof(struct info));
	/*
	 *  Adjust values to reflect that node i has been removed.
	 */
        for (j = 1; j < i; j++)
            order[j].value -= edges[order[j].index][order[i].index];
    }
}


/*
 * Comparison function for qsort to order in descending order.  This ensures
 * that the variables with the larger heuristic values are processed first
 * by the backtracking algorithm.
 */
static int
descending( a, b )
	struct info *a, *b;
{
	if     ( a->value > b->value ) return(-1);
	else if( a->value < b->value ) return( 1);
	else			       return( 0);
}


/*
 *  Simple sorting strategy.
 */
void
sort( n, edges, order )
	int	n, edges[N][N];
	struct	info order[N];
{
	qsort((char *)&order[1], n, sizeof(struct info), descending);
}


/*
 *  Apply the heuristic to the actual graph by reordering
 *  the graph such that solving the new graph in sequential
 *  order corresponds to solving the problem in the order
 *  specified by the heuristic.
 */
static void
apply_order( C, n, k, order )
	NETWORK	C;
	int	n, k;
	struct info order[N];
{
	NETWORK	temp;
	int	i, j, a, b, ii, jj;


	for( i = 1; i <= n; i++ ) { ii = order[i].index;
	for( j = 1; j <= n; j++ ) { jj = order[j].index;
		for( a = 0; a < k; a++ )
		for( b = 0; b < k; b++ )
			temp[i][j][a][b] = C[ii][jj][a][b];
	}}

	for( i = 1; i <= n; i++ )
	for( j = 1; j <= n; j++ )
		for( a = 0; a < k; a++ )
		for( b = 0; b < k; b++ )
			C[i][j][a][b] = temp[i][j][a][b];

}



/*
 * Create an order to instantiate the variables by applying a heuristic to the
 * original problem.  First, calculate the heuristic value for each edge and
 * keep a running total for each variable as well.  Then order the variables
 * according to one of the user-specified routines: order all variables at once
 * (i.e. sort), order from smallest last using gsl, or order from largest first
 * using glf.  Then, call the routine to actually change the "C" matrix
 * according to the results of the previous step.
 */
void
order_variables( heuristic, strategy, C, n, k )
int (* heuristic)(), n, k;
void (* strategy)();
NETWORK C;
{
    int i, j, edges[N][N];
    struct info order[N];

    for (i = 1; i <= n; i++) {
        order[i].index = i;
        order[i].value = 0;
        for (j = 1; j <= n; j++) if (i != j) {
            edges[i][j] = (*heuristic)(C[i][j], k);
            order[i].value += edges[i][j];
        }
    }
    (*strategy)(n, edges, order);

    apply_order(C, n, k, order);
}

