/*
 * WireWorld 2.1 (c) 1992,93 by Stefan Strack (stst@vuse.vanderbilt.edu)
 * (based on code by Kevin Dahlhausen (ap096@po.cwru.edu) posted on
 *  comp.theory.cell-automata)
 *
 * This program implements the cellular automata 'Wire World' as
 * described in the comp.theory.cell-automata newsgroup.
 *
 * v2.0:
 *      - direct screen memory addressing for faster speed
 *      - on-screen circuit editing
 *      - user-defined circuit symbols
 *      - optional wrap mode
 *      - save circuit to file
 * v2.1:
 *      - auto-detect of screensize
 *
 * Compiled with Turbo C 1.5
 * date of last change: 01-16-93
 *
 * This code and program is copyrighted freeware. Send me a copy if you
 * change/improve it.
 */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <dos.h>

/* Cell state characters */
#define BACK  ' '
#define WIRE  '+'
#define EHEAD '#'
#define ETAIL '*'

#define MAXX   132 /* Absolute maximum number of columns */
#define MAXY   60 /* Absolute maximum number of rows */
#define DEFAULTX 80 /* default x,y size */
#define DEFAULTY 25
#define CMDCHAR '!' /* prefaces circuit symbol definitions */

/* interface keys */
#define ENTER 13
#define CTRL_ENTER 10
#define ESCAPE 27
#define DELETE 83
#define INSERT 82
#define UPARROW 72
#define DOWNARROW 80
#define LEFTARROW 75
#define RIGHTARROW 77
#define HOME 71
#define END 79
#define PAGE_UP 73
#define PAGE_DOWN 81
#define F1 59
#define F2 60
#define CTRL_W 23
#define CTRL_B 2
#define CTRL_T 20
#define CTRL_H 8
#define ALT_W 17
#define ALT_S 31
#define ALT_C 46
#define ALT_K 37
#define ALT_H 35

/* Macros - replacing functions for speed
 */
/****************************************************************************/
/* define pointer to screen memory for fast read/write
 */
/* CGA,EGA,VGA */
#define SCREENADDR ((unsigned far *) 0xb8000000L)
#define SCREEN(x,y) (char) SCREENADDR[((y)*80)+(x)]
/* fast text output to screen
 */
#define PUTSCREEN(xstart,ystart,text)\
        do {\
                int x=(xstart), y=(ystart), i=0;\
                char c;\
                while ((c=text[i++]) != 0)\
                        if (c=='\n') { ++y; x=xstart; }\
                        else SCREEN(x++,y)=c;\
        } while(0)


/****************************************************************************/
/* req: 0<x<MAXX, 0<y<MAXY,
 * ens: GETXY(x, y, field)=character at field[x, y]
 */
#define GETXY(x, y, field) *(field+((y)*DimX)+(x))

/****************************************************************************/
/* req: 0<x<MAXX, 0<y<MAXY,
 * ens: field[x,y]=c, display updated
 */
#define SETXY(c,x,y,field)  *(field+((y)*DimX)+(x))=SCREEN((x),(y))=c

/****************************************************************************/
/* req: 0<x<MAXX, 0<y<MAXY
 * ens: ENEIGHBORS()=# of neighbors that are electron heads
 */
#define ENEIGHBORS( x, y, field)\
(wrap ?\
    ((GETXY(                 x, (y-1+DimY) % DimY, field)==ehead)\
    +(GETXY(      (x+1) % DimX, (y-1+DimY) % DimY, field)==ehead)\
    +(GETXY(      (x+1) % DimX,                 y, field)==ehead)\
    +(GETXY(      (x+1) % DimX,      (y+1) % DimY, field)==ehead)\
    +(GETXY(                 x,      (y+1) % DimX, field)==ehead)\
    +(GETXY( (x-1+DimX) % DimX,      (y+1) % DimX, field)==ehead)\
    +(GETXY( (x-1+DimX) % DimX,                 y, field)==ehead)\
    +(GETXY( (x-1+DimX) % DimX, (y-1+DimY) % DimY, field)==ehead))\
    :\
    (                              (y && (GETXY( x  , y-1, field)==ehead))\
    +                  ((y != DimY-1) && (GETXY( x  , y+1, field)==ehead))\
    +             ((x != DimX-1) && y && (GETXY( x+1, y-1, field)==ehead))\
    +                  ((x != DimX-1) && (GETXY( x+1, y  , field)==ehead))\
    +((x != DimX-1) &&  (y != DimY-1) && (GETXY( x+1, y+1, field)==ehead))\
    +              (x && (y !=DimY-1) && (GETXY( x-1, y+1, field)==ehead))\
    +                              (x && (GETXY( x-1, y  , field)==ehead))\
    +                         (x && y && (GETXY( x-1, y-1, field)==ehead))))

/* global vars
 */
char wire=WIRE, ehead=EHEAD, etail=ETAIL, back=BACK, wrap=0;
int current=0, DimX, DimY, curx=0, cury=0;
char *helptext="\
WireWorld 2.1 (c) 1992,93 by Stefan Strack (stst@vuse.vanderbilt.edu)\n\
\n\
Usage: ww circuitfile (loads circuitfile)\n\
       ww             (brings up this help screen)\n\
\n\
Commands:                                        Settings:\n\
\n\
       <Enter>       singlestep run                     !WIRE %c\n\
       <Ctrl-Enter>  continuous run                     !HEAD %c\n\
       <Alt-H>,<F1>  display this help screen           !TAIL %c\n\
       <Alt-S>,<F2>  save screen to SCRNDMP.WW          !BACK %c\n\
       <Alt-K>       kill electrons                     !CRSR %d %d\n\
       <Alt-C>       clear circuit                      !WRAP %s\n\
       <Alt-W>       toggle wrap mode\n\
       <Arrow keys>  move cursor\n\
       <Insert>      toggle pen up/down\n\
       <Ctrl-W>      draw wire\n\
       <Ctrl-H>      draw electron head\n\
       <Ctrl-T>      draw electron tail\n\
       <Ctrl-B>      draw background\n\
       <Escape>      exit\n\
(any other key writes this key at the current cursor position)\n";


/* low-level cursor control
 */
/************************************************************************/
void SetCursor(char start, char end)
/* requires: none
 * ensures: cursor start, end lines set as given
 */
{
    struct REGPACK regs;
    regs.r_ax=0x0100;
    regs.r_cx=(start*256)+end;
    intr(0x10, &regs);
}

/************************************************************************/
void pcursor(int x, int y)
{
    union REGS regs;

    regs.h.ah=0x02;
    regs.h.bh=0;
    regs.h.dh=y;
    regs.h.dl=x;
    int86(0x10, &regs, &regs);
}

/****************************************************************************/
void dorules(char *old, char *new)
/* req: old is an initialized circuit
 * ens: rules run on old to produce new, display updated
 */
{
    int x,y,ne;
    char c;

    for (y=0; y<DimY; y++) {
        for (x=0; x<DimX; x++) {
            if ((c=GETXY( x, y, old))==wire) {
                ne=ENEIGHBORS(x ,y, old);
                if ((ne==1)||(ne==2))
                    SETXY( ehead, x, y, new);
                else SETXY( wire, x, y, new);
            }
            else if (c==ehead) SETXY( etail, x, y, new);
            else if (c==etail) SETXY( wire, x, y, new);
        }
    }
}

/****************************************************************************/
void settobg(char *field)
/* req: none
 * ens: field[0..MAX-1, 0..MAXY-1]=BACK
 */
{
    int i;
    for (i=0; i<(MAXX*MAXY); i++) *(field+i)=back;
}

/****************************************************************************/
int readfield(char *name, char field[2][MAXX*MAXY])
/* req: name=name of field file
 * ens: file 'name' read into field, field displayed, returns !0 on error
 */
{
    FILE *infile;
    int x,y,i,j;
    char  c, buf[100], param[10], strng[5];

    if ((infile=fopen(name, "rt"))==NULL) return 1;
    else
        for (y=0; y<MAXY && (fgets(buf,MAXX+2,infile) != NULL);) {
            if (buf[0]==CMDCHAR) {
                switch(toupper(buf[1])) {
                case 'H':  /* HEAD c */
                    sscanf(buf+2,"%*s %c",&ehead);
                    break;
                case 'T':  /* TAIL c */
                    sscanf(buf+2,"%*s %c",&etail);
                    break;
                case 'W':  /* WIRE c || WRAP ON/OFF */
                    if (toupper(buf[2])=='I')
                        sscanf(buf+2,"%*s %c",&wire);
                    else {
                        sscanf(buf+2,"%*s %s",strng);
                        if (toupper(strng[1])=='N')
                                wrap=1;
                        }
                    break;
                case 'B':  /* BACK c */
                    sscanf(buf+2,"%*s %c",&back);
                    break;
/* v2.1:        case 'S':     SIZE x y
   (obsolete)       sscanf(buf+2,"%*s %d %d",&i,&j);
                    if ((i <= MAXX) && (j <=MAXY)) {
                            DimX=i;
                            DimY=j;
                    }
                    break;  */
                case 'C':  /* CRSR x y */
                    sscanf(buf+2,"%*s %d %d",&curx,&cury);
                    break;
                } /* switch */
            }
            else {
                for(x=0; (c=buf[x]) != '\n' && c != '\0'; x++) {
                    SETXY(c,x,y,field[0]);
                    SETXY(c,x,y,field[1]);
                }
                ++y;
            }
        }
    fclose(infile);
    return 0;
}

/****************************************************************************/
void dumpscreen()
{
    int x,y;
    FILE *outfile;

    if ((outfile=fopen("scrndmp.ww","wt"))==NULL) {
        fputs("cannot write to SCRNDMP.WW\n",stderr);
        exit(1);
    }
    fprintf(outfile,"!HEAD %c\n",ehead);
    fprintf(outfile,"!TAIL %c\n",etail);
    fprintf(outfile,"!WIRE %c\n",wire);
    fprintf(outfile,"!BACK %c\n",back);
/*  fprintf(outfile,"!SIZE %d %d\n",DimX,DimY);  v2.1: obsolete */;
    fprintf(outfile,"!CRSR %d %d\n",curx,cury);
    fprintf(outfile,"!WRAP %s\n",(wrap ? "ON" : "OFF"));

    for(y=0;y<DimY;y++) {
        for(x=0;x<DimX;x++)
            fputc(SCREEN(x,y),outfile);
        fputc('\n',outfile);
    }
    fclose(outfile);
}

void displayhelp( char *new)
{
        int i,j;
        char helpcurrent[DEFAULTX*DEFAULTY];

        sprintf(helpcurrent,helptext,wire,ehead,etail,back /*,DimX,DimY*/
                ,curx,cury,(wrap ? "ON" : "OFF"));
        clrscr();
        pcursor(-1,-1);             /* hide cursor */
        PUTSCREEN(0,0,helpcurrent); /* fast text output */
        while(!bioskey(1)) ;        /* wait for keypress */
        for (i=0;i<DimX;i++)        /* restore circuit screen */
            for (j=0;j<DimY;j++)
                SCREEN(i,j)=GETXY(i,j,new);
}

/****************************************************************************/
void main( int argc, char *argv[])
{
    int x,y,current=0;
    char field[2][MAXX*MAXY],curchar=0,c,done=0,singlestep=1,pause=0,penup=1;

    /* v2.1: get current textmode as defaults */
    DimX=peekb(0x0040, 0x004A);
    if ((DimY=peekb(0x0040, 0x0084)+1) < DEFAULTY) DimY=DEFAULTY;

    clrscr();
    settobg(field[0]);
    settobg(field[1]);

    if (argc!=2) displayhelp(field[1]);

    else if (readfield(argv[1], field)) {
        fprintf(stderr,"error opening %s\n", argv[1]);
        exit(1);
    }

    pcursor(curx,cury); /* upper left */

    while (!done) {
        if (bioskey(1)||singlestep) {
            if ((c=getch())==0) {
                switch (getch()) {
                case UPARROW:
                    if (!penup)
                        curchar=SCREEN(curx,cury);
                    if (wrap) cury=(cury-1+DimY) % DimY;
                        else if (cury>0) --cury;
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case DOWNARROW:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) cury=(cury+1) % DimY;
                        else if (cury<DimY-1) ++cury;
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case LEFTARROW:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) curx=(curx-1+DimX) % DimX;
                        else if (curx>0) --curx;
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case RIGHTARROW:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) curx=(curx+1) % DimX;
                        else if (curx<DimX-1) ++curx;
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case HOME:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) {
                        curx=(curx-1+DimX) % DimX;
                        cury=(cury-1+DimY) % DimY;
                    }
                    else if (curx>0 && cury>0) {
                        --curx;
                        --cury;
                    }
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case END:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) {
                        curx=(curx-1+DimX) % DimX;
                        cury=(cury+1) % DimY;
                    }
                    else if (curx>0 && (cury<DimY-1)) {
                        --curx;
                        ++cury;
                    }
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case PAGE_UP:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) {
                        curx=(curx+1) % DimX;
                        cury=(cury-1+DimY) % DimY;
                        }
                    else if ((curx<DimX-1) && cury>0) {
                        ++curx;
                        --cury;
                    }
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case PAGE_DOWN:
                    if (!penup) curchar=SCREEN(curx,cury);
                    if (wrap) {
                        curx=(curx+1) % DimX;
                        cury=(cury+1) % DimY;
                    }
                    else if ((curx<DimX-1) && (cury<DimY-1)) {
                        ++curx;
                        ++cury;
                    }
                    if (!penup) {
                        SETXY(curchar,curx,cury,field[0]);
                        SETXY(curchar,curx,cury,field[1]);
                    }
                    pause=1;
                    break;
                case INSERT:
                    penup = !penup;
                    if (!penup) {
                        if (curchar==0) curchar=SCREEN(curx,cury);
                        else {
                                SETXY(curchar,curx,cury,field[0]);
                                SETXY(curchar,curx,cury,field[1]);
                        }
                        SetCursor(0,7);
                    }
                    else SetCursor(6,7);
                    pause=1;
                    break;
                case F2:
                case ALT_S:
                    dumpscreen();
                    break;
                case ALT_C:
                    for (x=0;x<DimX;x++)
                        for (y=0;y<DimY;y++)
                            SCREEN(x,y)=back;
                    settobg(field[0]);
                    settobg(field[1]);
                    pause=1;
                    break;
                case ALT_K:
                    for(x=0;x<DimX;x++)
                        for(y=0;y<DimY;y++)
                            if ((SCREEN(x,y)==ehead) || (SCREEN(x,y)==etail)) {
                                SETXY(wire,x,y,field[0]);
                                SETXY(wire,x,y,field[1]);
                            }
                    break;
                case ALT_W: /* WRAP ON/OFF */
                    wrap = !wrap;
                    if (wrap) {
                        curx=DimX/2;
                        cury=DimY/2;
                    }
                    else {
                        curx=0;
                        cury=0;
                    }
                    break;
                case F1:
                case ALT_H: /* Help */
                    displayhelp( field[1-current]);
                    break;
                } /*switch*/
                pcursor(curx,cury);
            } /* if c==0 */
            else switch(c) {
            case CTRL_ENTER:
                singlestep=0;
                pause=0;
                break;
            case ENTER:
                singlestep=1;
                pause=0;
                break;
            case ESCAPE:
                done=1;
                break;
            case CTRL_W:
                SETXY(curchar=wire,curx,cury,field[0]);
                SETXY(curchar,curx,cury,field[1]);
                pause=1;
                break;
            case CTRL_T:
                SETXY(curchar=etail,curx,cury,field[0]);
                SETXY(curchar,curx,cury,field[1]);
                pause=1;
                break;
            case CTRL_H:
                SETXY(curchar=ehead,curx,cury,field[0]);
                SETXY(curchar,curx,cury,field[1]);
                pause=1;
                break;
            case CTRL_B:
                SETXY(curchar=back,curx,cury,field[0]);
                SETXY(curchar,curx,cury,field[1]);
                pause=1;
                break;
            default:
                SETXY(curchar=c,curx,cury,field[0]);
                SETXY(curchar,curx,cury,field[1]);
                pause=1;
                break;
            } /* switch */
        } /* if bioskey(1) */
        if (!pause) {
            current=1-current;
            dorules(field[current], field[1-current]);
        }
    } /* while (!done) */
    clrscr();
}
