/*
Copyright (C) 1992 University of Maryland.

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies, and
that both that copyright notice and this permission notice appear
in supporting documentation.  The author makes no representations
about the suitability of this software for any purpose.  It is
provided "as is" without express or implied warranty. Your use of
this software signifies that you are willing to take the risks of
using this software by yourself.

THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE
USE OR PERFORMANCE OF THIS SOFTWARE.
*/

/* oca.c */

/* author: Hui-Hsien Chou
           Dept. of Computer Science
           University of Maryland, College Park
*/

/* This program reads in the compressed rule sets from ck, and use it */
 /* to guide the simulation of the cellular automata world. Details */
 /* has been provided in the technical report which accompanies this */
 /* package. */

#include <stdio.h>
#include <string.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include <X11/Intrinsic.h> 
#include <X11/Shell.h>
#include <X11/Xutil.h>
#include <Xm/Xm.h>
#include <Xm/PushB.h>
#include <Xm/FileSB.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawingA.h>
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>
#include <Xm/BulletinB.h>
#include "libXs.h"

/* The define and declartion of struct MyXSizeHints was used to cope */
 /* with the newer version of X11 system. The motif version we have */
 /* now doesn't support these functions to work with the Window */
 /* Manager, but the underlying X11 routines does. So I need to */
 /* declare my own version to talk directly with X11 routines. */
#define PMinSize        (1L << 4) /* program specified minimum size */
#define PResizeInc      (1L << 6) /* program specified resize increments */
#define PBaseSize       (1L << 8) /* program specified base for incrementing */

typedef struct {
        long flags;     /* marks which fields in this structure are defined */
        int x, y;               /* obsolete for new window mgrs, but clients */
        int width, height;      /* should set so old wm's don't mess up */
        int min_width, min_height;
        int max_width, max_height;
        int width_inc, height_inc;
        struct {
                int x;  /* numerator */
                int y;  /* denominator */
        } min_aspect, max_aspect;
        int base_width, base_height;            /* added by ICCCM version 1 */
        int win_gravity;                        /* added by ICCCM version 1 */
} MyXSizeHints;

/* define constants used by the program. */
#define NONE 0         /* five different direction constants */
#define FORWARD 1
#define BACKWARD 2
#define FASTFORWARD 3
#define FASTBACKWARD 4
#define WORLDX 200    /* world size. change it to get a bigger world. */
#define WORLDY 80     
#define SCANX 40      /* initial window size. don't change them. */
#define SCANY 20
#define HTABLESIZE 507 /* hash table size. */
#define TRACESIZE 3000 /* trace steps. */
#define INITIALSTATES 8 /* default state number. */
#define hashit(i) ((i)%HTABLESIZE)

/* node structure used in the rule hash table. */
typedef struct node {
  struct node *child, *sibling;
  int key, match;
} node;

/* pre-declaration of functions. */
void clear_world(), quit_oca(), redisplay(), resize();
void forward(), fast_forward(), backward(), fast_backward(), stop();
void selection(), post_select(), show_stat(), ask();
void disable_track(), track_area(), file_choice(), cancel_file();

/* window components. */
Widget toplevel, form, commands, info, canvas, dialog, choice;

/* the hash table and the recyle pool for nodes. */
node *start, *pool=0;

GC white, black; /* the black and white color for drawing. */

/* the simulated world. we need two world array to store the old and */
 /* new cell status. we use two pointers to the data structure so that */
 /* the program code can reference to them indirectly. this facilitate */
 /* the swapping of worlds. */
int world1[WORLDX][WORLDY], world2[WORLDX][WORLDY];
int (*old)[WORLDX][WORLDY], (*new)[WORLDX][WORLDY];

/* stat. variables. */
int epoch=0, rules=0, trace=0;

/* option variables. */
int sym_symbols=0;

/* no. of states in the simulated world. */
int states=INITIALSTATES;

/* the trace board for storing simulation progresses. */
struct {
  long value;
  int change;
} traceboard[TRACESIZE];

/* trace file for storing simulation progresses. */
FILE *tracefile;

/* convenient place to compose program window name. */
char wname[50]; 

/* misc. variables. */
int selectx, selecty, pastx, pasty, no_care;
int direction, rangex = -1, rangey = -1;
int xori=WORLDX/2-SCANX/2, yori=WORLDY/2-SCANY/2;
int xscan=SCANX, yscan=SCANY;
int action;
char *pname;

/* the clipboard for storing things cut from the world. */
int clipold[WORLDX][WORLDY];

/* the menu for file operations. */
static xs_menu_struct file_menu[] = {
  {"Load", ask, (caddr_t)1, NULL, 0, NULL},
  {"Save", ask, (caddr_t)2, NULL, 0, NULL},
  {"Place Init", ask, (caddr_t)3, NULL, 0, NULL},
  {"Export Init", ask, (caddr_t)4, NULL, 0, NULL},
  {NULL, NULL, NULL, NULL, 0, NULL},
  {"Quit", quit_oca, NULL, NULL, 0, NULL}
};

/* the stat selector. */
static xs_menu_struct select_menu[] = {
  {".", selection, (caddr_t)0, NULL, 0, NULL},
  {"*", selection, (caddr_t)1, NULL, 0, NULL},
  {"L", selection, (caddr_t)2, NULL, 0, NULL},
  {"+", selection, (caddr_t)3, NULL, 0, NULL},
  {"-", selection, (caddr_t)4, NULL, 0, NULL},
  {"X", selection, (caddr_t)5, NULL, 0, NULL},
  {"O", selection, (caddr_t)6, NULL, 0, NULL},
  {"#", selection, (caddr_t)7, NULL, 0, NULL},
  {";", selection, (caddr_t)8, NULL, 0, NULL},
  {":", selection, (caddr_t)9, NULL, 0, NULL},
};

char *symbols[] = {
".", "*", "L", "+", "-", "X", "O", "#", ";", ":"
};

/* here comes the main program. */
main(argc, argv)
  int   argc;
  char *argv[];
{
  Arg          wargs[10];
  int          n, i, j;
  XGCValues    values;
  MyXSizeHints hint;
  XmString     name;
  Widget temp;

  /* initialize everything. */
  old = world1; new = world2;
  tracefile = tmpfile();
  start = 0;
  clear_world();

  /* parse command arguments. */
  for (i=1; i<argc; i++) {
    if (*argv[i] == '-')
      switch (argv[i][1]) {
      case 'r':
	sym_symbols = 1;
	continue;
      case 'n':
	i++;
	for (j=0; j<10 && argv[i][j]!='\0'; j++)
	  *(symbols[j]) = *(select_menu[j].name) = argv[i][j];
	if (argv[i][j]!='\0')
	  printf("%s: extra symbols %s ignored (max. 10 states allowed)!\n",
		 argv[0], &(argv[i][j]));
	states = j;
	continue;
      }
    printf("%s: unknown option %s ignored.\n", argv[0], argv[i]);
  }
  
  /* the main window. */
  argc=1;
  toplevel = XtInitialize(argv[0], "OCA", NULL, 0,
                          &argc, argv);

  n = 0;  /* setting the size of main window. */
  XtSetArg(wargs[n], XmNwidth, 246); n++;
  XtSetArg(wargs[n], XmNheight, 307); n++;
  XtSetValues(toplevel, wargs, n);

  /* create the parent of all subwindows in main window. */
  form = XtCreateManagedWidget("form",
			xmPanedWindowWidgetClass,
			toplevel, NULL, 0);

  /* create the information widget. */
  n = 0;
  XtSetArg(wargs[n], XmNshadowThickness, 0); n++;
  XtSetArg(wargs[n], XmNmarginHeight, 0); n++;
  XtSetArg(wargs[n], XmNmarginWidth, 0); n++;
  XtSetArg(wargs[n], XmNeditable, FALSE); n++;
  XtSetArg(wargs[n], XmNautoShowCursorPosition, FALSE); n++;
  XtSetArg(wargs[n], XmNcursorPositionVisible, FALSE); n++;
  info = XtCreateManagedWidget("info", xmTextWidgetClass, form, wargs, n);

  /* create the menu bar. */
  commands = XmCreateMenuBar(form, "commands", NULL, 0);
  XtManageChild(commands);
  temp = XmCreatePulldownMenu(commands, NULL, NULL, 0);
  XtSetArg(wargs[0], XmNsubMenuId, temp);
  XtCreateManagedWidget("File", xmCascadeButtonWidgetClass,
			commands, wargs, 1);
  xs_create_menu_buttons(NULL, temp, file_menu, XtNumber(file_menu));
  temp = XtCreateManagedWidget("<--", xmCascadeButtonWidgetClass, commands,
			NULL, 0);
  XtAddCallback(temp, XmNactivateCallback, fast_backward, 1);
  temp = XtCreateManagedWidget("<-", xmCascadeButtonWidgetClass, commands,
			NULL, 0);
  XtAddCallback(temp, XmNactivateCallback, backward, 1);
  temp = XtCreateManagedWidget("Stop", xmCascadeButtonWidgetClass, commands,
			NULL, 0);
  XtAddCallback(temp, XmNactivateCallback, stop, 1);
  temp = XtCreateManagedWidget("->", xmCascadeButtonWidgetClass, commands,
			NULL, 0);
  XtAddCallback(temp, XmNactivateCallback, forward, 1);
  temp = XtCreateManagedWidget("-->", xmCascadeButtonWidgetClass, commands,
			NULL, 0);
  XtAddCallback(temp, XmNactivateCallback, fast_forward, 1);

  /* create the display part */
  canvas = XtCreateManagedWidget("canvas", 
                                 xmDrawingAreaWidgetClass, 
                                 form, NULL, 0);
  XtAddCallback(canvas, XmNexposeCallback, redisplay, 0);
  XtAddCallback(canvas, XmNresizeCallback, resize, 0);

  /* create the state selection popup menu. */
  XtSetArg(wargs[0], XmNwhichButton, Button1);
  choice = XmCreatePopupMenu(canvas, "popup", wargs, 1);
  XtAddEventHandler(canvas, ButtonPressMask, FALSE, post_select,
		    NULL);
  xs_create_menu_buttons(NULL, choice, select_menu,
			 states); 

  /* create file selection popup menu. */
  n = 0;
  XtSetArg(wargs[n], XmNdirMask, XmStringCreate("*.world",
						XmSTRING_DEFAULT_CHARSET)); n++;
  dialog = XmCreateFileSelectionDialog(canvas, "dialog", wargs, n);
  XtAddCallback(dialog, XmNokCallback, file_choice, 0);
  XtAddCallback(dialog, XmNcancelCallback, cancel_file, 0);
  XtUnmanageChild(XmFileSelectionBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));

  /* create the white and blank colors for showing molecules. */
  n = 0;
  XtSetArg(wargs[n], XtNforeground, &values.foreground);n++;
  XtSetArg(wargs[n], XtNbackground, &values.background);n++;
  XtGetValues(canvas, wargs, n);
  white = XtGetGC(canvas, GCForeground | GCBackground, &values);
  n = values.foreground; values.foreground = values.background;
  values.background = n;
  black = XtGetGC(canvas, GCForeground | GCBackground, &values);
    
  XtRealizeWidget(toplevel);

  /* tell the window manager what we like our window to be resized. */
  hint.min_width = 78; hint.min_height = 79;
  hint.width_inc = 6; hint.height_inc = 12;
  hint.base_width = 6; hint.base_height = 67;
  hint.flags = PMinSize|PResizeInc|PBaseSize;
  XSetWMNormalHints(XtDisplay(toplevel), XtWindow(toplevel), &hint);

  /* name the program window. */
  sprintf(wname, "%s: Untitiled", argv[0]);
  XStoreName(XtDisplay(toplevel), XtWindow(toplevel), wname);
  pname=strdup(argv[0]);

  /* grab the sprite when a button is pressed. */
  XGrabButton(XtDisplay(canvas), AnyButton, AnyModifier,
	      XtWindow(canvas), True,
	      ButtonPressMask | ButtonMotionMask | ButtonReleaseMask,
	      GrabModeAsync, GrabModeAsync, XtWindow(canvas),
	      XCreateFontCursor(XtDisplay(canvas), XC_crosshair));

  /* then wait for user response. */
  show_stat();
  XtMainLoop();
}

/* redraw the window. */
void redisplay (w, data, call_data)
Widget w;
caddr_t data;
XmDrawingAreaCallbackStruct *call_data;
{
  register int i, j, x, y;
  int value;

  /* we only catch the last redisplaying events in a sequence. */
  if (call_data && call_data->event->xexpose.count) return;

  /* then clear the window. */
  if (data==NULL)
    XClearArea(XtDisplay(canvas), XtWindow(canvas), 0, 0, 0, 0, FALSE);

  /* then show all molecules within the window. */
  for (i=0, x=xori; i<xscan; i++, x++) 
    for (j=0, y=yori; j<yscan; j++, y++) {
      if ((*new)[x][y]>=100)
	XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
				   (j+1)*12, symbols[(*old)[x][y]], 1);
      else if (value = (*old)[x][y])
	XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
				   (j+1)*12, symbols[value], 1);
    }
}

/* redraw the part of the screen which has changed. */
void redraw()
{
  register int i, j, x, y, c, v;

  for (i=0, x=xori; i<xscan; i++, x++) 
    for (j=0, y=yori; j<yscan; j++, y++)
      if ((c=(*new)[x][y])>=100)
	XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			 (j+1)*12, symbols[(*old)[x][y]], 1);
      else if (c != (v=(*old)[x][y]))
	XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			 (j+1)*12, v ? symbols[v] : " ", 1);
}

/* recalculate some values when a window is resized. */
void resize(w, data, call_data)
Widget w;
caddr_t data;
XmDrawingAreaCallbackStruct *call_data;
{
  Arg wargs[10];
  int i, n;
  Dimension widget_width, widget_height;

  /* see if the window has been realized. */
  if (XtIsRealized(canvas)==0) return;

  n = 0;
  XtSetArg(wargs[n], XtNwidth,  &widget_width);n++;
  XtSetArg(wargs[n], XtNheight, &widget_height);n++;
  XtGetValues(canvas, wargs, n);

  yscan = widget_height/12;
  yori = WORLDY/2 - yscan/2;
  xscan = widget_width/6;
  xori = WORLDX/2 - xscan/2;

  /* clear the window. */
  XClearArea(XtDisplay(canvas), XtWindow(canvas), 0, 0, 0, 0, TRUE);
}

/* Show statistic on screen. */
void show_stat()
{
  char buf[100];

  sprintf(buf, "epoch=%d      rules=%d      trace=%d", epoch, rules, trace);
  XmTextSetString(info, buf);
}

/* clear the world data structure. */
void clear_world()
{
  int i, j;

  for (i=1; i<WORLDX-1; i++) 
    for (j=1; j<WORLDY-1; j++) {
      (*old)[i][j]=0; (*new)[i][j]=0;
    }
}

/* determine the next state for cell(i, j). */
int change(i, j)
int i, j;
{
  int buf[5];
  node *stack[5], *p;

  buf[0] = (*old)[i][j];
  buf[1] = (*old)[i][j-1];
  buf[2] = (*old)[i+1][j];
  buf[3] = (*old)[i][j+1];
  buf[4] = (*old)[i-1][j];
  for (stack[0]=start, i=0; i>=0; i--) {
    for (p=stack[i]; p; )
      if (p->key==buf[i] || p->key==no_care) {
        if (p->match)
          return p->child->key;
        stack[i] = p->sibling;
        i++;
        stack[i] = p->child;
        p = stack[i];
      } else 
        p = p->sibling;
  }
  return buf[0];
}

/* run one epoch for the whole world. */
void iterate()
{
  int i, j;

  for (i=1; i<WORLDX-1; i++) 
      for (j=1; j<WORLDY-1; j++)
	(*new)[i][j]=change(i, j);
}

/* quit the program. */
void quit_oca()
{
  XtCloseDisplay(XtDisplay(toplevel));
  fclose(tracefile);
  exit(0);
}

/* obtain a new node for transition tree. */
node *newnode(c, s, k, m)
node *c, *s;
int k, m;
{
  node *ptr;

  if (pool) {
    ptr = pool;
    pool = pool->sibling;
  } else
    ptr=(node*)malloc(sizeof(node));
  ptr->child=c; ptr->sibling=s;
  ptr->key=k; ptr->match=m;
  return ptr;
}

void killtree(p)
node *p;
{
  if (p) {
    killtree(p->sibling);
    killtree(p->child);
    p->sibling = pool;
    pool = p;
  }
}

int no_more(b, i)
char *b;
int i;
{
  for (++i; i<5; i++)
    if (b[i] != 11)
      return 0;
  return 1;
}

/* generate one rotation to the right. */
void rotate(a)
char *a;
{
  int i; char tmp;

  tmp = a[1];
  for (i=1; i<4; i++)
    a[i] = a[i+1];
  a[4] = tmp;

  if (sym_symbols) {
    for (i=0; i<6; i++)
      if (a[i]>1 && a[i] < 5)
        a[i]--;
      else if (a[i]==1)
        a[i] = 4;
  }
}

/* doing one interation and update the tracefile and screen. */
int proceed()
{
  int c, i, j, v;
  int (*temp)[WORLDX][WORLDY];

  iterate();
  traceboard[trace].value = ftell(tracefile);
  for (c=0, i=1; i<WORLDX-1; i++)
    for (j=1; j<WORLDY-1; j++)
      if ((v = (*old)[i][j]) != (*new)[i][j]){
	fprintf(tracefile, "%c%c%c", (char)i, (char)j, (char)v);
	c++;
      }
  traceboard[trace++].change = c;
  epoch++;
  temp = old; old = new; new = temp;
  show_stat();
  redraw();
  if (direction==FASTFORWARD)
    return FALSE;
  return TRUE;
}

/* doing one backtrack and update the screen. */
int backup()
{
  int d, hashvalue;
  long v;
  node *ptr;
  unsigned char i, j, c;
  int (*temp)[WORLDX][WORLDY];

  /* if nothing to backup, just return. */
  if (trace<=0)
    return TRUE;
  trace--;
  d = traceboard[trace].change;
  if (d>WORLDX*WORLDY)
    d -= (WORLDX*WORLDY+1); 
  else
    epoch--;
  for (i=1; i<WORLDX-1; i++)
    for (j=1; j<WORLDY-1; j++)
	(*new)[i][j] = (*old)[i][j];
  fseek(tracefile, traceboard[trace].value, 0);
  for (; d; d--) {
    fscanf(tracefile, "%c%c%c", &i, &j, &c);
    (*new)[i][j] = c;
  }
  fseek(tracefile, traceboard[trace].value, 0);
  temp = old; old = new; new = temp;
  redraw();
  show_stat();
  if (direction==FASTBACKWARD)
    return FALSE;
  return TRUE;
}

/* input a new rule or change the world. */
void selection(w, d, c)
Widget w;
int d;
XmAnyCallbackStruct *c;
{
  int i, j, v;

  i = xori+selectx; j = yori+selecty;
  traceboard[trace].value = ftell(tracefile);
  v = (*old)[i][j];
  fprintf(tracefile, "%c%c%c", (char)i, (char)j, (char)v);
  traceboard[trace++].change = WORLDX*WORLDY+2;
  (*old)[i][j] = d;
  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, selectx*6,
		   (selecty+1)*12, symbols[d], 1);
}

void disable_track(w, d, e)
Widget w;
Widget d;
XEvent *e;
{
  int i, j, x, y, c, v;

  XtRemoveEventHandler(canvas, ButtonReleaseMask, False, disable_track, NULL);
  XtRemoveEventHandler(canvas, ButtonMotionMask, False, track_area, NULL);
  rangex = pastx-selectx+1; rangey = pasty-selecty+1;
  traceboard[trace].value = ftell(tracefile);
  for (c=1, x=xori+selectx, i=0; i<rangex; i++, x++)
    for (y=yori+selecty, j=0; j<rangey; j++, y++) {
      v = clipold[i][j]  = (*old)[x][y];
      fprintf(tracefile, "%c%c%c", (char)x, (char)y, (char)v);
      (*old)[x][y] = (*new)[x][y] = 0;
      c++;
    }
  traceboard[trace++].change = c+(WORLDX*WORLDY+1);
  show_stat();
  redisplay(NULL, NULL, NULL);
}

void track_area(w, d, e)
Widget w;
Widget d;
XEvent *e;
{
  int nowx, nowy;
  int i, j, x, y, v;

  nowx = (e->xmotion.x) / 6;
  nowy = (e->xmotion.y-2) / 12;
  if (nowx>pastx)
    for (x=xori+pastx+1, i=pastx+1; i<=nowx; i++, x++)
      for (y=yori+selecty, j=selecty; j<=nowy; j++, y++)
	if ((*new)[x][y]>=100)
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, symbols[(*old)[x][y]], 1);
	else if (v=(*old)[x][y])
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, symbols[v], 1);
	else
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, " ", 1);
  else if (nowx<pastx)
    for (x=xori+nowx+1, i=nowx+1; i<=pastx; i++, x++)
      for (y=yori+selecty, j=selecty; j<=pasty; j++, y++)
	if ((*new)[x][y]>=100)
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, symbols[(*old)[x][y]], 1);
	else if (v=(*old)[x][y])
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, symbols[v], 1);
	else
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, " ", 1);
  if (nowy>pasty)
    for (x=xori+selectx, i=selectx; i<=nowx; i++, x++)
      for (y=yori+pasty+1, j=pasty+1; j<=nowy; j++, y++)
	if ((*new)[x][y]>=100)
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, symbols[(*old)[x][y]], 1);
	else if (v=(*old)[x][y])
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, symbols[v], 1);
	else
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, " ", 1);
  else if (nowy<pasty)
    for (x=xori+selectx, i=selectx; i<=pastx; i++, x++)
      for (y=yori+nowy+1, j=nowy+1; j<=pasty; j++, y++)
	if ((*new)[x][y]>=100)
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, i*6,
			   (j+1)*12, symbols[(*old)[x][y]], 1);
	else if (v=(*old)[x][y])
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, symbols[v], 1);
	else
	  XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, i*6,
			   (j+1)*12, " ", 1);
  pastx = nowx; pasty = nowy;
} 

/* act according to which button has been pressed. */
void post_select(w, m, e)
Widget w;
Widget m;
XEvent *e;
{
  int x, y, i, j, c, v;

  selectx = (e->xbutton.x) / 6;
  selecty = (e->xbutton.y-2) / 12;
  if (e->xbutton.button == Button1) {
    /* it's button 1, just post the selection menu for rule inputs. */
    XmMenuPosition(choice, e);
    XtManageChild(choice);
  } else if (e->xbutton.button == Button2) {
    /* it's button 2, prepare for tracking the mouse. */
    pastx = selectx; pasty = selecty;
    if ((*new)[xori+pastx][yori+pasty]>=100)
      XDrawImageString(XtDisplay(canvas), XtWindow(canvas), white, pastx*6,
		       (pasty+1)*12, 
		       symbols[(*old)[xori+pastx][yori+pasty]], 1);
    else if (v=(*old)[xori+pastx][yori+pasty])
      XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, pastx*6,
		       (pasty+1)*12, 
		       symbols[v], 1);
    else
      XDrawImageString(XtDisplay(canvas), XtWindow(canvas), black, pastx*6,
		       (pasty+1)*12, " ", 1);
    XtAddEventHandler(canvas, ButtonMotionMask, False, track_area, NULL);
    XtAddEventHandler(canvas, ButtonReleaseMask, False, disable_track, NULL);
  } else {
    /* it's button 3, paste the content of clipboard to the world. */
    if (rangex<=0 || rangey<=0) return;
    traceboard[trace].value = ftell(tracefile);
    for (c=1, x=xori+selectx, i=0; i<rangex; i++, x++)
      for (y=yori+selecty, j=0; j<rangey; j++, y++)
	if ((v = (*old)[x][y]) != clipold[i][j]){
	  fprintf(tracefile, "%c%c%c", (char)x, (char)y, (char)v);
	  (*new)[x][y] = (*old)[x][y];
	  (*old)[x][y] = clipold[i][j];
	  c++;
	}
    traceboard[trace++].change = c+(WORLDX*WORLDY+1);
    show_stat();
    redraw();
  }
}
  
int svalue(i)
char i;
{
  int j;

  if (i==' ')
    return 0;
  if (i=='_')
    return 11;
  for (j=0; j<states; j++)
    if (*symbols[j]==i) 
      return j;
  fprintf(stderr, "%s: unrecognized symbol %c in svalue!\n", pname, i);
  exit(1);
}

void load(world, rfile, tfile, sfile)
char *world, *rfile, *tfile, *sfile;
{
  FILE *f;
  int i, j, x, y, hashvalue, d;
  long v;
  node *p;
  int (*temp)[WORLDX][WORLDY];
  Arg wargs[1];
  char c, t, r, b, l, a, buf[100];
  char n, ne, e, se, s, sw, w, nw;

  trace=0; epoch=0; rules=0;
  killtree(start);
  start = newnode(0, 0, 0, 0);
  f = fopen(rfile, "r");
  while (fgets(buf, 100, f)) {
    for (j=0; j<5; j++)
      buf[j] = svalue(buf[j]);
    buf[5] = svalue(buf[9]);
    for (j=0; j<4; j++) {
      for (i=0, p=start; i<5; i++) {
        v = buf[i];
        if (p->child == 0)
          p->child = newnode(0, 0, v, 0);
        for (p = p->child; p->sibling; p = p->sibling) 
          if (p->key == v)
            break;
        if (p->key != v) {
          p->sibling = newnode(0, 0, v, 0);
          p = p->sibling;
        }
        if (no_more(buf, i)) {
          p->match = 1;
          if (p->child)
            fprintf(stderr, "hide out found: %d, %d\n", rules+1, i);
          p->child = newnode(0, 0, buf[5], 0);
          break;
        }
      }
      rotate(buf);
    }
    rules++;
  }
  fclose(f);
  p = start;
  start = start->child;
  p->sibling = pool;
  pool = p;
  no_care = svalue('_');
  if (f=fopen(world, "r")) {
    for (j=1; j<WORLDX-1; j++)
      for (d=1; d<WORLDY-1; d++) {
	fscanf(f, "%c", &a);
	(*old)[j][d] = svalue(a);
      }
    fclose(f);
  } else
    clear_world();
  if (f=fopen(sfile, "r")) {
    while (fgets(buf, 100, f)) {
      sscanf(buf, "%ld %d", &v, &d);
      traceboard[trace].value = v;
      traceboard[trace++].change = d;
      if (d<=(WORLDX*WORLDY))
	epoch++;
    }
    fclose(f);
    rewind(tracefile);
    f=fopen(tfile, "r");
    fread(&v, sizeof(long), 1, f);
    while (i=fread(buf, sizeof(char), 100, f)) 
      fwrite(buf, sizeof(char), i, tracefile);
    fclose(f);
    fseek(tracefile, v, 0);
  }
  show_stat();
  redisplay(NULL, NULL, NULL);
}

void save(world, tfile, sfile)
char *world, *tfile, *sfile;
{
  int i, j, d;
  long v;
  FILE *f;
  char c, t, r, b, l, buf[100];
  int st;
  char n, ne, e, se, s, sw, w, nw;

  f = fopen(sfile, "w");
  for (i=0; i<trace; i++) {
    d=traceboard[i].change;
    v = traceboard[i].value;
    fprintf(f, "%ld %d\n", v, d);
  }
  fclose(f);
  f = fopen(world, "w");
  for (j=1; j<WORLDX-1; j++)
    for (d=1; d<WORLDY-1; d++)
      fprintf(f, "%c", *symbols[(*old)[j][d]]);
  fclose(f);
  f = fopen(tfile, "w");
  v = ftell(tracefile);
  rewind(tracefile);
  fwrite(&v, sizeof(long), 1, f);
  while (i=fread(buf, sizeof(char), 100, tracefile))
    fwrite(buf, sizeof(char), i, f);
  fclose(f);
  fseek(tracefile, v, 0);
}

void place(init)
char *init;
{
  FILE *f;
  int i, j, d;
  char buf[100];

  f = fopen(init, "r"); 
  for (i=d=0; fgets(buf, 100, f); i++) {
    for (j=0; buf[j] && buf[j]!='\n'; j++) 
      clipold[j][i] = svalue(buf[j]);
    if (j>d)
      d=j;
  }
  rangex = d; rangey =i;
  fclose(f);
}

void export(init)
char *init;
{
  int i, j, found, minx, miny, maxx, maxy;
  FILE *f;

  minx=WORLDX; miny=WORLDY; maxx=0; maxy=0;
  for (j=1; j<WORLDY-1; j++) {
    found = 0;
    for (i=1; i<WORLDX-1; i++)
      if ((*old)[i][j]) {
	found = 1;
	if (i<minx) minx=i;
	if (i>maxx) maxx=i;
      }
    if (found) {
      if (j<miny) miny=j;
      if (j>maxy) maxy=j;
    }
  }
  f = fopen(init, "w");
  for (j=miny; j<=maxy; j++) {
    for (i=minx; i<=maxx; i++)
      fprintf(f, "%c", *symbols[(*old)[i][j]]);
    fprintf(f, "\n");
  }
  fclose(f);
}

void ask(w, data, call_data)
Widget w;
int data;
XmAnyCallbackStruct *call_data;
{
  action = data;
  XtManageChild(dialog);
}

void cancel_file()
{
  XtUnmanageChild(dialog);
}

void file_choice(w, data, call_data)
Widget w;
caddr_t data;
XmFileSelectionBoxCallbackStruct *call_data;
{
  char *ptr1, *ptr2;
  char buf1[150], buf2[150], buf3[150], buf4[150];

  XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET, &ptr1);
  if (ptr2=strstr(ptr1, ".world")) {
    if (strcmp(ptr2, ".world")==0)
      *ptr2 = '\0';
  } else if (ptr2=strstr(ptr1, ".init")) {
    if (strcmp(ptr2, ".init")==0)
      *ptr2 = '\0';
  } else if (ptr2=strstr(ptr1, ".rules.cp")) {
    if (strcmp(ptr2, ".rules.cp")==0)
      *ptr2 = '\0';
  } else 
    ptr2= ptr1+strlen(ptr1);
  for (ptr2--; ptr2>ptr1 && *ptr2!='/'; ptr2--);
  sprintf(buf1, "%s: %s", pname, ptr2+1);
  XStoreName(XtDisplay(toplevel), XtWindow(toplevel), buf1);
  sprintf(buf1, "%s.world", ptr1);
  sprintf(buf2, "%s.rules.cp", ptr1);
  sprintf(buf3, "%s.trace", ptr1);
  sprintf(buf4, "%s.stamp", ptr1);
  switch (action) {
  case 1:
    load(buf1, buf2, buf3, buf4);
    break;
  case 2:
    save(buf1, buf3, buf4);
    break;
  case 3:
    sprintf(buf1, "%s.init", ptr1);
    place(buf1);
    break;
  case 4:
    sprintf(buf1, "%s.init", ptr1);
    export(buf1);
    break;
  default:
    fprintf(stderr, "%s: fatal error in file_choice()\n", pname);
    exit(1);
  }
  XtUnmanageChild(dialog);
}

void forward()
{
  direction = FORWARD;
  proceed();
}

void fast_forward()
{
  direction = FASTFORWARD;
  XtAddWorkProc(proceed, 0);
}

void backward()
{
  direction = BACKWARD;
  backup();
}

void fast_backward()
{
  direction = FASTBACKWARD;
  XtAddWorkProc(backup, 0);
}

void stop()
{
  direction = NONE;
}
