/*****
 *
 *
 * File: cell.h
 *
 * Cellular automata simulator
 * Includes, definitions, and variable declarations/definitions.
 *
 *
 *****/

/*
 *
 * Cellsim copyright 1989, 1990 by Chris Langton and Dave Hiebeler
 * (cgl@lanl.gov, hiebeler@heretic.lanl.gov)
 *
 * This package may be freely distributed, as long as you don't:
 * - remove this notice
 * - try to make money by doing so
 * - prevent others from copying it freely
 * - distribute modified versions without clearly documenting your changes
 *   and notifying us
 *
 * Please contact either of the authors listed above if you have questions
 * or feel an exception to any of the above restrictions is in order.
 *
 * If you make changes to the code, or have suggestions for changes,
 * let us know!  If we use your suggestion, you will receive full credit
 * of course.
 */

/*****
 * Cellsim history:
 *
 * Cellsim was originally written on Apollo workstations by Chris Langton.
 *
 * Sun versions:
 *
 * - version 1.0
 *   by C. Ferenbaugh and C. Langton
 *   released 09/02/88
 *
 * - version 1.5
 *   by Dave Hiebeler and C. Langton  May - June 1989
 *   released 07/03/89
 *
 * - version 2.0
 *   by Dave Hiebeler and C. Langton  July - August 1989
 *   never officially released (unofficially released 09/08/89)
 *
 * - version 2.5
 *   by Dave Hiebeler and C. Langton  September '89 - February 1990
 *   released 02/26/90
 *****/



/*****
 *
 * Includes
 *
 *****/
  
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/time.h>

#include "cell.def"


/*****
 *
 * Defines
 *
 *****/

/* Used for the "Quick random" image generation */
#define RSIZE (1 << 24)
#define RMASK 0xFFFFFF			/* RSIZE - 1 */

/* Length of longest possible user-defined "sequence" */
#define MAX_SEQUENCE_LENGTH 128		/* longest sequence possible */
#define NUM_SEQUENCES	5		/* number of sequences */

#define CELLSIM_PORT 12700


/*****
 *
 * Typedefs
 *
 *****/

typedef unsigned char ((*func_ptr)());
typedef void ((*void_func_ptr)());
typedef int ((*int_func_ptr)());
typedef unsigned char byte;


/*****
 *
 * Function declarations
 *
 *****/

void
    which(),	/* look up full pathnames */
    quit(), quit2();	/* quit with error msg */

double
    compute_lambda();	/* compute current lambda value from LT */

void_func_ptr
    default_segv_func, emergency_LT_save_ptr;
void
    emergency_LT_save();


/*****
 *
 * External function declarations
 *
 *****/

extern char *getenv();		/* system stuff to get environment vars */
extern and_color(), and_planemask();
extern void def_CM_get_settings();



/*****
 *
 * External variables
 *
 *****/

/*** System stuff ***/
extern char *sys_errlist[];	/* System error msgs */
extern int errno;		/* Index into system error msgs */


/*** General variable ***/
extern State
    *ca, *ia;			/* current and image arrays */
extern int
    parm1, parm2,	/* extra parameters for computed-fcns */
    auto_image_change, auto_nhood_change,  /* whether or not to automatically
					 * switch nhood and image-size when
					 * reading in files */
    use_2D_always;	/* load/save 2-D images, even when in 1-D mode */

/*** CM-related variables ***/
extern int
    use_Sun_disp,	/* whether or not to use Sun display when using CM */
    use_FB,
    CM_disp_interval,	/* for skipping frames when running w/ display */
    CM_delay,	/* used to slow down CM */
    CM_zoom, CM_panx, CM_pany;


/*****
 *
 * Global variables
 *
 *****/

/*** Basic Parameters ***/
short
    S = 8,				/* # states per cell */
    L,					/* log S */
    B = 64,				/* array size */
    N,					/* number of neighbors */
    NHOOD = NH_VONN,			/* type of nhood in use */
    DIM,				/* dimension (1 or 2) */
    R = 1;				/* radius of neighborhood */
int
    table_symmetry = TRUE;		/* if true, store rotations of rules */


/*** Misc. related variables ***/
short
    BM1,				/* B - 1      */
    BPL1,				/* B + 1      */
    TBM1,				/* 2 * B - 1  */
    ABASE,				/* array base */
    RCOUNT;				/* B          */
int
    BSQR,				/* B * B      */
    BBUF,				/* BSQR + B   */
    ASIZE,				/* with buffers */
    ICBASE,				/* start of initial config. */
    ICSIZE;				/* size of initial config.  */

int
    TSIZE,				/* t-table size: S^N */
    TMASK,				/* t index mask */
    TMASK2;				/* second index mask */
short
    AMASK;				/* log S 1-bits */

/*** Socket vars ***/
int
    SOCKET_INUSE = FALSE,		/* boolean */
    portnum;				/* port # of receiving socket */
char
    hostname[BUFLEN] = "";		/* host of receiving machine */

/*** Pointers to nhood-specific routines ***/
int     (*get_address) (), (*to_nhood) (), (*to_index) (), (*rotate) (),
        (*auto_step) (), (*auto_screen) (), (*learn) ();

/*** State-variables, etc ***/
short
    setting_sequence,	/* 0 if not setting sequence, otherwise index into
			 * sequence table */
    sequence_length[NUM_SEQUENCES];	/* length of sequence */
int
    drawing = FALSE;	/* whether or not we're currently drawing something */


/*** Misc configuration vars ***/
byte
    save_images_compressed = 1,	/* save images in compressed form? */
    save_LT_compressed = 1;
short
    stop_on_mode,	/* when to automatically stop running, if ever */
    closeup,		/* in closeup mode? */
    shift_dist=8;	/* when user shifts image */
int
    blackwhite = FALSE,	/* boolean indicating display type */
    mag,		/* current magnification level */
    side,		/* how many rows/cols are currently displayed */
    xstart, ystart,	/* start coords in image, for displaying */
    MAXSIZE=512,	/* maximum allowed image size */
    save_in_image_dir=0,  /* save images in image-dir (instead of cwd) */
    save_in_LT_dir=0,	/* similarly for lookup tables */
    save_in_cmap_dir=0,   /* and cmaps */
    function,		/* using computed-function? */
    use_CM;		/* whether or not we're using (attached to) the CM */
char
    Image_dir[256],	/* directory to load/save images in */
    Table_dir[256],	/* similarly for LT's */
    Cmap_dir[256],	/* and cmaps */
    Fcn_dir[256],	/* and ".o" files (loading only, since can't save */
    current_working_dir[256];	/* current working directory */

/*** Misc global vars ***/
State
    *ta;		/* lookup table */
char
    buf1[BUFLEN], buf2[BUFLEN];
int
    stime = 0,		/* system time */
    *statecount;	/* state counter */
FILE
    *fp;		/* used to read/write files */

/*** Saved defaults for various things ***/
int
    saved_mag_factor,	/* to remember magnification during close-up */
    saved_skip = INTUNDEF,	/* interval when doing a "skip" */
    saved_display_interval = INTUNDEF,	/* display interval with "skip run" */
    saved_num_steps = INTUNDEF,	/* bound with "bound" or "skip/bound" runs */
    saved_side_length = 0, /* size of resized images when loading/saving */
    saved_rule_number = 90,	/* wolfram rule-numbers */
    saved_buffer = 1,	/* default buffer to use */
    saved_plane_mask = 0,	/* default plane mask */
    saved_density = 100,	/* density for "general random" proc */
    saved_ox=0, saved_oy=0,	/* origin for general random proc */
    saved_sx, saved_sy,		/* rectangle-size for general random proc */
    saved_min_val, saved_max_val,	/* ...min-max vals */
    saved_lambda_percent = INTUNDEF,	/* default lambda value */
    saved_lambda_increment = 0,		/* ...and lambda increment */
    saved_lambda_inc_set = 0,	/* whether or not lambda_inc was ever set */
    saved_rho_percent[16],	/* array of rho values */
    saved_xinc = INTUNDEF, 	/* parms for drawing shaded circles */
    saved_yinc = INTUNDEF,
    saved_synchro_interval = 1000;	/* synchronization interval for CM */
char
    saved_image_file[BUFLEN],	/* filename for images */
    saved_ienter_str[BUFLEN],	/* for entering initial configs in array */
    saved_rule_file[BUFLEN];	/* rule (LT or obj-code) filename */

/*** CM-related variables ***/
char
    CM_hostname[80];	/* hostname of CM front-end to attach to */
unsigned short
    CM_port;		/* port on CM FE to attach to */
long int
    CM_host;		/* CM FE address */


/*** Computed-function function-pointers (nhood routines need them) ***/
func_ptr
    update_function = NULL;
void_func_ptr
    before_function = NULL, after_function = NULL;

/*****
 *
 * Local variables
 *
 *****/

static char
    saved_x_offset[BUFLEN], saved_y_offset[BUFLEN],	/* offsets for
							 * resizing images */
    saved_nhood[BUFLEN],	/* nhood for altering LT */
    argv0[256];		/* pathname of Cellsim, used for dynamic linking */

static int
    local_run,	/* used to run things locally when attached to CM */
    non_local = 0,	/* occasionally used to flag a routine which would
			 * normally do something locally, to run on CM */
    set_just_pressed;	/* whether or not "set" button was just pressed */

static void_func_ptr
    initialization_function = NULL;

static int_func_ptr
    sequence_array[NUM_SEQUENCES][MAX_SEQUENCE_LENGTH];

static struct stat
    statbuf;	/* used by various routines to stat files */

