(* Elaboration *)
(* Author: Frank Pfenning *)

(*
 * Translates from external to abstract syntax,
 * creating internal names for types.
 * It also type-checks all declarations and runs 'exec' definitions
 *)

signature ELAB =
sig

    datatype env_n = Env of IntSyn.env * int
    (* Env(env, n), where "%n" is the next unused internal name *)

    val elab_env : env_n -> ExtSyn.env -> env_n (* may raise ErrorMsg.Error *)
    (* elab_env (Env(env,n)) raw = Env(env',n') elaborates raw environment
     * adding to the end of env
     *)
end

structure Elab :> ELAB =
struct

structure E = ExtSyn
structure I = IntSyn
structure Statics = Statics
structure Dynamics = Dynamics
(* structure Print = Print *)
val ERROR = ErrorMsg.ERROR

datatype env_n = Env of IntSyn.env * int
(* Env(env, n), where "%n" is the next unused internal name *)

fun check_def_tpname (Env(env,n)) t ext =
    if List.exists (fn I.TpDef(s,_,_) => s = t | _ => false) env then ()
    else ERROR ext ("undefined type name " ^ t)

fun check_def_raw_tpname raw t ext =
    if List.exists (fn E.TpDef(s,_,_) => s = t | _ => false) raw then ()
    else ERROR ext ("undefined type name " ^ t)

fun check_dup_label (l:ExtSyn.label) alts ext =
    if List.exists (fn (l',_) => l = l') alts
    then ERROR ext ("duplicate label " ^ I.pp_tag l)
    else ()

fun expand (Env(env,n)) t = I.expand env t

(* tp2tp raw env A ext = (env', A'), where A : ExtSyn.tp and A' : IntSyn.tp
 *
 * name_tp raw env A ext = (env', t) where t is a type name
 *
 * env' extends env (at the beginning) with the additional
 * internal type names %n that are introduced so that the components
 * of every structural type are type names.
 *
 * Note that env is partial because of mutual recursion between types
 * so we use the complete raw environment to make sure all mentioned
 * type name are defined
 *
 * We may reuse names x for types A if definitions x = A alreaddy exist in env.
 *)
fun tp2tp raw env (E.Tensor(A1,A2)) ext =
    let val (env1, t1) = name_tp raw env A1 ext
        val (env2, t2) = name_tp raw env1 A2 ext
    in (env2, I.Tensor(t1,t2)) end
  | tp2tp raw env (E.One) ext = (env, I.One)
  | tp2tp raw env (E.Plus(alts)) ext =
    let val (env', alts') = alts2alts raw env alts ext
    in (env', I.Plus(alts')) end
  | tp2tp raw env (E.Lolli(A1,A2)) ext =
    let val (env1, t1) = name_tp raw env A1 ext
        val (env2, s2) = name_tp raw env1 A2 ext
    in (env2, I.Lolli(t1,s2)) end
  | tp2tp raw env (E.With(alts)) ext =
    let val (env', alts') = alts2alts raw env alts ext
    in (env', I.With(alts')) end
  | tp2tp raw env (E.TpName(t)) ext =
    (* t = s, s must be previously defined *)
    (* this enforces contractivity but could be generalized *)
    let val () = check_def_tpname env t ext
    in (env, expand env t) end

and alts2alts raw env ((l,Al)::alts) ext =
    let val () = check_dup_label l alts ext
        val (env1, tl) = name_tp raw env Al ext
        val (env', alts') = alts2alts raw env1 alts ext
    in (env', (l,tl)::alts') end
  | alts2alts raw env nil ext = (env, nil)

and name_tp raw env (E.TpName(t)) ext =
    let val () = check_def_raw_tpname raw t ext (* check if globally defined *)
    in (env, t) end
  | name_tp raw env A ext =
    let val (Env(env',n), A') = tp2tp raw env A ext
    in case I.abbrev_tp env' A' (* reuse existing definitions t = A' *)
        of SOME(t) => (Env(env',n), t) (* already defined *)
         | NONE => let val t = "%" ^ Int.toString n
                   in (Env(I.TpDef(t,A',ext)::env', n+1), t) end
    end

fun ch2ch (E.Var(x)) = I.Var(x)

fun value2value (E.Channel(x)) = I.Channel(ch2ch x)
  | value2value (E.Label(k,V)) = I.Label(k,value2value V)
  | value2value (E.Pair(V1,V2)) = I.Pair(value2value V1, value2value V2)
  | value2value (E.Unit) = I.Unit
  | value2value (E.MarkedValue(marked_V)) =
    I.MarkedValue(Mark.mark'(value2value(Mark.data marked_V), Mark.ext marked_V))

fun proc2proc raw env (E.Send(x,V)) ext =
    (env, I.Send(ch2ch x, value2value V))
  | proc2proc raw env (E.Recv(x,E.Cont(K))) ext =
    let val (env', K') = cont2cont raw env K ext
    in (env', I.Recv(ch2ch x, I.Cont(K'))) end
  | proc2proc raw env (E.Fwd(x,y)) ext = (env, I.Fwd(ch2ch x, ch2ch y))
  | proc2proc raw env (E.Call(p, x, ys)) ext = (env, I.Call(p, ch2ch x, List.map ch2ch ys))
  | proc2proc raw env (E.Cut(x, NONE, P, Q)) ext =
    let val (env1, P') = proc2proc raw env P ext
        val (env2, Q') = proc2proc raw env1 Q ext
    in (env2, I.Cut(ch2ch x, NONE, P', Q')) end
  | proc2proc raw env (E.Cut(x, SOME(A), P, Q)) ext =
    let val (env0, t) = name_tp raw env A ext
        val (env1, P') = proc2proc raw env0 P ext
        val (env2, Q') = proc2proc raw env1 Q ext
    in (env2, I.Cut(ch2ch x, SOME(t), P', Q')) end
  | proc2proc raw env (E.Marked(marked_P)) ext =
    let val (env', P') = proc2proc raw env (Mark.data marked_P) (Mark.ext marked_P)
    in (env', I.Marked(Mark.mark'(P', Mark.ext marked_P))) end
and cont2cont raw env ((V,P)::K) ext =
    let val () = () (* check_dup_label l branches ext *) (* !!! *)
        val V' = value2value V
        val (env1, P') = proc2proc raw env P ext
        val (env2, K') = cont2cont raw env1 K ext
    in (env2, (V',P')::K') end
  | cont2cont raw env nil ext = (env, nil)

fun parm2parm raw env (E.Tp(x,A)) ext =
    let val (env', t) = name_tp raw env A ext
    in (env', I.Tp(ch2ch x,t)) end

fun parms2parms raw env (parm::parms) ext =
    let val (env1, parm') = parm2parm raw env parm ext
        val (env2, parms') = parms2parms raw env1 parms ext
    in (env2, parm'::parms') end
  | parms2parms raw env nil ext = (env, nil)

(* dec_ext d = ext, the extent of d in the source *)
fun dec_ext (E.TpDef(_,_,ext)) = ext
  | dec_ext (E.ProcDef(_,_,_,_,ext)) = ext
  | dec_ext (E.Exec(_,ext)) = ext
  | dec_ext (E.Fail(_,ext)) = ext
  | dec_ext (E.Error(ext)) = ext

(* raw2env' raw rawdecs env = env'
 * where raw,rawdecs : ExtSyn.env and env,env' : Ast.env
 *
 * raw is the environment with all declarations in external syntax
 * rawdecs are the declarations being processed
 *
 * this establishes the following additional invariants:
 * - all type names in env are defined (allowing arbitrary mutual recursion)
 * - there are no duplicate labels in internal or external choice
 * - there are no duplicate labels in pattern matches
 *
 * env' extends env with translated declarations from raw
 * and internal names that might be generated by type polarization
 *
 * Must extend env at the beginning so declarations are in
 * reverse order.  This actually matters for correctness in the
 * case of a 'fail <dec>' declarations.
 *)
fun raw2env' raw (E.TpDef(t,A,ext)::rawdecs) env =
    let val (Env(env',n'), A') = tp2tp raw env A ext
    in raw2env' raw rawdecs (Env(I.TpDef(t, A', ext)::env',n')) end
  | raw2env' raw (E.ProcDef(p,xA,yBs,P,ext)::rawdecs) env =
    let val (env1, xA') = parm2parm raw env xA ext
        val (env2, yBs') = parms2parms raw env1 yBs ext
        val (Env(env3,n'), P') = proc2proc raw env2 P ext
    in 
       raw2env' raw rawdecs (Env(I.ProcDef(p, xA', yBs', P', ext)::env3,n'))
    end
  | raw2env' raw (E.Exec(p,ext)::rawdecs) (Env(env,n)) =
    raw2env' raw rawdecs (Env(I.Exec(p,ext)::env,n))
  | raw2env' raw (E.Fail(d,ext)::rawdecs) (Env(env,n)) =
    raw2env' raw rawdecs
             (ErrorMsg.suppress (fn () =>
              let val Env(d'::env',n') = raw2env' raw [d] (Env(env,n))
              in Env(I.Fail(d',ext)::env',n') end (* might fail later, use env' *)
              handle ErrorMsg.Error => Env(I.Fail(I.Error(dec_ext d),ext)::env,n))) (* skip and use env *)
  | raw2env' raw (E.Error(ext)::rawdecs) (Env(env,n)) = raw2env' raw rawdecs (Env(I.Error(ext)::env,n))
  | raw2env' raw nil env = env

fun raw2env (Env(env, n)) raw =
    let val (Env(env',n')) = raw2env' raw raw (Env(List.rev env, n)) (* reverse for proper lookup! *)
    in Env(List.rev env', n') end (* reverse again for readability *)

(* dec_name d = "name", the name defined by declaration d, if any *)
fun dec_name (I.TpDef(t,_,_)) = "type " ^ t
  | dec_name (I.ProcDef(p,_,_,_,_)) = "proc " ^ p
  | dec_name (I.Exec(p,_)) = "exec " ^ p
  | dec_name (I.Fail(d,_)) = "fail " ^ dec_name d
  | dec_name (I.Error(_)) = "<error>"

(* check_dec env d = ()  if  d is valid
 * raises ErrorMsg.Error otherwise
 *)
fun check_dec env (d as I.TpDef(t,A,ext)) =
    let val () = if !Flags.verbosity = 1 andalso not (I.is_internal t) then print (dec_name d ^ "\n") else ()
        val () = if !Flags.verbosity >= 2 andalso not (I.is_internal t)
                 then print ("type " ^ t ^ " = " ^ I.pp_tp env A ^ "\n")
                 else ()
    in () end
  | check_dec env (d as I.ProcDef(p,I.Tp(x,nA),yBs,P,ext)) =
    let val () = if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else ()
        val Delta = List.map (fn I.Tp(x,nA) => I.TpN(x, nA, I.Must)) yBs
        val succ = I.TpN(x, nA, I.Right)
        val Delta' = Statics.check env Delta P (I.TpN(x, nA, I.Right)) ext
        (* no need to check Delta' here *)
    in () end

  | check_dec env (d as I.Exec(p,ext)) =
    (if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else () ;
     case I.lookup_procname env p
      of SOME(I.ProcDef(p, I.Tp(x,nA), nil, _, ext')) => ()
       | SOME(I.ProcDef(p, _, _, _, ext')) =>
         ERROR ext ("process " ^ p ^ " to be executed not closed")
       | NONE => ERROR ext ("process " ^ p ^ " undefined"))

  | check_dec env (d as I.Fail(d',ext)) =
    ( if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else () ;
      if ErrorMsg.suppress (fn () => ((check_dec env d' ; true) handle ErrorMsg.Error => false))
      then ERROR ext ("declaration '" ^ dec_name d' ^ "' unexpectedly succeeds")
      else () )

  | check_dec env (I.Error(ext)) = (* lex/parse error already accounted for: no message *)
    raise ErrorMsg.Error

(* checking for duplicate definitions *)
fun is_defined_type env t =
    List.exists (fn I.TpDef(s,_,_) => t = s | _ => false) env

fun is_defined_proc env p =
    List.exists (fn I.ProcDef(q,_,_,_,_) => p = q | _ => false) env

(* nodups env = () if there are no duplication definitions of type or expression names
 * raises ErrorMsg.Error otherwise
 *)
fun nodups (I.TpDef(t,_,ext)::env') =
    if is_defined_type env' t then ERROR ext ("type " ^ t ^ " defined more than once")
    else nodups env'
  | nodups (I.ProcDef(p,_,_,_,ext)::env') = 
    if is_defined_proc env' p then ERROR ext ("process " ^ p ^ " defined more than once")
    else nodups env'
  | nodups (I.Exec _::env') = nodups env'
  | nodups (I.Fail _::env')  = nodups env'
  | nodups (I.Error _::env') = nodups env'
  | nodups nil = ()

(* check_env env = ()  if  env is correctly polarized and typed
 * raise ErrorMsg.Error otherwise
 *)
fun check_env env =
    ( List.app (fn d => check_dec env d) env
    ; nodups env )

fun exec_dec env (I.Exec(p,ext)) =
    let val () = if !Flags.verbosity >= 1 then print ("% executing " ^ p ^ "\n") else ()
        val () = Dynamics.exec env p (* may raise an uncaught exception *)
            handle Dynamics.DynamicError(msg) => ERROR ext ("unexpected dynamic error " ^ msg)
        val v = if !Flags.verbosity >= 2
                then let val v = Observe.run_and_observe env p
                     in () end
                else () (* testing *)
    in () end
  | exec_dec env _ = ()

fun exec_env env =
    List.app (fn d => exec_dec env d) env 

(* elab_env env raw = env'
 * env is assumed to be valid and env' is its extension with
 * the external declations in raw.
 * raises ErrorMsg.Error if raw is not valid
 * May not terminate due to declarations 'exec p'
 *)
fun elab_env env_n raw =
    let 
        val Env(env',n') = raw2env env_n raw
        val () = check_env env'
        val () = exec_env env'
(*
        val () = if !Flags.verbosity >= 1 then Print.print_env env''
                 else ()
 *)
    in
        Env(env', n')
    end

end
