(* Top Level *)

(*
 * Use in a SML read/eval/print loop
 * and for building stand-along executables
 *)

signature TOP =
sig
    val load : Elab.env_n -> string list -> Elab.env_n  (* Top.load env_n [<file>,...] = env_n', may raise ErrorMsg.Error *)
    val mpass : string -> OS.Process.status      (* Top.mpass "<command line arguments>" = status *)
    val main : string * string list -> OS.Process.status (* for stand-alone executable *)
end (* signature TOP *)

structure Top :> TOP =
struct

structure G = GetOpt  (* from $/smlnj-lib/Util/getopt-sig.sml *)
structure E = ExtSyn

(************************)
(* Command Line Options *)
(************************)

datatype mpass_option =
         Subtp of bool
       | Depth of string
       | Verbose of int
       | Help of bool

(* printing error/success messages to stdErr *)
fun say s = TextIO.output (TextIO.stdErr, s ^ "\n")

val usage =
    if "sml" = #file (OS.Path.splitDirFile (CommandLine.name ()))
    then "Top.mpass \"<option>* <file>*\";"
    else "mpass <option>* <file>*"
val header = Flags.version ^ "\n" ^ "Usage: " ^ usage ^ "\nwhere <option> is"
val options : mpass_option G.opt_descr list =
    [
     {short = "s", long = ["subtyping"],
      desc = G.NoArg (fn () => Subtp(true)),
      help = "Allow subtyping"},
     {short = "l", long = ["depth"],
      desc = G.ReqArg ((fn d => Depth(d)), "<observation_depth>"),
      help = "Observation depth for 'exec' command"},
     {short = "v", long = ["verbose"],
      desc = G.NoArg (fn () => Verbose(2)),
      help = "Give verbose status messages"},
     {short = "q", long = ["quiet"],
      desc = G.NoArg (fn () => Verbose(0)),
      help = "Run quietly"},
     {short = "d", long = ["debug"],
      desc = G.NoArg (fn () => Verbose(3)),
      help = "Print some debugging information"},
     {short = "h", long = ["help"],
      desc = G.NoArg (fn () => Help(true)),
      help = "Give short usage message and exit"}
    ]

val usage_info = G.usageInfo {header = header, options = options}

exception OS_FAILURE
exception OS_SUCCESS

fun exit_failure msg = ( say msg ; raise OS_FAILURE )
fun exit_success msg = ( say msg ; raise OS_SUCCESS )

fun get_options (args) =
    G.getOpt {argOrder = G.RequireOrder,
              options = options,
              errFn = exit_failure}
             args

fun process_option (Subtp(b)) = Flags.subtp := b
  | process_option (Depth(str)) =
    (case Flags.parseDepth str
      of NONE => exit_failure ("observation depth '" ^ str ^ "' not a valid integer")
       | SOME(n) => Flags.depth := (if n < 0 then NONE else SOME(n)))
  | process_option (Verbose(level)) = Flags.verbosity := level
  | process_option (Help(true)) = exit_success usage_info
  | process_option (Help(false)) = ()

(*********************************)
(* Loading and Elaborating Files *)
(*********************************)

fun readable file = OS.FileSys.access (file, [OS.FileSys.A_READ])

(* command line options applied before execution *)
fun apply_options line =
    let val args = String.tokens Char.isSpace line
        val (options, filenames) = get_options args (* may exit_failure(msgs) *)
        val () = List.app process_option options
        val () = case filenames
                  of nil => ()
                   | (_::_) => exit_failure ("spurious options: "
                                             ^ List.foldr (fn (arg,msg) => arg ^ " " ^ msg) "" filenames)
    in () end

(* load env filenames = env'
 * where env' extends env with declarations from filenames
 * Assume env is valid and ensures env' is valid
 * may raise ErrorMsg.Error
 *)
fun load env_n (file::filenames) =
    let
        val () = ParseState.reset ()
        val () = ErrorMsg.reset ()
        val decs = Parse.parse file (* may raise ErrorMsg.Error *)
        val env_n' = Elab.elab_env env_n decs (* may raise ErrorMsg.error *)
    in (* do not allow for mutually recursive definitions between files *)
        load env_n' filenames
    end
  | load env_n nil = env_n

fun exit_on_empty_files nil = exit_success Flags.version
  | exit_on_empty_files (_::_) = ()

(* main function to run file *)
fun test raise_exn args =
    (* reset flags *)
    let val () = Flags.reset()
        (* val () = IntSyn.reset() *)
        (* get and apply options *)
        val (options, filenames) = get_options args
        val () = List.app process_option options
        val () = exit_on_empty_files filenames
        (* load files, returning an environment of declarations *)
        val env_n = load (Elab.Env(nil, 0)) filenames
            handle ErrorMsg.Error => exit_failure "% parsing or type-checking failed"
                 | e => if raise_exn then raise e   (* for debugging purposes *)
                        else exit_failure "% internal error (uncaught exception)"
    in
        exit_success "% success"
    end handle OS_SUCCESS => OS.Process.success
             | OS_FAILURE => OS.Process.failure

fun mpass argstring = test true (String.tokens Char.isSpace argstring)
fun main (name, args) = test false args

end (* structure Top *)
