signature OBSERVE =
sig

    datatype value = Inj of IntSyn.label * value
                   | Pair of value * value
                   | Unit
                   | Hidden
                   | DepthExceeded

    val run_and_observe : IntSyn.env -> IntSyn.procname -> value

end  (* signature OBSERVE *)

structure Observe :> OBSERVE =
struct

structure I = IntSyn
structure D = Dynamics

datatype value = Inj of I.label * value
               | Pair of value * value
               | Unit
               | Hidden
               | DepthExceeded

fun parens str = "(" ^ str ^ ")"
fun pp_value (Inj(k,v)) = I.pp_tag k ^ pp_value_paren v
  | pp_value (Pair(v1,v2)) = "(" ^ pp_value v1 ^ "," ^ pp_value v2 ^ ")"
  | pp_value (Unit) = "()"
  | pp_value (Hidden) = "-"
  | pp_value (DepthExceeded) = "?"
and pp_value_paren (v as Inj _) = parens (pp_value v)
  | pp_value_paren (v as Hidden) = parens (pp_value v)
  | pp_value_paren (v as DepthExceeded) = parens (pp_value v)
  | pp_value_paren v = pp_value v

fun observe env (a,depth) (D.Config(C,n)) =
   (case D.split a C nil
      of SOME(C1rev, P as I.Recv(a,match), C2) => (D.Config(P::List.revAppend(C1rev, C2),n), Hidden)
       | SOME(C1rev, I.Send(a,I.Unit,NONE), C2) => (D.Config(List.revAppend(C1rev, C2),n), Unit)
       | SOME(C1rev, I.Send(a,I.Label(k),SOME(P)), C2) =>
         let val (D.Config(C',n'), v) = next env (a,depth-1) (D.Config(P::List.revAppend(C1rev, C2),n))
         in (D.Config(C',n'), Inj(k, v)) end
       | SOME(C1rev, I.Send(A,I.Channel(b), SOME(P)), C2) =>
         let val (D.Config(C',n'), v2) = next env (a,depth-1) (D.Config(P::List.revAppend(C1rev, C2),n))
             val (D.Config(C'',n''), v1) = next env (b,depth-1) (D.Config(C',n'))
         in (D.Config(C'',n''), Pair(v1, v2)) end
       | NONE => raise D.DynamicError("channel " ^ I.pp_chan a ^ " unavailable"))
and next env (a,0) (D.Config(C,n)) = (D.Config(C,n), DepthExceeded)
  | next env (a,depth) (D.Config(C,n)) = observe env (a,depth) (D.iterate env (D.Config(C,n)))

fun run_and_observe env p =
    (case I.lookup_procname env p
      of SOME(I.ProcDef(p, I.Tp(x,nA), nil, P, ext)) =>
         let val P' = I.subst_proc [(I.Chan(0), x)] P
             val depth = case !Flags.depth of NONE => ~1 | SOME(n) => n (* negative = no bound *)
             val (D.Config(C,n), v) = next env (I.Chan(0),depth) (D.Config([P'],1))
             val () = if !Flags.verbosity >= 2 then print (pp_value v ^ "\n") else ()
         in v end
       | SOME(I.ProcDef(p, I.Tp(x,nA), parms, P, ext)) =>
         ErrorMsg.ERROR ext ("process " ^ p ^ " not closed")
       | NONE => ErrorMsg.ERROR NONE ("process " ^ p ^ " undefined"))

end (* structure Observe *)
