(* Simple Regression Testing *)
(* Frank Pfenning <fp@cs.cmu.edu> *)

signature TEST =
sig
    val test : string list -> OS.Process.status (* Test.test [<file>,...] = status *)
    val main : string * string list -> OS.Process.status (* for stand-alone executable *)
end

structure Test :> TEST =
struct

structure G = GetOpt        (* from $/smlnj-lib/Util/getopt-sig.sml *)

(* options for regression testing *)
(* printing: v - verbose, q - quiet, d - debug *)
datatype option =
         Subtp of bool
       | Verbose of int
       | Help of bool

fun say s = TextIO.output (TextIO.stdErr, s ^ "\n")

val usage =
    if "sml" = #file (OS.Path.splitDirFile (CommandLine.name ()))
    then "Test.test \"<option>* <file>*\";"
    else "mptest <option>* <file>*"
val header = Flags.version ^ "\n" ^ usage
val options : option G.opt_descr list =
    [
     {short = "s", long = ["subtyping"],
      desc = G.NoArg (fn () => Subtp(true)),
      help = "Allow subtyping"},
     {short = "v", long = ["verbose"],
      desc = G.NoArg (fn () => Verbose(2)),
      help = "Run verbosely"},
     {short = "q", long = ["quiet"],
      desc = G.NoArg (fn () => Verbose(0)),
      help = "Run quietly"},
     {short = "h", long = ["help"],
      desc = G.NoArg (fn () => Help(true)),
      help = "Give short usage message and exit"}
    ]
val usage_info = G.usageInfo {header = header, options = options}
exception OS_FAILURE
exception OS_SUCCESS
          
fun exit_failure msg = ( say msg ; raise OS_FAILURE )
fun exit_success msg = ( say msg ; raise OS_SUCCESS )

fun get_options (args) =
    G.getOpt {argOrder = G.RequireOrder,
              options = options,
              errFn = exit_failure}
             args

structure TestFlags :> sig
    val verbosity : int ref
    val reset : unit -> unit
end =
struct
    val verbosity = ref 1
    fun reset () = ( verbosity := 1 )
end (* structure TestFlags *)

(* set flags according to options given by the programmer *)
fun process_option (Subtp(b)) = Flags.subtp := b (* set global flag! *)
  | process_option (Verbose(level)) = TestFlags.verbosity := level
  | process_option (Help(true)) = exit_success usage_info
  | process_option (Help(false)) = ()

(* outcomes of compiling and executing programd *)
datatype outcome =
         Success                (* parses, type-checks, and runs successfully *)
       (* remaining ones are never expected, right now *)
       | StaticError            (* includes lexer, parser, type-checker error *)
       | UncaughtException      (* uncaught exception should always be a bug *)

fun pp_outcome outcome = case outcome of
    Success => "success"
  | StaticError => "static error (lexing, parsing, type-checking)"
  | UncaughtException => "uncaught exception"

exception Outcome of outcome * outcome (* expected, actual *)

fun accept_outcome (Success, Success) = true
  | accept_outcome (_, _) = false

fun test_file2 expected filename =
    let val env = Top.load (Elab.Env(nil, 0)) [filename]
            handle ErrorMsg.Error => raise Outcome(expected, StaticError)
    in
        raise Outcome(expected, Success)
    end handle e as Outcome(expected, actual) => raise e
             | e => raise Outcome(expected, UncaughtException)

fun test_file1 filename =
    let (* next two are reset in the Top.load function *)
        (* val () = ParseState.reset () *)
        (* val () = ErrorMsg.reset () *)
        val () = Flags.reset ()
        val () = Flags.verbosity := ~1 (* really quiet *)
    in
        test_file2 Success filename
    end handle e as Outcome(expected, actual) => raise e
             | e => raise Outcome(Success, UncaughtException)

val total = ref 0
val succeeded = ref 0
val failed = ref 0

fun success (expected, actual) =
    ( if !TestFlags.verbosity >= 1 then TextIO.print "[OK]\n" else ()
    ; succeeded := !succeeded+1 )

fun failure (expected, actual) =
    ( if !TestFlags.verbosity >= 1
      then ( TextIO.print ("[FAIL]\n")
           ; TextIO.print ("Expected: " ^ pp_outcome expected ^ "\n")
           ; TextIO.print ("Actual:   " ^ pp_outcome actual ^ "\n") )
      else ()
    ; failed := !failed+1 )

fun test_file filename =
    ( if !TestFlags.verbosity >= 1
      then ( TextIO.print (filename ^ "... ")
           ; TextIO.flushOut (TextIO.stdOut) )
      else ()
    ; test_file1 filename
      handle Outcome(expected, actual) =>
             ( total := !total+1
             ; if accept_outcome (expected,actual)
               then success (expected, actual)
               else failure (expected, actual) ))

fun reset_counts () =
    ( total := 0
    ; succeeded := 0
    ; failed := 0 )

fun print_results () =
    ( TextIO.print ("Total tests: " ^ Int.toString (!total) ^ "\n")
    ; TextIO.print ("Succeeded:   " ^ Int.toString (!succeeded) ^ "\n")
    ; TextIO.print ("Failed:      " ^ Int.toString (!failed) ^ "\n")
    )
      
fun test args =
    let val () = reset_counts ()
        val () = TestFlags.reset ()
        val (options, filenames) = get_options args
        val () = List.app process_option options
        val () = List.app test_file filenames
        val () = print_results ()
    in
        if !total = !succeeded
        then ( TextIO.print "% regression testing succeeded\n"
             ; OS.Process.success )
        else ( TextIO.print "% regression testing failed\n"
             ; OS.Process.failure )
    end

fun main (name, args) = test args

end (* structure Test *)
