(* Statics *)
(* implements a context-passing linear type-checker *)

signature STATICS =
sig

    val check : IntSyn.env -> IntSyn.ctx -> IntSyn.proc -> IntSyn.typing -> IntSyn.ext -> IntSyn.ctx
    (* may raise ErrorMsg.Error *)

end (* signature STATICS *)

structure Statics :> STATICS =
struct

fun ERROR(ext, msg) = ErrorMsg.ERROR ext msg

structure I = IntSyn

(* lookup that may raise ErrorMsg.Error *)

fun select k alts ext =
    (case I.select k alts
      of SOME(elem) => elem
       | NONE => ERROR(ext, "missing alternative " ^ I.pp_tag k))

fun lookup_procname env p ext =
    (case I.lookup_procname env p
      of SOME(d) => d
       | NONE => ERROR(ext, "process " ^ p ^ " undefined"))

fun lookup_ntp x Delta ext =
    (case I.lookup_tpn x Delta
      of SOME(I.TpN(x, nA, I.Used)) => ERROR(ext, "variable " ^ I.pp_chan x ^ " already used")
       | SOME(I.TpN(x, nA, _)) => nA
       | NONE => ERROR(ext, "variable " ^ I.pp_chan x ^ " unknown or out of scope"))

fun lookup_status x Delta ext =
    (case I.lookup_tpn x Delta
      of SOME(I.TpN(x, _, s)) => s
       | NONE => raise Match)

fun expand env nA ext =
    (case I.expand_opt env nA
      of SOME(A) => A
       | NONE => ERROR(ext, "type " ^ I.pp_tpn env nA ^ " undefined"))

fun lookup_tp env x Delta ext =
    expand env (lookup_ntp x Delta ext) ext

fun pp_xR env z nC = I.pp_parm env (I.Tp(z, nC))
fun pp_xL env x Delta =
    (case I.lookup_tpn x Delta
      of SOME(I.TpN(x, nA, _)) =>
         I.pp_parm env (I.Tp(x, nA))
       | _ => raise Match) (* must be defined *)

(************************)
(* context manipulation *)
(************************)

fun update (I.TpN(x, nA, s)) (I.TpN(y, nB, t)::Delta) =
    if x = y then I.TpN(x, nA, s)::Delta
    else I.TpN(y, nB, t)::update (I.TpN(x, nA, s)) Delta
  | update _ nil = raise Match

fun add (I.TpN(x, nA, s)) Delta ext =
    if List.exists (fn I.TpN(y, _, _) => x = y) Delta
    then ERROR(ext, "shadowing on " ^ I.pp_chan x)
    else I.TpN(x, nA, s)::Delta

fun remove x (I.TpN(y, nB, I.Used)::Delta) =
    if x = y then Delta else (I.TpN(y, nB, I.Used)::remove x Delta)
  | remove x (I.TpN(y, nB, s)::Delta) =
    I.TpN(y, nB, s)::remove x Delta
  | remove x nil = raise Match
    
fun mark_used xs (I.TpN(y, nB, s)::Delta) =
    if List.exists (fn x => x = y) xs
    then I.TpN(y, nB, I.Used)::mark_used xs Delta
    else I.TpN(y, nB, s)::mark_used xs Delta
  | mark_used xs nil = nil

fun all_used (I.TpN(x, nA, I.Must)::Delta) ext =
    ERROR(ext, "variable " ^ I.pp_chan x ^ " unused")
  | all_used (I.TpN(x, nA, s)::Delta) ext = (* May or Used *)
    I.TpN(x, nA, s)::all_used Delta ext
  | all_used nil ext = nil

fun check_status x Delta s ext =
    if lookup_status x Delta ext = s then ()
    else ERROR(ext, "inconsistent status of variable " ^ I.pp_chan x ^ " across branches")

fun check_subset (I.TpN(x, _, s)::Delta1) Delta2 ext =
    ( check_status x Delta2 s ext
    ; check_subset Delta1 Delta2 ext )
  | check_subset nil Delta2 ext = ()

fun check_equal Delta1 Delta2 ext =
    ( check_subset Delta1 Delta2 ext
    ; check_subset Delta2 Delta1 ext )

fun all_equal (Delta1::Delta2::Deltas) ext =
    ( check_equal Delta1 Delta2 ext
    ; all_equal (Delta2::Deltas) ext )
  | all_equal (Delta::nil) ext = Delta
  | all_equal nil ext = ERROR(ext, "at least one branch required")

fun must2may (I.TpN(x, nA, I.Must)::Delta) = I.TpN(x, nA, I.May)::must2may Delta
  | must2may (I.TpN(x, nA, s)::Delta) = I.TpN(x, nA, s)::must2may Delta (* s = Used or May *)
  | must2may nil = nil

fun may2must (I.TpN(x, nA, I.Must)::Delta) Delta' = (* was: Must, became May *)
    (case lookup_status x Delta' NONE               (* status should always be defined *)
      of I.May => may2must Delta (update (I.TpN(x, nA, I.Must)) Delta') (* revert to Must if unused *)
       | I.Used => may2must Delta Delta' (* remains Used *)
       | I.Must => raise Match) (* impossible *)
  | may2must (I.TpN(x, nA, s)::Delta) Delta' = (* s = Used or May *)
    may2must Delta Delta'
  | may2must nil Delta' = Delta'

(* minimal inference for cut *)

fun infer_cut env Delta (I.Cut(x, NONE, I.Call(p, x', ys), Q)) zC ext =
    let val () = if x = x' then ()
                 else ERROR(ext, "variable mismatch: " ^ I.pp_chan x' ^ " not equal provided channel " ^ I.pp_chan x)
        val I.ProcDef(p, I.Tp(_,nA), uBs, _, ext') = lookup_procname env p ext
    in nA end
  | infer_cut env Delta (I.Cut(x, NONE, I.Marked(marked_P), Q)) zC ext =
    infer_cut env Delta (I.Cut(x, NONE, Mark.data marked_P, Q)) zC ext (* don't update ext *)
  | infer_cut env Delta (I.Cut(x, NONE, P, Q)) zC ext =
    ERROR(ext, "ommitted type of " ^ I.pp_chan x ^ " not inferrable")

(* right_rule P z = true if P communicates along z, the provided channel
 * requires P is a send or receive
 *)
fun right_rule (I.Send(x,_,_)) z = (x = z)
  | right_rule (I.Recv(x,_)) z = (x = z)

(* check env Delta P zC ext = Delta'
 * checkR env Delta P zC ext = Delta' if P ends in a right rule
 * checkL env Delta P zC ext = Delta' if P ends in a right rule
 * where Delta - Delta' |- P :: (z : C)
 * raises ErrorMsg.Error if not such Delta' exists
 *
 * variables in Delta are marked as
 * Must (must be used, never in Delta')
 * May (may be used, could be May or Used in Delta'
 * Used (was used, so no longer available
 * Right (succedent/offered channel)
 *)
fun check env Delta (I.Fwd(x,y)) (I.TpN(x', nA, I.Right)) ext =
    ( if x = x' then () else ERROR(ext, "variable mismatch: " ^ I.pp_chan x ^ " not equal to provided channel " ^ I.pp_chan x') ;
      case lookup_ntp y Delta ext
       of nA' => if I.eqtp env nA' nA
                 then all_used (mark_used [y] Delta) ext
                 else ERROR(ext, "type mismatch: " ^ pp_xL env y Delta ^ " not equal " ^ pp_xR env x' nA))
                                      
  | check env Delta (I.Cut(x,SOME(nA),P,Q)) (I.TpN(z, nC, I.Right)) ext =
    let val () = if x <> z then () else ERROR(ext, "channel " ^ I.pp_chan x ^ " shadowing provided channel")
        val Delta1 = check env (must2may Delta) P (I.TpN(x, nA, I.Right)) ext
        val Delta2 = may2must Delta Delta1 (* revert May to Must *)
        val Delta' = check env (add (I.TpN(x, nA, I.Must)) Delta2 ext) Q (I.TpN(z, nC, I.Right)) ext
    in remove x Delta' end (* end of scope for x *)

  | check env Delta (I.Cut(x,NONE,P,Q)) (I.TpN(z, nC, I.Right)) ext =
    let val () = if x <> z then () else ERROR(ext, "channel " ^ I.pp_chan x ^ " shadowing provided channel")
        val nA = infer_cut env Delta (I.Cut(x, NONE, P, Q)) (I.TpN(z, nC, I.Right)) ext
    in check env Delta (I.Cut(x, SOME(nA), P, Q)) (I.TpN(z, nC, I.Right)) ext end

  | check env Delta (I.Call(p, x, ys)) (I.TpN(z, nC, I.Right)) ext =
    let val I.ProcDef(p, I.Tp(x',nA), uBs, _, ext') = lookup_procname env p ext
        val () = if x = z then ()
                 else ERROR(ext, "variable mismatch: " ^ I.pp_chan x ^ " does not match provided channel " ^ I.pp_chan z)
        val () = if I.eqtp env nA nC then ()
                 else ERROR(ext, "type mismatch: " ^ I.pp_parm env (I.Tp(x,nA)) ^ " not equal to " ^ pp_xR env z nC)
        val () = if List.length ys = List.length uBs then ()
                 else ERROR(ext, "process " ^ p ^ " requires " ^ Int.toString (List.length uBs) ^ " arguments "
                                 ^ " but given " ^ Int.toString (List.length ys))
        val yBs = List.map (fn y => I.Tp(y, lookup_ntp y Delta ext)) ys
        val () = ListPair.app (fn (I.Tp(y,nB'), I.Tp(u,nB)) =>
                                  if I.eqtp env nB' nB then ()
                                  else ERROR(ext, "type mismatch: " ^ I.pp_parm env (I.Tp(y,nB'))
                                                  ^ " not equal " ^ I.pp_parm env (I.Tp(u,nB))))
                              (yBs, uBs)
    in all_used (mark_used ys Delta) ext end

  | check env Delta (I.Marked(marked_P)) zC ext =
    check env Delta (Mark.data marked_P) zC (Mark.ext marked_P)

  | check env Delta P (I.TpN(z, nC, I.Right)) ext =
    if right_rule P z
    then checkR env Delta P (I.TpN(z, nC, I.Right)) ext
    else checkL env Delta P (I.TpN(z, nC, I.Right)) ext

and checkR env Delta (I.Send(x, I.Unit, NONE)) (I.TpN(z, nC, I.Right)) ext = (* x = z *)
    (case expand env nC ext
      of I.One => all_used Delta ext
       | _ => ERROR(ext, "type mismatch: " ^ pp_xR env z nC ^ " not equal 1"))

  | checkR env Delta (I.Send(x, I.Label(k), SOME(P))) (I.TpN(z, nC, I.Right)) ext = (* x = z *)
    (case expand env nC ext
      of I.Plus(alts) => (case select k alts ext
                           of nCk => check env Delta P (I.TpN(z, nCk, I.Right)) ext)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xR env z nC ^ " not an internal choice +{...}"))

  | checkR env Delta (I.Send(x, I.Channel(y), SOME(P))) (I.TpN(z, nC, I.Right)) ext =
    (case (lookup_ntp y Delta ext, expand env nC ext)
      of (nB', I.Tensor(nB,nA)) =>
         if I.eqtp env nB' nB
         then check env (update (I.TpN(y, nB', I.Used)) Delta) P (I.TpN(z, nA, I.Right)) ext
         else ERROR(ext, "type mismatch: " ^ pp_xL env y Delta ^ " not equal " ^ I.pp_tpn env nB)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xR env z nC ^ " not a tensor (_ * _)"))

  | checkR env Delta (I.Recv(x, I.MatchUnit(P))) zC ext =
    ERROR(ext, "type 'bot' not supported")

  | checkR env Delta (I.Recv(x, I.MatchLabel(branches))) (I.TpN(z, nC, I.Right)) ext =
    (case expand env nC ext
      of I.With(alts) =>
         all_equal (checkR_branches env Delta branches alts (I.TpN(z, nC, I.Right)) ext) ext
       | _ => ERROR(ext, "type mismatch: " ^ pp_xR env z nC ^ " is not an external choice &{...}"))

  | checkR env Delta (I.Recv(x, I.MatchChannel(y, P))) (I.TpN(z, nC, I.Right)) ext =
    (if y <> z then () else ERROR(ext, "channel " ^ I.pp_chan y ^ " shadowing provided channel") ;
     case expand env nC ext
      of I.Lolli(B,A) =>
         let val Delta' = check env (add (I.TpN(y, B, I.Must)) Delta ext) P (I.TpN(z, A, I.Right)) ext
         in remove y Delta' end (* end of scope of y *)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xR env z nC ^ " is not a linear function (_ -o _)"))

and checkR_branches env Delta branches ((l,Al)::alts) (I.TpN(z, nC, I.Right)) ext =
    check env Delta (select l branches ext) (I.TpN(z, Al, I.Right)) ext
    :: checkR_branches env Delta branches alts (I.TpN(z, nC, I.Right)) ext
  | checkR_branches env Delta branches nil zC ext = nil

and checkL env Delta (I.Send(x, I.Unit, NONE)) zC ext =
    ERROR(ext, "type 'bot' not supported")
    
  | checkL env Delta (I.Send(x,I.Label(k),SOME(P))) zC ext =
    (case lookup_tp env x Delta ext
      of I.With(alts) =>
         (case select k alts ext
           of nAk => check env (update (I.TpN(x, nAk, I.Must)) Delta) P zC ext)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xL env x Delta ^ " is not an external choice &{...}"))

  | checkL env Delta (I.Send(x,I.Channel(y),SOME(P))) zC ext =
    (if x <> y then () else ERROR(ext, "cannot send channel " ^ I.pp_chan x ^ " to itself") ;
     case (lookup_tp env x Delta ext, lookup_ntp y Delta ext)
      of (I.Lolli(nB,nA), nB') =>
         if I.eqtp env nB' nB
         then check env (update (I.TpN(x, nA, I.Must)) (mark_used [y] Delta)) P zC ext
         else ERROR(ext, "type mismatch: " ^ pp_xL env y Delta ^ " not equal to " ^ I.pp_tpn env nB)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xL env x Delta ^ " is not a linear function (_ -o _)"))

  | checkL env Delta (I.Recv(x,I.MatchLabel(branches))) zC ext =
    (case lookup_tp env x Delta ext
      of I.Plus(alts) => all_equal (checkL_branches env Delta x alts branches zC ext) ext
       | _ => ERROR(ext, "type mismatch: " ^ pp_xL env x Delta ^ " is not an internal choice +{...}"))

  | checkL env Delta (I.Recv(x,I.MatchUnit(P))) zC ext =
    (case lookup_tp env x Delta ext
      of I.One => check env (mark_used [x] Delta) P zC ext
       | _ => ERROR(ext, "type mismatch: " ^ pp_xL env x Delta ^ " is not the unit 1"))
    
  | checkL env Delta (I.Recv(x,I.MatchChannel(y,P))) (zC as I.TpN(z, _, _)) ext =
    (if y <> z then () else ERROR(ext, "channel " ^ I.pp_chan y ^ " shadowing succedent") ;
     case lookup_tp env x Delta ext
      of I.Tensor(nB, nA) =>
         let val Delta' = check env (add (I.TpN(y, nB, I.Must))
                                         (update (I.TpN(x, nA, I.Must)) Delta)
                                         ext)
                                P zC ext
         in remove y Delta' end (* end of scope for y *)
       | _ => ERROR(ext, "type mismatch: " ^ pp_xL env x Delta ^ " is not a tensor (_ * _)"))

and checkL_branches env Delta x ((l,Al)::alts) branches zC ext =
    check env (update (I.TpN(x,Al,I.Must)) Delta) (select l branches ext) zC ext
    :: checkL_branches env Delta x alts branches zC ext
  | checkL_branches env Delta x nil branches zC ext = nil

end
