(* Elaboration *)
(* Author: Frank Pfenning *)

(*
 * Translates from external to abstract syntax,
 * creating internal names for types.
 * It also type-checks all declarations and runs 'exec' definitions
 *)

signature ELAB =
sig

    val elab_env : IntSyn.env -> ExtSyn.env -> IntSyn.env (* may raise ErrorMsg.Error *)
    (* elab_env env raw = env' elaborates raw environment given env *)
end

structure Elab :> ELAB =
struct

structure E = ExtSyn
structure I = IntSyn
structure S = Statics
structure D = Dynamics
(* structure Print = Print *)
val ERROR = ErrorMsg.ERROR

datatype env = Env of I.env * int (* next internal name *)

fun is_tp (Env(env,n)) t ext =
    if List.exists (fn I.TpDef(s,_,_) => s = t | _ => false) env then ()
    else ERROR ext ("undefined type name " ^ t)

fun expand (Env(env,n)) t = I.expand env t

(* tp2tp env A ext = (env', A'), where A : ExtSyn.tp and A' : IntSyn.tp
 *
 * name_tp env A ext = (env', t) where t is a type name
 *
 * env' extends env (at the beginning) with the additional
 * internal type names %n that are introduced so that the components
 * of every structural type are type names.
 *
 * Note that env is partial because of mutual recursion between types
 *
 * We reuse names x for types A if definitions x = A alreaddy exist in env.
 *)
fun tp2tp env (E.Tensor(A1,A2)) ext =
    let val (env1, t1) = name_tp env A1 ext
        val (env2, t2) = name_tp env1 A2 ext
    in (env2, I.Tensor(t1,t2)) end
  | tp2tp env (E.One) ext = (env, I.One)
  | tp2tp env (E.Plus(alts)) ext =
    let val (env', alts') = alts2alts env alts ext
    in (env', I.Plus(alts')) end
  | tp2tp env (E.Lolli(A1,A2)) ext =
    let val (env1, t1) = name_tp env A1 ext
        val (env2, s2) = name_tp env1 A2 ext
    in (env2, I.Lolli(t1,s2)) end
  | tp2tp env (E.With(alts)) ext =
    let val (env', alts') = alts2alts env alts ext
    in (env', I.With(alts')) end
  | tp2tp env (E.TpName(t)) ext =
    let val () = is_tp env t ext (* FIX!!! *)
    in (env, expand env t) end

and alts2alts env ((k,Ak)::alts) ext =
    let val (env1, tk) = name_tp env Ak ext
        val (env', alts') = alts2alts env1 alts ext
    in (env', (k,tk)::alts') end
  | alts2alts env nil ext = (env, nil)

and name_tp env (E.TpName(t)) ext = (env, t)
  | name_tp env A ext =
    let val (Env(env',n), A') = tp2tp env A ext
    in case I.abbrev_tp env' A' (* reuse existing definitions t = A' *)
        of SOME(t) => (Env(env',n), t) (* already defined *)
         | NONE => let val t = "%" ^ Int.toString n
                   in (Env(I.TpDef(t,A',ext)::env', n+1), t) end
    end

fun ch2ch (E.Var(x)) = I.Var(x)

fun msg2msg (E.Label(k)) = I.Label(k)
  | msg2msg (E.Unit) = I.Unit
  | msg2msg (E.Channel(x)) = I.Channel(ch2ch x)

fun proc2proc env (E.Send(x,m,NONE)) ext =
    (env, I.Send(ch2ch x, msg2msg m, NONE))
  | proc2proc env (E.Send(x,m,SOME(P))) ext =
    let val (env', P') = proc2proc env P ext
    in (env', I.Send(ch2ch x, msg2msg m, SOME(P'))) end
  | proc2proc env (E.Recv(x,match)) ext =
    let val (env', match') = match2match env match ext
    in (env', I.Recv(ch2ch x, match')) end
  | proc2proc env (E.Fwd(x,y)) ext = (env, I.Fwd(ch2ch x, ch2ch y))
  | proc2proc env (E.Call(p, x, ys)) ext = (env, I.Call(p, ch2ch x, List.map ch2ch ys))
  | proc2proc env (E.Cut(x, NONE, P, Q)) ext =
    let val (env1, P') = proc2proc env P ext
        val (env2, Q') = proc2proc env1 Q ext
    in (env2, I.Cut(ch2ch x, NONE, P', Q')) end
  | proc2proc env (E.Cut(x, SOME(A), P, Q)) ext =
    let val (env0, t) = name_tp env A ext
        val (env1, P') = proc2proc env0 P ext
        val (env2, Q') = proc2proc env1 Q ext
    in (env2, I.Cut(ch2ch x, SOME(t), P', Q')) end
  | proc2proc env (E.Marked(marked_P)) ext =
    let val (env', P') = proc2proc env (Mark.data marked_P) (Mark.ext marked_P)
    in (env', I.Marked(Mark.mark'(P', Mark.ext marked_P))) end
and match2match env (E.MatchLabel(branches)) ext =
    let val (env', branches') = branches2branches env branches ext
    in (env', I.MatchLabel(branches')) end
  | match2match env (E.MatchUnit(P)) ext =
    let val (env', P') = proc2proc env P ext
    in (env', I.MatchUnit(P')) end
  | match2match env (E.MatchChannel(y,P)) ext =
    let val (env', P') = proc2proc env P ext
    in (env', I.MatchChannel(ch2ch y, P')) end
and branches2branches env ((l,Pl)::branches) ext =
    let val (env1, Pl') = proc2proc env Pl ext
        val (env2, branches') = branches2branches env1 branches ext
    in (env2, (l,Pl')::branches') end
  | branches2branches env nil ext = (env, nil)

fun parm2parm env (E.Tp(x,A)) ext =
    let val (env', t) = name_tp env A ext
    in (env', I.Tp(ch2ch x,t)) end

fun parms2parms env (parm::parms) ext =
    let val (env1, parm') = parm2parm env parm ext
        val (env2, parms') = parms2parms env1 parms ext
    in (env2, parm'::parms') end
  | parms2parms env nil ext = (env, nil)

(* dec_ext d = ext, the extent of d in the source *)
fun dec_ext (E.TpDef(_,_,ext)) = ext
  | dec_ext (E.ProcDef(_,_,_,_,ext)) = ext
  | dec_ext (E.Exec(_,ext)) = ext
  | dec_ext (E.Fail(_,ext)) = ext
  | dec_ext (E.Error(ext)) = ext

(* raw2env' raw env = env'  where raw : ExtSyn.env and env,env' : Ast.env
 *
 * raw is the environment with declarations in external syntax
 * env' extends env with translated declarations from raw
 * and internal names that might be generated by type polarization
 *
 * Must extend env at the beginning so declarations are in
 * reverse order.  This actually matters for correctness in the
 * case of a 'fail <dec>' declarations.
 *)
fun raw2env' (E.TpDef(t,A,ext)::raw) env =
    let val (Env(env',n'), A') = tp2tp env A ext
    in raw2env' raw (Env(I.TpDef(t, A', ext)::env',n')) end
  | raw2env' (E.ProcDef(p,xA,yBs,P,ext)::raw) env =
    let val (env1, xA') = parm2parm env xA ext
        val (env2, yBs') = parms2parms env1 yBs ext
        val (Env(env3,n'), P') = proc2proc env2 P ext
    in 
       raw2env' raw (Env(I.ProcDef(p, xA', yBs', P', ext)::env3,n'))
    end
  | raw2env' (E.Exec(p,ext)::raw) (Env(env,n)) =
    raw2env' raw (Env(I.Exec(p,ext)::env,n))
  | raw2env' (E.Fail(d,ext)::raw) (Env(env,n)) =
    raw2env' raw (ErrorMsg.suppress (fn () =>
                  let val Env(d'::env',n') = raw2env' [d] (Env(env,n))
                  in Env(I.Fail(d',ext)::env',n') end (* might fail later, use env' *)
                  handle ErrorMsg.Error => Env(I.Fail(I.Error(dec_ext d),ext)::env,n))) (* skip and use env *)
  | raw2env' (E.Error(ext)::raw) (Env(env,n)) = raw2env' raw (Env(I.Error(ext)::env,n))
  | raw2env' nil env = env

fun raw2env env raw =
    let val (Env(env',n')) = raw2env' raw (Env(List.rev env, 0))
    in List.rev env' end (* reverse for readability *)

(* closed_tp env A ext = ()  if A is closed and type names are defined
 * raise ErrorMsg.Error otherwise
 *)
fun is_tpn env t ext =
    if List.exists (fn I.TpDef(s,_,_) => t = s | _ => false) env then ()
    else ERROR ext ("type name " ^ t ^ " undefined")

fun closed_tp env (I.Tensor(t1,t2)) ext = ( is_tpn env t1 ext ; is_tpn env t2 ext )
  | closed_tp env (I.One) ext = ()
  | closed_tp env (I.Plus(alts)) ext = List.app (fn (k,tk) => is_tpn env tk ext) alts
  | closed_tp env (I.Lolli(t1,s2)) ext = ( is_tpn env t1 ext ; is_tpn env s2 ext )
  | closed_tp env (I.With(alts)) ext = List.app (fn (k,tk) => is_tpn env tk ext) alts

(* noduble alts ext = ()  if there are no duplicate tags in alts
 * raises ErrorMsg.Error otherwise
 *)
fun nodup_alts ((k,_)::alts) ext =
    if List.exists (fn (l,_) => k = l) alts
    then ERROR ext ("duplicate label " ^ I.pp_tag k)
    else nodup_alts alts ext
  | nodup_alts nil ext = ()

fun nodup_label (I.Plus(alts)) ext = nodup_alts alts ext
  | nodup_label (I.With(alts)) ext = nodup_alts alts ext
  | nodup_label A ext = ()

(* dec_name d = "name", the name defined by declaration d, if any *)
fun dec_name (I.TpDef(t,_,_)) = "type " ^ t
  | dec_name (I.ProcDef(p,_,_,_,_)) = "proc " ^ p
  | dec_name (I.Exec(p,_)) = "exec " ^ p
  | dec_name (I.Fail(d,_)) = "fail " ^ dec_name d
  | dec_name (I.Error(_)) = "<error>"

(* check_dec env d = ()  if  d is valid (well-polarized and well-typed)
 * raises ErrorMsg.Error otherwise
 *)
fun check_dec env (d as I.TpDef(t,A,ext)) =
    let val () = if !Flags.verbosity = 1 andalso not (I.is_internal t) then print (dec_name d ^ "\n") else ()
        val () = if !Flags.verbosity >= 2 andalso not (I.is_internal t)
                 then print ("type " ^ t ^ " = " ^ I.pp_tp env A ^ "\n")
                 else ()
        val () = closed_tp env A ext
        val () = nodup_label A ext
    in () end
  | check_dec env (d as I.ProcDef(p,I.Tp(x,nA),yBs,P,ext)) =
    let val () = if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else ()
        val Delta = List.map (fn I.Tp(x,nA) => I.TpN(x, nA, I.Must)) yBs
        val succ = I.TpN(x, nA, I.Right)
        val Delta' = Statics.check env Delta P (I.TpN(x, nA, I.Right)) ext
        (* no need to check Delta' here *)
    in () end

  | check_dec env (d as I.Exec(p,ext)) =
    (if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else () ;
     case I.lookup_procname env p
      of SOME(I.ProcDef(p, I.Tp(x,nA), nil, _, ext')) => ()
       | SOME(I.ProcDef(p, _, _, _, ext')) =>
         ERROR ext ("process " ^ p ^ " to be executed not closed")
       | NONE => ERROR ext ("process " ^ p ^ " undefined"))

  | check_dec env (d as I.Fail(d',ext)) =
    ( if !Flags.verbosity >= 1 then print (dec_name d ^ "\n") else () ;
      if ErrorMsg.suppress (fn () => ((check_dec env d' ; true) handle ErrorMsg.Error => false))
      then ERROR ext ("declaration '" ^ dec_name d' ^ "' unexpectedly succeeds")
      else () )

  | check_dec env (I.Error(ext)) = (* lex/parse error already accounted for: no message *)
    raise ErrorMsg.Error

(* checking for duplicate definitions *)
fun is_defined_type env t =
    List.exists (fn I.TpDef(s,_,_) => t = s | _ => false) env

fun is_defined_proc env p =
    List.exists (fn I.ProcDef(q,_,_,_,_) => p = q | _ => false) env

(* nodups env = () if there are no duplication definitions of type or expression names
 * raises ErrorMsg.Error otherwise
 *)
fun nodups (I.TpDef(t,_,ext)::env') =
    if is_defined_type env' t then ERROR ext ("type " ^ t ^ " defined more than once")
    else nodups env'
  | nodups (I.ProcDef(p,_,_,_,ext)::env') = 
    if is_defined_proc env' p then ERROR ext ("process " ^ p ^ " defined more than once")
    else nodups env'
  | nodups (I.Exec _::env') = nodups env'
  | nodups (I.Fail _::env')  = nodups env'
  | nodups (I.Error _::env') = nodups env'
  | nodups nil = ()

(* check_env env = ()  if  env is correctly polarized and typed
 * raise ErrorMsg.Error otherwise
 *)
fun check_env env =
    ( List.app (fn d => check_dec env d) env
    ; nodups env )

fun exec_dec env (I.Exec(p,ext)) =
    let val () = if !Flags.verbosity >= 0 then print ("% executing " ^ p ^ "\n") else ()
        val () = Dynamics.exec env p (* may raise an uncaught exception *)
            (* handle Dynamics.DynamicError(msg) => ERROR ext ("unexpected dynamic error " ^ msg) *)
    in () end
  | exec_dec env _ = ()

fun exec_env env =
    List.app (fn d => exec_dec env d) env 

(* elab_env env raw = env'
 * env is assumed to be valid and env' is its extension with
 * the external declations in raw.
 * raises ErrorMsg.Error if raw is not valid
 * May not terminate due to declarations 'exec p'
 *)
fun elab_env env raw =
    let 
        val env' = raw2env env raw
        val () = check_env env'
        val () = exec_env env'
(*
        val () = if !Flags.verbosity >= 1 then Print.print_env env''
                 else ()
 *)
    in
        env'
    end

end
