signature OBSERVE =
sig

    datatype value = Inj of IntSyn.label * value
                   | Pair of value * value
                   | Unit
                   | Hidden
                   | DepthExceeded

    val run_and_observe : IntSyn.env -> IntSyn.procname -> value

end  (* signature OBSERVE *)

structure Observe :> OBSERVE =
struct

structure I = IntSyn
structure D = Dynamics

datatype value = Inj of I.label * value
               | Pair of value * value
               | Unit
               | Hidden
               | DepthExceeded

fun parens str = "(" ^ str ^ ")"
fun pp_value (Inj(k,v)) = I.pp_tag k ^ pp_value_paren v
  | pp_value (Pair(v1,v2)) = "(" ^ pp_value v1 ^ "," ^ pp_value v2 ^ ")"
  | pp_value (Unit) = "()"
  | pp_value (Hidden) = "-"
  | pp_value (DepthExceeded) = "?"
and pp_value_paren (v as Inj _) = parens (pp_value v)
  | pp_value_paren (v as Hidden) = parens (pp_value v)
  | pp_value_paren (v as DepthExceeded) = parens (pp_value v)
  | pp_value_paren v = pp_value v

(* lookup a C, where C is final *)
fun lookup a (I.Send(b,V)::C) =
    if a = b then SOME(I.Send(b,V)) else lookup a C
  | lookup a (I.Recv(b,K)::C) =
    if a = b then SOME(I.Recv(b,K)) else lookup a C
  | lookup a (I.Marked(marked_P)::C) = lookup a (Mark.data marked_P::C)
  | lookup a (P::C) = NONE      (* P = Cut, Fwd, Call not in final config *)
  | lookup a nil = NONE

fun observe env (a,0) (D.Config(C,n)) = DepthExceeded
  | observe env (a,depth) (D.Config(C,n)) =
    (case lookup a C
      of SOME(I.Recv(a, K)) => Hidden
       | SOME(I.Send(a, V)) => observe_value env (V,depth) (D.Config(C,n))
       | NONE => raise D.DynamicError("channel " ^ I.pp_chan a ^ " unavailable"))
and observe_value env (V, 0) (D.Config(C,n)) = DepthExceeded
  | observe_value env (I.Unit, depth) (D.Config(C,n)) = Unit
  | observe_value env (I.Label(k,V), depth) (D.Config(C,n)) =
    Inj(k, observe_value env (V, depth-1) (D.Config(C,n)))
  | observe_value env (I.Pair(V1,V2), depth) (D.Config(C,n)) =
    Pair(observe_value env (V1, depth-1) (D.Config(C,n)),
         observe_value env (V2, depth-1) (D.Config(C,n)))
  | observe_value env (I.Channel(a), depth) (D.Config(C,n)) =
    observe env (a,depth) (D.Config(C,n))
  | observe_value env (I.MarkedValue(marked_V), depth) (D.Config(C,n)) =
    observe_value env (Mark.data marked_V, depth) (D.Config(C,n))

fun run_and_observe env p =
    (case I.lookup_procname env p
      of SOME(I.ProcDef(p, I.Tp(x,nA), nil, P, ext)) =>
         let val P' = I.subst_proc [(I.Chan(0), x)] P
             val depth = case !Flags.depth of NONE => ~1 | SOME(n) => n (* negative = no bound *)
             val v = observe env (I.Chan(0),depth) (D.iterate env (D.Config([P'],1)))
             val () = if !Flags.verbosity >= 2 then print (pp_value v ^ "\n") else ()
         in v end
       | SOME(I.ProcDef(p, I.Tp(x,nA), parms, P, ext)) =>
         ErrorMsg.ERROR ext ("process " ^ p ^ " not closed")
       | NONE => ErrorMsg.ERROR NONE ("process " ^ p ^ " undefined"))

end (* structure Observe *)
