//////////////////////////////////////////////////////////
// File  : symbolicHeuristic.cc
// Desc. : symbolic encoding of the heuristic function
//         h(v,x) as used by pure-BDDA* and BDDA*
// Author: Rune M. Jensen
// Date  : 5/7/02
//////////////////////////////////////////////////////////

#include <bdd.h>
#include <bvec.h>
#include <string.h>
#include <math.h>
#include "main.hpp"
#include "timer.hpp"
#include "set.hpp"
#include "bddLayout.hpp"
#include "reachInfo.hpp"
#include "numDomain.hpp"
#include "partition.hpp"
#include "symbolicHeuristic.hpp"


//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////
//
// FHinfo and BDDAinfo member functions 
//
//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////

void FHinfo::initialize(PDDLbddBDDAlayout& layout) {
  
	    Fcurrent = bddtrue;
	
	    for (int i = 0; i < layout.fVars.length; i++)
	      {
		Fcurrent &= bdd_ithvar(layout.fVars.current[i]);
	      }

	    Hcurrent = bddtrue;
	    Hnext    = bddtrue;
	    Hcurrent2next = bdd_newpair();
	    Hnext2current = bdd_newpair();

	    for (int i = 0; i < layout.hVars.length; i++)
	      {
		Hcurrent &= bdd_ithvar(layout.hVars.current[i]);
		Hnext    &= bdd_ithvar(layout.hVars.next[i]);
		bdd_setpair(Hcurrent2next,layout.hVars.current[i],layout.hVars.next[i]);
		bdd_setpair(Hnext2current,layout.hVars.next[i],layout.hVars.current[i]);
	      }

	    Hcurrent2Fcurrent = bdd_newpair();
	    for (int i = 0; i < layout.hVars.length; i++)
	      bdd_setpair(Hcurrent2Fcurrent,layout.hVars.current[i],layout.fVars.current[i]);	    	    	 	    
}






void PubFHinfo::initialize(PDDLbddPubBDDAlayout& layout) {
  
	    Fcurrent = bddtrue;
	    Fnext = bddtrue;
	    Fcurrent2next = bdd_newpair();
	    Fnext2current = bdd_newpair();
	
	    for (int i = 0; i < layout.fVars.length; i++)
	      {
		Fcurrent &= bdd_ithvar(layout.fVars.current[i]);
		Fnext    &= bdd_ithvar(layout.fVars.next[i]);
		bdd_setpair(Fcurrent2next,layout.fVars.current[i],layout.fVars.next[i]);
		bdd_setpair(Fnext2current,layout.fVars.next[i],layout.fVars.current[i]);
	      }

	    Hcurrent = bddtrue;
	    Hnext    = bddtrue;
	    Hcurrent2next = bdd_newpair();
	    Hnext2current = bdd_newpair();

	    for (int i = 0; i < layout.hVars.length; i++)
	      {
		Hcurrent &= bdd_ithvar(layout.hVars.current[i]);
		Hnext    &= bdd_ithvar(layout.hVars.next[i]);
		bdd_setpair(Hcurrent2next,layout.hVars.current[i],layout.hVars.next[i]);
		bdd_setpair(Hnext2current,layout.hVars.next[i],layout.hVars.current[i]);
	      }

	    Hcurrent2Fcurrent = bdd_newpair();
	    for (int i = 0; i < layout.hVars.length; i++)
	      bdd_setpair(Hcurrent2Fcurrent,layout.hVars.current[i],layout.fVars.current[i]);	    	    	 	    
}






void FHinfo::initialize(NADLbddBDDAlayout& layout) {
  
	    Fcurrent = bddtrue;
	
	    for (int i = 0; i < layout.fVars.length; i++)
	      {
		Fcurrent &= bdd_ithvar(layout.fVars.current[i]);
	      }

	    Hcurrent = bddtrue;
	    Hnext    = bddtrue;
	    Hcurrent2next = bdd_newpair();
	    Hnext2current = bdd_newpair();

	    for (int i = 0; i < layout.hVars.length; i++)
	      {
		Hcurrent &= bdd_ithvar(layout.hVars.current[i]);
		Hnext    &= bdd_ithvar(layout.hVars.next[i]);
		bdd_setpair(Hcurrent2next,layout.hVars.current[i],layout.hVars.next[i]);
		bdd_setpair(Hnext2current,layout.hVars.next[i],layout.hVars.current[i]);
	      }

	    Hcurrent2Fcurrent = bdd_newpair();
	    for (int i = 0; i < layout.hVars.length; i++)
	      bdd_setpair(Hcurrent2Fcurrent,layout.hVars.current[i],layout.fVars.current[i]);	    	    	 	    
}





void PubFHinfo::initialize(NADLbddPubBDDAlayout& layout) {

	    Fcurrent = bddtrue;
	    Fnext = bddtrue;
	    Fcurrent2next = bdd_newpair();
	    Fnext2current = bdd_newpair();

	    for (int i = 0; i < layout.fVars.length; i++)
	      {
		Fcurrent &= bdd_ithvar(layout.fVars.current[i]);
		Fnext    &= bdd_ithvar(layout.fVars.next[i]);
		bdd_setpair(Fcurrent2next,layout.fVars.current[i],layout.fVars.next[i]);
		bdd_setpair(Fnext2current,layout.fVars.next[i],layout.fVars.current[i]);
	      }

	    Hcurrent = bddtrue;
	    Hnext    = bddtrue;
	    Hcurrent2next = bdd_newpair();
	    Hnext2current = bdd_newpair();

	    for (int i = 0; i < layout.hVars.length; i++)
	      {
		Hcurrent &= bdd_ithvar(layout.hVars.current[i]);
		Hnext    &= bdd_ithvar(layout.hVars.next[i]);
		bdd_setpair(Hcurrent2next,layout.hVars.current[i],layout.hVars.next[i]);
		bdd_setpair(Hnext2current,layout.hVars.next[i],layout.hVars.current[i]);
	      }

	    Hcurrent2Fcurrent = bdd_newpair();
	    for (int i = 0; i < layout.hVars.length; i++)
	      bdd_setpair(Hcurrent2Fcurrent,layout.hVars.current[i],layout.fVars.current[i]);	    	    	 	    
}






//PDDL
void BDDAinfo::initialize(PDDLbddBDDAlayout& layout) {

  // make current and next var sets
  stateVarPairs = layout.stateVarPairs;
  
  currentVars = bddtrue;
  nextVars    = bddtrue;
  current2next = bdd_newpair();
  next2current = bdd_newpair();

  for (set< pair<int,int> >::iterator spi = stateVarPairs.begin(); 
       spi != stateVarPairs.end(); ++spi)
    {
      currentVars &= bdd_ithvar(spi->first);
      nextVars    &= bdd_ithvar(spi->second);
      bdd_setpair(current2next,spi->first,spi->second);
      bdd_setpair(next2current,spi->second,spi->first);
    }

  if (verbosity > 0)
    {
      cout << "Computing BDDA* expansion formulas:\n";
    }
  
  // statistics
  totalFormulaSize = 0;
	
  
  // generate BDDA* formulas for finding new f-values 
  bvec h =  bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm = bvec_varvec(layout.hVars.length,layout.hVars.next);
  bvec f =  bvec_varvec(layout.fVars.length,layout.fVars.current);

  formulaBackward.resize(layout.fhMax);
  // make a formula for each possible fmin values
  for (int i = 0; i < layout.fhMax; i++)
    {
      bvec sum = bvec(layout.hVars.length,i+1);
      // make f = (fmin + 1) + h' - h
      // (however since we are searching backwards the right form of the expression is:
      // f = (fmin + 1) + h - h' 
      sum = bvec_add(sum,h);
      sum = bvec_sub(sum,hm);

      formulaBackward[i] = (f == sum);

      if (verbosity > 0)
	{
	  cout << ".";
	  totalFormulaSize += bdd_nodecount(formulaBackward[i]);
	}
    }




  formulaForward.resize(layout.fhMax);
  // make a formula for each possible fmin values
  for (int i = 1; i < layout.fhMax; i++)
    {
      bvec sum = bvec(layout.hVars.length,i-1);
      // make f = (fmin - 1) + h' - h (this reverses the backward change in f)
      sum = bvec_add(sum,hm);
      sum = bvec_sub(sum,h);

      formulaForward[i] = (f == sum);

      if (verbosity > 0)
	{
	  cout << ".";
	  totalFormulaSize += bdd_nodecount(formulaForward[i]);
	}
    }



  if (verbosity > 0)
    {
      cout << "\n";
      cout << "TotalSize of formulas=" << totalFormulaSize << endl;
    }
}










// NADL
void BDDAinfo::initialize(NADLbddBDDAlayout& layout) {


  // make current and next var sets and
  // next state substitution sets
  currentVars = bddtrue;
  nextVars    = bddtrue;
  current2next = bdd_newpair();
  next2current = bdd_newpair();

  for (int i = 0; i < layout.var.size(); i++) 
    for (int j = 0; j < layout.var[i].length; j++)
      {
	pair<int,int> elem(layout.var[i].current[j],layout.var[i].next[j]);
	stateVarPairs.insert(elem);	 
	currentVars &= bdd_ithvar(layout.var[i].current[j]);
	nextVars    &= bdd_ithvar(layout.var[i].next[j]);
	bdd_setpair(current2next,layout.var[i].current[j],layout.var[i].next[j]);
	bdd_setpair(next2current,layout.var[i].next[j],layout.var[i].current[j]);
      }
  
  if (verbosity > 0)
    {
      cout << "Computing BDDA* expansion formulas:\n";
    }
  
  // statistics
  totalFormulaSize = 0;
	
  
  // generate BDDA* formulas for finding new f-values 
  bvec h =  bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm = bvec_varvec(layout.hVars.length,layout.hVars.next);
  bvec f =  bvec_varvec(layout.fVars.length,layout.fVars.current);

  formulaBackward.resize(layout.fhMax);
  // make a formula for each possible fmin values
  for (int i = 1; i < layout.fhMax; i++)
    {
      bvec sum = bvec(layout.hVars.length,i-1);
      // make f = (fmin - 1) + h - h'
      sum = bvec_add(sum,h);
      sum = bvec_sub(sum,hm);

      formulaBackward[i] = (f == sum);

      if (verbosity > 0)
	{
	  cout << ".";
	  totalFormulaSize += bdd_nodecount(formulaBackward[i]);
	}
    }




  formulaForward.resize(layout.fhMax);
  // make a formula for each possible fmin values
  for (int i = 0; i < layout.fhMax; i++)
    {
      bvec sum = bvec(layout.hVars.length,i+1);
      // make f = (fmin + 1) + h' - h 
      sum = bvec_add(sum,hm);
      sum = bvec_sub(sum,h);

      formulaForward[i] = (f == sum);

      if (verbosity > 0)
	{
	  cout << ".";
	  totalFormulaSize += bdd_nodecount(formulaForward[i]);
	}
    }

  if (verbosity > 0)
    {
      cout << "\n";
      cout << "TotalSize of formulas=" << totalFormulaSize << endl;
    }
}























// PDDL Pub
void PubBDDAinfo::initialize(PDDLbddPubBDDAlayout& layout) {

  // make current and next var sets
  stateVarPairs = layout.stateVarPairs;
  
  currentVars = bddtrue;
  nextVars    = bddtrue;
  current2next = bdd_newpair();
  next2current = bdd_newpair();

  for (set< pair<int,int> >::iterator spi = stateVarPairs.begin(); 
       spi != stateVarPairs.end(); ++spi)
    {
      currentVars &= bdd_ithvar(spi->first);
      nextVars    &= bdd_ithvar(spi->second);
      bdd_setpair(current2next,spi->first,spi->second);
      bdd_setpair(next2current,spi->second,spi->first);
    }

  if (verbosity > 0)
    {
      cout << "Computing BDDA* expansion formulas:\n";
    }
  
	  
  // generate BDDA* formulas for finding new f-values 
  bvec h =  bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm = bvec_varvec(layout.hVars.length,layout.hVars.next);
  bvec f =  bvec_varvec(layout.fVars.length,layout.fVars.current);
  bvec fm = bvec_varvec(layout.fVars.length,layout.fVars.next);

  // make backward formula

  bvec sum = bvec(layout.hVars.length,1);
  // make f = (f' + 1) + h' - h
  // (however since we are searching backwards the right form of the expression is:
  // f = (f' + 1) + h - h' 
  sum = bvec_add(sum,fm);
  sum = bvec_add(sum,h);
  sum = bvec_sub(sum,hm);

  formulaBackward = (f == sum);
  
  totalFormulaSize = bdd_nodecount(formulaBackward);


  if (verbosity > 0)
      cout << "TotalSize of formulas=" << totalFormulaSize << endl;
}





// NADL Pub
void PubBDDAinfo::initialize(NADLbddPubBDDAlayout& layout) {


  // make current and next var sets and
  // next state substitution sets
  currentVars = bddtrue;
  nextVars    = bddtrue;
  current2next = bdd_newpair();
  next2current = bdd_newpair();

  for (int i = 0; i < layout.var.size(); i++) 
    for (int j = 0; j < layout.var[i].length; j++)
      {
	pair<int,int> elem(layout.var[i].current[j],layout.var[i].next[j]);
	stateVarPairs.insert(elem);	 
	currentVars &= bdd_ithvar(layout.var[i].current[j]);
	nextVars    &= bdd_ithvar(layout.var[i].next[j]);
	bdd_setpair(current2next,layout.var[i].current[j],layout.var[i].next[j]);
	bdd_setpair(next2current,layout.var[i].next[j],layout.var[i].current[j]);
      }
  
  if (verbosity > 0)
    {
      cout << "Computing BDDA* expansion formulas:\n";
    }
  
  // statistics
  totalFormulaSize = 0;
	
  
  // generate BDDA* formulas for finding new f-values 
  bvec h  = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm = bvec_varvec(layout.hVars.length,layout.hVars.next);
  bvec f  = bvec_varvec(layout.fVars.length,layout.fVars.current);
  bvec fm = bvec_varvec(layout.fVars.length,layout.fVars.next);

  // make a forward formula 
  bvec sum = bvec(layout.hVars.length,1);
  // make f = (f' + 1) + h' - h 
  sum = bvec_add(sum,fm);
  sum = bvec_add(sum,hm);
  sum = bvec_sub(sum,h);

  formulaForward = (f == sum);

  totalFormulaSize = bdd_nodecount(formulaForward);

  if (verbosity > 0)
      cout << "TotalSize of formulas=" << totalFormulaSize << endl;
}























































































//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////
//
// Heuristic Function Encodings
//
//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////

//IN
// layout : BDDA* bbd layout
// rInfo  : fact reachability data
// numdom : numerical PDDL rep.
// fhInfo : structure with BDD variable info on h and f-encodings
//OUT
// symbolic encoding of the HSPr heuristic
bdd PDDLSymbolicHSPr(PDDLbddBDDAlayout& layout,reachInfo &rInfo,numDomain &numDom, FHinfo& fhInfo) {
  
  //     initially exp(h,x) = (h = 0)
  //     for each reqached fact 
  //             exp(h,h',x) := exp(h,x) /\ if (x = f1) then (h' = h + depth(f1)) else (h' = h) 
  //             exp(h',x) := exist h
  //             exp(h,x) := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building HSPr heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));
  

  // iterate over reached fluents
  set<int> reachedFluents = setDifference(rInfo.closedFacts,rInfo.initStaticFacts);
  for (set<int>::iterator si = reachedFluents.begin(); 
       si != reachedFluents.end(); ++si)
    {      
      // build h' = h + depth(f);
      int factId = *si;
      
      if (verbosity > 0)
	cout << ".";

      bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
			     bvec(layout.hVars.length,rInfo.depth[factId]));
      
      bdd r1 = hm == hPlusd;
      bdd r2 = hm == h;
      bdd l = layout.expr(numDom,factId,0);
      // build exp(h,h',x) := exp(h,x) /\ x = f1 =>  h' = h + depth(f1)
      exp = exp & bdd_ite(l,r1,r2);
      exp = bdd_exist(exp,fhInfo.Hcurrent);
      exp = bdd_replace(exp,fhInfo.Hnext2current);
    }
  
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  





// Version for published BDDA*
//IN
// layout : BDDA* bbd layout
// rInfo  : fact reachability data
// numdom : numerical PDDL rep.
// fhInfo : structure with BDD variable info on h and f-encodings
//OUT
// symbolic encoding of the HSPr heuristic
bdd PubPDDLSymbolicHSPr(PDDLbddPubBDDAlayout& layout,reachInfo &rInfo,numDomain &numDom, PubFHinfo& fhInfo) {
  
  //     initially exp(h,x) = (h = 0)
  //     for each reqached fact 
  //             exp(h,h',x) := exp(h,x) /\ if (x = f1) then (h' = h + depth(f1)) else (h' = h) 
  //             exp(h',x) := exist h
  //             exp(h,x) := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building HSPr heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));
  

  // iterate over reached fluents
  set<int> reachedFluents = setDifference(rInfo.closedFacts,rInfo.initStaticFacts);
  for (set<int>::iterator si = reachedFluents.begin(); 
       si != reachedFluents.end(); ++si)
    {      
      // build h' = h + depth(f);
      int factId = *si;
      
      if (verbosity > 0)
	cout << ".";

      bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
			     bvec(layout.hVars.length,rInfo.depth[factId]));
      
      bdd r1 = hm == hPlusd;
      bdd r2 = hm == h;
      bdd l = layout.expr(numDom,factId,0);
      // build exp(h,h',x) := exp(h,x) /\ x = f1 =>  h' = h + depth(f1)
      exp = exp & bdd_ite(l,r1,r2);
      exp = bdd_exist(exp,fhInfo.Hcurrent);
      exp = bdd_replace(exp,fhInfo.Hnext2current);
    }
  
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  


int anAbs(int x) {
  if (x > 0) 
    return x;
  else
    return -x;
}



//IN
// x,y    : board position 
// j      : tile number of tile occupying that position
// posNum : total number of positions in problem
//OUT
// the manhattan distance of the tile to it goal position
int manhattanDist(int x,int y,int j,int posNum) {
  
  int dim = int(sqrt(double(posNum)));

  // find goal x and y position
  int xg = (j-1) % dim;
  int yg = (dim - 1) - ((j-1) / dim);

  return ( anAbs(x - xg) + anAbs(y - yg) );
} 


    

//IN
// layout : BDDA* bbd layout
// rInfo  : fact reachability data
// numdom : numerical PDDL rep.
// fhInfo : structure with BDD variable info on h and f-encodings
//OUT
// symbolic encoding of the HSPr heuristic
bdd NADLSymbolicManhattan(NADLbddBDDAlayout& layout, FHinfo& fhInfo) {
  
  //     initially exp(h,state) = (h = 0)
  //     for each position for each tile
  //             exp(h,h',state) := exp(h,state) /\ if (state(position)) = tile then (h' = h + dist(position,tile)) else (h' = h) 
  //             exp(h',state) := exist h
  //             exp(h,state) := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building Manhattan heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));
  
  // find number of positions in problem
  // (= tile num + 1 since we also track the empty pos)
  int posNum =  layout.var.size();

  // iterate over tile numbers
  for (int i = 0; i < posNum; i++)
    {
      // find x and y of position
      char buf[32];
      strcpy(buf,(layout.var[i].name).c_str());

      int x = int(buf[1] - '0');
      int y = int(buf[2] - '0');

      // iterate over Tile numbers
      for (int j = 1; j < posNum; j++)
	{
	  if (verbosity > 0)
	    cout << ".";
	  
	  bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
				 bvec(layout.hVars.length,manhattanDist(x,y,j,posNum)));
	  
	  bdd r1 = hm == hPlusd;
	  bdd r2 = hm == h;
	  bdd l =  int2bdd(layout.var[i].current,layout.var[i].length,j);
	  // build exp(h,h',state) := exp(h,state) /\ pos(state) = j  then  h' = h + manhattanDist(pos,j) else h' = h
	  exp = exp & bdd_ite(l,r1,r2);
	  exp = bdd_exist(exp,fhInfo.Hcurrent);
	  exp = bdd_replace(exp,fhInfo.Hnext2current);
	}
    }
      
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  









//IN
// layout : BDDA* bbd layout
// rInfo  : fact reachability data
// numdom : numerical PDDL rep.
// fhInfo : structure with BDD variable info on h and f-encodings
//OUT
// symbolic encoding of the HSPr heuristic
bdd PubNADLSymbolicManhattan(NADLbddPubBDDAlayout& layout, PubFHinfo& fhInfo) {
  
  //     initially exp(h,state) = (h = 0)
  //     for each position for each tile
  //             exp(h,h',state) := exp(h,state) /\ if (state(position)) = tile then (h' = h + dist(position,tile)) else (h' = h) 
  //             exp(h',state) := exist h
  //             exp(h,state) := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building Manhattan heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));

  // find number of positions in problem
  // (= tile num + 1 since we also track the empty pos)
  int posNum =  layout.var.size();
  

  // iterate over positions
  for (int i = 0; i < posNum; i++)
    {
      // find x and y of position
      char buf[32];
      strcpy(buf,(layout.var[i].name).c_str());

      int x = int(buf[1] - '0');
      int y = int(buf[2] - '0');

      // iterate over tile numbers
      for (int j = 1; j < posNum; j++)
	{
	  if (verbosity > 0)
	    cout << ".";
	  
	  bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
				 bvec(layout.hVars.length,manhattanDist(x,y,j,posNum)));
	  
	  bdd r1 = hm == hPlusd;
	  bdd r2 = hm == h;
	  bdd l =  int2bdd(layout.var[i].current,layout.var[i].length,j);
	  // build exp(h,h',state) := exp(h,state) /\ pos(state) = j  then  h' = h + manhattanDist(pos,j) else h' = h
	  exp = exp & bdd_ite(l,r1,r2);
	  exp = bdd_exist(exp,fhInfo.Hcurrent);
	  exp = bdd_replace(exp,fhInfo.Hnext2current);
	}
    }
      
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  






//IN
// layout : BDDA* bbd layout
// fhInfo : structure with BDD variable info on h and f-encodings
// goal   : BDD encoding of goal
//OUT
// symbolic encoding of the HSPr heuristic
bdd symbolicMinHamming(NADLbddBDDAlayout& layout, FHinfo& fhInfo, bdd goal) {
  
  //     initially exp(h,state) = (h = 0)
  //     for each BDD variable, v
  //             exp(h,h',state) := exp(h,state) /\ if (state(v)) = not goal(v) then (h' = h + 1) else (h' = h) 
  //             exp(h',state)   := exist h
  //             exp(h,state)    := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building MinHamming heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));
  

  // iterate state BDD variables
  for (int i = 0; i < layout.var.size(); i++)
    for (int j = 0; j < layout.var[i].length; j++)
      {
	if (verbosity > 0)
	  cout << ".";
	  
	bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
			       bvec(layout.hVars.length,1));
	  
	bdd r1 = hm == hPlusd;
	bdd r2 = hm == h;
	bdd l;
	if ( (bdd_ithvar(layout.var[i].current[j]) & goal) == bddfalse )
	  l = bdd_ithvar(layout.var[i].current[j]);
	else
	  l = bdd_nithvar(layout.var[i].current[j]);
	// build exp(h,h',state) := exp(h,state) /\ if (state(v)) = not goal(v) then (h' = h + 1) else (h' = h)
	exp = exp & bdd_ite(l,r1,r2);
	exp = bdd_exist(exp,fhInfo.Hcurrent);
	exp = bdd_replace(exp,fhInfo.Hnext2current);
      }
  
  
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  







//IN
// layout : BDDA* bbd layout
// fhInfo : structure with BDD variable info on h and f-encodings
// goal   : BDD encoding of goal
//OUT
// symbolic encoding of the HSPr heuristic
bdd PubSymbolicMinHamming(NADLbddPubBDDAlayout& layout, PubFHinfo& fhInfo, bdd goal) {
  
  //     initially exp(h,state) = (h = 0)
  //     for each BDD variable, v
  //             exp(h,h',state) := exp(h,state) /\ if (state(v)) = not goal(v) then (h' = h + 1) else (h' = h) 
  //             exp(h',state)   := exist h
  //             exp(h,state)    := exp[h'/h] 

  if (verbosity > 0)
    cout << "Building MinHamming heuristic:";

  
  bvec h     = bvec_varvec(layout.hVars.length,layout.hVars.current);
  bvec hm    = bvec_varvec(layout.hVars.length,layout.hVars.next);

  
  // init exp
  bdd exp = (bvec_varvec(layout.hVars.length,layout.hVars.current) == bvec(layout.hVars.length,0));
  

  // iterate state BDD variables
  for (int i = 0; i < layout.var.size(); i++)
    for (int j = 0; j < layout.var[i].length; j++)
      {
	if (verbosity > 0)
	  cout << ".";
	  
	bvec hPlusd = bvec_add(bvec_varvec(layout.hVars.length,layout.hVars.current),
			       bvec(layout.hVars.length,1));
	  
	bdd r1 = hm == hPlusd;
	bdd r2 = hm == h;
	bdd l;
	if ( (bdd_ithvar(layout.var[i].current[j]) & goal) == bddfalse )
	  l = bdd_ithvar(layout.var[i].current[j]);
	else
	  l = bdd_nithvar(layout.var[i].current[j]);
	// build exp(h,h',state) := exp(h,state) /\ if (state(v)) = not goal(v) then (h' = h + 1) else (h' = h)
	exp = exp & bdd_ite(l,r1,r2);
	exp = bdd_exist(exp,fhInfo.Hcurrent);
	exp = bdd_replace(exp,fhInfo.Hnext2current);
      }
  
  
  if (verbosity > 0)
    cout << "\n";

  return exp;
}
  































































//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////
//
// Misc Aux. functions
//
//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////



//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the fMin value or -1 if fMin does not exist
int fMin(PDDLbddBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i = layout.fVars.length - 1; i >= 0; i--)
    {
      bdd b = min & bdd_nithvar(layout.fVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.fVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}












//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the fMin value or -1 if fMin does not exist
int fMin(NADLbddBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i = layout.fVars.length - 1; i >= 0; i--)
    {
      bdd b = min & bdd_nithvar(layout.fVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.fVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}








//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the fMin value or -1 if fMin does not exist
int PubfMin(NADLbddPubBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i = layout.fVars.length - 1; i >= 0; i--)
    {
      bdd b = min & bdd_nithvar(layout.fVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.fVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}








// Public BDDA* version
//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the fMin value or -1 if fMin does not exist
int PubfMin(PDDLbddPubBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i = layout.fVars.length - 1; i >= 0; i--)
    {
      bdd b = min & bdd_nithvar(layout.fVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.fVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}








//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the hMin value or -1 if hMin does not exist
int hMin(PDDLbddBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i=layout.hVars.length-1; i >=0 ; i--)
    {
      bdd b = min & bdd_nithvar(layout.hVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.hVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}









// Public BDDA* version
//IN
// layout : bdd variable structure
// min    : (implicit output) open constrained with f = fmin
//          Obs: fmin assumed to be given in current state variables
// open   : the open queue of BDDA* and pure-BDDA*
//OUT
// the hMin value or -1 if hMin does not exist
int PubhMin(PDDLbddPubBDDAlayout& layout,bdd& min,bdd& open) {
  
  int res = 0;
  min = open;
  
  for (int i=layout.hVars.length-1; i >=0 ; i--)
    {
      bdd b = min & bdd_nithvar(layout.hVars.current[i]);
      if (b != bddfalse)
	{
	  min = b;
	  res = 2*res;
	}
      else
	{
	  b = min & bdd_ithvar(layout.hVars.current[i]);
	  if (b != bddfalse)
	    {
	      min = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}




//IN
// var       : integer BDD variable encoding that has been
//             generated using bvec (LSB first in ordering)
// VarLength : Number of bits in var 
// state     : state with encoding of the value 
//OUT
// the value of var, if it exist. Otherwise -1. Notice that this function
// non-deterministic will choose one of the values if several are possible  
int bvecEcoding2int(int varLength, int* var,bdd state) {
  
  int res = 0;
  bdd c = state;
  
  for (int i = varLength - 1; i >= 0; i--)
    {
      bdd b = c & bdd_nithvar(var[i]);
      if (b != bddfalse)
	{
	  c = b;
	  res = 2*res;
	}
      else
	{
	  b = c & bdd_ithvar(var[i]);
	  if (b != bddfalse)
	    {
	      c = b;
	      res = 2*res + 1;
	    }
	  else
	    return -1;
	}
    } 	       
  return res;
}









//IN 
// T : disjunctive partitioning (also output)
// This function adds IV expressions to all 
// untouched variables of each partition.
// This is needed since BDDA* require a representation
// of (x,x') in order to assign the right h and h' values
void IVextend(BDDAinfo& bddaInfo,TRel& T) {
  
  // go through each partition expression
  // and constrain it with an IV expression
  for (int i = 0; i < T.p.size(); i++)
    T.p[i].exp &= IV(setDifference(bddaInfo.stateVarPairs,T.p[i].modify));
}







































































//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////
//
// Misc. Image and PreImage functions
//
//////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////







//IN
// bddaInfo   : partitioned  transition relation
// fhInfo     : BDD variable position of f- and h-values
// T          : optimized disjunctive transition relation
// h          : BDD encoding of heuristic (h current state vars)
// hm         : BDD encoding of heuristic (h next state vars)
// min        : states (in current vars) with minimum f-value (the f-value is not represented)
// fMin       : the minimum f-value (this is the way the f value is given of min)
//OUT
// states (current vars) reached backward from min, paired with their f-value (in current vars)
bdd preImageOpen(BDDAinfo& bddaInfo, FHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min, int fMin) {


  bdd newOpen = bddfalse;  // set of (f,s) pairs of the open queue of BDDA*

  // change min to next variables (we are searching backwards)
  min = bdd_replace(min,bddaInfo.current2next);

  // increment newOpen for each partition in turn
  for (int i=0; i<T.p.size(); i++)
    {
      // make a backward step 
      bdd currentNextVarPairs = T.p[i].exp & min;
      // add heuristic of next vars and quantify next variables
      bdd currentVarsHm = bdd_exist(currentNextVarPairs & hm,bddaInfo.nextVars);
      // add heuristic of current vars
      bdd currentVarsHmH = currentVarsHm & h;
      
      // apply formula to get new f and quantify the two h values
      bdd openFrac = bddaInfo.formulaBackward[fMin] & currentVarsHmH;
      openFrac = bdd_exist(openFrac,fhInfo.Hcurrent);
      openFrac = bdd_exist(openFrac,fhInfo.Hnext);
      newOpen |= openFrac;      
    }
  
  return newOpen;
}







// Public BDDA* version
//IN
// bddaInfo   : varaible info
// fhInfo     : BDD variable position of f- and h-values
// T          : monolithic transition relation
// h          : BDD encoding of heuristic (h current state vars)
// hm         : BDD encoding of heuristic (h next state vars)
// min        : fmin * states  (f, states pair in open with min f value) (both in current vars)
//OUT
// states (current vars) reached backward from min, paired with their f-value (in current vars)
bdd PubPreImageOpen(PubBDDAinfo& bddaInfo, PubFHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min) {

  
  bdd newOpen;  // set of (f,s) pairs of the open queue of BDDA*
  
  // change min to next state variables (we are searching backwards)
  min = bdd_replace(min,bddaInfo.current2next);
  
  // change min to next state f values
  min = bdd_replace(min,fhInfo.Fcurrent2next);
  

  // compute newOpen for the single partition
  
  // associate states in a backward step 
  bdd currentNextVarPairsFm = T.p[0].exp & min;
  // add heuristic of next vars and quantify next variables
  bdd currentVarsHmFm = bdd_exist(currentNextVarPairsFm & hm,bddaInfo.nextVars);
  // add heuristic of current vars
  bdd currentVarsHmH = currentVarsHmFm & h;
  
  // apply formula to get new f and quantify the two h values and f'
  newOpen = bddaInfo.formulaBackward & currentVarsHmH;
  newOpen = bdd_exist(newOpen,fhInfo.Hcurrent);
  newOpen = bdd_exist(newOpen,fhInfo.Hnext);
  newOpen = bdd_exist(newOpen,fhInfo.Fnext);
  
  return newOpen;
}





//IN
// bddaInfo   : partitioned  transition relation
// fhInfo     : BDD variable position of f- and h-values
// T          : optimized disjunctive transition relation
// h          : BDD encoding of heuristic (h and state in current state vars)
// hm         : BDD encoding of heuristic (h and state in  next state vars)
// min        : states (in current vars) with minimum f-value (the f-value is NOT represented)
// fMin       : the minimum f-value (this is the way the f value is given of min)  
//OUT
// states (current vars) reached forward from min, paired with their f-value (in current vars)
bdd imageOpen(BDDAinfo& bddaInfo, FHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min, int fMin) {

  bdd newOpen = bddfalse;  // set of (f,s) pairs of the open queue of BDDA*


  // increment newOpen for each partition in turn
  for (int i=0; i<T.p.size(); i++)
    {
      // make a forward step 
      bdd currentNextVarPairs = min & T.p[i].exp;
      // add heuristic of current vars and quantify current variables
      bdd nextVarsH = bdd_exist(currentNextVarPairs & h,bddaInfo.currentVars);
      // add heuristic of next vars
      bdd nextVarsHmH = nextVarsH & hm;
      
      // apply formula to get new f and quantify the two h values
      bdd openFrac = bddaInfo.formulaForward[fMin] & nextVarsHmH;
      openFrac = bdd_exist(openFrac,fhInfo.Hcurrent);
      openFrac = bdd_exist(openFrac,fhInfo.Hnext);
      newOpen |= openFrac;      
    }



  // change newOpen to current variables (we are searching forward)
  newOpen = bdd_replace(newOpen,bddaInfo.next2current);
  
  return newOpen;
}





//IN
// bddaInfo   : partitioned  transition relation
// fhInfo     : BDD variable position of f- and h-values
// T          : optimized disjunctive transition relation (assumed monolithic)
// h          : BDD encoding of heuristic (h and state in current state vars)
// hm         : BDD encoding of heuristic (h and state in  next state vars)
// min        : states paired with f-values (f and state in current vars) 
//OUT
// states (current vars) reached forward from min, paired with their f-value (in current vars)
bdd PubImageOpen(PubBDDAinfo& bddaInfo, PubFHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min) {

  bdd newOpen;  // set of (f,s) pairs of the open queue of BDDA*

  // change min to next state f values
  min = bdd_replace(min,fhInfo.Fcurrent2next);

  // make a forward step 
  bdd currentNextVarPairsFm = min & T.p[0].exp;
  // add heuristic of current vars and quantify current variables
  bdd nextVarsHFm = bdd_exist(currentNextVarPairsFm & h,bddaInfo.currentVars);
  // add heuristic of next vars
  bdd nextVarsHmHFm = nextVarsHFm & hm;
  
  // apply formula to get new f and quantify the two h values and f'
  newOpen = bddaInfo.formulaForward & nextVarsHmHFm;
  newOpen = bdd_exist(newOpen,fhInfo.Hcurrent);
  newOpen = bdd_exist(newOpen,fhInfo.Hnext);
  newOpen = bdd_exist(newOpen,fhInfo.Hnext);
  newOpen = bdd_exist(newOpen,fhInfo.Fnext);
  

  // change newOpen to current variables (we are searching forward)
  newOpen = bdd_replace(newOpen,bddaInfo.next2current);
  
  return newOpen;
}










//IN
// bddaInfo   : partitioned  transition relation
// fhInfo     : BDD variable position of f- and h-values
// T          : optimized disjunctive transition relation
// h          : BDD encoding of heuristic (h current state vars)
// hm         : BDD encoding of heuristic (h next state vars)
// min        : states (in current vars) with minimum f-value (the f-value is NOT represented)
// fMin       : the minimum f-value (this is the way the f value is given of min)  
// act        : what partition in T to use
//OUT
// states (current vars) reached by partition act forward from min, paired with their f-value (in current vars)
bdd successorOfActOpen(BDDAinfo& bddaInfo, FHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min, int fMin, int act) {

  bdd newOpen = bddfalse;  // set of (f,s) pairs of the open queue of BDDA*

  // make a forward step 
  bdd currentNextVarPairs = min & T.p[act].exp;
  // add heuristic of current vars and quantify current variables
  bdd nextVarsH = bdd_exist(currentNextVarPairs & h,bddaInfo.currentVars);

  // add heuristic of next vars
  bdd nextVarsHmH = nextVarsH & hm;
      
  // apply formula to get new f and quantify the two h values
  bdd openFrac = bddaInfo.formulaForward[fMin] & nextVarsHmH;
  openFrac = bdd_exist(openFrac,fhInfo.Hcurrent);
  openFrac = bdd_exist(openFrac,fhInfo.Hnext);
  newOpen |= openFrac;      
  
  // change newOpen to current variables (we are searching forward)
  newOpen = bdd_replace(newOpen,bddaInfo.next2current);
  
  return newOpen;
}









//IN
// bddaInfo   : partitioned  transition relation
// fhInfo     : BDD variable position of f- and h-values
// T          : optimized disjunctive transition relation
// h          : BDD encoding of heuristic (h current state vars)
// hm         : BDD encoding of heuristic (h next state vars)
// min        : states (in current vars) with minimum f-value (the f-value is not represented)
// fMin       : the minimum f-value (this is the way the f value is given of min)
// act        : action number that should be applied
//OUT
// states (current vars) reached backward from min, paired with their f-value (in current vars)
bdd predecessorOfActOpen(BDDAinfo& bddaInfo, FHinfo& fhInfo, TRel& T, bdd h, bdd hm, bdd min, int fMin,int act) {


  bdd newOpen = bddfalse;  // set of (f,s) pairs of the open queue of BDDA*

  // change min to next variables (we are searching backwards)
  min = bdd_replace(min,bddaInfo.current2next);

  // make a backward step 
  bdd currentNextVarPairs = T.p[act].exp & min;
  // add heuristic of next vars and quantify next variables
  bdd currentVarsHm = bdd_exist(currentNextVarPairs & hm,bddaInfo.nextVars);
  // add heuristic of current vars
  bdd currentVarsHmH = currentVarsHm & h;
  
  // apply formula to get new f and quantify the two h values
  bdd openFrac = bddaInfo.formulaBackward[fMin] & currentVarsHmH;
  openFrac = bdd_exist(openFrac,fhInfo.Hcurrent);
  openFrac = bdd_exist(openFrac,fhInfo.Hnext);
  newOpen |= openFrac;      
   
  
  return newOpen;
}


