//////////////////////////////////////////////////////////
// File  : singleState.search.cc
// Desc. : single state search algorithms based on PDDL
//         and STL hashset manipulation
// Author: Rune M. Jensen
// Date  : 05/03/02
//////////////////////////////////////////////////////////

#include <list>
#include "numDomain.hpp"
#include "reachInfo.hpp"


vector<singleStateAction> mkActions(numDomain &numdom, reachInfo &rInfo, heuristicType heuristic) {
  
  // for each executable action instantiation 
  for (int i=0; i < numDom.act.size(); i++)
    // for each posible instantiation of the action
    for (set<int>::iterator proj = rInfo.act[i].oldProjections.begin(); 
	 proj != rInfo.act[i].oldProjections.end(); ++proj)
      {

	// clean action
	singleStateAction act;
	act.pre.clear();
	act.add.clear();
	act.del.clear();
	act.dg = 0;
	act.dh = 0;
		
	// make/check precondition
	bool check = true;
	for (int j=0; j < numDom.act[i].pre.size(); j++)
	  {
	    int factId = instantiate(numDom.act[i].pre[j],*proj,numDom.act[i].paramNum,
				     rInfo.objMask,rInfo.predMask,
				     rInfo.predShift,rInfo.objShift);
	    
	    if (setMember(rInfo.staticPreds,(numDom.act[i].pre[j])[0]))
	      { // check if fact is reachable if fact is static
		if (!setMember(rInfo.closedFacts,factId))
		  check = false;
	      }
	    else
	      {
		// add the fact to expression
		act.pre.insert(factId);
	      }
	  }

	if (check)
	  {

	    // make delta g and h-value according to Heuristic function
	    switch (heuristic) 
	      {
	      case ht_HSPr:
	        act.dg = 1;
		act.dh = 0;
		break;
	    
	      case ht_minHamming:
		act.dg = 1;
		act.dh = 0;
		break;
		
		
	      case ht_none: 
		// needed by blind BDD search
		act.dg = 1;
		break;
		
	      default:
		cout << "singleState.search.cc : PDDLmkAct: option not covered\nExiting\n";
		exit(-1);
		break;
	      }


	    // add add-list
	    for (int j=0; j < numDom.act[i].add.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].add[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		// add fact to next state
		act part.exp &= layout.expr(numDom,factId,1);
		
		// assume fact is false in current state
		part.exp &= !layout.expr(numDom,factId,0);
		

		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // all add facts don't exist in the current state thus
		    // they contribute to the h-value in the next state
		    part.dh += rInfo.depth[factId];		  
		    break;
		    
		  case ht_minHamming :
		    break;
		    
		  case ht_none :
		    break;
		    
		  default:
		    cout << "partition.cc : PDDLmkAct: option not covered\nExiting\n";
		    exit(-1);
		    break;
		}


		
		// update modify pair set
		PDDLvarDef v(layout.bddVars(factId));
		for (int k=0; k < v.length; k++)
		  {
		    pair<int,int> elem(v.current[k],v.next[k]);
		    part.modify.insert(elem);
		  }
	      }

	    

	    // add del-list
	    for (int j=0; j < numDom.act[i].del.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].del[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // add delta-h of fact (since it is a subset of pre
		    // each of these facts indeed are deleted from the current state) 
		    part.dh -= rInfo.depth[factId];
		    break;
		    
		  case ht_minHamming :
		    break;
		    
		  case ht_none :
		    break;
		  }

		// add the fact to expression
		part.exp &= !layout.expr(numDom,factId,1);
		
		// update modify set
		PDDLvarDef v(layout.bddVars(factId));
		for (int k=0; k < v.length; k++) 
		  {
		    pair<int,int> elem(v.current[k],v.next[k]);
		    part.modify.insert(elem);
		  }
	      }
	    
	    

	    ////////////////////////////////////////////
	    //  Make modify varsets of partition 
	    ////////////////////////////////////////////

	    part.varSetf = bddtrue;
	    part.varSetb = bddtrue;
	    part.subVarSetf = bdd_newpair();
	    part.subVarSetb = bdd_newpair();
	
	    for (set< pair<int,int> >::iterator si = part.modify.begin(); 
		 si != part.modify.end(); ++si)
	      {
		part.varSetf &= bdd_ithvar(si->first); 
		part.varSetb &= bdd_ithvar(si->second);  
		bdd_setpair(part.subVarSetf,si->second,si->first);
		bdd_setpair(part.subVarSetb,si->first,si->second);
	      }
	    
	    
	    // make action name
	    string name = numDom.act[i].name;
	    vector<int> projection;
	    int projId = *proj;
	    id2proj(projId,projection,rInfo.objMask,rInfo.objShift,
		    numDom.act[i].paramNum);
	    for (int k=0; k < projection.size(); k++)
	      name = name + "_" + numDom.obj[ projection[k]-1 ];
	    
	    part.actName = name;

	    if (verbosity > 1)
	      cout << "Action : " << name << " dh=" << part.dh << endl;

	    if (part.exp != bddfalse)
	      {
		// store partition in Tact
                // obs. if exp = bdd_false then it's because we had a contradiction
                // according to balanced sets a la' !at(here)' /\ at(here)'
                // so these can just be ignored
		Tact.p.push_back(part);		
	      }	     	
	  }
      }
}	     	


  

list<string> Astar(numDomain &numdom, reachInfo &rInfo  vector<singleStateAction> action) {



) 

  


/////////////////////////////////
//
// Action generation functions
//
/////////////////////////////////

// IN/OUT
//  act : vector of grounded actions
void makeActions(vector<action> &act) {

  action a;
  char buffer[16];

  // A1_1
  a.name = "A1_1";
  a.pre.insert(Istar2id());
  a.add.insert(Gj2id(1));
  act.push_back(a);

  // A1_2-16
  for (int j = 2; j <= 16; j++)
    {
      sprintf(buffer,"A1_%d",j);
      a.name = string(buffer);
      a.pre.clear();
      a.add.clear();
      a.del.clear();
      a.pre.insert(Istar2id());
      a.pre.insert(Gj2id(j-1));
      a.add.insert(Gj2id(j));
      act.push_back(a);
    }
      

  // A2_1-16
  for (int j = 1; j <= 16; j++)
    {
      sprintf(buffer,"A2_%d",j);
      a.name = string(buffer);
      a.pre.clear();
      a.add.clear();
      a.del.clear();
      a.add.insert(Ij2id(j));
      a.del.insert(Istar2id());
      act.push_back(a);
    }
}


/////////////////////////////////
//
// Heuristic computation
//
/////////////////////////////////

// IN
//  state : set of integers making a state
int hFunc(set<int> &state,set<int> &goal) {

  int res = 0;

  // add one for each fact in goal not present in 
  // the state (this corresponds to the minimum Hamming 
  // distance to a goal state
  for (set<int>::iterator si = goal.begin(); si != goal.end(); ++si)
    {
      int elem = *si;
      if (!setmember(state,elem)) res++;
    }

  return res;
}




int main(int argc,char **argv)  {

  int k = atoi(argv[1]);
  
  vector<action> act;
  makeActions(act);

  //  for (int i=0; i < act.size(); i++) {
  //    cout << "Action " << i << endl;
  //    act[i].print();
  //    cout << endl;
  //  }

  // set up init and goal state
  set<int> init;
  init.insert(Istar2id());
  set<int> goal;
  for (int j = k + 1; j <= 16; j++)
    goal.insert(Gj2id(j));


  // A* search algorithm
  
  // priority Q
  priority_queue<Qvalue,vector<Qvalue>,Qvaluecmp> Q;
  
  // cycle detection (we refuse to store a state with a higher g-value 
  // than we have seen so far)
  map<set<int>,int> G;

  // initialize Q and G
  int h = hFunc(init,goal);
  int g = 0;
  int f = g + h;
  Qvalue Init(f,g,h,init);
  G[init] = g;
  Q.push(Init);

  int it = 0;

  // search loop
  // don't worry we jump out when a solution is found
  while (!Q.empty()) 
    {
      Qvalue top = Q.top();
      Q.pop();
      
      // jump out if goal found
      if (hFunc(top.state,goal) == 0) 
	  break;
      else
	{
	  it++;
	  //cout << "Expanding state=";
	  //printIdSet(top.state);
	  //cout << " (f=" << top.f << ",g=" << top.g << ",h=" << top.h << ") to:\n";
	  // else expand node
	  for (int i = 0; i < act.size(); i++)
	    if ( setdifference(act[i].pre,top.state).empty())
	      {
		// action is applicable, apply it
		set<int> state = top.state;
		state = setdifference(state,act[i].del);
		state = setunion(state,act[i].add);
		h = hFunc(state,goal);
		g = top.g + 1;
		f = g + h;
		Qvalue child(f,g,h,state);
		
		if (G.count(child.state) == 0)
		  {
		    // if state has not been seen before then insert it
		    //cout << "  ";
		    //printIdSet(child.state);
		    //cout << endl;
		    Q.push(child);
		    // and update G
		    G[child.state] = child.g;
		  }
		else if (child.g < G[child.state] )
		  {
		    // else if we haven't seen such a low g value before then insert it
		    //cout << "  ";
		    //printIdSet(child.state);
		    //cout << endl;
		    Q.push(child);
		    // and update G
		    G[child.state] = child.g;
		  }
		// otherwise skip the child
	      }
	}
    }
  cout << it << endl;
  return 0;
}

