//////////////////////////////////////////////////////////
// File  : singleState.search.cc
// Desc. : single state search algorithms based on PDDL
//         and STL set manipulation
// Author: Rune M. Jensen
// Date  : 05/03/02
//////////////////////////////////////////////////////////

#include <list>
#include <set>
#include <map>
#include <queue>
#include "set.hpp"
#include "numDomain.hpp"
#include "reachInfo.hpp"
#include "singleState.search.hpp"
#include "singleState.actions.hpp"
#include "timer.hpp"




//////////////////////////////////////////////////////////
// OBS: THIS CODE IS NOT CHANGED FROM f = (1-w)g + wh
// f = wg*g + wh*h
//
// Regular single-state weighted A* search algorithm
// -------------------------------------------------
//
// Properties
//  Only searches in the space fluents. Static preds are
//  tested and instantiated at action generation time
//
//  A state is a set of facts each fact is encoded as an 
//  integer and a set of fact is represented by set<int>
//
//  Grounded actions are assumed to change the heuristic 
//  value by the same amount no matter what state they are 
//  applied in.
//  
//  This of course restricts the applicability of this 
//  Astar algorithm. On the other hand, it allows us to 
//  bypass the computing the h-value for each state. 
//  We simple associate a dg and dh-value with each action 
//  as for SetA*.  
//
//  cycle detection is encoded by refusing a state seen 
//  before with lower g-value
//
// Assumptions
//  del <= pre
//  add disjoint with pre 
//  (=> add != del)
//
//////////////////////////////////////////////////////////


// IN
// action : vector of grounded actions
// init   : set of facts (fluents) in the initial state
// goal   : set of facts in the goal state
// hinit  : h-value of the initial state
// w      : weight of weighted A*
// OUT
//  list of actions in solution if any
list<string> Astar(numDomain& numDom, reachInfo& rInfo,vector<singleStateAction> &action, 
		   set<int> init, set<int> goal,int hinit,double w) {

  timer searchClk,extractClk;


  if (verbosity > 0)
    searchClk.start();

  
  // priority Queue
  priority_queue<singleQvalue,vector<singleQvalue>,singleQvaluecmp> Q;
  
  // cycle detection (we refuse to store a state with a higher g-value 
  // than we have seen so far)
  map<set<int>,int> G; 

  // initialize Q and G
  int g = 0;
  int h = hinit;
  double f = (1-w)*g + w*h;
  singleQvalue Init(f,g,h,-1,NULL,init);
  G[init] = g;
  Q.push(Init);

  int it = 0;
  int maxQsize = 0;

  singleQvalue parent;
  bool solutionFound = false;
  
  // search loop
  // don't worry we jump out when a solution is found
  while (!Q.empty()) 
    {
      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      // pop Queue
      parent = Q.top();
      Q.pop();

      singleQvalue* parentRef = new singleQvalue(parent); // we need to keep a ref. for solution extraction
      
      // jump out if goal found
      if (setSubset(goal,parent.state))
	{
	  solutionFound = true;
	  break;
	}
      else
	{
	  it++;
	  if (verbosity > 0)
	    {
	      // debug
	      cout << "it=" << it << " |Q|=" << Q.size() << " Expanding \n";
	      printFactSet(parent.state,rInfo.predShift,rInfo.objShift,
		  rInfo.predMask,rInfo.objMask,rInfo.predArity,numDom);
	      cout << "\n (f=" << parent.f << ",g=" << parent.g << ",h=" << parent.h << ") [";
	      if (Q.size() > maxQsize) maxQsize = Q.size();
	    }
	  for (int i = 0; i < action.size(); i++)
	    if ( setSubset(action[i].pre,parent.state) && 
		 setIntersection(action[i].add,parent.state).empty() )
	      {
		// action is applicable: (pre <= S and no add-facts exist in S)
		// apply it
		set<int> state = setDifference(parent.state,action[i].del);
		state = setUnion(state,action[i].add);
		g = parent.g + action[i].dg;
		h = parent.h + action[i].dh;
		f = (1-w)*g + w*h;
		singleQvalue child(f,g,h,i,parentRef,state);
		
		if (G.count(child.state) == 0)
		  {
		    // state has not been seen before insert it
		    // debug
		    cout << "\n Applied: " << action[i].name << " (not seen before)\n"; 
		    cout << "    New state: ";
		    printFactSet(state,rInfo.predShift,rInfo.objShift,
				     rInfo.predMask,rInfo.objMask,rInfo.predArity,numDom);
		    cout << endl;
		    Q.push(child);
		    // and update G
		    G[child.state] = child.g;
		    if (verbosity > 0)
		      if (action[i].dh < 0)			
			cout << "+";
		      else if (action[i].dh > 0)
			cout << "-";
		      else
			cout << "0";
		  }
		else if (child.g < G[child.state] )
		  {
		    // debug
		    cout << "\n Applied: " << action[i].name << " (not seen before with such low g)\n"; 		   
		    cout << "    New state: ";
		    printFactSet(state,rInfo.predShift,rInfo.objShift,
				     rInfo.predMask,rInfo.objMask,rInfo.predArity,numDom);
		    cout << endl; 
		    // we haven't seen such a low g value before, insert it
		    Q.push(child);
		    // and update G
		    G[child.state] = child.g;
		    if (verbosity > 0)
		      if (action[i].dh < 0)			
			cout << "+";
		      else if (action[i].dh > 0)
			cout << "-";
		      else
			cout << "0";
		  }		
		else
		  {
		    // otherwise skip the child
		    if (verbosity > 0)
		      cout << "e";
		    // debug
		    cout << "\n Could not apply: " << action[i].name << " (state seen before with lower g)\n"; 		    
		  }
	      }
	 
	  if (verbosity > 0)
	    cout << "]\n";
	}
    }
      

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
      
  list<string> res;

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return res;

  // extract solution if solution is found
  
  if (solutionFound)
    {
      singleQvalue* currentNode = &parent;
      while (currentNode != NULL && currentNode->actionNo != -1)
	{
	  res.push_front(action[currentNode->actionNo].name);
	  currentNode = currentNode->parent;
	}
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  
  return res;
}








//////////////////////////////////////////////////////////
//
// Regular single-state backward weighted A* search 
// algorithm
// -------------------------------------------------
//
// Searches backward for e.g. applying the HSPr heuristic
//
// Properties
//  Only searches in the space fluents. Static preds are
//  tested and instantiated at action generation time
//
//  A state is a set of facts each fact is encoded as an 
//  integer and a set of fact is represented by set<int>
//
//  Grounded actions are assumed to change the heuristic 
//  value by the same amount no matter what state they are 
//  applied in.
//  
//  This of course restricts the applicability of this 
//  Astar algorithm. On the other hand, it allows us to 
//  bypass the computing the h-value for each state. 
//  We simple associate a dg and dh-value with each action 
//  as for SetA*.  
//
//  cycle detection is encoded by refusing a state seen 
//  before with lower g-value
//
// Assumptions
//  del <= pre
//  add disjoint with pre 
//  (=> add != del)
//
//////////////////////////////////////////////////////////


// IN
// action : vector of grounded actions
// init   : set of facts (fluents) in the initial state
//          (=the goal state of the PDDL problem. Obs: we assume 
//           there is only a single goal state)
// goal   : set of facts in the goal state
//          (the initial state of the PDDL prolem)
// hinit  : h-value of the initial state (= PDDL goal state)
// wg, wh : weight of weighted A* f = wg*g + wh*h
// OUT
//  list of actions in solution if any
list<string> AstarBackward(numDomain& numDom, reachInfo& rInfo,vector<singleStateAction> &action, 
		   set<int> init, set<int> goal,int hinit,double wg,double wh) {

  
  timer searchClk,extractClk;


  if (verbosity > 0)
    searchClk.start();

  
  // priority Queue
  priority_queue<singleQvalue,vector<singleQvalue>,singleQvaluecmp> Q;
  
  // cycle detection (we refuse to store a state with a higher g-value 
  // than we have seen so far)
  map<set<int>,int> G; 

  // initialize Q and G
  int g = 0;
  int h = hinit;
  double f = wg*g + wh*h;
  singleQvalue Init(f,g,h,-1,NULL,init);
  G[init] = g;
  Q.push(Init);

  int it = 0;
  int maxQsize = 0;

  singleQvalue parent;
  bool solutionFound = false;
  
  // search loop
  // don't worry we jump out when a solution is found
  while (!Q.empty()) 
    {

      if (verbosity > 0)
	if (searchClk.stop() > timeout)
	  break;

      // pop Queue
      parent = Q.top();
      Q.pop();

      singleQvalue* parentRef = new singleQvalue(parent); // we need to keep a ref. for solution extraction
      
      // jump out if goal found
      if (goal == parent.state)
	{
	  solutionFound = true;
	  break;
	}
      else
	{
	  it++;
	  if (verbosity > 0)
	    {
	      cout << " it=" << it << " |Q|=" << Q.size() << " Expanding";
	      cout << " (f=" << parent.f << ",g=" << parent.g << ",h=" << parent.h << ") [";

	      if (Q.size() > maxQsize) maxQsize = Q.size();
	    }
	  for (int i = 0; i < action.size(); i++)
	    {
	      // make the backward precondition :	  
	      // 1) pda <= S',  where pda = (pre - del) + add 
	      // 2) no del-facts exists in S'
                                                                
	      set<int> pda = setDifference(action[i].pre,action[i].del);
	      pda = setUnion(pda,action[i].add);
		
	      if ( setSubset(pda,parent.state) &&
		   setIntersection(action[i].del,parent.state).empty() )
		{		  
		  // action is backward applicable, apply it backward
		  set<int> newState = setUnion(parent.state,action[i].del);
		  newState = setDifference(newState,action[i].add);

		  g = parent.g + action[i].dg;
		  h = parent.h - action[i].dh; // we subtract dh since it was generated in forward direction
		  f = wg*g + wh*h;
		  singleQvalue child(f,g,h,i,parentRef,newState);
		  
		  if (G.count(child.state) == 0)
		    {
		      // state has not been seen before insert it
		      Q.push(child);
		      // and update G
		      G[child.state] = child.g;
			if (verbosity > 0)
			  if (action[i].dh < 0)			
			    cout << "-";
			  else if (action[i].dh > 0)
			    cout << "+";
			  else
			    cout << "0";
		    }
		  else if (child.g < G[child.state] )
		    {
		      // we haven't seen such a low g value before, insert it
		      Q.push(child);
		      // and update G
		      G[child.state] = child.g;
			if (verbosity > 0)
			  if (action[i].dh < 0)			
			    cout << "-";
			  else if (action[i].dh > 0)
			    cout << "+";
			  else
			    cout << "0";
		    }		
		  else
		    {
		      // otherwise skip the child
			if (verbosity > 0)
			  cout << "e";
		    }
		}
	    }
	 
	  if (verbosity > 0)
	    cout << "]\n";
	}
    }
      

  if (verbosity > 0)
    {
      Tsearch = searchClk.stop();
      cout << "Total search time: " << Tsearch << " seconds\n";
      cout << "max Q size: " << maxQsize << endl;
      iterationNum = it;
      maxSearchQueueSize = maxQsize;
      extractClk.start();
    }
      
  list<string> res;
  // extract solution if solution is found

  if (verbosity > 0)
    if (searchClk.stop() > timeout)
      return res;

  
  if (solutionFound)
    {
      singleQvalue* currentNode = &parent;
      while (currentNode != NULL && currentNode->actionNo != -1)
	{
	  res.push_back(action[currentNode->actionNo].name);
	  currentNode = currentNode->parent;
	}
    }
  
  if (verbosity > 0)
    {
      Textract = extractClk.stop();
      cout << "Total solution extraction time: " 
	   << Textract << " seconds\n";
    }
  
  return res;
}

