//////////////////////////////////////////////////////////
// File  : singleState.actions.cc
// Desc. : single state search algorithms based on PDDL
//         and STL set manipulation
// Author: Rune M. Jensen
// Date  : 05/03/02
//////////////////////////////////////////////////////////

#include <list>
#include <set>
#include "set.hpp"
#include "main.hpp"
#include "numDomain.hpp"
#include "reachInfo.hpp"
#include "singleState.actions.hpp"



//////////////////////////////////////////////////////////
//
// singleStateAction member fundtion
//
//
//////////////////////////////////////////////////////////
 
void singleStateAction::print(numDomain& numDom, reachInfo& rInfo) {
  cout << "  Name: " << name << endl;
  cout << "   pre: ";
  printFactSet(pre,rInfo.predShift,rInfo.objShift,rInfo.predMask,rInfo.objMask,
	       rInfo.predArity,numDom);  	   
  cout << "\n   add: ";				   
  printFactSet(add,rInfo.predShift,rInfo.objShift,rInfo.predMask,rInfo.objMask,
	       rInfo.predArity,numDom);		   
  cout << "\n   del: ";				   
  printFactSet(del,rInfo.predShift,rInfo.objShift,rInfo.predMask,rInfo.objMask,
	       rInfo.predArity,numDom);
  cout << "\n   dg=" << dg;
  cout << "\n   dh=" << dh;
  cout << endl;
  
}


//////////////////////////////////////////////////////////
//
// Function for making grounded actiojn vector
//
//
//////////////////////////////////////////////////////////


// IN
//  action : (implicit output) vector of actions
//  numdom : nummerical PDDL domain representation
//  rinfo  : fact reachability information
//  heuristic : heuristic applied
void mkActions(vector<singleStateAction> &action,numDomain &numDom, reachInfo &rInfo, heuristicType heuristic) {
  
  // for each executable action instantiation 
  for (int i=0; i < numDom.act.size(); i++)
    // for each posible instantiation of the action
    for (set<int>::iterator proj = rInfo.act[i].oldProjections.begin(); 
	 proj != rInfo.act[i].oldProjections.end(); ++proj)
      {

	// clean action
	singleStateAction act;
	act.pre.clear();
	act.add.clear();
	act.del.clear();
	act.dg = 0;
	act.dh = 0;
		
	// make/check precondition
	bool check = true;
	for (int j=0; j < numDom.act[i].pre.size(); j++)
	  {
	    int factId = instantiate(numDom.act[i].pre[j],*proj,numDom.act[i].paramNum,
				     rInfo.objMask,rInfo.predMask,
				     rInfo.predShift,rInfo.objShift);
	    
	    if (setMember(rInfo.staticPreds,(numDom.act[i].pre[j])[0]))
	      { // check if fact is reachable if fact is static
		if (!setMember(rInfo.closedFacts,factId))
		  check = false;
	      }
	    else
	      {
		// add the fact to expression
		act.pre.insert(factId);
	      }
	  }

	if (check)
	  {

	    // make delta g and h-value according to Heuristic function
	    switch (heuristic) 
	      {
	      case ht_HSPr:
	        act.dg = 1;
		act.dh = 0;
		break;
	    
	      case ht_minHamming:
		act.dg = 1;
		act.dh = 0;
		break;
		
		
	      case ht_none: 
		// needed by blind BDD search
		act.dg = 1;
		break;
		
	      default:
		cout << "singleState.search.cc : mkActions: option not covered\nExiting\n";
		exit(-1);
		break;
	      }


	    // add add-list
	    for (int j=0; j < numDom.act[i].add.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].add[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		// add fact to next state
		act.add.insert(factId);
		
		
		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // all add facts are assumed not to exist in the current state thus
		    // they contribute to the h-value in the next state
		    act.dh += rInfo.depth[factId];		  
		    break;
		    
		  case ht_minHamming :
		    // need to do something here
		    break;
		    
		  case ht_none :
		    break;
		    
		  default:
		    cout << "singleState.actions.cc : mkAtions: option not covered\nExiting\n";
		    exit(-1);
		    break;
		}
    	      }

	    

	    // add del-list
	    for (int j=0; j < numDom.act[i].del.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].del[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // add delta-h of fact (since it is a subset of pre
		    // each of these facts indeed are deleted from the current state) 
		    act.dh -= rInfo.depth[factId];
		    break;
		    
		  case ht_minHamming :
		    break;
		    
		  case ht_none :
		    break;
		  }

		// add the fact to expression
		act.del.insert(factId);		
	      }
	   	    
	    
	    // make action name
	    string name = numDom.act[i].name;
	    vector<int> projection;
	    int projId = *proj;
	    id2proj(projId,projection,rInfo.objMask,rInfo.objShift,
		    numDom.act[i].paramNum);
	    for (int k=0; k < projection.size(); k++)
	      name = name + "_" + numDom.obj[ projection[k]-1 ];
	    
	    act.name = name;


	    // only add action if delete and add set is disjoint
	    if ( setIntersection(act.add,act.del).empty() )
	      {
                // obs. if add and del overlaps then it's because we had a contradiction
                // according to balanced sets a la' !at(here)' /\ at(here)'
                // so these can just be ignored

		// however before adding the ation we also want to make a sanity check
                // assuring that add is disjoint from pre and that del is a subset of 
                // pre as we assume		
		if ( setIntersection(act.add,act.pre).empty() && setSubset(act.del,act.pre) )
		  {
		    if (verbosity > 1)
		      cout << "Action : " << name << " (dg=" << act.dg << ",dh=" << act.dh << ")\n";

		    // sanity check succeeded add action 
		    action.push_back(act);
		  }
		else
		  {
		    // sanity check failed abort !
		    cout << "singleState.actions.cc : mkActions : action " << act.name 
			 << " failed the del < pre, del != add, add != pre assumption\nexiting\n";
		    exit(1);
		  }
	      }	     	
	  }
      }
}	     	



//////////////////////////////////////////////////////////
//
// Init and goal translation functions
//
//////////////////////////////////////////////////////////



//IN
// rInfo : reachability analysis data
//OUT
// set of IDs of fluent facts in init state
set<int> init2intSet(reachInfo &rInfo) {
  
  set<int> res;
  for (set<int>::iterator si = rInfo.initFluents.begin();
       si != rInfo.initFluents.end(); ++si)
    res.insert(*si);
  return res;
}



//IN
// rInfo : reachability analysis data
// goal  : goal facts of PDDL problem
//         Obs: this is assumed to be just a single state
//OUT
// set of IDs of facts in the goal state
set<int> goal2intSet(reachInfo &rInfo,vector< vector<int> > &goal) {

  set<int> res;

  //make init expression
  for (int i=0; i < goal.size(); i++)
    {
      int id = fact2id(goal[i],rInfo.predShift,rInfo.objShift);
      res.insert(id);
    }

  return res;
}








//IN
// rInfo     : data from reachability analysis
// s         : vector of true facts (obs s can only represent a single state)
//OUT
// HSPr-value of the state
int hHSPrSingle(reachInfo &rInfo,vector< vector<int> > s) {

  int sum = 0;

  //sum depths of facts in s
  for (int i=0; i < s.size(); i++)
    {
      int id = fact2id(s[i],rInfo.predShift,rInfo.objShift);
      sum += rInfo.depth[id];
    }
  return sum;  
}







