///////////////////////////////////////////////////////
// File  : searchQueue.hpp  
// Desc. : Header for searchQueue.cc
// Author: Rune M. Jensen CS, CMU
// Date  : 10/28/02 
///////////////////////////////////////////////////////

#ifndef SEARCHQUEUEHPP
#define SEARCHQUEUEHPP

#include <list>
#include <utility>
#include <queue>

#include "bddLayout.hpp"
#include "partition.hpp"
#include "universal.hpp"



struct stateSet {
  bdd *state;
  int size;
  stateSet(bdd *st,int s) {state = st; size=s;}
  stateSet(int s) {state = new bdd[s]; size=s;}
};


///////////////////////////////////////////////////////
// Structures for Priority Queue datastructure
///////////////////////////////////////////////////////


struct bddNode {
  int size;
  bdd b;
  bddNode(int si, bdd bi) {size = si; b = bi;}
};


struct bddNodeCmp {
  bool operator() (bddNode n1,bddNode n2) 
  {return (n1.size > n2.size);}
};


struct Mvalue {
  priority_queue<bddNode,vector<bddNode>,bddNodeCmp> q;      
                      // BDD priority queue
  bool goalOverlap;   // true if some state in the node overlap with the goal states
  Mvalue(priority_queue<bddNode,vector<bddNode>,bddNodeCmp> qi,bool goalOverlapi)
  {q = qi; goalOverlap = goalOverlapi;}
  Mvalue() {};        // default value needed for map
    // debug { goalOverlap = false; }
};


struct Qvalue {
  double f;
  int g;
  int h;
  Qvalue(double fi,int gi,int hi) 
  {f = fi; g = gi; h = hi;} 
};


// lowest f value gets highest priority
// ties are broken in favor of lowest h value
struct QvalueCmp {
  bool operator() (Qvalue v1,Qvalue v2) 
  {if (v1.f == v2.f) return (v1.h > v2.h);
  else return (v1.f > v2.f);}
};


// ghqueue abstract datatype
struct ghQueue {
  int upperBound;         // coalescing upper bound
  double wg;
  double wh;
  int bddNum;             // number of BDDs in the queue 
  map< pair<int,int> , Mvalue > M;                   // mapping from (g,h) -> bdd node priority queue
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp> Q; // pqueue keeping track of (g,h) entry 
                                                     // with highest priority 
  ghQueue(int u, double wgi, double whi) 
  {upperBound = u; wg = wgi; wh = whi; bddNum = 0;}
  bdd popTop(double& f,int& g,int& h);
  void insert(bdd set,bool goalOverlap,int g,int h);
  bool empty() { Q.empty(); };
  void clear();
  bool topHasGoalOverlap();
  int  gOfTop();
  int size() { return bddNum; }
  void print();
  
};




// lowest f value gets highest priority
// ties are broken in favor of lowest g value
struct QvalueCmp2 {
  bool operator() (Qvalue v1,Qvalue v2) 
  {if (v1.f == v2.f) return (v1.g > v2.g);
  else return (v1.f > v2.f);}
};



// ghSCqueue abstract datatype
// this queue is used by guided fault tolerant planning
// and has another tiebreaking rule than 
// ghQueue: given equal f-value, the node with least 
// g-value is expanded. in this way any promissing node at
// weak preimage level i is expanded before any at level i+1
// when computing a preComponent
struct ghSCQueue {
  int upperBound;         // coalescing upper bound
  double wg;
  double wh;
  int bddNum;             // number of BDDs in the queue 
  map< pair<int,int> , Mvalue > M;                   // mapping from (g,h) -> bdd node priority queue
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp2> Q; // pqueue keeping track of (g,h) entry 
                                                     // with highest priority 
  ghSCQueue(int u, double wgi, double whi) 
  {upperBound = u; wg = wgi; wh = whi; bddNum = 0;}
  bdd popTop(double& f,int& g,int& h);
  void insert(bdd set,bool goalOverlap,int g,int h);
  bool empty() { Q.empty(); };
  void clear();
  bool topHasGoalOverlap();
  int  gOfTop();
  int size() { return bddNum; }
  void print();
  
};




///////////////////////////////////////////////////////
// Structure for loop detection
///////////////////////////////////////////////////////


// within reachable sets for pruning A* search paths
struct  reachedStates {
  vector<bdd> withinSet;      // keeps the set of states seen within the given g-level        
  int seenDepth;
  void update(bdd states,int g); 
  bdd prune(bdd states,int g);
  bdd within(int g); // returns the within set of g
  list<string> extractSolution(TRel& TAct, bdd goal, int gLevel);
  list<string> extractSolutionBackward(TRel& TAct, bdd goal, int gLevel);
  void print();
  reachedStates(int maxDepth)  // init maxDepth to above the max search depth
    {seenDepth=0; withinSet.resize(maxDepth+2,bddfalse);}
};



///////////////////////////////////////////////////////
// Data structure for guided strong cyclic planning
///////////////////////////////////////////////////////

struct Smap {
  map<int,bdd > S;
  Smap(int hGoal, bdd goal);
  bdd expandGuided(TRel& T, ActInfo& Ainfo, bdd AccSA, bdd goal);
};
    

#endif  
