////////////////////////////////////////////////////////
// File  : searchQueue.cc  
// Desc. : Implementation of search queue and reached structure 
// Author: Rune M. Jensen CS, CMU
// Date  : 10/28/02
////////////////////////////////////////////////////////

#include <bdd.h>
#include <math.h>
#include <queue>
#include <list>
#include "main.hpp"
#include "numDomain.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"
#include "searchQueue.hpp"
#include "searchAlgAuxFunc.hpp"



////////////////////////////////////////////////////////
//
// G-H Priority Queue datastructure
// implementation
//
////////////////////////////////////////////////////////


void ghQueue::clear() {
  M.clear();
  while (!Q.empty())
    Q.pop();
  bddNum = 0;
}
  



//IN
// (implicit outputs)
// f  :  returns f of top
// g  :  returns g of top
// h  :  returns h of top
//OUT
// a bdd of top states 
bdd ghQueue::popTop(double& f,int& g,int& h) {


  bdd res;

  if (Q.empty())
    return bddfalse; // the queue is empty
  else
    {

      // get g and h of node with highest priority
      Qvalue Qtop = Q.top();
      bddNum--;
      
      g = Qtop.g;
      h = Qtop.h;
      f = Qtop.f;
      
      pair<int,int> gh(g,h);
      // lookup bdd of node
      
      Mvalue Mentry = M[gh];

      // pop the top bdd node of the BDD queue 
      bddNode topBddNode(Mentry.q.top()); 
      res = topBddNode.b;
      Mentry.q.pop();

      // update Q and M
      if (Mentry.q.empty())
	{ 
	  // the BDD queue has been emptied so pop the (g,h) of Mentry from Q
	  Q.pop();
	  // and erase Mentry from M
	  M.erase(gh);
	}
      else
	{ 
	  // update Mentry in M (don't change Q since the entry still exists)
	  M[gh] = Mentry;
	}
      
      return res;
    }
}



void ghQueue::insert(bdd set,bool goalOverlap,int g, int h) {

  pair<int,int> gh(g,h);
  
  if (M.count(gh) > 0)
    {
      // key exists in M : update it's BDD queue with "set"
      Mvalue Mentry = M[gh];

      // try to merge it with the top node, otherwise insert "set" as an independent
      // node in the BDD queue  
      bddNode topBddNode(Mentry.q.top()); 
      if (bdd_nodecount(set) + topBddNode.size < upperBound)
	{
	  // coalecs set and topbddnode
	  Mentry.q.pop();
	  topBddNode.b |= set;
	  topBddNode.size = bdd_nodecount(topBddNode.b);
	  Mentry.q.push(topBddNode);
	}
      else
	{ // insert "set" as a new node in the BDD queue 
	  // (obs reusing topnode to create the new node)
	  bddNum++;
	  topBddNode.size = bdd_nodecount(set);
	  topBddNode.b = set;
	  Mentry.q.push(topBddNode);
	}

      
      bool NewGoalOverLap = ( Mentry.goalOverlap || goalOverlap );
      // you cannot make this assignment directly, it won't work
      Mentry.goalOverlap = NewGoalOverLap; 
            
      // finally update value in M
      M[gh] = Mentry;           
    }
  else
    {
      // key does not exist: create it both in Q and M
      bddNum++;
      
      // update M
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> q;
      q.push(bddNode(bdd_nodecount(set),set));
      Mvalue Mentry(q,goalOverlap);
      M[gh] = Mentry;
    
      // update Q
      Q.push(Qvalue(wg*g + wh*h,g,h));
    }
}


bool ghQueue::topHasGoalOverlap() {

  // get g and h of node with highest priority
  Qvalue Qtop = Q.top();
    
  pair<int,int> gh(Qtop.g,Qtop.h);
  
  // lookup entry in M
  
  Mvalue Mentry = M[gh];

  return Mentry.goalOverlap;
}
  

int ghQueue::gOfTop() { 
  Qvalue Qtop = Q.top();
  return Qtop.g;
}


void ghQueue::print() {
  cout << "SetA* ghQueue structure\n";
  cout << " Upper bound=" << upperBound << endl;
  cout << " Queue content:\n";
  
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp> q = Q;
  
  // print each entry in Q 
  while (!q.empty()) 
    {
      Qvalue qtop = q.top();
      q.pop();
      
      pair<int,int> gh(qtop.g,qtop.h);
      Mvalue Mentry = M[gh];
      cout << "  (f=" << qtop.f << ",g=" << qtop.g << ",h=" << qtop.h << ") : goalOverLap=" << Mentry.goalOverlap;
      
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> bddq = Mentry.q;
      
      //print info for each BDD node 
      while (!bddq.empty())
	{
	  bddNode node = bddq.top();
	  bddq.pop();
	  cout << " (size=" << node.size << ")";
	}
      
      cout << endl;
    }      
}









////////////////////////////////////////////////////////
//
// Fault tolerant G-H Priority Queue datastructure
// implementation
//
////////////////////////////////////////////////////////


void ghSCQueue::clear() {
  M.clear();
  while (!Q.empty())
    Q.pop();
  bddNum = 0;
}
  



//IN
// (implicit outputs)
// f  :  returns f of top
// g  :  returns g of top
// h  :  returns h of top
//OUT
// a bdd of top states 
bdd ghSCQueue::popTop(double& f,int& g,int& h) {


  bdd res;

  if (Q.empty())
    return bddfalse; // the queue is empty
  else
    {

      // get g and h of node with highest priority
      Qvalue Qtop = Q.top();
      bddNum--;
      
      g = Qtop.g;
      h = Qtop.h;
      f = Qtop.f;
      
      pair<int,int> gh(g,h);
      // lookup bdd of node
      
      Mvalue Mentry = M[gh];

      // pop the top bdd node of the BDD queue 
      bddNode topBddNode(Mentry.q.top()); 
      res = topBddNode.b;
      Mentry.q.pop();

      // update Q and M
      if (Mentry.q.empty())
	{ 
	  // the BDD queue has been emptied so pop the (g,h) of Mentry from Q
	  Q.pop();
	  // and erase Mentry from M
	  M.erase(gh);
	}
      else
	{ 
	  // update Mentry in M (don't change Q since the entry still exists)
	  M[gh] = Mentry;
	}
      
      return res;
    }
}



void ghSCQueue::insert(bdd set,bool goalOverlap,int g, int h) {

  pair<int,int> gh(g,h);
  
  if (M.count(gh) > 0)
    {
      // key exists in M : update it's BDD queue with "set"
      Mvalue Mentry = M[gh];

      // try to merge it with the top node, otherwise insert "set" as an independent
      // node in the BDD queue  
      bddNode topBddNode(Mentry.q.top()); 
      if (bdd_nodecount(set) + topBddNode.size < upperBound)
	{
	  // coalecs set and topbddnode
	  Mentry.q.pop();
	  topBddNode.b |= set;
	  topBddNode.size = bdd_nodecount(topBddNode.b);
	  Mentry.q.push(topBddNode);
	}
      else
	{ // insert "set" as a new node in the BDD queue 
	  // (obs reusing topnode to create the new node)
	  bddNum++;
	  topBddNode.size = bdd_nodecount(set);
	  topBddNode.b = set;
	  Mentry.q.push(topBddNode);
	}

      
      bool NewGoalOverLap = ( Mentry.goalOverlap || goalOverlap );
      // you cannot make this assignment directly, it won't work
      Mentry.goalOverlap = NewGoalOverLap; 
            
      // finally update value in M
      M[gh] = Mentry;           
    }
  else
    {
      // key does not exist: create it both in Q and M
      bddNum++;
      
      // update M
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> q;
      q.push(bddNode(bdd_nodecount(set),set));
      Mvalue Mentry(q,goalOverlap);
      M[gh] = Mentry;
    
      // update Q
      Q.push(Qvalue(wg*g + wh*h,g,h));
    }
}


bool ghSCQueue::topHasGoalOverlap() {

  // get g and h of node with highest priority
  Qvalue Qtop = Q.top();
    
  pair<int,int> gh(Qtop.g,Qtop.h);
  
  // lookup entry in M
  
  Mvalue Mentry = M[gh];

  return Mentry.goalOverlap;
}
  

int ghSCQueue::gOfTop() { 
  Qvalue Qtop = Q.top();
  return Qtop.g;
}


void ghSCQueue::print() {
  cout << "SetA* ghQueue structure\n";
  cout << " Upper bound=" << upperBound << endl;
  cout << " Queue content:\n";
  
  priority_queue<Qvalue,vector<Qvalue>,QvalueCmp2> q = Q;
  
  // print each entry in Q 
  while (!q.empty()) 
    {
      Qvalue qtop = q.top();
      q.pop();
      
      pair<int,int> gh(qtop.g,qtop.h);
      Mvalue Mentry = M[gh];
      cout << "  (f=" << qtop.f << ",g=" << qtop.g << ",h=" << qtop.h << ") : goalOverLap=" << Mentry.goalOverlap;
      
      priority_queue<bddNode,vector<bddNode>,bddNodeCmp> bddq = Mentry.q;
      
      //print info for each BDD node 
      while (!bddq.empty())
	{
	  bddNode node = bddq.top();
	  bddq.pop();
	  cout << " (size=" << node.size << ")";
	}
      
      cout << endl;
    }      
}





////////////////////////////////////////////////////////
//
// Reachedstates datatype implementation
//
////////////////////////////////////////////////////////


//IN
// states      : the set of new states that should be inserted into the set of reached states
// g           : the g value=depth of=image number of the states
// OUT
// the reached states data structure updated with the new states
void reachedStates::update(bdd states,int g) {

  if (g > seenDepth) {
    // update maxDepth
    for (int i=seenDepth+1; i<=g; i++)
      withinSet[i] |= withinSet[seenDepth];
    seenDepth = g;
  }

  // update reachablestates with new states
  for (int i=g; i<=seenDepth; i++)
  withinSet[i] |= states;
}
  


//IN
// states      : states from which states already reached with lower or same cost should be pruned
// g           : the g value=depth of=image number of the states
// OUT
// a bdd representing the subset of states from states that has not been 
// reached earlier with a lower or equal g-value
bdd reachedStates::prune(bdd states,int g) {
  if (g > seenDepth)
    return (states & !withinSet[seenDepth]);
  else
    return (states & !withinSet[g]);
}
      

//IN
// g           : the g level of within set
// OUT
// the states reached within the g-level.
// (corrected for the case where g is above seen depth)
bdd reachedStates::within(int g) {
  if (g > seenDepth)
    return withinSet[seenDepth];
  else
    return withinSet[g];
}



//IN
// def           : domain definition
// goal          : bdd of goal set
// glevel        : the g level the goal overlap is found at
// OUT
// a solution given as a list of action names
// OBS: this function desctructs the within sets !!
list<string> reachedStates::extractSolution(TRel& Tact, bdd goal, int gLevel) {

  list<string> solution;
  
  // generate a set of explored images from the within set
  for (int i=gLevel; i > 0; i--)
      withinSet[i] != withinSet[i-1];
  
  // constrain goal set to goal states and choose one
  // OBS we may not extract the exact path that was found by the last node of A*
  // but the generated path is as least as cheap
  withinSet[gLevel] &= goal;
  withinSet[gLevel] = bdd_satone(withinSet[gLevel]);
  bdd previousState;

  for (int i = gLevel; i > 0; i--)
    { 
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = predecessorOfAct(Tact,withinSet[i-1],withinSet[i],j);	  
	}
      if (verbosity > 1) cout << "Extracted action: " << Tact.p[j].actName << endl;
      cout.flush();
      withinSet[i-1] = previousState;
      solution.push_front(Tact.p[j].actName);
    }

  return solution;
}




//IN
// def           : domain definition
// goal          : bdd of goal set
// glevel        : the g level the goal overlap is found at
// OUT
// a solution given as a list of action names
// OBS: this function desctructs the within sets !!
list<string> reachedStates::extractSolutionBackward(TRel& Tact, bdd goal, int gLevel) {

  list<string> solution;
  
  // generate a set of explored images from the within set
  for (int i=gLevel; i > 0; i--)
      withinSet[i] != withinSet[i-1];
  
  // constrain goal set to goal states and choose one
  // OBS we may not extract the exact path that was found by the last node of A*
  // but the generated path is as least as cheap
  withinSet[gLevel] &= goal;
  withinSet[gLevel] = bdd_satone(withinSet[gLevel]);
  bdd previousState;

  for (int i = gLevel; i > 0; i--)
    { 
      int j=-1;
      previousState = bddfalse;
      while (previousState == bddfalse)
	{
	  j++;
	  previousState = successorOfAct(Tact,withinSet[i],withinSet[i-1],j);	  
	}
      if (verbosity > 1) cout << "Extracted action: " << Tact.p[j].actName << endl;
      cout.flush();
      withinSet[i-1] = previousState;
      solution.push_back(Tact.p[j].actName);      
    }

  return solution;
}



  
void reachedStates::print() {
  
  cout << "depth:bddNode#";
  for (int i=0; i <= seenDepth; i++)
    cout << i << ":" << bdd_nodecount(withinSet[i]);
}










////////////////////////////////////////////////////////
//
// Strong Cyclic g-h map datastructure
//
////////////////////////////////////////////////////////

// IN
//  hGoal : h-value of goal
//  goal  : goal states 
// OUT side effect
//  - goal unioned with states in h-entry of S
//  - statesInS updated  
Smap::Smap(int hGoal, bdd goal) {

  if (S.count(hGoal) == 1)
    S[hGoal] |= goal;	      
  else
    S[hGoal] = goal;	      
}


// IN
//  T    : disjunctive branching partitioning
//  A    : action bdd-var position info.
// AccSA : SAs currently in AccSA
// OUT
//  SAs expanded from S with lowest h-value
//
//  Algorithm:
//  1) compute and merge children of S (SA pruned for 
//     SA in AccSA and goal states)
//  2) add states of child to S with lowest h-value
//  3) return SAs of child with lowest h-value
bdd Smap::expandGuided(TRel& T, ActInfo& Ainfo, bdd AccSA, bdd goal) {

  map<int,bdd> child;


  //  1) compute and merge children of S (SA pruned for 
  //     SA in AccSA and goal states)

  if (verbosity > 0)
    cout << " children=[";  

  for (map<int,bdd>::iterator mi = S.begin(); mi != S.end(); ++mi)
    {
      if (verbosity > 0)
	cout << "(";  

      // go through each branching partition and add the new nodes to searchQueue
      for (int i = 0; i < T.p.size(); i++)
	{
	  // compute child i
	  // a) compute partion preImage
	  bdd childSA = partitionPreImage(T,mi->second,i);
	  
	  // b) prune child from states in goal
	  childSA &= !goal;

	  childSA &= !AccSA;
	    	  
	  // c) add child to map if nonempty
	  if (childSA == bddfalse)
	    {
	      // child should not be added, but write a dot to tell that a child was processed
	      if (verbosity > 0)
		{
		  cout << ".";	   
		  cout.flush();
		}
	    }
	  else
	    {
	      // write child summary
	      if (verbosity > 0)
		{
		  if (T.p[i].dh > 0)
		    cout << "+";
		  else if (T.p[i].dh < 0)
		    cout << "-";
		  else
		    cout << "0";		      
		  cout.flush();
		}
	      // add child to search queue
	      
	      if (S.count(mi->first + T.p[i].dh) == 1)
		child[mi->first + T.p[i].dh] |= childSA;	      
	      else
		child[mi->first + T.p[i].dh] = childSA;	      
	    }
	}
      
      if (verbosity > 0)
	cout << ")";
      
    }
  
  if (verbosity > 0)
    cout << "]";
  
  if ( child.empty() )
    return AccSA;
  else
    {
      if (verbosity > 0)
	cout << " h=" << child.begin()->first << " size=" << bdd_nodecount(child.begin()->second);
      
      //  2) add states of child with lowest h-value to S

      if (S.count(child.begin()->first) == 1)
	S[child.begin()->first] |= bdd_exist(child.begin()->second,Ainfo.Avars);
      else
	S[child.begin()->first] = bdd_exist(child.begin()->second,Ainfo.Avars);

      
      //  3) return SAs of child unioned AccSA
      return child.begin()->second | AccSA;
    }
}

