////////////////////////////////////////////////////////
// File  : searchAlgAuxFunc.cc  
// Desc. : Auxiliary functions for search algorithms
//         for deterministic and nondeterministic 
//         search domains
// Author: Rune M. Jensen CS, CMU
// Date  : 10/28/02
////////////////////////////////////////////////////////

#include <bdd.h>
#include <math.h>
#include <queue>
#include <list>
#include "main.hpp"
#include "numDomain.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"
#include "searchQueue.hpp"
#include "searchAlgAuxFunc.hpp"
#include "universal.hpp"
#include "timer.hpp"





bdd pruneStates(bdd preImage, bdd v) {
  return preImage & !v;
}



// IN
//  T    : partitioned  transition relation
//  v    : currently visited states/ last preImage (encoded in current state variables)
// OUT
//  preImage of V encoded in current state vars 
//  calculation equal to Clark et. al. p. 80
bdd preImage(TRel& T, bdd v) {
  bdd res;
  bdd vr;

  res = bddfalse;

  for (int i=0; i<T.p.size(); i++)
    {
      vr = bdd_replace(v,T.p[i].subVarSetb);
      res |= bdd_appex(vr,T.p[i].exp,bddop_and,T.p[i].varSetb);
    }
  
  return res;
}



// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
// OUT
//  image of V encoded in current state vars
//  calculation equal to research journal 8/16-99 p. 60.
bdd image(TRel& T, bdd v) {
  bdd res;
  bdd u;
  int i;
  
  res = bddfalse;

  for (i=0; i<T.p.size(); i++)
    {
      u = bdd_appex(v,T.p[i].exp,bddop_and,T.p[i].varSetf);
      res |= bdd_replace(u,T.p[i].subVarSetf);
    }
  
  return res;
}



// IN
//  Tact     : disjunctive partitioned transition relation where the each partition 
//             corresponds to an action 
//  set1     : set of state the action leads from
//  set2     : set of states (only one state) the action leads to 
//  act      : action number of the applied action
// OUT
//  predecessor within set1 of the single state in set2 when applying act  
//  backwards (calculation equal to Clark et. al. p. 80)
bdd predecessorOfAct(TRel& Tact,bdd set1,bdd set2,int act) {

  bdd res;
  bdd vr;
  
  vr = bdd_replace(set2,Tact.p[act].subVarSetb);
  res = bdd_appex(vr,Tact.p[act].exp,bddop_and,Tact.p[act].varSetb);
  res &= set1;
  return res;
}




// IN
//  Tact  : disjunctive partitioned transition relation where the each partition 
//             corresponds to an action
//  set1     : set of states the action leads from
//  set2     : set of states (only one state) the action leads to 
//  act      : action number of the applied action
// OUT
//  successor within set2 of the single state in set1 when applying act  
//  forward (calculation equal to Clark et. al. p. 80)
bdd successorOfAct(TRel& Tact,bdd set1,bdd set2,int act) {

  bdd res;
  bdd u;

  u = bdd_appex(set1,Tact.p[act].exp,bddop_and,Tact.p[act].varSetf);
  res = bdd_replace(u,Tact.p[act].subVarSetf);
  res &= set2;
  return res;
}





// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
//  i    : partition image should be made of
// OUT 
//  Image of parition i of disjunctive partitioned transition relation T
bdd partitionImage(TRel& T, bdd v, int i) {
  bdd vr;

  vr = bdd_appex(v,T.p[i].exp,bddop_and,T.p[i].varSetf);
  return bdd_replace(vr,T.p[i].subVarSetf);
}


// IN
//  T    : partitioned transition relation
//  v    : currently visited states/ last image (encoded in current state variables)
//  i    : partition image should be made of
// OUT 
//  preImage of parition i of disjunctive partitioned transition relation T
bdd partitionPreImage(TRel &T, bdd v, int i) {
  bdd vr;
  
  vr = bdd_replace(v,T.p[i].subVarSetb);
  return bdd_appex(vr,T.p[i].exp,bddop_and,T.p[i].varSetb);
}












////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
//
// Non-deterministic search algorithms
//
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////


// IN
//  T     : partitioned transition relation
//  sa    : state-actions we want to find next states of
//  Ainfo : action ID variable info needed to cut Act info
// OUT
//  next states of SAs represented in current state variables
bdd nextStates(TRel& T, bdd sa, ActInfo& Ainfo) {
  bdd res;
  bdd u;
  int i;
  
  res = bddfalse;

  for (i=0; i<T.p.size(); i++)
    {
      u = bdd_appex(sa,T.p[i].exp,bddop_and,T.p[i].varSetf);
      res |= bdd_replace(u,T.p[i].subVarSetf);
    }
  
  return bdd_exist(res,Ainfo.Avars);
}




// IN
//  T         : Disjunctive partitioning 
//  s         : set of states preimage should be made of and strong
//              with repsect to 
// OUT
//  strong preimage of s
bdd strongPreImage(TRel& T, bdd s) {
 
  return preImage(T,s) & !preImage(T,!s);
}



// IN
//  T           : disjunctive branching partitioning where transitions 
//                are associated with action identifiers
//  searchQueue : ghQueue where the bdd in the search node represents
//                state-action sets 
//  parentS     : states of parent node
//  parentG     : g-value of parent node 
//  parentH     : h-value of parent node
//  pruneStates : states that should be pruned from children SAs
// OUT
//  side effect : insert weak unpruned children SAs of parent on queue
void insertPrunedChildren(TRel& T,ghSCQueue& searchQueue,
		    bdd parentS,int parentG,int parentH, bdd pruneStates) {

  if (verbosity > 0)
    cout << "[";  
  
  // go through each branching partition and add the new nodes to searchQueue
  for (int i = 0; i < T.p.size(); i++)
    {
      // compute child i
      // 1) compute partion preImage
      bdd childSA = partitionPreImage(T,parentS,i);

      // 2) prune states from children SAs in the set of prune states
      childSA &= ! pruneStates;
                        
      // 3) add child to search queue if nonempty
      if (childSA == bddfalse)
	{
	  // child should not be added, but write a dot to tell that a child was processed
	  if (verbosity > 0)
	    {
	      cout << ".";	   
	      cout.flush();
	    }
	}
      else
	{
	  // write child summary
	  if (verbosity > 0)
	    {
	      if (T.p[i].dh > 0)
		cout << "+";
	      else if (T.p[i].dh < 0)
		cout << "-";
	      else
		cout << "0";		      
	      cout.flush();
	    }
	  // add child to search queue
	  searchQueue.insert(childSA,false,parentG + 1,parentH + T.p[i].dh);	  
	}
    }
  
  if (verbosity > 0)
    cout << "]";  
  
}

/////////////////////////////////////////////////////////////////
//
// Strong Cyclic Planning aux. functions
// Using identical function names as
// Revisiting strong Cyclic planning, Vardi et al, ECP-99 
//
/////////////////////////////////////////////////////////////////




//IN
// AInfo : action variable information structure
// SA    : state action pairs to find states of 
//OUT
// SA with action info. abstracted
bdd States(ActInfo& Ainfo,bdd SA) {
  return bdd_exist(SA,Ainfo.Avars);
}





//IN
// T     : partitioned transition relation
// AInfo : action variable information structure
// G     : set of states in current state variables
// SA    : state action pairs (state in current state variables)
//OUT
// preimage of (states of SA unioned with G) minus goal states 
bdd OneStepBack(TRel& T, ActInfo& Ainfo,bdd G,bdd SA) {
  return preImage(T,States(Ainfo,SA) | G) & !G; 
}





//IN
// T     : partitioned transition relation
// AInfo : action variable information structure
// G     : set of states in current state variables
// SA    : state action pairs (state in current state variables)
//OUT
// preimage of complement of (states of SA unioned with G) 
bdd ComputeOutgoing(TRel& T, ActInfo& Ainfo,bdd G,bdd SA) {
  return preImage(T, !(States(Ainfo,SA) | G)  ) & SA; 
}



//IN
// T     : partitioned transition relation
// AInfo : action variable information structure
// G     : set of states in current state variables
// SA    : state action pairs (state in current state variables)
//OUT
// preimage of complement of (states of SA unioned with G) 
bdd PruneOutgoing(TRel& T, ActInfo& Ainfo,bdd G,bdd SA) {

  if (verbosity > 0)
    cout << " PO:";
  
  bdd Outgoing = ComputeOutgoing(T,Ainfo,G,SA);
  while (Outgoing != bddfalse)
    {
      SA = SA & !Outgoing;
      Outgoing = ComputeOutgoing(T,Ainfo,G,SA);

      if (verbosity > 0)
	cout << ".";
    }
  return SA;
}



//IN
// T     : partitioned transition relation
// AInfo : action variable information structure
// G     : set of states in current state variables
// SA    : state action pairs (state in current state variables)
//OUT
// preimage of complement of (states of SA unioned with G) 
bdd PruneUnconnected(TRel& T, ActInfo& Ainfo,bdd G,bdd SA) {

  if (verbosity > 0)
    cout << " PU:";


  bdd ConnectedToG = bddfalse;
  bdd OldConnectedToG = !ConnectedToG; // using negation as bottom element
  while (ConnectedToG != OldConnectedToG)
    {
      OldConnectedToG = ConnectedToG;
      ConnectedToG = SA & OneStepBack(T,Ainfo,G,ConnectedToG) ;

      if (verbosity > 0)
	cout << ".";
    }
  return ConnectedToG;
}


//////////////////////////////////////////////////////
//
// Aux functions for adversarial planning
//
//////////////////////////////////////////////////////






// IN
//  T    : disjunctive partitioned  transition relation
//         where actions are labelled with system and environment
//         action ID 
// OUT
//  App(s,a_e) : applicable environement actions in s 
//  calculation equal to Clark et. al. p. 80
bdd App(TRel& Tenv, ActInfo& Ainfo) {

  bdd res = bddfalse;
  for (int i=0; i < Tenv.p.size(); i++)
      res |= bdd_appex(bddtrue,Tenv.p[i].exp,bddop_and,Tenv.p[i].varSetb);
  return bdd_exist(res,Ainfo.Avars);
}





// IN
//  Tenv : disjunctive partitioned  transition relation
//         where actions are labelled with system and environment
//         action ID
//  A    : bdd A(s,a_e) of applicable environement actions
//  SA   : states-action table SA(s,a_s) we want to prune for 
//         states unfair wrt. C
//   C   : States C(s) SAs in SA should be fair wrt.
// OUT
//  NewSA with SAs from SA fair wrt. C(s,a_e)
bdd Fair(TRel& Tenv, ActInfo& Ainfo, bdd A, bdd SA, bdd C) {

  bdd JPreImgC = preImage(Tenv,C);
  
  bdd res = SA & !C;
  res &= bdd_forall( A >> bdd_exist( SA & JPreImgC , Ainfo.Avars ) , Ainfo.AvarsEnv ); 

  return res;
}


// Weak preComponent
bdd WPreComp(TRel& T, bdd C) {
  return preImage(T,C) & !C;
}



//  T    : disjunctive partitioned  transition relation
//         with system action ID
//  Tenv : disjunctive partitioned  transition relation
//         where actions are labelled with system and environment
//         action ID
//  A    : bdd A(s,a_e) of applicable environement actions
//   C   : States C to find weak adversarial precomponent of
// OUT
//  Weak adversarial precomponent
bdd WAPreComp(TRel& T,TRel& Tenv, ActInfo& Ainfo, bdd A,  bdd C) {
  return Fair(Tenv,Ainfo,A,preImage(T,C),C);
}




//IN
// T     : partitioned transition relation
// Tenv  : partitioned transition relation with environment actions present
// AInfo : action variable information structure
// A     : A(s,e_a) applicable env. actions 
// C     : set of states covered by the plan  
// SA    : candidate state-action pairs
//OUT
// SA prund for SAs of unfair states
bdd PruneUnfair(TRel& T,TRel& Tenv,ActInfo& Ainfo,bdd A, bdd C,bdd SA) {

  if (verbosity > 0)
    cout << " PUf:";

  bdd NewSA = bddfalse;
  bdd OldSA = !NewSA; // using negation as bottom element
  while (NewSA != OldSA)
    {
      OldSA = NewSA;
      bdd FairStates = States(Ainfo,NewSA) | C;
      NewSA |= Fair(Tenv,Ainfo,A,SA,FairStates);

      if (verbosity > 0)
	cout << ".";
    }
  return NewSA;
}



























// IN
//  T             : disjunctive branching partitioning where transitions are 
//                  associated with action identifiers
//  Ainfo         : action identifier BDD variable info 
//
//  preCompSA     : precomponent SAs (output)
//  preCompS      : states in precomponent (output)
//  preCompCandSA : precomponent candidate SAs
//  preCompCandS  : states in precomponent candidate 
//  coveredStates : states covered by current strong cyclic plan
// OUT
//  fixedPoint preComp
void SCfixedPoint(TRel& T, ActInfo& Ainfo,bdd& preCompSA,bdd& preCompS,
		  bdd preCompCandSA,bdd preCompCandS,bdd coveredStates) {

  bdd preCompCandSOld;

  do 
    {

      if (verbosity > 0)
	{
	  cout << ".";
	  cout.flush();
	}
      
      preCompCandSOld = preCompCandS;
      
      // find outgoing SAs from preCompCandS and covered states;
      bdd outgoingSAs = preImage(T,!(preCompCandS | coveredStates));
      // remove these SAs from preComCand 
      preCompCandSA = preCompCandSA & ! outgoingSAs;
      preCompCandS = bdd_exist(preCompCandSA,Ainfo.Avars);
    }
  while (preCompCandSOld != preCompCandS);

  preCompSA = preCompCandSA;
  preCompS = preCompCandS;  
}






/////////////////////////////////////////////////////////////////
//
// Fault Tolerant Strong Cyclic Planning aux. functions
// Using identical function names as
// Revisiting strong Cyclic planning, Vardi et al, ECP-99 
//
/////////////////////////////////////////////////////////////////



//IN
// Teff       : Disjunctive partitioned transition relation 
//              of the NADL eff expression
// USpart     : The fault tolerant plan (U) partitioned wrt.  
//              heuristic values
// US         : States in the current fault tolerant plan
// USfrontier : (implicit output) frontier SAs partitioned
//              according to h-value
//OUT
// partitioned frontier via USfronier
void computeUSfrontier(TRel& Teff, map<int,bdd>& USpart, bdd US, map<int,bdd>& USfrontier) {
 
  USfrontier.clear();

  // go through all partitions of Teff and USpart
  for (map<int,bdd>::iterator ibi = USpart.begin(); ibi != USpart.end(); ++ibi)
    for (int i = 0; i < Teff.p.size(); i++)
      {

	// 1) compute partion preImage and prune for states already in US
	bdd SA = partitionPreImage(Teff,ibi->second,i) & !US;
	
	// 2) add SAs to to the backward frontier of US if non-empty
	if (SA != bddfalse)
	  {
	    // OBS: in ICAPS and IJCAI papers I assume that delta h is given in
	    // forward direction !!!
	    if (USfrontier.count(ibi->first + Teff.p[i].dh) == 1)
	      USfrontier[ibi->first + Teff.p[i].dh] |= SA;	      
	    else
	      USfrontier[ibi->first + Teff.p[i].dh] = SA;	      
	  }
      }    
  
  if (verbosity > 0)
	{
	  cout << " front=[";
	  for (map<int,bdd>::iterator ibi = USfrontier.begin(); ibi != USfrontier.end(); ++ibi)
	    if (ibi == USfrontier.begin())
	      cout << ibi->first;
	    else
	      cout << "," << ibi->first;
	  cout << "]";
	  cout.flush();
	}
}





//IN
// Teff               : Disjunctive partitioned transition relation 
//                      of the NADL eff expression
// RSA                : Current recovery plan
// RS                 : States of RSA
// USAPreCompCandSerr : (implicit output) frontier SAs partitioned
//                      according to h-value
//OUT
// side-effect: RSA extended a number of times equal 
// to the current minimum Manhattan distance to error
// states
void extendRSA2(TRel& Teff, ActInfo& Ainfo, bdd& RSA, bdd& RS, bdd& USAPreCompCandSerr) {

  cout << "No support for BuDDy mod\n";
  exit(1);

  if (verbosity > 0)
    cout << "(";

  bdd uncoveredSerr = bddfalse; // USAPreCompCandSerr & !RS;


  if (uncoveredSerr != bddfalse)
    {
      int L =1; //bdd_lowerboundH(RS,uncoveredSerr);

      for (int i = 0; i < L; i++)
	{
	  
	  if (verbosity > 0)
	    cout << ".";
      	
	  bdd RSAPreComp = preImage(Teff,RS) & !RS;
	  bdd RSAPreCompS = bdd_exist(RSAPreComp,Ainfo.Avars);
	  bdd withinL = bddfalse; //bdd_stateswithinH(RSAPreCompS,uncoveredSerr,L);
	  RS |= withinL;
	  RSA |= RSAPreComp & withinL;
	  uncoveredSerr &= !RS;
	}
    }

  if (verbosity > 0)
    cout << ")";

} 





//IN
// Teff               : Disjunctive partitioned transition relation 
//                      of the NADL eff expression
// RSA                : Current recovery plan
// RS                 : States of RSA
// USAPreCompCandSerr : (implicit output) frontier SAs partitioned
//                      according to h-value
//OUT
// side-effect: RSA extended a number of times equal 
// to the current minimum Manhattan distance to error
// states
void extendRSA3timed(TRel& Teff,TRel& Terr, ActInfo& Ainfo, 
		     bdd& RSA, bdd& RS, bdd& USAPreCompCandSA,double lastExpand ) {

  timer expandClk;
  expandClk.start();

  if (verbosity > 0)
    cout << "(";

  bdd uncoveredSerr = bddfalse; // USAPreCompCandSerr & !RS;


  if (uncoveredSerr != bddfalse)
    {
      int L =1; //bdd_lowerboundH(RS,uncoveredSerr);

      for (int i = 0; i < L; i++)
	{
	  
	  if (verbosity > 0)
	    cout << ".";
      	
	  bdd RSAPreComp = preImage(Teff,RS) & !RS;
	  bdd RSAPreCompS = bdd_exist(RSAPreComp,Ainfo.Avars);
	  bdd withinL = bddfalse; //bdd_stateswithinH(RSAPreCompS,uncoveredSerr,L);
	  RS |= withinL;
	  RSA |= RSAPreComp & withinL;
	  uncoveredSerr &= !RS;
	}
    }

  if (verbosity > 0)
    cout << ")";

} 
