/////////////////////////////////////////////////////////
// File  : reachInfo.hpp
// Desc. : Performs an explicit fix point analysis
//	   on which grounded predicates can be reached 
//	   from the initial state. Obs: this analysis 
//	   is relaxed by ignoring the delete list
//	   (otherwise it would be as complex as the
//	    planning problem itself)
//	   The algorithm was suggested by Edelkamp and
//	   Helmert.
// 
//	   - Makes a depth computation of each fluent
//	     used for a HSPr approximation (also suggested
//	     Edelkamp & Helmert)
//  
// Author: Rune M. Jensen, CS, CMU
// Date  : 12/8/01
/////////////////////////////////////////////////////////

#ifndef REACHINFOHPP
#define REACHINFOHPP


#include<vector>
#include<set>
#include "pddl.hpp"
#include "numDomain.hpp"


// Action information for the reachability analysis algorithm
//
// Comments:
// - The precondition predicates are sorted such that the predicates gets fully
//   instantiated in ascending order given that the parameters are instantiated in
//   asscending order. i.e. the predicates are level. Predicates at level 0 has no 
//   paramters in their support. Predicates at level 1 has parameter 1 in their support.
//   Predicates at level 2 has parameter {1,2} in their support (maybe only parameter 2) etc.
//
// - The static facts of arity 0 and one are removed from the
//   precondition after finding the parameter domain (since the truthvalue never needs
//   to be checked again). However, static facts of arity > 1 must be rechecked.
//   The idea is to cancel an instantiation as early as possible  
struct reachAction {
  string name;
  vector<int> predArity; 
  set<int> predsInPre;                 // predicates in precondition
  int paramNum;                        // number of parameters of action
  vector< set<int> > paramDom;         // domain of each parameter as given by the static facts objects
  vector< vector< vector<int> > > pre; // levels of predicates in precondition, according to support
                                       // level 0 has no support, level support of param 1, level 2,
                                       // support of param {1,2} etc.
  vector< vector<int> > add;  
  set<int> oldProjections;             // a projection is an instantiation of the of parameters 
                                       // of the action stored by its bitvector representation
  reachAction(numAction &act, set<int> &staticPreds, numDomain &numDom);  
  void print(numDomain &numDom,int objShift,int objMask);
};



// reachability info struct. Used by the main reachability analysis
// function to keep track of the search.
//
// comments
// - grounded predicates are represented by bit vectors
//   on the form: [argN,argN-1,...,arg1,predName]
//   The expanded form is a vector<int>:
//   [0:PredName,1:Arg1,2:Arg,...,N:ArgN]
struct reachInfo {

  // bit vector representation vars
  int predShift;
  int objShift;                 
  int predMask;
  int objMask;
  int maxArity;
  vector<int> predArity;

  // search vars
  set<int> initFluents;         // set of init fluents (bit vector rep)
  set<int> initStaticFacts;     // set of init static facts (bit vector rep)
  set<int> staticPreds;         // set of preds not changed by any add del list (predno: 1,2,3,...)
  vector<reachAction> act;      // actions
  list<int> openList;            // openlist of facts (necessary to guarantee BFS depth)
  set<int> openFacts;           // open set of search           
  set<int> closedFacts;         // closed set of search all facts seen
  map<int,int> depth;           // map from factid to depth (used for HSPr heuristic)
  void analyse(numDomain &numDom, double squizeFactor); // *main function* calls the below functions 
  void print(numDomain &numDom); 
  void fixedPoint();            // do the reachability analysis
  void range(double squizeFactor);  // change the depths according to the new number of depth levels
                                // this weakens the depth information 
};



// A projection is an instantiation of the parameters of an action
// this abstract datastructure is used to generate all the possible 
// projections of an action.
// 
// comment
// - projections are represented by bit vectors on the form: [paramN,paramN-1,...,param1]
//   The expanded form is a vector<int>:
//   [0:Param1,1:Param2,...,N-1:ParamN]
struct projection {
  vector < set<int>::iterator > pi;       // iterator to each paramter domain
  vector < set<int>::iterator > pBegin;   // begin iterator to each paramter domain
  vector < set<int>::iterator > pEnd;     // end iterator to each paramter domain

  int next(int &param);     // next projection changing parameter param
                            // -1 if no new projection (otherwise the 
                            // index of the smallest changed parameter 
  int id(int &objshift);    // id of current projection)
  int instantiate(vector<int> &fact,int &predShift,int &objShift);
                            // instantiate a fact with the current projection 
  bool empty() { return pi[0] == pEnd[0]; } // true if no more possible projections
  projection(vector< set<int> > &paramDom); // intializes the iterators to traverse
};                                          // the parameter domain given by paramdom



// prototypes
inline int fact2id(vector<int> &fact,int &predShift, int &objShift);
inline void id2fact(int &id, vector<int> &fact, 
		    int &objMask,int &predMask,
		    int &predShift, int &objShift,
		    vector<int> &predarity);
inline void id2proj(int &id, vector<int> &proj, 
		    int &objMask, int &objShift,
		    int &projArity);

void printFact(vector<int> &fact,numDomain &numDom);
void printFactSet(set<int> s,int predShift, int objShift,
		  int predMask,int objMask, vector<int> predArity,
		  numDomain &numDom);


int instantiate(vector<int> pred,int projId,int projArity,int objMask,int predMask,
		int predShift,int objShift);


#endif  
