////////////////////////////////////////////////////////
// File  : partition.cc   
// Desc. : partioning functions 
// Author: Rune M. Jensen
// Date  : 1/11/02
//          
////////////////////////////////////////////////////////
// Obs: BuDDySplit has discontinued to be included 
//      in order    line 140. 
////////////////////////////////////////////////////////

#include <stream.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <limits.h>
#include <bdd.h>
#include <fstream.h>
#include <set>
#include <string>
#include <vector>
#include <list>
#include <map>
#include "pddl.hpp"
#include "main.hpp"
#include "numDomain.hpp"
#include "set.hpp"
#include "reachInfo.hpp"
#include "balancedInfo.hpp"
#include "bddLayout.hpp"
#include "partition.hpp"


////////////////////////////////////////////////////////
//
// TRel member functions
//
////////////////////////////////////////////////////////


int TRel::size() {
  int res = 0;
  for (int i = 0; i < p.size(); i++)
    res += bdd_nodecount(p[i].exp);
  return res;
}

////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
//
// Shared Functions 
//
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////




//IN
// T         : Partitioned transition relation
// threshold : least number of nodes in optimized transition relation
// opT       : (implicit output) transition relation where partitions are 
//             unified until partition size is above threshold 
//             (not taking g and h-value changes into account)		       
void optimize(TRel& T, TRel& Topt, int threshold) {

  // aux.
  branchPartition dp;

  // collapse partitiones s.t. the node count of each is above the threshold
  int i = 0;
  while (i < T.p.size())
    {
      dp.exp     = T.p[i].exp;
      dp.modify  = T.p[i].modify;
      dp.dg = T.p[i].dg;
      dp.dh = T.p[i].dh;
      // poor mans solution: combine partitions until exp > threshold
      while (i < T.p.size() - 1 && bdd_nodecount(dp.exp) < threshold) 
	{
	  i++;	  
	  dp.exp   = dp.exp &      IV(setDifference(T.p[i].modify,dp.modify)) | 
		     T.p[i].exp &  IV(setDifference(dp.modify,T.p[i].modify));
	  dp.modify = setUnion(dp.modify,T.p[i].modify);
	}

      
      // post: dp full or partitions exhausted
      // generate the final expression for the partition 
      dp.varSetb = bddtrue;
      dp.varSetf = bddtrue;
      dp.subVarSetf = bdd_newpair();
      dp.subVarSetb = bdd_newpair();

      
      for (set< pair<int,int> >::iterator si = dp.modify.begin(); si != dp.modify.end(); ++si)
	{
	  dp.varSetf &= bdd_ithvar(si->first); 
	  dp.varSetb &= bdd_ithvar(si->second);  
	  bdd_setpair(dp.subVarSetf,si->second,si->first);
	  bdd_setpair(dp.subVarSetb,si->first,si->second);
	}

      Topt.p.push_back(dp);

      if (verbosity > 0)
	cout << "Optimized partition " << Topt.p.size() << " size=" << bdd_nodecount(dp.exp) << endl; 
      i++;
    }      

}	  





// IN
//  Tact      : a disjunctive partitioning where each parition corrsponds to 
//              a single action. It forms the base for the split
//  Tbranch   : (implicit output) A branching partitioning 
//     		disjunctive partition where each partition is associated with 
//     		a split element of one action. An action split element improves
//     		the MIN HAMMING distance to the goal with some value [...,-1,0,1,...]
//     	       
//     		The asynchronous trick (Clark et al. p. 80) is assumed to be 
//     		used for each partition. 
//  goal      : goal states that the actions should be
//              splitted according to
void splitTactPartitions(TRel& Tact,TRel& Tbranch,bdd goal) {    

  cout << "partition.cc: splitTactPartitions: In order to use this function you need to\n";
  cout << "change the BuDDy package to BuDDy20Split and make appropriate changes in the code\n";
  cout << "Follow CHANGE HERE comments\n";
  exit(1);
  
  // go through the action partitioning
  for (int i=0; i < Tact.p.size(); i++) 
    {
      if (verbosity > 0)
	cout << "Splitting Action " << Tact.p[i].actName << " : ";
      
      // make splits of action 
      // dh lies within  
      // [-|bits(mod1)+...+bits(modN)|,...,0,...,+|bits(mod1)+...+bits(modN)|]
      int c = Tact.p[i].modify.size();
      for (int j = -c; j <= c; j++)
	{	  
	  // compute the action part with improvement j
	  // CHANGE HERE 
	  bdd as =  bddtrue;  // bdd_split(Tact.p[i].exp,goal,j);
	  
	  // only add a non_empty action
	  if (as != bddfalse)
	    {
	      branchPartition part;
	      part.dg = Tact.p[i].dg;
	      part.dh = -j;
	      part.exp = as;
	      part.modify = Tact.p[i].modify; 
	      part.varSetf = Tact.p[i].varSetf;
	      part.varSetb = Tact.p[i].varSetb;
	      part.subVarSetf = Tact.p[i].subVarSetf;
	      part.subVarSetb = Tact.p[i].subVarSetb;
	      
	      if (verbosity > 0)
		{
		  cout << " ([dg=" << part.dg << ",dh=" << -j << "] size=" << bdd_nodecount(as) << ")";
 		  cout.flush();
	 	}
	      
	      
	      // insert partition at the improvment of j entry
	      Tbranch.p.push_back(part);
	    }
	}
      if (verbosity > 0)
	cout << endl;
    }
}





//IN
// Tbranch    : Branching partitioning of transition relation
// threshold  : least number of nodes in optimized transition relation
// bddLayout  : domain definition
// TbranchOpt : (implicit output) transition relation where partitions are unified
//              until partition size is above threshold, each dg - dh pair
//              handled individually		       
void optimizeBranching(TRel& Tbranch, TRel& TbranchOpt, int threshold) {

  map< pair<int,int>, TRel> dghMap;


  // sort partitions according to dh and dg 
  for (int i=0; i < Tbranch.p.size(); i++)
    {
      pair<int,int> key(Tbranch.p[i].dg,Tbranch.p[i].dh);
      dghMap[key].p.push_back(Tbranch.p[i]);    
    }

  // optimize each partition in turn

  TRel Taux;
  TRel TauxOpt;
  
  for (map< pair<int,int>, TRel >::iterator mi = dghMap.begin(); mi != dghMap.end(); ++mi)
    {
      TauxOpt.p.clear();
      Taux = mi->second;
      optimize(Taux,TauxOpt,threshold);
      
      if (verbosity > 0)
	cout << "Added " << Taux.p.size() << " partitions with [dg=" << 
	  mi->first.first << ",dh=" << mi->first.second << "] " 
	     << " optimized to " << TauxOpt.p.size() << " partitions\n"  ;
      
      for (int i=0; i < TauxOpt.p.size(); i++)
	TbranchOpt.p.push_back(TauxOpt.p[i]);
    }
}






////////////////////////////////////////////////////////
// Shared Print functions
////////////////////////////////////////////////////////

void TRel::print() {

  for (int i=0; i < p.size(); i++)
    {
      cout << "  Partition # : " << i << "\n";
      p[i].print();
    }
}

void branchPartition::print() {
  cout << "   dg  : " << dg << endl;
  cout << "   dh  : " << dh << endl;
  cout << "   action name : " << actName << endl;
  cout << "   exp nodecount : " << bdd_nodecount(exp) << "\n";
  cout << "   modify : "; 
  for (set< pair<int,int> >::iterator si = modify.begin(); si != modify.end(); ++si)
    cout << " [" << si->first << "," << si->second << "]"; 
  cout << endl;
  cout << "   varSetf : "; bdd_printset(varSetf); cout << "\n";
  cout << "   varSetb : "; bdd_printset(varSetb); cout << "\n";
  cout << "   subVarSetf (address) : " << (int) subVarSetf << "\n";
  cout << "   subVarSetb (address) : " << (int) subVarSetb  << "\n";
}








































////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
// 
// PDDL 
// Functions
// 
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////



////////////////////////////////////////////////////////
// 
// Functions for building disjunctive partitioned
// transition relations and disjunctive split transition 
// relations and action partition relations
// 
////////////////////////////////////////////////////////






//      mkactT is used to build a transition relation
//      for each action (assuming Ed. Clarks asynchronous trick
//      to be applied as for dispartitions) used to find actions
//      in a classical plan. The rep. corresponds to a conjuntive 
//      partition where each partition only represents a single action.
//
 //
//      NOT IMPLEMENTED YET
//        The actions are sorted such that actions constraining the same 
//        state variables are in sequence (this improves the optimization 
//        of the transition relation)
//
//
//
// IN
//  layout   : bdd variable layout
//  rInfo    : reachability info
//  numDom   : nummerical PDDL domain info 
//  Tact     : (implicit output) : disjunctive transition relation partitioning
//             with each partition associated with exactly one action.
//             For the minHamming dist each action has a particular h-value change
//             Thus Tact also works as a branching partitioning
//          	 
//
// heuristic : Type of heuristic for branching partitioning
//
//             HSPr : 
//               this heuristic is for backward search. But dh will still
//               contain it's change in forward direction
//               For each action we assume that each application
//               satisfy:
//                1) del <= pre
//                2) add != pre
//                3) No add facts exists in the current state
//                1) + 2) => add != del
//               Domains not satisfying these assumptions needs actions to be applied
//               in counterintuitive where facts are added already being true (3). Its not
//               clear what such actions represents even though they can be defined in
//               the language (if taking such applications into account in the BDD model, the 
//               backward images are likely to get unstructured). Similarly actions where 
//               not add != del have unclear semantics (2) (and normally won't exist in a domain)
//               In order to use our HSP heuristic, it must be checked that the domain
//               fullfils 1) and 2)
//               Obs: negative values of h during search may happen if the goal state
//                    is a set of states with diffferent h-value (to be correct we need
//                    to explicitly take don't care facts into account when computing the
//                    h-value of the goal states)
//
//                    
void PDDLmkTact(PDDLbddLayout& layout,reachInfo& rInfo,numDomain& numDom,
		TRel& Tact,heuristicType heuristic) {
 
  // for each executable action instantiation 
  //   make one Tact partition
  //   make n partitions of the splitted action
  for (int i=0; i < numDom.act.size(); i++)
    for (set<int>::iterator proj = rInfo.act[i].oldProjections.begin(); 
	 proj != rInfo.act[i].oldProjections.end(); ++proj)
      {
	branchPartition part;

	part.exp = bddtrue;
	
	////////////////////////////////////////////
	//  Make bdd expression of partition 
	////////////////////////////////////////////

	// make/check precondition
	// Obs: notice the variables in the precondition is
        // not added to the modify set: these are only 
        // next state variables (add and del list) 
	bool check = true;
	for (int j=0; j < numDom.act[i].pre.size(); j++)
	  {
	    int factId = instantiate(numDom.act[i].pre[j],*proj,numDom.act[i].paramNum,
				     rInfo.objMask,rInfo.predMask,
				     rInfo.predShift,rInfo.objShift);
	    
	    if (setMember(rInfo.staticPreds,(numDom.act[i].pre[j])[0]))
	      { // check if fact is reachable if fact is static
		if (!setMember(rInfo.closedFacts,factId))
		  check = false;
	      }
	    else
	      {
		// add the fact to expression
		part.exp &= layout.expr(numDom,factId,0);
	      }
	  }

	if (check)
	  {

	    // make delta g and h-value according to Heuristic function
	    switch (heuristic) 
	      {
	      case ht_HSPr :
		part.dg = 1;
		part.dh = 0;
		break;
	    
	      case ht_minHamming :
		part.dg = 1;
		break;
		
	      case ht_none :
		part.dg = 1;
		break;
		
	      default:
		cout << "partition.cc : PDDLmkAct: option not covered\nExiting\n";
		exit(-1);
		break;
	      }

	    // add add-list
	    for (int j=0; j < numDom.act[i].add.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].add[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		// add fact to next state
		part.exp &= layout.expr(numDom,factId,1);
		
		// assume fact is false in current state
		part.exp &= !layout.expr(numDom,factId,0);
		

		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // all add facts don't exist in the current state thus
		    // they contribute to the h-value in the next state
		    part.dh += rInfo.depth[factId];		  
		    break;
		    
		  case ht_minHamming :
		    break;
		    
		  case ht_none :
		    break;
		    
		  default:
		    cout << "partition.cc : PDDLmkAct: option not covered\nExiting\n";
		    exit(-1);
		    break;
		}


		
		// update modify pair set
		PDDLvarDef v(layout.bddVars(factId));
		for (int k=0; k < v.length; k++)
		  {
		    pair<int,int> elem(v.current[k],v.next[k]);
		    part.modify.insert(elem);
		  }
	      }

	    

	    // add del-list
	    for (int j=0; j < numDom.act[i].del.size(); j++)
	      {
		int factId = instantiate(numDom.act[i].del[j],*proj,numDom.act[i].paramNum,
					 rInfo.objMask,rInfo.predMask,
					 rInfo.predShift,rInfo.objShift);
		
		switch (heuristic) 
		  {
		  case ht_HSPr :
		    // add delta-h of fact (since it is a subset of pre
		    // each of these facts indeed are deleted from the current state) 
		    part.dh -= rInfo.depth[factId];
		    break;
		    
		  case ht_minHamming :
		    break;
		    
		  case ht_none :
		    break;
		  }

		// add the fact to expression
		part.exp &= !layout.expr(numDom,factId,1);
		
		// update modify set
		PDDLvarDef v(layout.bddVars(factId));
		for (int k=0; k < v.length; k++) 
		  {
		    pair<int,int> elem(v.current[k],v.next[k]);
		    part.modify.insert(elem);
		  }
	      }
	    
	    

	    ////////////////////////////////////////////
	    //  Make modify varsets of partition 
	    ////////////////////////////////////////////

	    part.varSetf = bddtrue;
	    part.varSetb = bddtrue;
	    part.subVarSetf = bdd_newpair();
	    part.subVarSetb = bdd_newpair();
	
	    for (set< pair<int,int> >::iterator si = part.modify.begin(); 
		 si != part.modify.end(); ++si)
	      {
		part.varSetf &= bdd_ithvar(si->first); 
		part.varSetb &= bdd_ithvar(si->second);  
		bdd_setpair(part.subVarSetf,si->second,si->first);
		bdd_setpair(part.subVarSetb,si->first,si->second);
	      }
	    
	    
	    // make action name
	    string name = numDom.act[i].name;
	    vector<int> projection;
	    int projId = *proj;
	    id2proj(projId,projection,rInfo.objMask,rInfo.objShift,
		    numDom.act[i].paramNum);
	    for (int k=0; k < projection.size(); k++)
	      name = name + "_" + numDom.obj[ projection[k]-1 ];
	    
	    part.actName = name;

	    if (part.exp != bddfalse)
	      {

		if (verbosity > 1)
		  cout << "Action : " << name << " (dg=" << part.dg << ",dh=" << part.dh << ")\n";

		// store partition in Tact
                // obs. if exp = bdd_false then it's because we had a contradiction
                // according to balanced sets a la' !at(here)' /\ at(here)'
                // so these can just be ignored
		Tact.p.push_back(part);		
	      }	     	
	  }
      }
}	     	




////////////////////////////////////////////////////////
// PDDL 
// Aux. functions for making initial and goal state
// (only used by main)
////////////////////////////////////////////////////////


//IN
// layout : BDD domain def
// numdom : numerical def. of PDDL domain
// init   : initial facts of problem
//OUT
// bdd rep of init (obs all facts not mentioned in init are
// false by convention)
bdd init2bdd(PDDLbddLayout& layout, numDomain& numDom,vector< vector<int> >& init) {

  bdd res = bddtrue;

  set<int> unModifiedVars = layout.currentVars;
  
  //make init expression
  for (int i=0; i < init.size(); i++)
    if (setMember(layout.fluents,(init[i])[0]))
      {
	int id = fact2id(init[i],layout.predShift,layout.objShift);
	res &= layout.expr(numDom,id,0);
	
	// erase modified from unmodified vars
	PDDLvarDef v(layout.bddVars(id));
	for (int j=0; j < v.length; j++)
	  unModifiedVars.erase(v.current[j]);
      }

  // since all balanced facts are instantiated the variables
  // in unmodified must represent unbalanced facts
  // thus set them all to false
  for (set<int>::iterator si = unModifiedVars.begin(); si != unModifiedVars.end(); ++si)
    res &= bdd_nithvar(*si);

  return res;
}


//IN
// layout : BDD domain def
// numdom : numerical def. of PDDL domain
// goal   : initial facts of problem
//OUT
// bdd representation of goal. 
bdd goal2bdd(PDDLbddLayout& layout, numDomain& numDom,vector< vector<int> >& goal) {

  bdd res = bddtrue;

  //make goal expression
  for (int i=0; i < goal.size(); i++)
    if (setMember(layout.fluents,(goal[i])[0]))
      {
	int id = fact2id(goal[i],layout.predShift,layout.objShift);
	res &= layout.expr(numDom,id,0);
      }

  return res;
}


////////////////////////////////////////////////////////
// PDDL 
// Aux. functions for changing transition relation
// to Fmin Astar format
////////////////////////////////////////////////////////


//IN
// T   : (implicit output) branching partition 
//OUT
// iterates through T and reconfigure dg and dh to
// dg = wg*dg - wh*dh (recall dh is computed in forward direction.
// thus df = wg*dg - wh*dh and not df = wg*dg + wh*dh
// In this way dg equals df for Astar
// (we scale up with WEIGHTRESOLUTION to represent the 
//  result as an integer !)
void FminModify(TRel& T,double wg,double wh) {
  
  for (int i = 0; i < T.p.size(); i++)
    {
      T.p[i].dg = int( WEIGHTRESOLUTION * (wg*double(T.p[i].dg) - wh*double(T.p[i].dh)) );
      T.p[i].dh = 0;
    }
}




//IN
// T   : (implicit output) branching partition 
//OUT
// iterates through T and reconfigure dg and dh to
// dg = wg*dg - wh*dh (recall dh is computed in forward direction.
// thus df = wg*dg - wh*dh and not df = wg*dg + wh*dh
// In this way dg equals df for Astar
// (we scale up with WEIGHTRESOLUTION to represent the 
//  result as an integer !)
void NADLFminModify(TRel& T,double wg,double wh) {
  
  for (int i = 0; i < T.p.size(); i++)
    {
      T.p[i].dg = int( WEIGHTRESOLUTION * (wg*double(T.p[i].dg) + wh*double(T.p[i].dh)) );
      T.p[i].dh = 0;
    }
}






//IN
// T   : (implicit output) branching partition 
//OUT
// iterates through T and reconfigure dg and dh to
// dg = dh
// dh = 0
// In this way dg equals df for Best-first search
void pureFminModify(TRel& T) {
  for (int i = 0; i < T.p.size(); i++)
    {
      T.p[i].dg = -T.p[i].dh;
      T.p[i].dh = 0;
    }
}












































































////////////////////////////////////////////////////////
////////////////////////////////////////////////////////
// 
// NADL 
// Functions
//
// Pre: The NADL domain only has one system agent 
//      with deterministic actions.
// 
////////////////////////////////////////////////////////
////////////////////////////////////////////////////////




//      NADLmkTact is used to build a transition relation
//      for each action (assuming Ed. Clarks asynchronous trick
//      to be applied) used to find actions
//      in a classical plan. The rep. corresponds to a disjunctive
//      partition where each partition only represents one action.
//
//      The actions are sorted such that actions constraining the same 
//      state variables are in sequence (this improves the optimization 
//      of the transition relation)
// IN
//  layout    : Definition of BDD variables
//  Tact      : (implicit output)  disjunctive partition where each partition 
//              is associated with only one action sorted according to 
//              constrained variables
void NADLmkTact(NADLbddLayout& layout,TRel& Tact) {    


  // sort actions according to constrained variables 
  map< set<string> , set<int> > partitionMap;

  // cluster actions that modify the same variable
  for (int i = 0; i < layout.agt.act.size(); i++)
    partitionMap[layout.agt.act[i].tran[0].mod].insert(i);


  // make action partitions 
  // go through the map
  for (map< set<string> , set<int> >::iterator mi = partitionMap.begin(); mi != partitionMap.end(); ++mi)
    for (set<int>::iterator si = mi->second.begin(); si != mi->second.end(); ++si)
      {
	branchPartition part;
	
	part.exp = A(layout,*si);
	part.actName = layout.agt.act[*si].name;
	part.dg = layout.agt.act[*si].tran[0].dg;
	
	for (set<string>::iterator ilp = layout.agt.act[*si].tran[0].mod.begin();
	     ilp != layout.agt.act[*si].tran[0].mod.end(); ++ilp)
	  {
	    NADLvarDef v(layout.varMap[*ilp]);
	    for (int i = 0; i < v.length; i++)
	      {
		pair<int,int> elem(v.current[i],v.next[i]);
		part.modify.insert(elem);
	      }
	  }



      part.varSetb = bddtrue;
      part.varSetf = bddtrue;
      part.subVarSetf = bdd_newpair();
      part.subVarSetb = bdd_newpair();

      
      for (set< pair<int,int> >::iterator si = part.modify.begin(); si != part.modify.end(); ++si)
	{
	  part.varSetf &= bdd_ithvar(si->first); 
	  part.varSetb &= bdd_ithvar(si->second);  
	  bdd_setpair(part.subVarSetf,si->second,si->first);
	  bdd_setpair(part.subVarSetb,si->first,si->second);
	}

	
	if (verbosity > 1)
	  cout << "Made action partition of " << part.actName 
	       << " size = " << bdd_nodecount(part.exp) << endl; 

	// store partition
	Tact.p.push_back(part);
      }	
}



//IN
// Tact : (implicit output) action branching partitioning
//        corrected to dh = dg - 10 (- 10 neded since negative numbers 
//        cannot be represented in NADL and dg = 1 for the 15 puzzle
void modifyTact15puzzle(TRel& Tact) {
  for (int i = 0; i < Tact.p.size(); i++)
    {
      Tact.p[i].dh = Tact.p[i].dg - 10;
      Tact.p[i].dg = 1;
    }
}





///////////////////////////////////////////////////////////////////
//
// Non deterministic search
// partition algorithms
//
///////////////////////////////////////////////////////////////////


//IN
// layout : NonDet BDD layout
//          one system agent
//          at most one environment agent 
// T      : implicit output Transisiton relation
//OUT 
// A disjunctive partitioned transition relation 
// where each transition is labled with action identifiers
// Actions of the environment are modeled as nondeterminism  
void NADLNonDetmkT(NADLNonDetbddLayout& layout,TRel& T) {


  // 1) compute vector of joint actions
  vector<jointAction> jointAct;

  if ( layout.envActive )
    {
      // domain has a single system agent with n actions
      // and a single environment agent with m actions
      // the n*m joint actions are any possible combination 
      // of them

      for (int iE = 0; iE < layout.agtEnv.act.size(); iE++)
	for (int jE = 0; jE < layout.agtEnv.act[iE].tran.size(); jE++)
	  for (int iS = 0; iS < layout.agt.act.size(); iS++)
	    for (int jS = 0; jS < layout.agt.act[iS].tran.size(); jS++)
	      {
		jointAction jA(iE,jE,iS,jS);
		jointAct.push_back(jA);
	      }
    }
  else
    {
      // domain has a single system agent with n actions
      // and no environment agent
      // Thus joint actions equals system agents actions

      for (int iS = 0; iS < layout.agt.act.size(); iS++)
	for (int jS = 0; jS < layout.agt.act[iS].tran.size(); jS++)
	      {
		jointAction jA(-1,-1,iS,jS);
		jointAct.push_back(jA);
	      }

    }


  // 2) sort joint actions according to constrained variables 
  map< set<string> , set<int> > partitionMap;

  // cluster joint actions that modify the same variable
  for (int i = 0; i < jointAct.size(); i++)
    if (layout.envActive)
      partitionMap[ setUnion(layout.agtEnv.act[jointAct[i].envAct].tran[jointAct[i].envTr].mod,
			     layout.agt.act[jointAct[i].sysAct].tran[jointAct[i].sysTr].mod) ].insert(i);
    else
      partitionMap[ layout.agt.act[jointAct[i].sysAct].tran[jointAct[i].sysTr].mod ].insert(i);
      


  // 3) make joint action partitions 
  // go through the map
  for (map< set<string> , set<int> >::iterator mi = partitionMap.begin(); mi != partitionMap.end(); ++mi)
    for (set<int>::iterator si = mi->second.begin(); si != mi->second.end(); ++si)
      {
	branchPartition part;
	
	if ( layout.envActive )
	  {
	    part.exp = int2bdd(layout.actIDenv.current,layout.actIDenv.length,jointAct[*si].envAct);
	    part.exp &= int2bdd(layout.actID.current,layout.actID.length,jointAct[*si].sysAct);
	    part.exp &= trsGroup(layout,jointAct[*si].sysAct,jointAct[*si].sysTr);
	    part.exp &= trsGroupEnv(layout,jointAct[*si].envAct,jointAct[*si].envTr); 
	  }
	else
	  part.exp = 
	    int2bdd(layout.actID.current,layout.actID.length,jointAct[*si].sysAct) &
	    trsGroup(layout,jointAct[*si].sysAct,jointAct[*si].sysTr);
	
	// the name is the name of the system agent action
	part.actName = layout.agt.act[ jointAct[*si].sysAct ].name;

	// set modified variables
	for (set<string>::iterator ilp = mi->first.begin(); ilp != mi->first.end(); ++ilp)
	  {
	    NADLvarDef v(layout.varMap[*ilp]);
	    for (int i = 0; i < v.length; i++)
	      {
		pair<int,int> elem(v.current[i],v.next[i]);
		part.modify.insert(elem);
	      }
	  }

	part.varSetb = bddtrue;
	part.varSetf = bddtrue;
	part.subVarSetf = bdd_newpair();
	part.subVarSetb = bdd_newpair();

      
	for (set< pair<int,int> >::iterator si = part.modify.begin(); si != part.modify.end(); ++si)
	  {
	    part.varSetf &= bdd_ithvar(si->first); 
	    part.varSetb &= bdd_ithvar(si->second);  
	    bdd_setpair(part.subVarSetf,si->second,si->first);
	    bdd_setpair(part.subVarSetb,si->first,si->second);
	  }

	
	if (verbosity > 1)
	  cout << "Made action partition of " << part.actName 
	       << " size = " << bdd_nodecount(part.exp) << endl; 

	// store partition
	T.p.push_back(part);
      }	
}





//IN
// layout : NonDet BDD layout
//          one system agent
//          at most one environment agent 
// T      : implicit output Transisiton relation
//OUT 
// each partition quantified for environement actions.
// ok, since it is a disjunctive partitioning
void abstractEnvAct(TRel& T,ActInfo& Ainfo) {
  for (int i = 0; i < T.p.size(); i++)
    T.p[i].exp = bdd_exist(T.p[i].exp,Ainfo.AvarsEnv);
}



//IN
// layout : NonDet BDD layout
// T      : implicit output Transisiton relation
//OUT 
// A disjunctive partitioned transition relation 
// where each transition is labled with action identifiers
// effect of transition is their error effect ONLY !  
void NADLNonDetmkTErr(NADLNonDetbddLayout& layout,TRel& T) {

  if (layout.envActive)
    {
      cout << "partition.cc : NADLNonDetmkTErr : domain can not have an active env!\nexiting\n";
      exit(1);
    }
	  

  // sort actions according to constrained variables 
  map< set<string> , set< pair<int,int> > > partitionMap;

  // cluster actions that modify the same variable
  for (int i = 0; i < layout.agt.act.size(); i++)
    for (int j = 0; j < layout.agt.act[i].tran.size(); j++)
      {
	pair<int,int> key(i,j);
	partitionMap[layout.agt.act[i].tran[j].mod].insert(key);
      }


  // make action partitions 
  // go through the map
  for (map< set<string> , set< pair<int,int> > >::iterator mi = partitionMap.begin(); mi != partitionMap.end(); ++mi)
    for (set< pair<int,int> >::iterator si = mi->second.begin(); si != mi->second.end(); ++si)
      if (layout.agt.act[si->first].tran[si->second].errOpt)
	{
	  branchPartition part;
	  
	  part.exp = int2bdd(layout.actID.current,layout.actID.length,si->first) & 
	             trsGroupErr(layout,si->first,si->second);
	  part.actName = layout.agt.act[si->first].name;
	  
	  // set modified variables
	  for (set<string>::iterator ilp = layout.agt.act[si->first].tran[si->second].mod.begin();
	       ilp != layout.agt.act[si->first].tran[si->second].mod.end(); ++ilp)
	    {
	      NADLvarDef v(layout.varMap[*ilp]);
	      for (int i = 0; i < v.length; i++)
		{
		  pair<int,int> elem(v.current[i],v.next[i]);
		  part.modify.insert(elem);
		}
	    }
	  
	  
	  
	  part.varSetb = bddtrue;
	  part.varSetf = bddtrue;
	  part.subVarSetf = bdd_newpair();
	  part.subVarSetb = bdd_newpair();
	  
      
	  for (set< pair<int,int> >::iterator si = part.modify.begin(); si != part.modify.end(); ++si)
	    {
	      part.varSetf &= bdd_ithvar(si->first); 
	      part.varSetb &= bdd_ithvar(si->second);  
	      bdd_setpair(part.subVarSetf,si->second,si->first);
	      bdd_setpair(part.subVarSetb,si->first,si->second);
	    }

	
	  if (verbosity > 1)
	    cout << "Made action partition of " << part.actName 
		 << " size = " << bdd_nodecount(part.exp) << endl; 
	  
	  // store partition
	  T.p.push_back(part);
	}	
}



//IN
// layout : NonDet BDD layout
// T      : implicit output Transisiton relation
//OUT 
// A disjunctive partitioned transition relation 
// where each transition is labled with action identifiers +
// heuristic info from the description  
void NADLNonDetHeumkT(NADLNonDetbddLayout& layout,TRel& T) {

  if (layout.envActive)
    {
      cout << "partition.cc : NADLNonDetmkTErr : domain can not have an active env!\nexiting\n";
      exit(1);
    }


  // sort actions according to constrained variables 
  map< set<string> , set< pair<int,int> > > partitionMap;

  // cluster actions that modify the same variable
  for (int i = 0; i < layout.agt.act.size(); i++)
    for (int j = 0; j < layout.agt.act[i].tran.size(); j++)
      {
	pair<int,int> key(i,j);
	partitionMap[layout.agt.act[i].tran[j].mod].insert(key);
      }


  // make action partitions 
  // go through the map
  for (map< set<string> , set< pair<int,int> > >::iterator mi = partitionMap.begin(); mi != partitionMap.end(); ++mi)
    for (set< pair<int,int> >::iterator si = mi->second.begin(); si != mi->second.end(); ++si)
      {
	branchPartition part;
	
	part.exp = int2bdd(layout.actID.current,layout.actID.length,si->first) & trsGroup(layout,si->first,si->second);
	part.actName = layout.agt.act[si->first].name;
	if (!layout.agt.act[si->first].tran[si->second].dgOpt)
	  {
	    cout << "partition.cc : NADLNonDetHeumkT : dg: entry needed for action \""
		 << layout.agt.act[si->first].name << "\" transition group number " << si->second + 1
		 << "\nexiting\n";
	    exit(1);
	  }
	part.dg = layout.agt.act[si->first].tran[si->second].dg;

	if (!layout.agt.act[si->first].tran[si->second].dhOpt)
	  {
	    cout << "partition.cc : NADLNonDetHeumkT : dh: entry needed for action \""
		 << layout.agt.act[si->first].name << "\" transition group number " << si->second + 1
		 << "\nexiting\n";
	    exit(1);
	  }
	part.dh = layout.agt.act[si->first].tran[si->second].dh;

	// set modified variables
	for (set<string>::iterator ilp = layout.agt.act[si->first].tran[si->second].mod.begin();
	     ilp != layout.agt.act[si->first].tran[si->second].mod.end(); ++ilp)
	  {
	    NADLvarDef v(layout.varMap[*ilp]);
	    for (int i = 0; i < v.length; i++)
	      {
		pair<int,int> elem(v.current[i],v.next[i]);
		part.modify.insert(elem);
	      }
	  }

	part.varSetb = bddtrue;
	part.varSetf = bddtrue;
	part.subVarSetf = bdd_newpair();
	part.subVarSetb = bdd_newpair();

      
	for (set< pair<int,int> >::iterator si = part.modify.begin(); si != part.modify.end(); ++si)
	  {
	    part.varSetf &= bdd_ithvar(si->first); 
	    part.varSetb &= bdd_ithvar(si->second);  
	    bdd_setpair(part.subVarSetf,si->second,si->first);
	    bdd_setpair(part.subVarSetb,si->first,si->second);
	  }

	
	if (verbosity > 1)
	  cout << "Made action partition of " << part.actName 
	       << " size = " << bdd_nodecount(part.exp) << endl; 

	// store partition
	T.p.push_back(part);
      }	
}


