////////////////////////////////////////////////////////
// File  : numDomain.cc
// Desc. : Functions for making a numerical representation 
//	   of a PDDL domain and planning problem 
//	   performing dynamic type checking along the way
// Author: Rune M. Jensen
// Date  : 12/8/01
////////////////////////////////////////////////////////

#include <map>
#include <set>
#include <stream.h>
#include "numDomain.hpp"
#include "pddl.hpp"



////////////////////////////////////////////////////////
// Aux functions
////////////////////////////////////////////////////////

// translatetion function for literals in
// action pre, del and add
vector<int> literal2numRep(literal &lit, 
			   map<string,int> predNo, 
			   map<string,int> objNo,
			   map<string,int> paramNo,
			   vector<literal> &domPred) {
  
  if (!lit.isTrue)
    {
      cout << "numDomain.cc:literal2numRep: Syntax error ! literal \n";
      lit.print();
      cout << "\n is negative \nexiting\n";
      exit(1);
    }
  
  vector<int> pred; 
  // make pred name
  if (predNo[ lit.name ])
    pred.push_back(predNo[ lit.name ]);
  else
    {
      cout << "numDomain.cc:literal2numRep: Syntax error ! unknown predicate name in\n";
      lit.print();
      cout << "\nexiting\n";
      exit(1);
    }

  if ( lit.arg.size() != domPred[predNo[lit.name]-1].arg.size() )
    {
      cout << "numDomain.cc:fact2numRep: Syntax error ! argument number of \n";
      lit.print();
      cout << "is wrong \nexiting\n";
      exit(1);
    }

  // make arguments
  for (int i=0; i < lit.arg.size(); i++)
    if (paramNo[lit.arg[i]])
      pred.push_back(-paramNo[lit.arg[i]]);
    else if (objNo[lit.arg[i]])
      pred.push_back(objNo[lit.arg[i]]);
    else
      {
	cout << "numDomain.cc:literal2numRep: Syntax error ! argument " << i+1 << " in ";
	lit.print();
	cout << "\nneither refers to a parameter nor object\nexiting\n";
	exit(1);
      }
  
  return pred;
}
  


// translatetion function for init and goal facts
vector<int> fact2numRep(literal &lit, 
			map<string,int> predNo, 
			map<string,int> objNo,
			vector<literal> &domPred) {
  if (!lit.isTrue)
    {
      cout << "numDomain.cc:fact2numRep: Syntax error ! literal \n";
      lit.print();
      cout << "\n is negative \nexiting\n";
      exit(1);
    }
    
  vector<int> pred; 
  // make pred name
  if (predNo[ lit.name ])
    pred.push_back(predNo[ lit.name ]);
  else
    {
      cout << "numDomain.cc:fact2numRep: Syntax error ! unknown predicate name in\n";
      lit.print();
      cout << "\nexiting\n";
      exit(1);
    }

  if ( lit.arg.size() != domPred[predNo[lit.name]-1].arg.size() )
    {
      cout << "numDomain.cc:fact2numRep: Syntax error ! argument number of \n";
      lit.print();
      cout << "is wrong \nexiting\n";
      exit(1);
    }


  // make arguments
  for (int i=0; i < lit.arg.size(); i++)
    if (objNo[lit.arg[i]])
      pred.push_back(objNo[lit.arg[i]]);
    else
      {
	cout << "numDomain.cc:literal2numRep: Syntax error ! argument " << i+1 << " in ";
	lit.print();
	cout << "\n refers to unknown object\nexiting\n";
	exit(1);
      }
  
  return pred;
}


void printPredList(vector< vector<int> > predLst) {
  for (int i=0; i < predLst.size(); i++)
    {
      cout << "(";
      for (int j=0; j< predLst[i].size(); j++)
	{
	  cout << (predLst[i])[j];
	  if (j < predLst[i].size() - 1) cout << ",";
	}
      cout << ")";
      if (i < predLst.size() - 1) cout << ", ";
    }
}

  

// make a numerical action description from the usual 
// action description
numAction::numAction(PDDLaction &act, 
		     map<string,int> &predNo, 
		     map<string,int> &objNo,
		     vector<literal> &domPred) {
  name = act.name;
  map<string,int> paramNo;

  paramNum = act.param.size();
  for (int i=0; i < paramNum; i++)
    paramNo[act.param[i]] = i+1;
 
  // make numerical rep. of pre, add and del
  for (int i=0; i < act.pre.size();i++)
    pre.push_back(literal2numRep(act.pre[i],predNo,objNo,paramNo,domPred));
  for (int i=0; i < act.add.size();i++)
    add.push_back(literal2numRep(act.add[i],predNo,objNo,paramNo,domPred));
  for (int i=0; i < act.del.size();i++)
    del.push_back(literal2numRep(act.del[i],predNo,objNo,paramNo,domPred));
}



void numAction::print() {
  cout << name << endl;
  cout << "  Pre: ";printPredList(pre);cout << endl;
  cout << "  Add: ";printPredList(add);cout << endl;
  cout << "  Del: ";printPredList(del);cout << endl;
}

void numDomain::print() {
  cout << "PredNum: " << predNum << endl;
  cout << "ObjNum: " << objNum << endl;
  cout << "MaxArity : " << maxArity << endl; 
  cout << "PredArity :\n";
  for (int i=0; i<predArity.size(); i++) 
    cout << "  pred" << i+1 << ": " << predArity[i] << endl;
  cout << "\nActions:\n\n";
  for (int i=0; i < act.size(); i++)
    {
      act[i].print();
      cout << endl;
    }
  cout << "Init: ";printPredList(init);cout << endl;
  cout << "Goal: ";printPredList(goal);cout << endl;
}




////////////////////////////////////////////////////////
// Main function
////////////////////////////////////////////////////////


// translate domain to its numerical representation
// and perform static type check
void numDomain::initialize(PDDLdomain &dom, PDDLproblem &prob) {

  
  map<string,int> objNo;
  map<string,int> predNo;

  // init misc. structures
  obj = prob.obj;
  maxArity = 0;
  predNum = dom.pred.size();
  for (int i=0; i<dom.pred.size(); i++) {
    predNo[dom.pred[i].name] = i+1;
    pred.push_back(dom.pred[i].name);
    predArity.push_back(dom.pred[i].arg.size());
    if (predArity[i] > maxArity) maxArity = predArity[i];
  }

  objNum = prob.obj.size();
  for (int i=0; i<prob.obj.size(); i++)
    objNo[prob.obj[i]] = i+1;
  
  for (int i=0; i<dom.act.size(); i++) 
    act.push_back(numAction(dom.act[i],predNo,objNo,dom.pred));
  
  for (int i=0; i<prob.init.size(); i++) 
    init.push_back(fact2numRep(prob.init[i],predNo,objNo,dom.pred));

  for (int i=0; i<prob.goal.size(); i++) 
    goal.push_back(fact2numRep(prob.goal[i],predNo,objNo,dom.pred));
}  

